#!/bin/bash

set -o errexit
set -o nounset
set -o pipefail

if [[ ${TRACE-0} == "1" ]]; then set -o xtrace; fi

trap 'echo "Error occurred at line $LINENO. Command: $BASH_COMMAND"' ERR

# Get current dir
CURRENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" >/dev/null 2>&1 && pwd)"
readonly CURRENT_DIR

# Game grids file (dart GameData object)
GAME_GRIDS_FILE="${CURRENT_DIR}/../lib/data/game_data.dart"
# Grids as raw strings
GRIDS="$(cat "${GAME_GRIDS_FILE}" | grep "',$" | sed "s/ //g" | sed "s/,//g" | sed "s/'//g")"

readonly GAME_GRIDS_FILE
readonly GRIDS

# Parameters
ALLOWED_BLOCK_SIZE_VALUES="${1:-}"
if [[ -z "${ALLOWED_BLOCK_SIZE_VALUES}" ]]; then
  ALLOWED_BLOCK_SIZE_VALUES="2x2 3x2 3x3"
fi
CHART_WIDTH="$( echo "$(tput cols) - 8" | bc)"
CHART_HEIGHT=20

readonly ALLOWED_BLOCK_SIZE_VALUES
readonly CHART_WIDTH
readonly CHART_HEIGHT

# Get block size from grid string
function get_block_size() {
  local GRID="${1:-}"
  local LOCAL_GRID_LENGTH

  LOCAL_GRID_LENGTH="$(echo -n "${GRID}" | wc -c)"
  case "${LOCAL_GRID_LENGTH}" in
    "16")
        echo "2x2"
        ;;
    "36")
        echo "3x2"
        ;;
    "81")
        echo "3x3"
        ;;
    *)
        echo "0x0"
        ;;
  esac
}

# Count "0" in grid, compute "emptiness" ratio
function get_emptiness() {
  local LOCAL_GRID="${1:-}"
  local LOCAL_CELLS_COUNT
  local LOCAL_STRING
  local LOCAL_ZEROS_COUNT
  local LOCAL_RATIO

  LOCAL_CELLS_COUNT=$(echo "($(get_block_size "${LOCAL_GRID}"))^2" | sed 's/x/\*/g' | bc)
  LOCAL_STRING="${LOCAL_GRID//[^0]}"
  LOCAL_ZEROS_COUNT="${#LOCAL_STRING}"
  LOCAL_RATIO=$(echo "100*${LOCAL_ZEROS_COUNT}/${LOCAL_CELLS_COUNT}" | bc)

  # normalize values according to chart width
  LOCAL_RATIO=$(echo "${LOCAL_RATIO}*${CHART_WIDTH}/100" | bc)

  printf "%02d" "${LOCAL_RATIO}"
}

for BLOCK_SIZE in ${ALLOWED_BLOCK_SIZE_VALUES}; do
  echo "Analyzing grids [${BLOCK_SIZE}]..."

  # Pick only grids with expected "block size", compute emptiness
  GRIDS_FOR_CURRENT_SIZE=""
  for GRID in ${GRIDS}; do
    GRID_BLOCK_SIZE="$(get_block_size "${GRID}")"
    if [[ "${GRID_BLOCK_SIZE}" == "${BLOCK_SIZE}" ]]; then
      # compute emptyness
      EMPTYNESS="$(get_emptiness "${GRID}")"
      # add grid to selected grids for this size
      GRIDS_FOR_CURRENT_SIZE="$(echo -e "${GRIDS_FOR_CURRENT_SIZE}\n${EMPTYNESS};${GRID}")"
    fi
  done

  # remove duplicates, sort, remove empty lines
  GRIDS_FOR_CURRENT_SIZE="$(echo "${GRIDS_FOR_CURRENT_SIZE}" | sed '/^$/d' | sort | uniq)"

  GRIDS_COUNT=$(echo "${GRIDS_FOR_CURRENT_SIZE}" | wc -l | awk '{print $1}')
  echo " found ${GRIDS_COUNT} grids"

  # keep emptiness values
  VALUES="$(echo "${GRIDS_FOR_CURRENT_SIZE}" | cut -d";" -f1 | awk '{print $1}')"
  # count values
  COUNTS="$(echo "${VALUES}" | sort | uniq --count)"
  # get max count
  MAX_COUNT="$(echo "${COUNTS}" | awk '{print $1}' | sort --numeric-sort --reverse | head -n 1)"

  echo ""
  for VERTICAL_VALUE in $(seq ${CHART_HEIGHT} -1 0); do
    NORMALIZED_VERTICAL_VALUE="$(echo "${VERTICAL_VALUE}*${MAX_COUNT}/${CHART_HEIGHT}" | bc)"
    NORMALIZED_VERTICAL_VALUE="$(printf "%3s" "${NORMALIZED_VERTICAL_VALUE}")"
    LINE="${NORMALIZED_VERTICAL_VALUE} "
    for HORIZONTAL_VALUE in $(seq 0 "${CHART_WIDTH}"); do
        FORMATTED_HORIZONTAL_VALUE="$(printf "%02d" "${HORIZONTAL_VALUE}")"
        COUNT="$(echo "${COUNTS}" | (grep " ${FORMATTED_HORIZONTAL_VALUE}$" || true) | awk '{print $1}')"
        NORMALIZED_COUNT_VALUE="$(echo "0${COUNT}*${CHART_HEIGHT}/${MAX_COUNT}" | bc)"
        if [[ ${NORMALIZED_COUNT_VALUE} -gt ${VERTICAL_VALUE} ]]; then
          LINE="${LINE}X"
        else
          LINE="${LINE}."
        fi
    done
    echo "${LINE}"
  done
  echo "  $(printf "%-${CHART_WIDTH}s" "0%") 100%"
  echo ""

  echo "done for ${BLOCK_SIZE}."
  echo ""
done

