#!/bin/bash -i

set -o errexit
set -o nounset
set -o pipefail

if [[ ${TRACE-0} == "1" ]]; then set -o xtrace; fi

trap 'echo "Error occurred at line $LINENO. Command: $BASH_COMMAND"' ERR

# Get current dir
CURRENT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" >/dev/null 2>&1 && pwd)"
readonly CURRENT_DIR

# Check python dependency
command -v python >/dev/null 2>&1 || {
  echo >&2 "I require python but it's not available. Aborting."
  exit 1
}

ALLOWED_BLOCK_SIZE_VALUES="${1:-}"
if [[ -z "${ALLOWED_BLOCK_SIZE_VALUES}" ]]; then
  ALLOWED_BLOCK_SIZE_VALUES="2x2 3x2 3x3"
fi
ALLOWED_DIFFICULTY_VALUES="${2:-}"
if [[ -z "${ALLOWED_DIFFICULTY_VALUES}" ]]; then
  ALLOWED_DIFFICULTY_VALUES="easy medium hard"
fi

MAX_GRIDS_PER_SIZE_AND_DIFFICULTY=100 # max attempts per size/difficulty

readonly ALLOWED_BLOCK_SIZE_VALUES
readonly ALLOWED_DIFFICULTY_VALUES
readonly MAX_GRIDS_PER_SIZE_AND_DIFFICULTY

VERBOSE=1
readonly VERBOSE

# Game grids file (dart GameData object)
GAME_GRIDS_FILE="${CURRENT_DIR}/../lib/data/game_data.dart"
readonly GAME_GRIDS_FILE

# Get block size from grid string
function get_block_size() {
  local GRID="${1:-}"
  local LOCAL_GRID_LENGTH

  LOCAL_GRID_LENGTH="$(echo -n "${GRID}" | wc -c)"
  case "${LOCAL_GRID_LENGTH}" in
    "16")
        echo "2x2"
        ;;
    "36")
        echo "3x2"
        ;;
    "81")
        echo "3x3"
        ;;
    *)
        echo "0x0"
        ;;
  esac
}

# Count "0" in grid, compute "emptiness" ratio
function get_emptiness() {
  local LOCAL_GRID="${1:-}"
  local LOCAL_CELLS_COUNT
  local LOCAL_STRING
  local LOCAL_ZEROS_COUNT
  local LOCAL_RATIO

  LOCAL_CELLS_COUNT=$(echo "($(get_block_size "${LOCAL_GRID}"))^2" | sed 's/x/\*/g' | bc)
  LOCAL_STRING="${LOCAL_GRID//[^0]}"
  LOCAL_ZEROS_COUNT="${#LOCAL_STRING}"
  LOCAL_RATIO=$(echo "100*${LOCAL_ZEROS_COUNT}/${LOCAL_CELLS_COUNT}" | bc)

  echo "$(printf "%02d" "${LOCAL_RATIO}")"
}

for BLOCK_SIZE in ${ALLOWED_BLOCK_SIZE_VALUES}; do
  for DIFFICULTY in ${ALLOWED_DIFFICULTY_VALUES}; do
    echo-blue "Block size: ${BLOCK_SIZE} / Difficulty: ${DIFFICULTY}"
    for ITERATION in $(seq ${MAX_GRIDS_PER_SIZE_AND_DIFFICULTY}); do
      ITERATION_STRING="${ITERATION}/${MAX_GRIDS_PER_SIZE_AND_DIFFICULTY} ${BLOCK_SIZE} ${DIFFICULTY}"
      if [[ ${VERBOSE} -eq 1 ]]; then
          echo " (${ITERATION_STRING})..."
      fi

      # Generate grid candidate
      GRID="$(python "${CURRENT_DIR}"/generate.py "${BLOCK_SIZE}" "${DIFFICULTY}" | tail -n 1)"

      # Ensure grid can be resolved without any assumption
      CAN_BE_SOLVED="$(python "${CURRENT_DIR}"/solve.py "${BLOCK_SIZE}" "${GRID}" | tail -n 1)"
      if [ "${CAN_BE_SOLVED}" == "Ok" ]; then
        if [[ ${VERBOSE} -eq 1 ]]; then
          EMPTYNESS="$(get_emptiness "${GRID}")"
          echo " (${ITERATION_STRING}) -> $(echo-green "OK / ${EMPTYNESS}%")"
        fi

        # Add grid to data file
        sed -i "/\/\/ put new templates here/a \ \ \ \ \ \ \ \ '${GRID}'," "${GAME_GRIDS_FILE}"
      else
        if [[ ${VERBOSE} -eq 1 ]]; then
          echo " (${ITERATION_STRING}) -> $(echo-yellow "FAILED")"
        fi
      fi
    done
  done
done
