#!/usr/bin/python
# -*- coding: iso-8859-15 -*-

#
# Naive sudoku grid solver
#
# Will try to solve sudoku grid without making any assumption
# -> pick and fill first cell where only one value is allowed
#    and loop until grid is filled
#
# Will fail if solution needs any assumption on a cell value
#

import sys

if (len(sys.argv) != 3):
    sys.stdout.write('Usage: solve.py block-size grid')
    sys.stdout.write('block-size: [2x2|3x2|3x3|4x4]')
    exit()

blocksize, gridTemplate = sys.argv[1], sys.argv[2]

if blocksize not in ['2x2', '3x2', '3x3', '4x4']:
    sys.stdout.write('wrong size given')
    exit()

splitted_blocksize = blocksize.split('x')
size_horizontal = int(splitted_blocksize[0])
size_vertical = int(splitted_blocksize[1])

boardSize = size_horizontal * size_vertical

if (len(gridTemplate) != boardSize * boardSize):
    sys.stdout.write('wrong grid length (should be ' +
                     str(boardSize * boardSize) + ')')
    exit()

debugSolveGrid = False

############################################################################

sys.stdout.write('Will solve grid: [' + str(size_horizontal) +
                 'x' + str(size_vertical) + '] // ' + gridTemplate + '\n')

stringValues = '0123456789ABCDEFG'


# draw grid (array style)
def draw_grid(grid):
    grid_vertical_size = len(grid)
    grid_horizontal_size = len(grid[0])
    horizontal_line_length = ((size_horizontal + 1) * size_vertical) + 1

    for row in range(grid_horizontal_size):
        if ((row % size_vertical) == 0):
            sys.stdout.write(('═' * horizontal_line_length) + '\n')
        for col in range(grid_vertical_size):
            if ((col % size_horizontal) == 0):
                sys.stdout.write('║')
            if grid[row][col] != 0:
                sys.stdout.write(stringValues[grid[row][col]])
            else:
                sys.stdout.write(' ')
        sys.stdout.write('║\n')
    sys.stdout.write(('═' * horizontal_line_length) + '\n')
    sys.stdout.write('\n')


# (deep) copy of grid
def copy_grid(grid):
    copied_grid = []
    for row in range(len(grid)):
        copied_grid.append([])
        for col in range(len(grid[row])):
            copied_grid[row].append(grid[row][col])
    return copied_grid


# Init grid from given template
def init_grid(board_size, grid_template):
    grid = []
    index = 0
    for row in range(board_size):
        grid.append([])
        for _ in range(board_size):
            grid[row].append(stringValues.index(grid_template[index]))
            index += 1
    return grid


# Check if grid is fully completed, without any empty cell
def is_fully_completed(grid):
    for row in range(len(grid)):
        for col in range(len(grid[row])):
            if grid[row][col] == 0:
                return False
    return True


# Check if a list contains duplicates (conflicts)
def contains_duplicates(list):
    tmp_set = set(list)
    return (len(list) != len(tmp_set))


# Check if given grid contains conflicts
def has_conflict(grid, size_horizontal, size_vertical):
    # Check horizontal conflicts
    for row in range(len(grid)):
        values = []
        for col in range(len(grid[row])):
            value = grid[row][col]
            if value != 0:
                values.append(value)
        if contains_duplicates(values):
            # sys.stdout.write('Horizontal conflict found')
            return True

    # Check vertical conflicts
    for col in range(len(grid[0])):
        values = []
        for row in range(len(grid)):
            value = grid[row][col]
            if value != 0:
                values.append(value)
        if contains_duplicates(values):
            # sys.stdout.write('Vertical conflict found')
            return True

    # Check sub-blocks conflicts
    for block_row in range(size_horizontal):
        for block_col in range(size_vertical):
            # Get sub-square
            values = []
            for row_in_block in range(size_vertical):
                for col_in_block in range(size_horizontal):
                    row = (block_row * size_vertical) + row_in_block
                    col = (block_col * size_horizontal) + col_in_block
                    value = grid[row][col]
                    if value != 0:
                        values.append(value)
            if contains_duplicates(values):
                # sys.stdout.write('Sub-block conflict found')
                return True

    return False


# Check if a value is allowed in a cell (without conflicting)
def is_value_allowed(grid, size_horizontal, size_vertical, row, col, candidate_value):
    test_grid = copy_grid(grid)
    test_grid[row][col] = candidate_value
    if not has_conflict(test_grid, size_horizontal, size_vertical):
        return True
    return False


# Get allowed values in a cell (witjout conflicting)
def find_allowed_values_for_cell(grid, size_horizontal, size_vertical, row, col):
    allowed_values = []
    if not has_conflict(grid, size_horizontal, size_vertical):
        for candidate_value in range(1, size_horizontal * size_vertical + 1):
            if is_value_allowed(grid, size_horizontal, size_vertical, row, col, candidate_value):
                allowed_values.append(candidate_value)
    return allowed_values


# Globally solve grid
def solve(grid, size_horizontal, size_vertical):
    iterations = 0
    max_iterations = 500

    # Loop until grid is fully completed
    while True:
        iterations += 1
        if is_fully_completed(grid) or (iterations > max_iterations):
            break

        if debugSolveGrid:
            sys.stdout.write('===================================')
            sys.stdout.write('Iteration: ' + str(iterations))

        # Get first/next cell with only one allowed value
        candidates = []
        if debugSolveGrid:
            sys.stdout.write('Searching for empty cells...')
        for row in range(len(grid)):
            for col in range(len(grid[row])):
                if grid[row][col] == 0:
                    if debugSolveGrid:
                        sys.stdout.write(
                            'Found empty cell [' + str(col) + ',' + str(row) + ']')
                    candidates.append([row, col])

        if len(candidates):
            for candidate in candidates:
                candidate_row = candidate[0]
                candidate_col = candidate[1]
                allowed_values = find_allowed_values_for_cell(
                    grid, size_horizontal, size_vertical, candidate_row, candidate_col)
                if debugSolveGrid:
                    sys.stdout.write('Allowed values for cell [' + str(candidate_col) + ',' + str(
                        candidate_row) + ']: ' + str(allowed_values))
                if len(allowed_values) != 1:
                    if debugSolveGrid:
                        sys.stdout.write(
                            ' Non unique allowed value for cell. Skip to next cell')
                else:
                    value = allowed_values[0]
                    grid[candidate_row][candidate_col] = value
                    if debugSolveGrid:
                        sys.stdout.write(' Found unique allowed value for cell [' + str(
                            candidate_col) + ',' + str(candidate_row) + ']: ' + str(value))
                        draw_grid(grid)

#########################


sys.stdout.write('Building start grid:\n')
grid = init_grid(boardSize, gridTemplate)
draw_grid(grid)

sys.stdout.write('Checking grid:\n')
if has_conflict(grid, size_horizontal, size_vertical):
    sys.stdout.write(' - oups, initial conflict found.\n')
else:
    sys.stdout.write(' - ok, no initial conflict found.\n')
    sys.stdout.write('\n')

    sys.stdout.write('Solving grid...\n')
    solve(grid, size_horizontal, size_vertical)

    if is_fully_completed(grid):
        sys.stdout.write('Ok, solved grid:\n')
        draw_grid(grid)
        sys.stdout.write('Ok\n')
    else:
        sys.stdout.write('Failed to solve grid\n')
