#!/usr/bin/python
# -*- coding: iso-8859-15 -*-

import sys
from random import randint, shuffle

if (len(sys.argv) != 3):
    sys.stdout.write('Usage: generate.py block-size difficulty')
    sys.stdout.write('block-size: [2x2|3x2|3x3|4x4]')
    sys.stdout.write('difficulty: [easy|medium|hard|nightmare]')
    exit()

blocksize, difficulty = sys.argv[1], sys.argv[2]

if blocksize not in ['2x2', '3x2', '3x3', '4x4']:
    sys.stdout.write('wrong size given')
    exit()

splitted_blocksize = blocksize.split('x')
size_horizontal = int(splitted_blocksize[0])
size_vertical = int(splitted_blocksize[1])

boardSize = size_horizontal * size_vertical

if difficulty not in ['easy', 'medium', 'hard', 'nightmare']:
    sys.stdout.write('wrong difficulty given')
    exit()

debugFillGrid = False
debugSolveGrid = False
debugComputeGameGrid = False

############################################################################

sys.stdout.write('Will generate grid: ')
sys.stdout.write(' [' + str(size_horizontal) + 'x' + str(size_vertical) + ']')
sys.stdout.write(' (' + difficulty + ')\n')

ratios = {
    '2x2' : {
        'easy': 30,
        'medium': 50,
        'hard': 70,
        'nightmare': 90,
    },
    '3x2' : {
        'easy': 30,
        'medium': 50,
        'hard': 70,
        'nightmare': 90,
    },
    '3x3' : {
        'easy': 30,
        'medium': 50,
        'hard': 70,
        'nightmare': 90,
    },
    '4x4' : {
        'easy': 30,
        'medium': 50,
        'hard': 70,
        'nightmare': 90,
    },
}
expected_empty_cells_ratio = ratios.get(blocksize).get(difficulty)

# low max attemps count -> will fail quickly if no solution is found
attemps = {
    '2x2' : 12,
    '3x2' : 10,
    '3x3' : 8,
    '4x4' : 1,
}
max_attemps = attemps.get(blocksize)

# and some randomization to empty cells ratio
expected_empty_cells_ratio = expected_empty_cells_ratio + randint(-9, 9)

sys.stdout.write('  -> expected empty cells ratio: ' +
                 str(expected_empty_cells_ratio) + '\n')
sys.stdout.write('  (max attemps: ' + str(max_attemps) + ')\n')
sys.stdout.write('\n')

stringValues = '0123456789ABCDEFG'
solutions_count = 1


# draw grid (array style)
def draw_grid(grid):
    grid_vertical_size = len(grid)
    grid_horizontal_size = len(grid[0])
    horizontal_line_length = ((size_horizontal + 1) * size_vertical) + 1

    for row in range(grid_horizontal_size):
        if ((row % size_vertical) == 0):
            sys.stdout.write(('-' * horizontal_line_length) + '\n')
        for col in range(grid_vertical_size):
            if ((col % size_horizontal) == 0):
                sys.stdout.write('|')
            if grid[row][col] != 0:
                sys.stdout.write(stringValues[grid[row][col]])
            else:
                sys.stdout.write(' ')
        sys.stdout.write('|\n')
    sys.stdout.write(('-' * horizontal_line_length) + '\n')
    sys.stdout.write('Empty cells ratio: ' +
                     str(compute_empty_cells_ratio(grid)) + '% (expected ' + str(expected_empty_cells_ratio) + '%)\n')
    sys.stdout.write('\n')


# draw grid (inline style)
def draw_grid_inline(grid):
    for row in range(len(grid)):
        for col in range(len(grid[row])):
            sys.stdout.write(stringValues[grid[row][col]])
    sys.stdout.write('\n')


# initialise empty grid
def generate_empty_grid(board_size):
    empty_grid = []
    for row in range(board_size):
        empty_grid.append([])
        for _ in range(board_size):
            empty_grid[row].append(0)
    return empty_grid


# A check if the grid is full
def check_fully_completed_grid(grid):
    for row in range(len(grid)):
        for col in range(len(grid[row])):
            if grid[row][col] == 0:
                return False
    return True


# (deep) copy of grid
def copy_grid(grid):
    copied_grid = []
    for row in range(len(grid)):
        copied_grid.append([])
        for col in range(len(grid[row])):
            copied_grid[row].append(grid[row][col])
    return copied_grid


# A backtracking/recursive function to check all
# possible combinations of numbers until a solution is found
def solve_grid(grid, iteration_solve_count):
    if debugSolveGrid:
        sys.stdout.write('solveGrid / ' + str(iteration_solve_count) + '\n')
    grid_size = len(grid)
    cells_count = len(grid) * len(grid[0])
    number_list = [(value + 1) for value in range(grid_size)]

    global solutions_count

    # Find next empty cell
    for i in range(0, cells_count):
        row = i // grid_size
        col = i % grid_size
        if grid[row][col] == 0:
            shuffle(number_list)
            for value in number_list:
                if debugSolveGrid:
                    sys.stdout.write(
                        'solveGrid: '
                        + '[' + str(row) + ',' + str(col) + ']'
                        + ' try with value ' + str(value)
                        + '\n'
                    )
                # Check that this value has not already be used on this row
                if (value not in grid[row]):
                    # Check that this value has not already be used on this column
                    found_in_column = False
                    for r in range(0, grid_size):
                        if (value == grid[r][col]):
                            found_in_column = True

                    if not found_in_column:
                        # Get sub-square
                        block_col_from = size_horizontal * \
                            int(col / size_horizontal)
                        block_row_from = size_vertical * \
                            int(row / size_vertical)
                        square = [grid[i][block_col_from:block_col_from + size_horizontal]
                                  for i in range(block_row_from, block_row_from + size_vertical)]

                        # Check that this value has not already be used on this sub square
                        if not any(value in squareLine for squareLine in square):
                            grid[row][col] = value
                            if check_fully_completed_grid(grid):
                                if debugSolveGrid:
                                    sys.stdout.write(
                                        'solveGrid: grid complete, found solution\n')
                                iteration_solve_count += 1
                                solutions_count += 1
                                break
                            else:
                                if debugSolveGrid:
                                    sys.stdout.write('solveGrid: recursive call (solutionsCount=' + str(
                                        solutions_count) + ', iteration_solve_count=' + str(iteration_solve_count) + ')\n')
                                if solve_grid(grid, iteration_solve_count + 1):
                                    if debugSolveGrid:
                                        sys.stdout.write(
                                            'solveGrid: still searching for solution\n')
                                    return True
            break
    grid[row][col] = 0


# A backtracking/recursive function to check all possible combinations of numbers until a solution is found
def fill_grid(grid, board_size, iteration_fill_count):
    if debugFillGrid:
        sys.stdout.write('fillGrid / ' + str(iteration_fill_count) + '\n')
        draw_grid(grid)

    board_size = len(grid)
    cells_count = len(grid) * len(grid[0])
    number_list = [(value + 1) for value in range(board_size)]

    global solutions_count

    # Find next empty cell
    for i in range(0, cells_count):
        row = i // board_size
        col = i % board_size

        # Ensure cell is not already set
        if grid[row][col] == 0:
            # Try to fill cell with random numbers, iteratively
            shuffle(number_list)
            for value in number_list:
                if debugFillGrid:
                    sys.stdout.write(
                        'fillGrid: [' + str(row) + ',' + str(col) + '] -> try with value ' + str(value) + '\n')
                # Check that this value has not already be used on this row
                if (value not in grid[row]):
                    # Check that this value has not already be used on this column
                    found_in_column = False
                    for r in range(0, board_size):
                        if (value == grid[r][col]):
                            found_in_column = True

                    if not found_in_column:
                        # Get sub-square
                        block_col_from = size_horizontal * \
                            int(col / size_horizontal)
                        block_row_from = size_vertical * \
                            int(row / size_vertical)
                        square = [grid[i][block_col_from:block_col_from + size_horizontal]
                                  for i in range(block_row_from, block_row_from + size_vertical)]

                        # Check that this value has not already be used on this sub square
                        if not any(value in squareLine for squareLine in square):
                            if debugFillGrid:
                                sys.stdout.write(
                                    'fillGrid: [' + str(row) + ',' + str(col) + '] <- ' + str(value) + ' / ok, no conflict\n')
                            grid[row][col] = value
                            if check_fully_completed_grid(grid):
                                if debugFillGrid:
                                    sys.stdout.write(
                                        'fillGrid: found final solution\n')
                                return True
                            else:
                                if debugFillGrid:
                                    sys.stdout.write(
                                        'fillGrid: recursive call (iteration_fill_count=' + str(iteration_fill_count) + ')\n')
                                iteration_fill_count += 1
                                if fill_grid(grid, board_size, iteration_fill_count):
                                    return True
            break
    if debugFillGrid:
        sys.stdout.write(
            'fillGrid: no solution found [' + str(row) + ',' + str(col) + '] <- 0\n')
    grid[row][col] = 0


def compute_empty_cells_ratio(grid):
    empty_cells_count = 0

    grid_size = len(grid)
    cells_count = len(grid) * len(grid[0])

    for i in range(0, cells_count):
        row = i // grid_size
        col = i % grid_size
        if grid[row][col] == 0:
            empty_cells_count += 1

    return round(100 * empty_cells_count/cells_count)


def compute_resolvable_grid(grid, max_attemps, empty_cells_ratio):
    global solutions_count

    # A higher number of attemps will end up removing more numbers from the grid
    # Potentially resulting in more difficiult grids to solve!

    # Start Removing Numbers one by one
    remaining_attemps = max_attemps
    while (remaining_attemps > 0) and (compute_empty_cells_ratio(grid) < empty_cells_ratio):
        if debugComputeGameGrid:
            sys.stdout.write('Remaining attemps: ' +
                             str(remaining_attemps) + '.\n')

        # Select a random cell that is not already empty
        row = randint(0, boardSize - 1)
        col = randint(0, boardSize - 1)
        while grid[row][col] == 0:
            row = randint(0, boardSize - 1)
            col = randint(0, boardSize - 1)

        # Remove value in this random cell
        saved_cell_value = grid[row][col]
        grid[row][col] = 0

        solutions_count = 0
        if debugComputeGameGrid:
            sys.stdout.write('Remove value in [' + str(
                row) + ',' + str(col) + '] (was ' + str(saved_cell_value) + ').\n')
            draw_grid(grid)
            sys.stdout.write('Check grid unique solution...\n')

        solve_grid(copy_grid(grid), 0)

        # Non unique solution => restore this cell value
        if solutions_count != 1:
            if debugComputeGameGrid:
                sys.stdout.write(
                    'Failed to solve grid (multiple solutions). Will try with clearing another cell.\n')
            grid[row][col] = saved_cell_value
            remaining_attemps -= 1
        else:
            if debugComputeGameGrid:
                sys.stdout.write('Ok found unique solution.\n')

    if debugComputeGameGrid:
        sys.stdout.write('Ok found solvable grid.\n')
        sys.stdout.write(' Expected empty cells ratio: ' +
                         str(empty_cells_ratio) + '\n')
        sys.stdout.write(' Real empty cells ratio: ' +
                         str(compute_empty_cells_ratio(grid)) + '\n')
        sys.stdout.write('\n')

###########################################################################


grid = generate_empty_grid(boardSize)

sys.stdout.write('Building grid...\n')
fill_grid(grid, boardSize, 0)

sys.stdout.write('Solved grid:\n')
draw_grid(grid)

sys.stdout.write('Generating solvable grid:\n')
compute_resolvable_grid(grid, max_attemps, expected_empty_cells_ratio)

sys.stdout.write('Generated grid:\n')
draw_grid(grid)

sys.stdout.write(
    'Inline grid [' + str(size_horizontal) + 'x' + str(size_vertical) + ']'
    + ', '
    + 'difficulty: ' + difficulty
    + ':'
    + '\n'
)
draw_grid_inline(grid)
