import 'package:flutter/material.dart';

import 'package:flutter_custom_toolbox/flutter_toolbox.dart';

class PageParameters extends StatelessWidget {
  const PageParameters({
    super.key,
    required this.appConfig,
    required this.canBeResumed,
  });

  final ApplicationConfigDefinition appConfig;
  final bool canBeResumed;

  final double separatorHeight = 4.0;

  @override
  Widget build(BuildContext context) {
    // Activity settings (top)
    final List<Widget> blockTop = [];
    for (ApplicationSettingsParameter parameter in appConfig.activitySettings) {
      if (parameter.displayedOnTop) {
        for (List<Widget> itemsLine in buildParametersLines(
          parameter: parameter,
          isEnabled: !canBeResumed,
        )) {
          blockTop.add(Row(
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            children: itemsLine,
          ));
        }
        blockTop.add(SizedBox(height: separatorHeight));
      }
    }

    // Activity settings (bottom)
    final List<Widget> blockBottom = [];
    for (ApplicationSettingsParameter parameter in appConfig.activitySettings) {
      if (!parameter.displayedOnTop) {
        for (List<Widget> itemsLine in buildParametersLines(
          parameter: parameter,
          isEnabled: !canBeResumed,
        )) {
          blockBottom.add(Row(
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            children: itemsLine,
          ));
        }
        blockBottom.add(SizedBox(height: separatorHeight));
      }
    }

    // Actions buttons (middle block)
    final double buttonBaseSize = MediaQuery.of(context).size.width / 4;
    final List<Widget> blockActions = [];
    if (canBeResumed) {
      blockActions.add(Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        crossAxisAlignment: CrossAxisAlignment.center,
        children: [
          // Delete saved activity
          SizedBox.fromSize(
            size: Size(buttonBaseSize, buttonBaseSize),
            child: ActivityButtonDeleteSaved(
              onPressed: () {
                appConfig.deleteCurrentActivity(context);
              },
            ),
          ),
          // Resume activity
          SizedBox.fromSize(
            size: Size(2.9 * buttonBaseSize, buttonBaseSize),
            child: ActivityButtonResumeSaved(
              onPressed: () {
                appConfig.resumeActivity(context);
              },
            ),
          ),
        ],
      ));
    } else {
      // Start new activity
      blockActions.add(
        SizedBox.fromSize(
          size: Size(4 * buttonBaseSize, buttonBaseSize),
          child: ActivityButtonStartNew(
            onPressed: () {
              appConfig.startNewActivity(context);
            },
          ),
        ),
      );
    }
    blockActions.add(SizedBox(height: separatorHeight));

    // Global blocks list
    final List<Widget> blocks = [
      Expanded(
        child: appConfig.allowScrollParametersList
            ? ListView(
                children: blockTop,
              )
            : Column(
                children: blockTop,
              ),
      ),
      SizedBox(height: separatorHeight),
    ];

    blocks.addAll(blockActions);

    if (blockBottom.isNotEmpty) {
      blocks.addAll(blockBottom);
    } else {
      // Add bottom margin
      blocks.add(SizedBox(height: separatorHeight));
      blocks.add(SizedBox(height: separatorHeight));
    }

    return Column(children: blocks);
  }

  List<List<Widget>> buildParametersLines({
    required ApplicationSettingsParameter parameter,
    required bool isEnabled,
  }) {
    final List<ApplicationSettingsParameterItemValue> items = parameter.values;
    final List<Widget> parameterButtons = [];

    if (items.length <= 1) {
      return [];
    }

    final int itemsPerLine =
        parameter.itemsPerLine != 0 ? parameter.itemsPerLine : items.length;

    // Build linear list
    for (ApplicationSettingsParameterItemValue item in items) {
      final Widget parameterButton = BlocBuilder<ActivitySettingsCubit, ActivitySettingsState>(
        builder: (BuildContext context, ActivitySettingsState activitySettingsState) {
          final ActivitySettingsCubit activitySettingsCubit =
              BlocProvider.of<ActivitySettingsCubit>(context);

          final double displayWidth = MediaQuery.of(context).size.width;
          final double itemWidth = displayWidth / itemsPerLine - 4;

          return SizedBox.square(
            dimension: itemWidth,
            child: parameter.buildParameterItem(
              context: context,
              parameter: parameter,
              itemValue: item,
              size: itemWidth,
              onPressed: isEnabled
                  ? () {
                      if (parameter.allowMultipleValues) {
                        activitySettingsCubit.toggle(parameter.code, item.value);
                      } else {
                        activitySettingsCubit.set(parameter.code, item.value);
                      }
                    }
                  : null,
            ),
          );
        },
      );

      parameterButtons.add(parameterButton);
    }

    // Split buttons in lines according to items count limit per line
    final List<List<Widget>> lines = [];

    List<Widget> line = [];
    for (var parameterButton in parameterButtons) {
      line.add(parameterButton);
      if (line.length == itemsPerLine) {
        lines.add(line);
        line = [];
      }
    }
    if (line.length != itemsPerLine) {
      lines.add(line);
    }

    return lines;
  }
}
