import 'package:flutter/material.dart';

import 'package:flutter_custom_toolbox/flutter_toolbox.dart';

/// A screen item,
///
/// Normalized screens are: "activity", "settings" and "about".
///
/// These are set in [ApplicationNavigation] with
/// - [ApplicationNavigation.screenActivity]
/// - [ApplicationNavigation.screenSettings]
/// - [ApplicationNavigation.screenAbout]
///
class ScreenItem {
  final String code;
  final Icon icon;
  final Widget Function({required ApplicationConfigDefinition appConfig}) screen;

  const ScreenItem({
    required this.code,
    required this.icon,
    required this.screen,
  });
}

/// A page in [ScreenItem] given in [ApplicationNavigation.screenActivity]
///
/// These are set in [ApplicationNavigation] with
/// [ApplicationNavigation.activityPages].
///
class ActivityPageItem {
  final String code;
  final Icon? icon;
  final Widget Function({required ApplicationConfigDefinition appConfig}) builder;

  const ActivityPageItem({
    required this.code,
    this.icon,
    required this.builder,
  });

  factory ActivityPageItem.empty() {
    return ActivityPageItem(
      code: '',
      builder: ({required ApplicationConfigDefinition appConfig}) => Text(''),
    );
  }
}

/// Custom AppBar configuration
///
class AppBarConfiguration {
  final bool? hideApplicationTitle;
  final bool? pushQuitActivityButtonLeft;
  final bool? hideQuitActivityButton;
  final List<AppBarButton> Function(BuildContext context)? topBarButtonsBuilder;

  const AppBarConfiguration({
    this.hideApplicationTitle = false,
    this.pushQuitActivityButtonLeft = false,
    this.hideQuitActivityButton = false,
    this.topBarButtonsBuilder,
  });
}

/// Custom AppBar button (will generate a widget [IconButton])
///
class AppBarButton {
  final Icon icon;
  final Function(BuildContext context)? onPressed;
  final Function(BuildContext context)? onLongPress;

  const AppBarButton({
    required this.icon,
    this.onPressed,
    this.onLongPress,
  });

  Widget render(BuildContext context) {
    return TextButton(
      onPressed: () => onPressed!(context),
      onLongPress: () => onLongPress!(context),
      style: TextButton.styleFrom(
        shape: const CircleBorder(),
        iconColor: Theme.of(context).colorScheme.onSurface,
      ),
      child: icon,
    );
  }
}

/// Navigation configuration for application
///
class ApplicationNavigation {
  final ScreenItem screenActivity;
  final ScreenItem screenSettings;
  final ScreenItem screenAbout;
  final AppBarConfiguration? appBarConfiguration;
  final Map<int, ActivityPageItem> activityPages;
  final bool displayBottomNavBar;

  const ApplicationNavigation({
    required this.screenActivity,
    required this.screenSettings,
    required this.screenAbout,
    this.appBarConfiguration,
    required this.activityPages,
    this.displayBottomNavBar = false,
  });

  static const indexActivity = 0;
  static const indexSettings = 1;
  static const indexAbout = 2;

  ScreenItem getScreen(int screenIndex) {
    switch (screenIndex) {
      case indexActivity:
        return screenActivity;
      case indexSettings:
        return screenSettings;
      case indexAbout:
        return screenAbout;
      default:
        return screenActivity;
    }
  }

  ScreenItem getActivityScreen() {
    return screenActivity;
  }

  Widget getScreenWidget({
    required ApplicationConfigDefinition appConfig,
    required int screenIndex,
  }) {
    return getScreen(screenIndex).screen(appConfig: appConfig);
  }

  bool isActivityPageIndexAllowed(int pageIndex) {
    return activityPages.keys.contains(pageIndex);
  }
}
