/*
 * Jitsi, the OpenSource Java VoIP and Instant Messaging client.
 *
 * Distributable under LGPL license. See terms of license at gnu.org.
 */
package org.jivesoftware.smackx.colibri;

import org.atalk.service.neomedia.MediaDirection;
import org.atalk.util.MediaType;
import org.jivesoftware.smack.packet.IQ;
import org.jivesoftware.smack.util.StringUtils;
import org.jivesoftware.smackx.jingle_rtp.element.IceUdpTransport;
import org.jivesoftware.smackx.jingle_rtp.JingleUtils;
import org.jivesoftware.smackx.jingle_rtp.element.PayloadType;
import org.jivesoftware.smackx.jingle_rtp.element.RtpDescription;
import org.jivesoftware.smackx.jingle_rtp.element.RtpHeader;
import org.jivesoftware.smackx.jingle_rtp.element.SdpSource;
import org.jivesoftware.smackx.jingle_rtp.element.SdpSourceGroup;
import org.jivesoftware.smackx.jingle.element.JingleContent;
import org.jxmpp.jid.Jid;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import timber.log.Timber;

/**
 * Utility class for building Colibri queries. It can be used to allocate,
 * expire Colibri channels and update its ICE transport info on the bridge.
 * The flow is as follows:<br/>
 * <ol>
 * <li>
 * Add one or multiple requests of the same type by calling
 * {@link #addAllocateChannelsReq(boolean, String, String, boolean, java.util.List)}}
 * or {@link #addExpireChannelsReq(ColibriConferenceIQ)}
 * or {@link #addRtpDescription(RtpDescription, String, ColibriConferenceIQ.Channel)}
 * and {@link #addSourceGroupsInfo(Map, ColibriConferenceIQ)}
 * and {@link #addSourceInfo(Map, ColibriConferenceIQ)}.
 * </li>
 * <li>
 * Compile the request by calling {@link #getRequest(Jid)}. Then send it to the bridge.
 * </li>
 * <li>
 * Use {@link ColibriAnalyser} to extract particular channels info or create response to the client.
 * </li>
 * <li>
 * Call {@link #reset()} and start next query.
 * </li>
 * </ol>
 * The important thing is to share the same {@link ColibriConferenceIQ} instance
 * between the {@link ColibriAnalyser} and {@link ColibriBuilder}. The builder
 * needs to have conference ID set once the first response from JVB is received.
 * Otherwise it will be allocating new conferences for each allocate request and
 * it will fail on other request types because it will not supply valid conference ID.
 *
 * @author Pawel Domas
 * @author Boris Grozev
 * @author Eng Chong Meng
 */
public class ColibriBuilder
{
    /**
     * Copies the transport info from a {@link JingleContent} to a
     * {@link ColibriConferenceIQ.ChannelCommon}.
     *
     * @param content the {@link JingleContent} from which to copy.
     * @param channel the {@link ColibriConferenceIQ.ChannelCommon} to which to copy.
     */
    private static void copyTransport(JingleContent content, ColibriConferenceIQ.ChannelCommon channel)
    {
        IceUdpTransport transport = content.getFirstChildElement(IceUdpTransport.class);
        channel.setTransport(IceUdpTransport.cloneTransportAndCandidates(transport, true));
    }

    /**
     * Copies the contents of the description element from a
     * {@link JingleContent} to a {@link ColibriConferenceIQ.Channel}.
     *
     * @param content the {@link JingleContent} from which to copy.
     * @param channel the {@link ColibriConferenceIQ.Channel} to which to copy.
     */
    private static boolean copyDescription(JingleContent content, ColibriConferenceIQ.Channel channel)
    {
        RtpDescription description = content.getFirstChildElement(RtpDescription.class);
        if (description != null) {
            return copyDescription(description, channel);
        }
        return false;
    }

    /**
     * Copies the contents of a {@link RtpDescription} to a {@link ColibriConferenceIQ.Channel}.
     *
     * @param description the {@link RtpDescription} from which to copy.
     * @param channel the {@link ColibriConferenceIQ.Channel} to which to copy.
     */
    private static boolean copyDescription(RtpDescription description, ColibriConferenceIQ.Channel channel)
    {
        boolean added = false;
        for (PayloadType payloadType : description.getChildElements(PayloadType.class)) {
            channel.addPayloadType(PayloadType.clone(payloadType));
            added = true;
        }
        for (RtpHeader rtpHdrExt : description.getChildElements(RtpHeader.class)) {
            channel.addRtpHeaderExtension(RtpHeader.clone(rtpHdrExt));
            added = true;
        }
        return added;
    }

    /**
     * Sets the {@link SdpSource}s of a particular {@link ColibriConferenceIQ.Channel}.
     *
     * @param channel the channel to which to add sources.
     * @param sources the list of sources to add.
     */
    private static void addSources(ColibriConferenceIQ.Channel channel, List<SdpSource> sources)
    {
        for (SdpSource source : sources) {
            channel.addSource(SdpSource.clone(source));
        }
        if ((channel.getSources() == null) || channel.getSources().isEmpty()) {
            // Put an empty source to remove all sources
            channel.addSource(SdpSource.getBuilder()
                    .setSsrc(-1L)
                    .build()
            );
        }
    }

    /**
     * Adds a list of {@link SdpSourceGroup}s to a particular {@link ColibriConferenceIQ.Channel}.
     *
     * @param channel the channel to which to add source groups.
     * @param sourceGroups the source groups to add.
     * @param video whether the channel's media type is video or not.
     * @return {@code true} iff any source groups we added to the channel.
     */
    private static boolean addSourceGroups(ColibriConferenceIQ.Channel channel,
            List<SdpSourceGroup> sourceGroups, boolean video)
    {
        boolean hasAnyChanges = false;
        if (sourceGroups.isEmpty() && video) {
            hasAnyChanges = true;

            // Put an empty source group to turn off simulcast layers
            channel.addSourceGroup(SdpSourceGroup.getBuilder()
                    .setSemantics(SdpSourceGroup.SEMANTICS_SIMULCAST)
                    .build()
            );
        }

        for (SdpSourceGroup sourceGroup : sourceGroups) {
            hasAnyChanges = true;
            channel.addSourceGroup(sourceGroup);
        }
        return hasAnyChanges;
    }

    /**
     * Colibri IQ that holds conference state.
     */
    private final ColibriConferenceIQ conferenceState;

    /**
     * The type of the request currently being build.
     * {@link RequestType#UNDEFINED} if this instance is in the "zero" state.
     * To get into the "zero" state call {@link #reset()}.
     */
    private RequestType requestType = RequestType.UNDEFINED;

    /**
     * Object holding the request currently being constructed.
     */
    private ColibriConferenceIQ request;

    /**
     * Field used to hold info if there are any channels to be expired
     */
    private boolean hasAnyChannelsToExpire = false;

    /**
     * Channel 'last-n' option that will be added when channels are created. Set to <code>null</code> in order to omit.
     */
    private Integer channelLastN;

    /**
     * Channel 'simulcast-mode' option that will be added when channels are created.
     * Set to <code>null</code> in order to omit.
     */
    private SimulcastMode simulcastMode;

    /**
     * Specifies the audio packet delay that will be set on all created audio channels.
     * When set to <code>null</code> the builder will clear the attribute which stands for 'undefined'.
     **/
    private Integer audioPacketDelay;

    /**
     * Channel 'rtp-level-relay-type' option that will be used with all created
     * audio channel. Possible values: mixer or translator (default).
     */
    private RTPLevelRelayType rtpLevelRelayType;

    /**
     * Creates new instance of {@link ColibriBuilder} for given
     *
     * @param conferenceState the conference state which will be taken into
     * account when constructing specific requests(the need to
     * allocate new conference or re-use the one currently in progress).
     */
    public ColibriBuilder(ColibriConferenceIQ conferenceState)
    {
        this.conferenceState = Objects.requireNonNull(conferenceState, "conferenceState");
        reset();
    }

    /**
     * Resets this builder state to the "zero" state when no request info is stored.
     */
    public void reset()
    {
        requestType = RequestType.UNDEFINED;

        // Copy conference ID if any otherwise null or empty will be placed
        // which means that new conference will be allocated
        request = new ColibriConferenceIQ();
        request.setID(conferenceState.getID());
        request.setName(conferenceState.getName());
        request.setGID(conferenceState.getGID());
        request.setType(IQ.Type.set);
    }

    /**
     * Adds next channel allocation request to {@link RequestType#ALLOCATE_CHANNELS} query currently being built.
     *
     * @param useBundle <code>true</code> if allocated channels should all use the same bundle.
     * @param endpointId name of the endpoint for which Colibri channels will
     * @param peerIsInitiator the value that will be set in 'initiator'
     * attribute ({@link ColibriConferenceIQ.Channel#initiator}).
     * @param contents the list of {@link JingleContent} describing channels media.
     * @return {@code true} if the request yields any changes in Colibri
     * channels state on the bridge or {@code false} otherwise. In general when
     * {@code false} is returned for all combined requests it makes no sense to send it.
     */
    public boolean addAllocateChannelsReq(boolean useBundle, String endpointId, String statsId,
            boolean peerIsInitiator, List<JingleContent> contents)
    {
        return addAllocateChannelsReq(useBundle, endpointId, statsId, peerIsInitiator, contents,
                null, null, null);
    }

    /**
     * Adds next channel allocation request to {@link RequestType#ALLOCATE_CHANNELS} query currently being built.
     *
     * @param useBundle <code>true</code> if allocated channels should all use the same bundle.
     * @param endpointId name of the endpoint for which Colibri channels will
     * @param statsId the stats ID of the endpoint for which channels are to be allocated.
     * @param peerIsInitiator the value that will be set in 'initiator'
     * attribute ({@link ColibriConferenceIQ.Channel#initiator}).
     * @param contents the list of {@link JingleContent} describing channels media.
     * @param sourceMap a map of content name to the list of
     * {@link SdpSource}s which are to be added to the channel allocation request for this content.
     * @param sourceGroupMap a map of content name to the list of
     * {@link SdpSourceGroup}s which are to be added to the channel allocation request for this content.
     * @param octoRelayIds the optional list of Octo relay IDs to be added to
     * the channel. If this parameter is non-null, the allocated channels will be of type Octo.
     * @return {@code true} if the request yields any changes in Colibri
     * channels state on the bridge or {@code false} otherwise. In general when
     * {@code false} is returned for all combined requests it makes no sense to send it.
     */
    public boolean addAllocateChannelsReq(boolean useBundle, String endpointId, String statsId, boolean peerIsInitiator,
            List<JingleContent> contents, Map<String, List<SdpSource>> sourceMap,
            Map<String, List<SdpSourceGroup>> sourceGroupMap, List<String> octoRelayIds)
    {
        Objects.requireNonNull(contents, "contents");
        assertRequestType(RequestType.ALLOCATE_CHANNELS);
        boolean hasAnyChanges = false;

        for (JingleContent content : contents) {
            MediaType mediaType = JingleUtils.getMediaType(content);
            String contentName = mediaType.toString();
            ColibriConferenceIQ.Content requestContent = request.getOrCreateContent(contentName);

            ColibriConferenceIQ.ChannelCommon requestChannel;
            if (mediaType == MediaType.DATA) {
                requestChannel = new ColibriConferenceIQ.SctpConnection();
            }
            else if (octoRelayIds != null) {
                requestChannel = new ColibriConferenceIQ.OctoChannel();
            }
            else {
                requestChannel = new ColibriConferenceIQ.Channel();
            }

            if (endpointId != null) {
                requestChannel.setEndpoint(endpointId);
            }
            requestChannel.setInitiator(peerIsInitiator);

            if (useBundle && endpointId != null) {
                requestChannel.setChannelBundleId(endpointId);
            }

            if (requestChannel instanceof ColibriConferenceIQ.Channel) {
                ColibriConferenceIQ.Channel requestRtpChannel = (ColibriConferenceIQ.Channel) requestChannel;
                copyDescription(content, requestRtpChannel);

                // Config options
                requestRtpChannel.setLastN(channelLastN);
                requestRtpChannel.setSimulcastMode(simulcastMode);
                if (MediaType.AUDIO.equals(mediaType)) {
                    // When audioPacketDelay is null it will clear the attribute
                    requestRtpChannel.setPacketDelay(audioPacketDelay);
                    // Set rtp packet relay type for this channel
                    requestRtpChannel.setRTPLevelRelayType(rtpLevelRelayType);
                }

                // Copy the sources and source groups
                if ((sourceMap != null) && sourceMap.get(contentName) != null) {
                    addSources(requestRtpChannel, sourceMap.get(contentName));
                }
                if ((sourceGroupMap != null) && sourceGroupMap.get(contentName) != null) {
                    addSourceGroups(requestRtpChannel, sourceGroupMap.get(contentName), MediaType.VIDEO.equals(mediaType));
                }
            }

            if (requestChannel instanceof ColibriConferenceIQ.OctoChannel) {
                ((ColibriConferenceIQ.OctoChannel) requestChannel).setRelays(octoRelayIds);
            }

            // Copy transport
            if (!useBundle) {
                copyTransport(content, requestChannel);
            }

            if (requestChannel instanceof ColibriConferenceIQ.Channel) {
                requestContent.addChannel((ColibriConferenceIQ.Channel) requestChannel);
            }
            else {
                requestContent.addSctpConnection((ColibriConferenceIQ.SctpConnection) requestChannel);
            }
            hasAnyChanges = true;
        }

        if (useBundle && endpointId != null && contents.size() >= 1) {
            // Copy first transport to bundle
            ColibriConferenceIQ.ChannelBundle bundle = new ColibriConferenceIQ.ChannelBundle(endpointId);

            JingleContent firstContent = contents.get(0);
            IceUdpTransport transport = firstContent.getFirstChildElement(IceUdpTransport.class);
            if (transport != null) {
                hasAnyChanges = true;
                bundle.setTransport(IceUdpTransport.cloneTransportAndCandidates(transport, true));
            }
            request.addChannelBundle(bundle);
        }

        if (endpointId != null) {
            // Set the endpoint
            ColibriConferenceIQ.Endpoint endpoint = new ColibriConferenceIQ.Endpoint(endpointId, statsId, null);
            request.addEndpoint(endpoint);
        }
        return hasAnyChanges;
    }

    /**
     * Adds a {@link ColibriConferenceIQ.ChannelBundle} with a specific ID and a specific {@code transport}
     * element to a {@link RequestType#CHANNEL_INFO_UPDATE} query.
     *
     * @param transport the transport element to add to the bundle.
     * @param channelBundleId the ID of the bundle
     * @return {@code true} if the request yields any changes in Colibri
     * channels state on the bridge or {@code false} otherwise. In general when
     * {@code false} is returned for all combined requests it makes no sense to send it.
     */
    public boolean addBundleTransportUpdateReq(IceUdpTransport transport, String channelBundleId)
            throws IllegalArgumentException
    {
        Objects.requireNonNull(transport, "transport");
        if ((conferenceState == null) || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet.

            // XXX by returning false we silently indicate that there is no
            // need to send a packet, while in fact the state of the conference
            // does NOT reflect the information that this method call was
            // supposed to add. At least log something.
            Timber.w("Not adding a transport bundle, not initialized");
            return false;
        }

        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);
        ColibriConferenceIQ.ChannelBundle channelBundleRequest = new ColibriConferenceIQ.ChannelBundle(channelBundleId);
        channelBundleRequest.setTransport(IceUdpTransport.cloneTransportAndCandidates(transport, true));
        request.addChannelBundle(channelBundleRequest);

        // Note that we don't actually check whether the addition of the bundle
        // made any changes. We return true, because it is safe. It might lead
        // us to send an unnecessary request, but it will not fail to send a
        // request when we need it. We may need to add a check for modification
        // of the content of the channel bundle extension.
        return true;
    }

    /**
     * Adds next expire channel request to {@link RequestType#EXPIRE_CHANNELS} query currently being built.
     *
     * @param channelInfo the {@link ColibriConferenceIQ} instance that contains
     * info about the channels to be expired.
     * @return <code>true</code> if the request yields any changes in Colibri
     * channels state on the bridge or <code>false</code> otherwise.
     * In general when <code>false</code> is returned for all combined requests it makes no sense to send it.
     */
    public boolean addExpireChannelsReq(ColibriConferenceIQ channelInfo)
    {
        Objects.requireNonNull(channelInfo, "channelInfo");
        // Formulate the ColibriConferenceIQ request which is to be sent.
        if ((conferenceState == null) || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            return false;
        }

        assertRequestType(RequestType.EXPIRE_CHANNELS);
        for (ColibriConferenceIQ.Content expiredContent : channelInfo.getContents()) {
            ColibriConferenceIQ.Content stateContent = conferenceState.getContent(expiredContent.getName());

            if (stateContent != null) {
                ColibriConferenceIQ.Content requestContent = request.getOrCreateContent(stateContent.getName());

                for (ColibriConferenceIQ.Channel expiredChannel : expiredContent.getChannels()) {
                    ColibriConferenceIQ.Channel stateChannel = stateContent.getChannel(expiredChannel.getID());

                    if (stateChannel != null) {
                        ColibriConferenceIQ.Channel requestChannel = new ColibriConferenceIQ.Channel();
                        requestChannel.setExpire(0);
                        requestChannel.setID(stateChannel.getID());
                        requestContent.addChannel(requestChannel);
                        hasAnyChannelsToExpire = true;
                    }
                }
                for (ColibriConferenceIQ.SctpConnection expiredConn : expiredContent.getSctpConnections()) {
                    ColibriConferenceIQ.SctpConnection stateConn = stateContent.getSctpConnection(expiredConn.getID());

                    if (stateConn != null) {
                        ColibriConferenceIQ.SctpConnection requestConn = new ColibriConferenceIQ.SctpConnection();

                        requestConn.setID(stateConn.getID());
                        requestConn.setExpire(0);

                        // FIXME: can be removed once we do not handle SCTP by endpoint anymore
                        requestConn.setEndpoint(stateConn.getEndpoint());
                        requestContent.addSctpConnection(requestConn);
                        hasAnyChannelsToExpire = true;
                    }
                }
            }
        }

        /*
         * Remove the channels which are to be expired from the internal state of the conference
         * managed by this ColibriBuilder.
         */
        for (ColibriConferenceIQ.Content requestContent : request.getContents()) {
            ColibriConferenceIQ.Content stateContent = conferenceState.getContent(requestContent.getName());

            for (ColibriConferenceIQ.Channel requestChannel : requestContent.getChannels()) {
                ColibriConferenceIQ.Channel stateChannel = stateContent.getChannel(requestChannel.getID());
                stateContent.removeChannel(stateChannel);

                /*
                 * If the last remote channel is to be expired, expire the local channel as well.
                 */
                /*
                 * if (stateContent.getChannelCount() == 1) { stateChannel =
                 * stateContent.getChannel(0);
                 *
                 * ColibriConferenceIQ.Channel channelRequest = new ColibriConferenceIQ.Channel();
                 * channelRequest.setExpire(0); channelRequest.setID(stateChannel.getID());
                 * requestContent.addChannel(channelRequest);
                 *
                 * hasAnyChannelsToExpire = true;
                 *
                 * stateContent.removeChannel(stateChannel); break; }
                 */
            }
            for (ColibriConferenceIQ.SctpConnection requestConn : requestContent.getSctpConnections()) {
                ColibriConferenceIQ.SctpConnection stateConn = stateContent.getSctpConnection(requestConn.getID());

                stateContent.removeSctpConnection(stateConn);
                /*
                 * If the last remote channel is to be expired, expire the local channel as well.
                 */
                /*if (stateContent.getSctpConnections().size() == 1)
                {
                    stateConn = stateContent.getSctpConnections().get(0);

                    ColibriConferenceIQ.SctpConnection requestConn
                        = new ColibriConferenceIQ.SctpConnection();
                    requestConn.setID(stateConn.getID());
                    requestConn.setExpire(0);
                    requestConn.setEndpoint(stateConn.getEndpoint());
                    requestContent.addSctpConnection(requestConn);

                    hasAnyChannelsToExpire = true;

                    stateContent.removeSctpConnection(stateConn);

                    break;
                }*/
            }
        }
        return hasAnyChannelsToExpire;
    }

    /**
     * Adds an {@link RtpDescription} to a specific channel in
     * the request which is currently being built. The channel in the request
     * is identified by the content name and the ID of {@code channel}.
     *
     * @param description the {@link RtpDescription} to add.
     * @param contentName the name of the content to which the channel belongs.
     * @param channel the the channel used to match the channel in the request
     * to which an {@link RtpDescription} will be added.
     * @return {@code true} if the request yields any changes in Colibri
     * channels state on the bridge or {@code false} otherwise. In general when
     * {@code false} is returned for all combined requests it makes no sense to send it.
     */
    public boolean addRtpDescription(RtpDescription description, String contentName,
            ColibriConferenceIQ.Channel channel)
    {
        Objects.requireNonNull(description, "description");
        Objects.requireNonNull(contentName, "contentName");
        Objects.requireNonNull(channel, "channel");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet

            // XXX by returning false we silently indicate that there is no
            // need to send a packet, while in fact the state of the conference
            // does NOT reflect the information that this method call was
            // supposed to add. At least log something.
            Timber.w("Not adding description to a channel, not initialized");
            return false;
        }

        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);
        ColibriConferenceIQ.Channel requestChannel = getRequestChannel(contentName, channel);
        return copyDescription(description, requestChannel);
    }

    /**
     * Adds next source information update request to
     * {@link RequestType#CHANNEL_INFO_UPDATE} query currently being built.
     *
     * @param sourceMap the map of content name to the list of <code>SdpSourceGroup</code>.
     * @param localChannelsInfo {@link ColibriConferenceIQ} holding info about
     * Colibri channels to be updated.
     * @return <code>true</code> if the request yields any changes in Colibri
     * channels state on the bridge or <code>false</code> otherwise. In general when
     * <code>false</code> is returned for all combined requests it makes no sense to send it.
     */
    public boolean addSourceInfo(Map<String, List<SdpSource>> sourceMap, ColibriConferenceIQ localChannelsInfo)
    {
        Objects.requireNonNull(sourceMap, "sourceMap");
        Objects.requireNonNull(localChannelsInfo, "localChannelsInfo");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet
            return false;
        }

        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);
        boolean hasAnyChanges = false;

        // Go over sources
        for (String contentName : sourceMap.keySet()) {
            // Get channel from local channel info
            ColibriConferenceIQ.Channel channel = getChannel(localChannelsInfo, contentName);
            if (channel == null) {
                // There's no channel for this content name in localChannelsInfo
                continue;
            }

            hasAnyChanges = true;

            // Ok we have channel for this content, let's add sources
            ColibriConferenceIQ.Channel requestChannel = getRequestChannel(contentName, channel);
            addSources(requestChannel, sourceMap.get(contentName));
        }
        return hasAnyChanges;
    }

    /**
     * Sets the Octo relays for specific channels in the request currently being built.
     *
     * @param octoRelays the map of content name to the list of <code>SdpSourceGroup</code>.
     * @param localChannelsInfo {@link ColibriConferenceIQ} holding info about Colibri channels to be updated.
     * @return <code>true</code> if the request yields any changes in Colibri
     * channels state on the bridge or <code>false</code> otherwise. In general when
     * <code>false</code> is returned for all combined requests it makes no sense to send it.
     */
    public boolean addOctoRelays(List<String> octoRelays, ColibriConferenceIQ localChannelsInfo)
    {
        Objects.requireNonNull(octoRelays, "octoRelays");
        Objects.requireNonNull(localChannelsInfo, "localChannelsInfo");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet
            return false;
        }

        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);
        boolean hasAnyChanges = false;

        for (ColibriConferenceIQ.Content content : conferenceState.getContents()) {
            String contentName = content.getName();
            // Get channel from local channel info
            ColibriConferenceIQ.Channel channel = getChannel(localChannelsInfo, contentName);
            if (channel == null || !(channel instanceof ColibriConferenceIQ.OctoChannel)) {
                // There's no channel for this content name in localChannelsInfo
                continue;
            }

            ColibriConferenceIQ.OctoChannel requestChannel
                    = getRequestChannel(contentName, (ColibriConferenceIQ.OctoChannel) channel);
            requestChannel.setRelays(octoRelays);
            hasAnyChanges = true;
        }
        return hasAnyChanges;
    }

    /**
     * Adds next SSRC group information update request to
     * {@link RequestType#CHANNEL_INFO_UPDATE} query currently being built.
     *
     * @param sourceGroupMap the map of content name to the list of <code>SdpSourceGroup</code>.
     * @param localChannelsInfo {@link ColibriConferenceIQ} holding info about Colibri channels to be updated.
     * @return <code>true</code> if the request yields any changes in Colibri channels state on the
     * bridge or <code>false</code> otherwise. In general when <code>false</code> is returned for all
     * combined requests it makes no sense to send it.
     */
    public boolean addSourceGroupsInfo(Map<String, List<SdpSourceGroup>> sourceGroupMap,
            ColibriConferenceIQ localChannelsInfo)
    {
        Objects.requireNonNull(sourceGroupMap, "sourceGroupMap");
        Objects.requireNonNull(localChannelsInfo, "localChannelsInfo");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet
            return false;
        }

        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);
        boolean hasAnyChanges = false;

        // Go over source groups
        for (String contentName : sourceGroupMap.keySet()) {
            // Get channel from local channel info
            ColibriConferenceIQ.Channel channel = getChannel(localChannelsInfo, contentName);
            if (channel == null) {
                // There's no channel for this content name in localChannelsInfo
                continue;
            }

            // Ok we have channel for this content, let's add sources
            ColibriConferenceIQ.Channel requestChannel = getRequestChannel(contentName, channel);

            hasAnyChanges |= addSourceGroups(requestChannel, sourceGroupMap.get(contentName),
                    "video".equalsIgnoreCase(contentName));
        }
        return hasAnyChanges;
    }

    /**
     * Adds next ICE transport update request to
     * {@link RequestType#CHANNEL_INFO_UPDATE} query currently being built.
     *
     * Note that this should only be used for channels which do NOT use bundle.
     * For the bundle case use {@link #addBundleTransportUpdateReq} instead.
     *
     * @param transportMap the map of content name to transport extensions. Maps transport to media types.
     * @param localChannelsInfo {@link ColibriConferenceIQ} holding info about Colibri channels to be updated.
     * @return <code>true</code> if the request yields any changes in Colibri channels state on the
     * bridge or <code>false</code> otherwise. In general when <code>false</code> is returned for all
     * combined requests it makes no sense to send it.
     */
    public boolean addTransportUpdateReq(Map<String, IceUdpTransport> transportMap,
            ColibriConferenceIQ localChannelsInfo)
    {
        Objects.requireNonNull(transportMap, "transportMap");
        Objects.requireNonNull(localChannelsInfo, "localChannelsInfo");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet
            return false;
        }

        boolean hasAnyChanges = false;
        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);

        for (Map.Entry<String, IceUdpTransport> e : transportMap.entrySet()) {
            String contentName = e.getKey();
            ColibriConferenceIQ.ChannelCommon channel = getChannelCommon(localChannelsInfo, contentName);

            if (channel != null) {
                IceUdpTransport transport
                        = IceUdpTransport.cloneTransportAndCandidates(e.getValue(), true);

                ColibriConferenceIQ.ChannelCommon requestChannel = channel instanceof ColibriConferenceIQ.Channel
                        ? new ColibriConferenceIQ.Channel() : new ColibriConferenceIQ.SctpConnection();

                requestChannel.setID(channel.getID());
                requestChannel.setEndpoint(channel.getEndpoint());
                requestChannel.setTransport(transport);

                request.getOrCreateContent(contentName).addChannelCommon(requestChannel);
                hasAnyChanges = true;
            }
        }
        return hasAnyChanges;
    }

    /**
     * Adds next request to {@link RequestType#CHANNEL_INFO_UPDATE} query.
     *
     * @param mediaDirectionMap the map of content name to media direction. Maps media direction to media types.
     * @param localChannelsInfo {@link ColibriConferenceIQ} holding info about Colibri channels to be updated.
     * @return <code>true</code> if the request yields any changes in Colibri channels state on the
     * bridge or <code>false</code> otherwise. In general when <code>false</code> is returned for all
     * combined requests it makes no sense to send it.
     * @deprecated Please refactor accordingly if/when this API starts to be used.
     */
    @Deprecated
    public boolean addDirectionUpdateReq(Map<String, MediaDirection> mediaDirectionMap,
            ColibriConferenceIQ localChannelsInfo)
    {
        Objects.requireNonNull(mediaDirectionMap, "mediaDirectionMap");
        Objects.requireNonNull(localChannelsInfo, "localChannelsInfo");

        if (conferenceState == null || StringUtils.isNullOrEmpty(conferenceState.getID())) {
            // We are not initialized yet
            return false;
        }

        boolean hasAnyChanges = false;
        assertRequestType(RequestType.CHANNEL_INFO_UPDATE);

        for (String contentName : mediaDirectionMap.keySet()) {
            ColibriConferenceIQ.Content content = localChannelsInfo.getContent(contentName);
            if (content != null) {
                for (ColibriConferenceIQ.Channel channel : content.getChannels()) {
                    ColibriConferenceIQ.Channel requestChannel = getRequestChannel(contentName, channel);
                    requestChannel.setDirection(mediaDirectionMap.get(contentName));
                    hasAnyChanges = true;
                }
            }
        }
        return hasAnyChanges;
    }

    /**
     * Finds the first {@link ColibriConferenceIQ.ChannelCommon} with a given
     * content name in a specific {@link ColibriConferenceIQ}.
     *
     * @param colibriConferenceIQ the {@link ColibriConferenceIQ} from which to find a channel.
     * @param contentName the name of the content.
     * @return the first {@link ColibriConferenceIQ.ChannelCommon} in the
     * content with name {@code contentName} in {@code colibriConferenceIQ}.
     */
    private ColibriConferenceIQ.ChannelCommon getChannelCommon(ColibriConferenceIQ colibriConferenceIQ, String contentName)
    {
        ColibriConferenceIQ.Content content = colibriConferenceIQ.getContent(contentName);

        if (content == null) {
            return null;
        }

        if (content.getChannelCount() > 0) {
            return content.getChannel(0);
        }

        if (content.getSctpConnections().size() > 0) {
            return content.getSctpConnections().get(0);
        }
        return null;
    }

    /**
     * Finds the first {@link ColibriConferenceIQ.Channel} with a given content
     * name in a specific {@link ColibriConferenceIQ}.
     *
     * @param localChannelsInfo the {@link ColibriConferenceIQ} from which to find a channel.
     * @param contentName the name of the content.
     * @return the first {@link ColibriConferenceIQ.Channel} in the
     * content with name {@code contentName} in {@code colibriConferenceIQ}.
     */
    private ColibriConferenceIQ.Channel getChannel(ColibriConferenceIQ localChannelsInfo, String contentName)
    {
        ColibriConferenceIQ.ChannelCommon channel = getChannelCommon(localChannelsInfo, contentName);
        return channel instanceof ColibriConferenceIQ.Channel ? (ColibriConferenceIQ.Channel) channel : null;
    }


    /**
     * Finishes query construction and returns it. It does not reset this instance state and
     * {@link #reset()} must be called to start new query. Otherwise we can continue adding next
     * requests into current query.
     *
     * @param videobridge the JID of videobridge to which this query is directed.
     * @return constructed query directed to given <code>videobridge</code>. If request type of current
     * query is {@link RequestType#EXPIRE_CHANNELS} and there are no channels to be expired
     * then <code>null</code> is returned which signals that there's nothing to be done.
     */
    public ColibriConferenceIQ getRequest(Jid videobridge)
    {
        Objects.requireNonNull(videobridge, "videobridge");

        request.setTo(videobridge);
        if (requestType == RequestType.EXPIRE_CHANNELS) {
            if (!hasAnyChannelsToExpire)
                return null;
            hasAnyChannelsToExpire = false;
        }
        return request;
    }

    /**
     * Returns {@link ColibriBuilder.RequestType} of the query currently being constructed or
     * {@link RequestType#UNDEFINED} if no query construction has been started yet.
     */
    public RequestType getRequestType()
    {
        return requestType;
    }

    /**
     * Makes sure that we do not modify the request type after construction has been started.
     *
     * @param currentReqType the current request type to be used.
     */
    private void assertRequestType(RequestType currentReqType)
    {
        if (requestType == RequestType.UNDEFINED) {
            requestType = currentReqType;
        }
        if (requestType != currentReqType) {
            throw new IllegalStateException("Request type already set to " + requestType
                    + ", can not change to " + currentReqType);
        }
    }

    /**
     * Returns <code>true</code> if {@link RequestType#EXPIRE_CHANNELS} request is being constructed
     * and there are valid channels to be expired.
     */
    public boolean hasAnyChannelsToExpire()
    {
        return hasAnyChannelsToExpire;
    }

    /**
     * Channel 'last-n' option that will be added when channels are created. Set to
     * <code>null</code> in order to omit. Value is reset after {@link #reset} is called.
     *
     * @return an integer value or <code>null</code> if option is unspecified.
     */
    public Integer getChannelLastN()
    {
        return channelLastN;
    }

    /**
     * Sets channel 'last-n' option that will be added to the request when channels are created.
     *
     * @param channelLastN an integer value to specify 'last-n' option or <code>null</code> in order to omit in requests.
     */
    public void setChannelLastN(Integer channelLastN)
    {
        this.channelLastN = channelLastN;
    }

    /**
     * Returns an <code>Integer</code> which stands for the audio packet delay that will be set on all created audio
     * channels or <code>null</code> if the builder should leave not include the XML attribute at all.
     */
    public Integer getAudioPacketDelay()
    {
        return audioPacketDelay;
    }

    /**
     * Configures audio channels packet delay.
     *
     * @param audioPacketDelay an <code>Integer</code> value which stands for
     * the audio packet delay that will be set on all created audio channels or
     * <code>null</code> if the builder should not set that channel property to any value.
     */
    public void setAudioPacketDelay(Integer audioPacketDelay)
    {
        this.audioPacketDelay = audioPacketDelay;
    }

    /**
     * Sets channel 'simulcast-mode' option that will be added to the request when channels are created.
     *
     * @param simulcastMode a <code>SimulcastMode</code> value to specify
     * 'simulcast-mode' option or <code>null</code> in order to omit in requests.
     */
    public void setSimulcastMode(SimulcastMode simulcastMode)
    {
        this.simulcastMode = simulcastMode;
    }

    /**
     * Returns the channel from {@link #request} which matches a particular
     * content name and a particular {@link ColibriConferenceIQ.ChannelCommon}
     * instance (i.e. their IDs match).
     * If the request does not contain such a channel, a new instance is created
     * and added to {@link #request}, and it is initialized only with the fields
     * required to identify the particular Colibri channel on the bridge (i.e.
     * just the ID is specified). The exact run-time type of the instance (
     * {@link ColibriConferenceIQ.Channel},
     * {@link ColibriConferenceIQ.SctpConnection}, or
     * {@link ColibriConferenceIQ.OctoChannel}) depends on the type of
     * {@code channel}.
     *
     * The returned value is always non-null.
     *
     * @param contentName the name of the content.
     * @param channel the channel to match.
     * @return the channel from {@link #request} with the specified content
     * name and ID and run-time type matching this of {@code channel}.
     */
    private <T extends ColibriConferenceIQ.ChannelCommon> T getRequestChannel(String contentName, T channel)
    {
        ColibriConferenceIQ.Content requestContent = request.getOrCreateContent(contentName);
        ColibriConferenceIQ.ChannelCommon requestChannel = requestContent.getChannel(channel.getID());
        if (requestChannel == null) {
            requestChannel = requestContent.getSctpConnection(channel.getID());
        }

        if (requestChannel == null) {
            if (channel instanceof ColibriConferenceIQ.SctpConnection) {
                requestChannel = new ColibriConferenceIQ.SctpConnection();
            }
            else if (channel instanceof ColibriConferenceIQ.OctoChannel) {
                requestChannel = new ColibriConferenceIQ.OctoChannel();
            }
            else {
                requestChannel = new ColibriConferenceIQ.Channel();
            }
            requestChannel.setID(channel.getID());
            requestContent.addChannelCommon(requestChannel);
        }

        try {
            return (T) requestChannel;
        } catch (ClassCastException cce) {
            throw new IllegalStateException("Channel type mismatch: requested " + channel.getClass().getSimpleName()
                    + ", found a channel of class " + requestChannel.getClass().getSimpleName());
        }
    }

    /**
     * Configures RTP-level relay (RFC 3550, section 2.3).
     *
     * @param rtpLevelRelayType an <code>RTPLevelRelayType</code> value which
     * stands for the rtp level relay type that will be set on all created audio channels.
     */
    public void setRTPLevelRelayType(RTPLevelRelayType rtpLevelRelayType)
    {
        this.rtpLevelRelayType = rtpLevelRelayType;
    }

    /**
     * Configures RTP-level relay (RFC 3550, section 2.3).
     *
     * @param rtpLevelRelayType a <code>String</code> value which
     * stands for the rtp level relay type that will be set on all created audio channels.
     */
    public void setRTPLevelRelayType(String rtpLevelRelayType)
    {
        setRTPLevelRelayType(RTPLevelRelayType.parseRTPLevelRelayType(rtpLevelRelayType));
    }

    /**
     * The types of request that can be built with {@link ColibriBuilder}.
     */
    public enum RequestType
    {
        /**
         * Allocates new channels on the bridge request.
         */
        ALLOCATE_CHANNELS,

        /**
         * An update request which is meant to modify some values of existing Colibri channels on the bridge.
         */
        CHANNEL_INFO_UPDATE,

        /**
         * Expires specified Colibri channels.
         */
        EXPIRE_CHANNELS,

        /**
         * Undefined means {@link ColibriBuilder} "zero" state. The point where builder instance
         * can be used to start next query.
         */
        UNDEFINED;
    }
}
