/*
 * Copyright @ 2015 Atlassian Pty Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package org.atalk.impl.neomedia.jmfext.media.protocol.rtpdumpfile;

import org.atalk.impl.neomedia.jmfext.media.protocol.AbstractVideoPullBufferStream;
import org.atalk.service.neomedia.RawPacket;

import java.io.IOException;

import javax.media.Buffer;
import javax.media.Format;
import javax.media.control.FormatControl;
import javax.media.format.AudioFormat;
import javax.media.format.VideoFormat;

import timber.log.Timber;

/**
 * Implements a <code>PullBufferStream</code> which read an rtpdump file to generate a RTP stream from
 * the payloads recorded in a rtpdump file.
 *
 * @author Thomas Kuntz
 */
public class RtpdumpStream extends AbstractVideoPullBufferStream<DataSource>
{
    /**
     * The <code>RawPacketScheduler</code> responsible for throttling our RTP packet reading.
     */
    private final RawPacketScheduler rawPacketScheduler;

    /**
     * Boolean indicating if the last call to <code>doRead</code> return a marked rtp packet (to
     * know if <code>timestamp</code> needs to be updated).
     */
    private boolean lastReadWasMarked = true;

    /**
     * The <code>RtpdumpFileReader</code> used by this stream to get the rtp payload.
     */
    private RtpdumpFileReader rtpFileReader;

    /**
     * The timestamp to use for the timestamp of the next <code>Buffer</code> filled in
     * {@link #doRead(javax.media.Buffer)}
     */
    private long timestamp;

    /**
     * Initializes a new <code>RtpdumpStream</code> instance
     *
     * @param dataSource the <code>DataSource</code> which is creating the new instance so that it becomes one of
     * its <code>streams</code>
     * @param formatControl the <code>FormatControl</code> of the new instance which is to specify the format in which
     * it is to provide its media data
     */
    RtpdumpStream(DataSource dataSource, FormatControl formatControl)
    {
        super(dataSource, formatControl);
        /*
         * NOTE: We use the sampleRate or frameRate field of the format to piggyback the RTP clock
         * rate. See RtpdumpMediaDevice#createRtpdumpMediaDevice.
         */
        Format format = getFormat();
        long clockRate;
        if (format instanceof AudioFormat) {
            clockRate = (long) ((AudioFormat) format).getSampleRate();
        }
        else if (format instanceof VideoFormat) {
            clockRate = (long) ((VideoFormat) format).getFrameRate();
        }
        else {
            Timber.w("Unknown format. Creating RtpdumpStream with clock rate 1 000 000 000.");
            clockRate = 1000 * 1000 * 1000;
        }

        this.rawPacketScheduler = new RawPacketScheduler(clockRate);
        String rtpdumpFilePath = dataSource.getLocator().getRemainder();
        this.rtpFileReader = new RtpdumpFileReader(rtpdumpFilePath);
    }

    /**
     * Reads available media data from this instance into a specific <code>Buffer</code>.
     *
     * @param buffer the <code>Buffer</code> to write the available media data into
     * @throws IOException if an I/O error has prevented the reading of available media data from this instance
     * into the specified <code>Buffer</code>
     */
    @Override
    protected void doRead(Buffer buffer)
            throws IOException
    {
        Format format;
        format = buffer.getFormat();
        if (format == null) {
            format = getFormat();
            if (format != null)
                buffer.setFormat(format);
        }

        RawPacket rtpPacket = rtpFileReader.getNextPacket(true);
        byte[] data = rtpPacket.getPayload();

        buffer.setData(data);
        buffer.setOffset(rtpPacket.getOffset());
        buffer.setLength(rtpPacket.getPayloadLength());

        buffer.setFlags(Buffer.FLAG_SYSTEM_TIME | Buffer.FLAG_LIVE_DATA);
        if (lastReadWasMarked) {
            timestamp = System.nanoTime();
        }
        lastReadWasMarked = rtpPacket.isPacketMarked();
        if (lastReadWasMarked) {
            buffer.setFlags(buffer.getFlags() | Buffer.FLAG_RTP_MARKER);
        }
        buffer.setTimeStamp(timestamp);

        try {
            rawPacketScheduler.schedule(rtpPacket);
        } catch (InterruptedException e) {
        }
    }
}
