/*
 * aTalk, android VoIP and Instant Messaging client
 * Copyright 2014 Eng Chong Meng
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.atalk.impl.appnotification;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.Intent;
import android.provider.Settings;

import java.util.List;

import org.atalk.android.R;

/**
 * Helper class to manage notification channels, and create notifications.
 *
 * @author Eng Chong Meng
 */
public class NotificationHelper extends ContextWrapper {
    private NotificationManager notificationManager = null;

    private static final int LED_COLOR = 0xff00ff00;

    /**
     * Registers notification channels, which can be used later by individual notifications.
     *
     * @param ctx The application context
     */
    public NotificationHelper(Context ctx) {
        super(ctx);

        // Init the system service NotificationManager
        notificationManager = ctx.getSystemService(NotificationManager.class);

        // Delete any unused channel IDs or all if force is true.
        deleteObsoletedChannelIds(false);

        final NotificationChannel nCall = new NotificationChannel(AppNotifications.CALL_GROUP,
                getString(R.string.noti_channel_call_group), NotificationManager.IMPORTANCE_HIGH);
        nCall.setSound(null, null);
        nCall.setShowBadge(false);
        nCall.setLightColor(LED_COLOR);
        nCall.enableLights(true);
        nCall.setLockscreenVisibility(Notification.VISIBILITY_PUBLIC);
        notificationManager.createNotificationChannel(nCall);

        final NotificationChannel nMessage = new NotificationChannel(AppNotifications.MESSAGE_GROUP,
                getString(R.string.noti_channel_message_group), NotificationManager.IMPORTANCE_HIGH);
        nMessage.setSound(null, null);
        nMessage.setShowBadge(true);
        nMessage.setLightColor(LED_COLOR);
        nMessage.enableLights(true);
        // nMessage.setAllowBubbles(true);
        nMessage.setLockscreenVisibility(Notification.VISIBILITY_PRIVATE);
        notificationManager.createNotificationChannel(nMessage);

        final NotificationChannel nFile = new NotificationChannel(AppNotifications.FILE_GROUP,
                getString(R.string.noti_channel_file_group), NotificationManager.IMPORTANCE_HIGH);
        nFile.setSound(null, null);
        nFile.setShowBadge(true);
        nFile.setLightColor(LED_COLOR);
        nFile.enableLights(true);
        nFile.setLockscreenVisibility(Notification.VISIBILITY_PRIVATE);
        notificationManager.createNotificationChannel(nFile);

        final NotificationChannel nDefault = new NotificationChannel(AppNotifications.DEFAULT_GROUP,
                getString(R.string.noti_channel_default_group), NotificationManager.IMPORTANCE_LOW);
        nDefault.setSound(null, null);
        nDefault.setShowBadge(false);
        // nDefault.setLightColor(Color.WHITE);
        nDefault.setLockscreenVisibility(Notification.VISIBILITY_PRIVATE);
        notificationManager.createNotificationChannel(nDefault);

        final NotificationChannel nQuietHours = new NotificationChannel(AppNotifications.SILENT_GROUP,
                getString(R.string.noti_channel_silent_group), NotificationManager.IMPORTANCE_LOW);
        nQuietHours.setSound(null, null);
        nQuietHours.setShowBadge(true);
        nQuietHours.setLightColor(LED_COLOR);
        nQuietHours.setLockscreenVisibility(Notification.VISIBILITY_PRIVATE);
        notificationManager.createNotificationChannel(nQuietHours);
    }

    /**
     * Send a notification.
     *
     * @param id The ID of the notification
     * @param notification The notification object
     */
    public void notify(int id, Notification.Builder notification) {
        notificationManager.notify(id, notification.build());
    }

    /**
     * Send Intent to load system Notification Settings for this app.
     */
    public void goToNotificationSettings() {
        Intent i = new Intent(Settings.ACTION_APP_NOTIFICATION_SETTINGS);
        i.putExtra(Settings.EXTRA_APP_PACKAGE, getPackageName());
        startActivity(i);
    }

    /**
     * Send intent to load system Notification Settings UI for a particular channel.
     *
     * @param channel Name of notification channel.
     */
    public void goToNotificationSettings(String channel) {
        Intent intent = new Intent(Settings.ACTION_CHANNEL_NOTIFICATION_SETTINGS);
        intent.putExtra(Settings.EXTRA_APP_PACKAGE, getPackageName());
        intent.putExtra(Settings.EXTRA_CHANNEL_ID, channel);
        startActivity(intent);
    }

    /**
     * Delete any unused (or all if force is true) channel IDs.
     * Must force to re-init all notification channels if higher priority changes are made to any existing channel.
     * See <a href="https://developer.android.com/develop/ui/views/notifications/channels#UpdateChannel">...</a>
     *
     * @param force force delete the notification channel
     */
    private void deleteObsoletedChannelIds(boolean force) {
        List<NotificationChannel> channelGroups = notificationManager.getNotificationChannels();
        for (NotificationChannel nc : channelGroups) {
            if (force || !AppNotifications.notificationIds.contains(nc.getId())) {
                notificationManager.deleteNotificationChannel(nc.getId());
            }
        }
    }
}
