/*
 * Jitsi, the OpenSource Java VoIP and Instant Messaging client.
 *
 * Distributable under LGPL license. See terms of license at gnu.org.
 */
package org.atalk.android.gui.account;

import android.app.Service;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.text.Editable;
import android.text.InputFilter;
import android.text.TextWatcher;
import android.view.ContextMenu;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.inputmethod.InputMethodManager;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.DatePicker;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.Spinner;
import android.widget.Toast;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultCaller;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.yalantis.ucrop.UCrop;

import net.java.sip.communicator.service.protocol.AccountID;
import net.java.sip.communicator.service.protocol.OperationFailedException;
import net.java.sip.communicator.service.protocol.OperationSetPresence;
import net.java.sip.communicator.service.protocol.OperationSetServerStoredAccountInfo;
import net.java.sip.communicator.service.protocol.PresenceStatus;
import net.java.sip.communicator.service.protocol.ProtocolProviderService;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.AboutMeDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.AddressDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.BirthDateDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.CityDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.CountryDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.DisplayNameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.EmailAddressDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.FirstNameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.GenderDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.GenericDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.ImageDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.JobTitleDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.LastNameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.MiddleNameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.MobilePhoneDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.NicknameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.PhoneNumberDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.PostalCodeDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.ProvinceDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.URLDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.WorkEmailAddressDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.WorkOrganizationNameDetail;
import net.java.sip.communicator.service.protocol.ServerStoredDetails.WorkPhoneDetail;
import net.java.sip.communicator.service.protocol.globalstatus.GlobalStatusService;
import net.java.sip.communicator.util.ServiceUtils;
import net.java.sip.communicator.util.account.AccountUtils;

import org.atalk.android.BaseActivity;
import org.atalk.android.R;
import org.atalk.android.gui.AppGUIActivator;
import org.atalk.android.gui.account.settings.AccountPreferenceActivity;
import org.atalk.android.gui.actionbar.ActionBarUtil;
import org.atalk.android.gui.contactlist.ContactInfoActivity;
import org.atalk.android.gui.dialogs.DialogActivity;
import org.atalk.android.gui.dialogs.ProgressDialog;
import org.atalk.android.gui.util.ViewUtil;
import org.atalk.android.gui.util.event.EventListener;
import org.atalk.android.util.AppImageUtil;
import org.atalk.util.SoftKeyboard;
import org.jivesoftware.smackx.avatar.AvatarManager;

import timber.log.Timber;

/**
 * Activity allows user to set presence status, status message, change the user avatar
 * and all the vCard-temp information for the {@link Account}.
 * <p>
 * The main panel that allows users to view and edit their account information.
 * Different instances of this class are created for every registered
 * <code>ProtocolProviderService</code>.
 * Currently, supported account details are first/middle/last names, nickname,
 * street/city/region/country address, postal code, birth date, gender,
 * organization name, job title, about me, home/work email, home/work phone.
 * <p>
 * The {@link #mAccount} is retrieved from the {@link Intent} extra by it's
 * {@link AccountID#getAccountUid()}
 *
 * @author Pawel Domas
 * @author Eng Chong Meng
 */
public class AccountInfoPresenceActivity extends BaseActivity
        implements EventListener<AccountEvent>, DialogActivity.DialogListener,
        SoftKeyboard.SoftKeyboardChanged, DatePicker.OnDateChangedListener {
    private DatePicker mDatePicker;

    private static final String AVATAR_ICON_REMOVE = "Remove Picture";

    // avatar default image size
    private static final int AVATAR_PREFERRED_SIZE = 64;
    private static final int CROP_MAX_SIZE = 108;

    /**
     * Intent's extra's key for account ID property of this activity
     */
    static public final String INTENT_ACCOUNT_ID = "account_id";

    /**
     * The account's {@link OperationSetPresence} used to perform presence operations
     */
    private OperationSetPresence accountPresence;

    /**
     * The instance of {@link Account} used for operations on the account
     */
    private Account mAccount;

    /**
     * Flag indicates if there were any uncommitted changes that shall be applied on exit
     */
    private boolean hasChanges = false;

    /**
     * Flag indicates if there were any uncommitted status changes that shall be applied on exit
     */
    private boolean hasStatusChanges = false;

    /**
     * Mapping between all supported by this plugin <code>ServerStoredDetails</code> and their
     * respective <code>EditText</code> that are used for modifying the details.
     */
    private final Map<Class<? extends GenericDetail>, EditText> detailToTextField = new HashMap<>();

    /**
     * The <code>ProtocolProviderService</code> that this panel is associated with.
     */
    ProtocolProviderService protocolProvider;

    /**
     * The operation set giving access to the server stored account details.
     */
    private OperationSetServerStoredAccountInfo accountInfoOpSet;

    /*
     * imageUrlField contains the link to the image or a command to remove avatar
     */
    private EditText imageUrlField;

    private EditText urlField;
    private EditText aboutMeArea;
    private EditText ageField;
    private EditText birthDateField;
    private Button mApplyButton;

    private EditTextWatcher editTextWatcher;

    private DisplayNameDetail displayNameDetail;
    private FirstNameDetail firstNameDetail;
    private MiddleNameDetail middleNameDetail;
    private LastNameDetail lastNameDetail;
    private NicknameDetail nicknameDetail;
    private URLDetail urlDetail;
    private AddressDetail streetAddressDetail;
    private CityDetail cityDetail;
    private ProvinceDetail regionDetail;
    private PostalCodeDetail postalCodeDetail;
    private CountryDetail countryDetail;
    private PhoneNumberDetail phoneDetail;
    private WorkPhoneDetail workPhoneDetail;
    private MobilePhoneDetail mobilePhoneDetail;
    private EmailAddressDetail emailDetail;
    private WorkEmailAddressDetail workEmailDetail;
    private WorkOrganizationNameDetail organizationDetail;
    private JobTitleDetail jobTitleDetail;
    private AboutMeDetail aboutMeDetail;
    private GenderDetail genderDetail;
    private BirthDateDetail birthDateDetail;

    private ImageView avatarView;
    private ImageDetail avatarDetail;
    private DateFormat dateFormat;

    private ActivityResultLauncher<String> mGetContent;

    /**
     * Container for apply and cancel buttons; auto- hide when field text entry is active
     */
    private View mButtonContainer;

    private ImageView mCalenderButton;
    private SoftKeyboard softKeyboard;
    private long pDialogId;
    private boolean isRegistered;

    @Override
    protected void onCreate(android.os.Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Set the main layout
        setContentView(R.layout.account_info_presence_status);
        mButtonContainer = findViewById(R.id.button_Container);

        mGetContent = getAvatarContent();
        avatarView = findViewById(R.id.accountAvatar);
        registerForContextMenu(avatarView);
        avatarView.setOnClickListener(v -> openContextMenu(avatarView));

        // Get account ID from intent extras; and find account for given account ID
        String accountIDStr = getIntent().getStringExtra(INTENT_ACCOUNT_ID);
        AccountID accountID = AccountUtils.getAccountIDForUID(accountIDStr);

        if (accountID == null) {
            Timber.e("No account found for: %s", accountIDStr);
            finish();
            return;
        }

        mAccount = new Account(accountID, AppGUIActivator.bundleContext, this);
        mAccount.addAccountEventListener(this);
        protocolProvider = mAccount.getProtocolProvider();

        editTextWatcher = new EditTextWatcher();
        initPresenceStatus();
        initSoftKeyboard();

        accountInfoOpSet = protocolProvider.getOperationSet(OperationSetServerStoredAccountInfo.class);
        if (accountInfoOpSet != null) {
            initSummaryPanel();

            // May still be in logging if user enters preference edit immediately after account is enabled
            if (!protocolProvider.isRegistered()) {
                try {
                    Thread.sleep(3000);
                } catch (InterruptedException e) {
                    Timber.e("Account Registration State wait error: %s", protocolProvider.getRegistrationState());
                }
                Timber.d("Account Registration State: %s", protocolProvider.getRegistrationState());
            }

            isRegistered = protocolProvider.isRegistered();
            setFieldEditState(isRegistered);
            mApplyButton.setVisibility(isRegistered ? View.VISIBLE: View.GONE);
            if (!isRegistered) {
                Toast.makeText(this, R.string.accountinfo_not_registered_message, Toast.LENGTH_LONG).show();
            }
            else {
                loadDetails();
            }
            // Reset hasChanges to false on entry
            hasChanges = false;
        }
        getOnBackPressedDispatcher().addCallback(backPressedCallback);
    }

    @Override
    protected void onResume() {
        super.onResume();
        // setPrefTitle(R.string.plugin_accountinfo_TITLE);
        ActionBarUtil.setTitle(this, getString(R.string.accountinfo_title));

    }

    @Override
    protected void onStop() {
        super.onStop();
        if (ProgressDialog.isShowing(pDialogId))
            ProgressDialog.dismiss(pDialogId);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (softKeyboard != null) {
            softKeyboard.unRegisterSoftKeyboardCallback();
            softKeyboard = null;
        }
    }

    OnBackPressedCallback backPressedCallback = new OnBackPressedCallback(true) {
        @Override
        public void handleOnBackPressed() {
            if (hasChanges || hasStatusChanges) {
                checkUnsavedChanges();
            }
        }
    };

    /**
     * Create and initialize the view with actual values
     */
    private void initPresenceStatus() {
        this.accountPresence = mAccount.getPresenceOpSet();

        // Check for presence support
        if (accountPresence == null) {
            Toast.makeText(this, getString(R.string.presence_not_supported,
                    mAccount.getAccountName()), Toast.LENGTH_LONG).show();
            finish();
            return;
        }

        // Account properties
        ActionBarUtil.setSubtitle(this, mAccount.getAccountName());

        // Create spinner with status list
        Spinner statusSpinner = findViewById(R.id.presenceStatusSpinner);

        // Create list adapter
        List<PresenceStatus> presenceStatuses = accountPresence.getSupportedStatusSet();
        StatusListAdapter statusAdapter = new StatusListAdapter(this,
                R.layout.account_presence_status_row, presenceStatuses);
        statusSpinner.setAdapter(statusAdapter);

        // Selects current status
        PresenceStatus presenceStatus = accountPresence.getPresenceStatus();
        ActionBarUtil.setStatusIcon(this, presenceStatus.getStatusIcon());

        statusSpinner.setSelection(statusAdapter.getPosition(presenceStatus), false);
        statusSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parentView, View selectedItemView, int position, long id) {
                hasStatusChanges = true;
            }

            @Override
            public void onNothingSelected(AdapterView<?> parentView) {
            }
        });

        // Sets current status message
        EditText statusMessageEdit = findViewById(R.id.statusMessage);
        statusMessageEdit.setText(accountPresence.getCurrentStatusMessage());

        // Watch the text for any changes
        statusMessageEdit.addTextChangedListener(editTextWatcher);
    }

    /**
     * Initialized the main panel that contains all <code>ServerStoredDetails</code> and update
     * mapping between supported <code>ServerStoredDetails</code> and their respective
     * <code>EditText</code> that are used for modifying the details.
     */
    private void initSummaryPanel() {
        imageUrlField = findViewById(R.id.ai_ImageUrl);
        detailToTextField.put(ImageDetail.class, imageUrlField);

        EditText displayNameField = findViewById(R.id.ai_DisplayNameField);
        View displayNameContainer = findViewById(R.id.ai_DisplayName_Container);
        if (accountInfoOpSet.isDetailClassSupported(DisplayNameDetail.class)) {
            displayNameContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(DisplayNameDetail.class, displayNameField);
        }

        EditText firstNameField = findViewById(R.id.ai_FirstNameField);
        detailToTextField.put(FirstNameDetail.class, firstNameField);

        EditText middleNameField = findViewById(R.id.ai_MiddleNameField);
        detailToTextField.put(MiddleNameDetail.class, middleNameField);

        EditText lastNameField = findViewById(R.id.ai_LastNameField);
        detailToTextField.put(LastNameDetail.class, lastNameField);

        EditText nicknameField = findViewById(R.id.ai_NickNameField);
        View nickNameContainer = findViewById(R.id.ai_NickName_Container);
        if (accountInfoOpSet.isDetailClassSupported(NicknameDetail.class)) {
            nickNameContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(NicknameDetail.class, nicknameField);
        }

        urlField = findViewById(R.id.ai_URLField);
        View urlContainer = findViewById(R.id.ai_URL_Container);
        if (accountInfoOpSet.isDetailClassSupported(URLDetail.class)) {
            urlContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(URLDetail.class, urlField);
        }

        EditText genderField = findViewById(R.id.ai_GenderField);
        View genderContainer = findViewById(R.id.ai_Gender_Container);
        if (accountInfoOpSet.isDetailClassSupported(GenderDetail.class)) {
            genderContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(GenderDetail.class, genderField);
        }

        birthDateField = findViewById(R.id.ai_BirthDateField);
        detailToTextField.put(BirthDateDetail.class, birthDateField);
        birthDateField.setEnabled(false);

        ageField = findViewById(R.id.ai_AgeField);
        ageField.setEnabled(false);

        EditText streetAddressField = findViewById(R.id.ai_StreetAddressField);
        View streetAddressContainer = findViewById(R.id.ai_StreetAddress_Container);
        if (accountInfoOpSet.isDetailClassSupported(AddressDetail.class)) {
            streetAddressContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(AddressDetail.class, streetAddressField);
        }

        EditText cityField = findViewById(R.id.ai_CityField);
        View cityContainer = findViewById(R.id.ai_City_Container);
        if (accountInfoOpSet.isDetailClassSupported(CityDetail.class)) {
            cityContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(CityDetail.class, cityField);
        }

        EditText regionField = findViewById(R.id.ai_RegionField);
        View regionContainer = findViewById(R.id.ai_Region_Container);
        if (accountInfoOpSet.isDetailClassSupported(ProvinceDetail.class)) {
            regionContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(ProvinceDetail.class, regionField);
        }

        EditText postalCodeField = findViewById(R.id.ai_PostalCodeField);
        View postalCodeContainer = findViewById(R.id.ai_PostalCode_Container);
        if (accountInfoOpSet.isDetailClassSupported(PostalCodeDetail.class)) {
            postalCodeContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(PostalCodeDetail.class, postalCodeField);
        }

        EditText countryField = findViewById(R.id.ai_CountryField);
        View countryContainer = findViewById(R.id.ai_Country_Container);
        if (accountInfoOpSet.isDetailClassSupported(CountryDetail.class)) {
            countryContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(CountryDetail.class, countryField);
        }

        EditText emailField = findViewById(R.id.ai_EMailField);
        detailToTextField.put(EmailAddressDetail.class, emailField);

        EditText workEmailField = findViewById(R.id.ai_WorkEmailField);
        View workEmailContainer = findViewById(R.id.ai_WorkEmail_Container);
        if (accountInfoOpSet.isDetailClassSupported(WorkEmailAddressDetail.class)) {
            workEmailContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(WorkEmailAddressDetail.class, workEmailField);
        }

        EditText phoneField = findViewById(R.id.ai_PhoneField);
        detailToTextField.put(PhoneNumberDetail.class, phoneField);

        EditText workPhoneField = findViewById(R.id.ai_WorkPhoneField);
        View workPhoneContainer = findViewById(R.id.ai_WorkPhone_Container);
        if (accountInfoOpSet.isDetailClassSupported(WorkPhoneDetail.class)) {
            workPhoneContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(WorkPhoneDetail.class, workPhoneField);
        }

        EditText mobilePhoneField = findViewById(R.id.ai_MobilePhoneField);
        View mobileContainer = findViewById(R.id.ai_MobilePhone_Container);
        if (accountInfoOpSet.isDetailClassSupported(MobilePhoneDetail.class)) {
            mobileContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(MobilePhoneDetail.class, mobilePhoneField);
        }

        EditText organizationField = findViewById(R.id.ai_OrganizationNameField);
        View organizationNameContainer = findViewById(R.id.ai_OrganizationName_Container);
        if (accountInfoOpSet.isDetailClassSupported(WorkOrganizationNameDetail.class)) {
            organizationNameContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(WorkOrganizationNameDetail.class, organizationField);
        }

        EditText jobTitleField = findViewById(R.id.ai_JobTitleField);
        View jobDetailContainer = findViewById(R.id.ai_JobTitle_Container);
        if (accountInfoOpSet.isDetailClassSupported(JobTitleDetail.class)) {
            jobDetailContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(JobTitleDetail.class, jobTitleField);
        }

        aboutMeArea = findViewById(R.id.ai_AboutMeField);
        View aboutMeContainer = findViewById(R.id.ai_AboutMe_Container);
        if (accountInfoOpSet.isDetailClassSupported(AboutMeDetail.class)) {
            aboutMeContainer.setVisibility(View.VISIBLE);
            detailToTextField.put(AboutMeDetail.class, aboutMeArea);

            // aboutMeArea.setEnabled(false); cause auto-launch of softKeyboard creating problem
            InputFilter[] filterArray = new InputFilter[1];
            filterArray[0] = new InputFilter.LengthFilter(ContactInfoActivity.ABOUT_ME_MAX_CHARACTERS);
            aboutMeArea.setFilters(filterArray);
            aboutMeArea.setBackgroundResource(R.drawable.alpha_blue_01);
        }

        // Setup and initialize birthday calendar basic parameters
        dateFormat = DateFormat.getDateInstance();
        Calendar today = Calendar.getInstance();
        int mYear = today.get(Calendar.YEAR);
        int mMonth = today.get(Calendar.MONTH);
        int mDay = today.get(Calendar.DAY_OF_MONTH);
        mDatePicker = findViewById(R.id.datePicker);
        mDatePicker.init(mYear, mMonth, mDay, this);

        mCalenderButton = findViewById(R.id.datePickerBtn);
        mCalenderButton.setEnabled(false);
        mCalenderButton.setOnClickListener(v -> {
            if (mDatePicker.getVisibility() == View.GONE)
                mDatePicker.setVisibility(View.VISIBLE);
            else
                mDatePicker.setVisibility(View.GONE);
        });

        mApplyButton = findViewById(R.id.button_Apply);
        mApplyButton.setOnClickListener(v -> {
            if (hasChanges || hasStatusChanges)
                launchApplyProgressDialog();
            else
                finish();
        });

        Button mCancelButton = findViewById(R.id.button_Cancel);
        mCancelButton.setOnClickListener(v -> checkUnsavedChanges());
    }

    /**
     * Setup textFields' editable state and addTextChangedListener if enabled
     */
    private void setFieldEditState(boolean editState) {
        // Setup textFields' editable state and addTextChangedListener if enabled
        boolean isEditable;
        for (Class<? extends GenericDetail> classDetail : detailToTextField.keySet()) {
            EditText field = detailToTextField.get(classDetail);
            isEditable = editState && accountInfoOpSet.isDetailClassEditable(classDetail);

            if (classDetail.equals(BirthDateDetail.class))
                mCalenderButton.setEnabled(isEditable);
            else if (classDetail.equals(ImageDetail.class))
                avatarView.setEnabled(isEditable);
            else {
                if (field != null) {
                    field.setEnabled(isEditable);
                    if (isEditable)
                        field.addTextChangedListener(editTextWatcher);
                }
            }
        }
    }

    /**
     * check for any unsaved changes and alert user
     */
    private void checkUnsavedChanges() {
        if (hasChanges) {
            DialogActivity.showConfirmDialog(this,
                    R.string.unsaved_changes_title,
                    R.string.unsaved_changes,
                    R.string.save, this);
        }
        else {
            finish();
        }
    }

    /**
     * Fired when user clicks the dialog's confirm button.
     *
     * @param dialog source <code>DialogActivity</code>.
     */
    public boolean onConfirmClicked(DialogActivity dialog) {
        return mApplyButton.performClick();
    }

    /**
     * Fired when user dismisses the dialog.
     *
     * @param dialog source <code>DialogActivity</code>
     */
    public void onDialogCancelled(DialogActivity dialog) {
        finish();
    }

    @Override
    public void onDateChanged(DatePicker datePicker, int year, int monthOfYear, int dayOfMonth) {
        Calendar mDate = Calendar.getInstance();

        int age = mDate.get(Calendar.YEAR) - year;
        if (mDate.get(Calendar.MONTH) < monthOfYear)
            age--;
        if ((mDate.get(Calendar.MONTH) == monthOfYear)
                && (mDate.get(Calendar.DAY_OF_MONTH) < dayOfMonth))
            age--;

        String ageDetail = Integer.toString(age);
        ageField.setText(ageDetail);

        mDate.set(year, monthOfYear, dayOfMonth);
        birthDateField.setText(dateFormat.format(mDate.getTime()));

        // internal program call is with dialog == null
        hasChanges = (datePicker != null);
    }

    /**
     * Loads all <code>ServerStoredDetails</code> which are currently supported by this plugin.
     * Note that some <code>OperationSetServerStoredAccountInfo</code> implementations may support
     * details that are not supported by this plugin. In this case they will not be loaded.
     */
    private void loadDetails() {
        try (ExecutorService sThread = Executors.newSingleThreadExecutor()) {
            sThread.execute(() -> {
                final Iterator<GenericDetail> allDetails = accountInfoOpSet.getAllAvailableDetails();

                // ANR from FFR: Seems not need to run on UI Thread!
                // runOnUiThread(() -> {
                    if (allDetails != null) {
                        while (allDetails.hasNext()) {
                            GenericDetail detail = allDetails.next();
                            loadDetail(detail);
                        }
                    }
                    // get user avatar via XEP-0084
                    getUserAvatarData();
                    // Logger.getLogger("LoadDetail").warn("Load Details done!!!");
                });
            // });

            sThread.shutdown();
            try {
                if (!sThread.awaitTermination(1500, TimeUnit.MILLISECONDS)) {
                    Timber.w("DetailsLoadWorker shutDown on timeout!");
                    sThread.shutdownNow();
                }
            } catch (InterruptedException ex) {
                sThread.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    /**
     * Loads a single <code>GenericDetail</code> obtained from the
     * <code>OperationSetServerStoredAccountInfo</code> into this plugin.
     * <p>
     * If VcardTemp contains <photo/>, it will be converted to XEP-0084 avatarData &
     * avatarMetadata, and remove it from VCardTemp.
     *
     * @param detail the loaded detail for extraction.
     */
    private void loadDetail(GenericDetail detail) {
        if (detail instanceof BirthDateDetail) {
            birthDateDetail = (BirthDateDetail) detail;
            Object objBirthDate = birthDateDetail.getDetailValue();

            // default to today if birthDate is null
            if (objBirthDate instanceof Calendar) {
                Calendar birthDate = (Calendar) objBirthDate;

                int bYear = birthDate.get(Calendar.YEAR);
                int bMonth = birthDate.get(Calendar.MONTH);
                int bDay = birthDate.get(Calendar.DAY_OF_MONTH);
                // Preset calendarDatePicker date
                mDatePicker.updateDate(bYear, bMonth, bDay);

                // Update BirthDate and Age
                onDateChanged(null, bYear, bMonth, bDay);
            }
            else if (objBirthDate != null) {
                birthDateField.setText((String) objBirthDate);
            }
            return;
        }

        Class<? extends GenericDetail> classDetail = detail.getClass();
        if (classDetail.equals(AboutMeDetail.class)) {
            aboutMeDetail = (AboutMeDetail) detail;
            aboutMeArea.setText((String) detail.getDetailValue());
            return;
        }

        EditText field = detailToTextField.get(classDetail);
        if (field != null) {
            if (detail instanceof ImageDetail) {
                avatarDetail = (ImageDetail) detail;
                byte[] avatarImage = avatarDetail.getBytes();
                Bitmap bitmap = BitmapFactory.decodeByteArray(avatarImage, 0, avatarImage.length);
                avatarView.setImageBitmap(bitmap);
            }
            else if (detail instanceof URLDetail) {
                urlDetail = (URLDetail) detail;
                urlField.setText(urlDetail.getURL().toString());
            }
            else {
                Object obj = detail.getDetailValue();
                if (obj instanceof String)
                    field.setText((String) obj);
                else if (obj != null)
                    field.setText(obj.toString());

                if (classDetail.equals(DisplayNameDetail.class))
                    displayNameDetail = (DisplayNameDetail) detail;
                else if (classDetail.equals(FirstNameDetail.class))
                    firstNameDetail = (FirstNameDetail) detail;
                else if (classDetail.equals(MiddleNameDetail.class))
                    middleNameDetail = (MiddleNameDetail) detail;
                else if (classDetail.equals(LastNameDetail.class))
                    lastNameDetail = (LastNameDetail) detail;
                else if (classDetail.equals(NicknameDetail.class))
                    nicknameDetail = (NicknameDetail) detail;
                else if (classDetail.equals(GenderDetail.class))
                    genderDetail = (GenderDetail) detail;
                else if (classDetail.equals(AddressDetail.class))
                    streetAddressDetail = (AddressDetail) detail;
                else if (classDetail.equals(CityDetail.class))
                    cityDetail = (CityDetail) detail;
                else if (classDetail.equals(ProvinceDetail.class))
                    regionDetail = (ProvinceDetail) detail;
                else if (classDetail.equals(PostalCodeDetail.class))
                    postalCodeDetail = (PostalCodeDetail) detail;
                else if (classDetail.equals(CountryDetail.class))
                    countryDetail = (CountryDetail) detail;
                else if (classDetail.equals(PhoneNumberDetail.class))
                    phoneDetail = (PhoneNumberDetail) detail;
                else if (classDetail.equals(WorkPhoneDetail.class))
                    workPhoneDetail = (WorkPhoneDetail) detail;
                else if (classDetail.equals(MobilePhoneDetail.class))
                    mobilePhoneDetail = (MobilePhoneDetail) detail;
                else if (classDetail.equals(EmailAddressDetail.class))
                    emailDetail = (EmailAddressDetail) detail;
                else if (classDetail.equals(WorkEmailAddressDetail.class))
                    workEmailDetail = (WorkEmailAddressDetail) detail;
                else if (classDetail.equals(WorkOrganizationNameDetail.class))
                    organizationDetail = (WorkOrganizationNameDetail) detail;
                else if (classDetail.equals(JobTitleDetail.class))
                    jobTitleDetail = (JobTitleDetail) detail;
            }
        }
    }

    /**
     * Retrieve avatar via XEP-0084 and override vCard <photo/> content if avatarImage not null
     */
    private void getUserAvatarData() {
        byte[] avatarImage = AvatarManager.getAvatarImageByJid(mAccount.getJid().asBareJid());
        if (avatarImage != null && avatarImage.length > 0) {
            Bitmap bitmap = BitmapFactory.decodeByteArray(avatarImage, 0, avatarImage.length);
            avatarView.setImageBitmap(bitmap);
        }
        else {
            avatarView.setImageResource(R.drawable.person_photo);
        }
    }

    /**
     * Attempts to upload all <code>ServerStoredDetails</code> on the server using
     * <code>OperationSetServerStoredAccountInfo</code>
     */
    private void SubmitChangesAction() {
        if (!isRegistered || !hasChanges)
            return;

        if (accountInfoOpSet.isDetailClassSupported(ImageDetail.class)) {
            String sCommand = ViewUtil.toString(imageUrlField);
            if (sCommand != null) {
                ImageDetail newDetail;

                /*
                 * command to remove avatar photo from vCardTemp. XEP-0084 support will always
                 * init imageUrlField = AVATAR_ICON_REMOVE
                 */
                if (AVATAR_ICON_REMOVE.equals(sCommand)) {
                    newDetail = new ImageDetail("avatar", new byte[0]);
                    changeDetail(avatarDetail, newDetail);
                }
                else {
                    try {
                        Uri imageUri = Uri.parse(sCommand);
                        Bitmap bmp = AppImageUtil.scaledBitmapFromContentUri(this,
                                imageUri, AVATAR_PREFERRED_SIZE, AVATAR_PREFERRED_SIZE);

                        // Convert to bytes if not null
                        if (bmp != null) {
                            final byte[] rawImage = AppImageUtil.convertToBytes(bmp, 100);

                            newDetail = new ImageDetail("avatar", rawImage);
                            changeDetail(avatarDetail, newDetail);
                        }
                        else
                            showAvatarChangeError();
                    } catch (IOException e) {
                        Timber.e(e, "%s", e.getMessage());
                        showAvatarChangeError();
                    }
                }
            }
        }

        if (accountInfoOpSet.isDetailClassSupported(DisplayNameDetail.class)) {
            String text = getText(DisplayNameDetail.class);

            DisplayNameDetail newDetail = null;
            if (text != null)
                newDetail = new DisplayNameDetail(text);

            if (displayNameDetail != null || newDetail != null)
                changeDetail(displayNameDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(FirstNameDetail.class)) {
            String text = getText(FirstNameDetail.class);

            FirstNameDetail newDetail = null;
            if (text != null)
                newDetail = new FirstNameDetail(text);

            if (firstNameDetail != null || newDetail != null)
                changeDetail(firstNameDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(MiddleNameDetail.class)) {
            String text = getText(MiddleNameDetail.class);

            MiddleNameDetail newDetail = null;
            if (text != null)
                newDetail = new MiddleNameDetail(text);

            if (middleNameDetail != null || newDetail != null)
                changeDetail(middleNameDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(LastNameDetail.class)) {
            String text = getText(LastNameDetail.class);
            LastNameDetail newDetail = null;

            if (text != null)
                newDetail = new LastNameDetail(text);

            if (lastNameDetail != null || newDetail != null)
                changeDetail(lastNameDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(NicknameDetail.class)) {
            String text = getText(NicknameDetail.class);

            NicknameDetail newDetail = null;
            if (text != null)
                newDetail = new NicknameDetail(text);

            if (nicknameDetail != null || newDetail != null)
                changeDetail(nicknameDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(URLDetail.class)) {
            String text = getText(URLDetail.class);

            URL url;
            URLDetail newDetail = null;

            if (text != null) {
                try {
                    url = new URL(text);
                    newDetail = new URLDetail("URL", url);
                } catch (MalformedURLException e1) {
                    Timber.d("URL field has malformed URL; save as text instead.");
                    newDetail = new URLDetail("URL", text);
                }
            }
            if (urlDetail != null || newDetail != null)
                changeDetail(urlDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(GenderDetail.class)) {
            String text = getText(GenderDetail.class);
            GenderDetail newDetail = null;

            if (text != null)
                newDetail = new GenderDetail(text);

            if (genderDetail != null || newDetail != null)
                changeDetail(genderDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(BirthDateDetail.class)) {
            String text = ViewUtil.toString(birthDateField);
            BirthDateDetail newDetail = null;

            if (text != null) {
                Calendar birthDate = Calendar.getInstance();
                try {
                    Date mDate = dateFormat.parse(text);
                    birthDate.setTime(mDate);
                    newDetail = new BirthDateDetail(birthDate);
                } catch (ParseException e) {
                    // Save as String value
                    newDetail = new BirthDateDetail(text);
                }
            }
            if (birthDateDetail != null || newDetail != null)
                changeDetail(birthDateDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(AddressDetail.class)) {
            String text = getText(AddressDetail.class);

            AddressDetail newDetail = null;
            if (text != null)
                newDetail = new AddressDetail(text);

            if (streetAddressDetail != null || newDetail != null)
                changeDetail(streetAddressDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(CityDetail.class)) {
            String text = getText(CityDetail.class);

            CityDetail newDetail = null;
            if (text != null)
                newDetail = new CityDetail(text);

            if (cityDetail != null || newDetail != null)
                changeDetail(cityDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(ProvinceDetail.class)) {
            String text = getText(ProvinceDetail.class);

            ProvinceDetail newDetail = null;
            if (text != null)
                newDetail = new ProvinceDetail(text);

            if (regionDetail != null || newDetail != null)
                changeDetail(regionDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(PostalCodeDetail.class)) {
            String text = getText(PostalCodeDetail.class);

            PostalCodeDetail newDetail = null;
            if (text != null)
                newDetail = new PostalCodeDetail(text);

            if (postalCodeDetail != null || newDetail != null)
                changeDetail(postalCodeDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(CountryDetail.class)) {
            String text = getText(CountryDetail.class);

            CountryDetail newDetail = null;
            if (text != null)
                newDetail = new CountryDetail(text);

            if (countryDetail != null || newDetail != null)
                changeDetail(countryDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(EmailAddressDetail.class)) {
            String text = getText(EmailAddressDetail.class);

            EmailAddressDetail newDetail = null;
            if (text != null)
                newDetail = new EmailAddressDetail(text);

            if (emailDetail != null || newDetail != null)
                changeDetail(emailDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(WorkEmailAddressDetail.class)) {
            String text = getText(WorkEmailAddressDetail.class);

            WorkEmailAddressDetail newDetail = null;
            if (text != null)
                newDetail = new WorkEmailAddressDetail(text);

            if (workEmailDetail != null || newDetail != null)
                changeDetail(workEmailDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(PhoneNumberDetail.class)) {
            String text = getText(PhoneNumberDetail.class);

            PhoneNumberDetail newDetail = null;
            if (text != null)
                newDetail = new PhoneNumberDetail(text);

            if (phoneDetail != null || newDetail != null)
                changeDetail(phoneDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(WorkPhoneDetail.class)) {
            String text = getText(WorkPhoneDetail.class);

            WorkPhoneDetail newDetail = null;
            if (text != null)
                newDetail = new WorkPhoneDetail(text);

            if (workPhoneDetail != null || newDetail != null)
                changeDetail(workPhoneDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(MobilePhoneDetail.class)) {
            String text = getText(MobilePhoneDetail.class);

            MobilePhoneDetail newDetail = null;
            if (text != null)
                newDetail = new MobilePhoneDetail(text);

            if (mobilePhoneDetail != null || newDetail != null)
                changeDetail(mobilePhoneDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(WorkOrganizationNameDetail.class)) {
            String text = getText(WorkOrganizationNameDetail.class);

            WorkOrganizationNameDetail newDetail = null;
            if (text != null)
                newDetail = new WorkOrganizationNameDetail(text);

            if (organizationDetail != null || newDetail != null)
                changeDetail(organizationDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(JobTitleDetail.class)) {
            String text = getText(JobTitleDetail.class);

            JobTitleDetail newDetail = null;
            if (text != null)
                newDetail = new JobTitleDetail(text);

            if (jobTitleDetail != null || newDetail != null)
                changeDetail(jobTitleDetail, newDetail);
        }

        if (accountInfoOpSet.isDetailClassSupported(AboutMeDetail.class)) {
            String text = ViewUtil.toString(aboutMeArea);

            AboutMeDetail newDetail = null;
            if (text != null)
                newDetail = new AboutMeDetail(text);

            if (aboutMeDetail != null || newDetail != null)
                changeDetail(aboutMeDetail, newDetail);
        }

        try {
            //mainScrollPane.getVerticalScrollBar().setValue(0);
            accountInfoOpSet.save();
        } catch (OperationFailedException e1) {
            showAvatarChangeError();
        }
    }

    /**
     * get the class's editText string value or null (length == 0)
     *
     * @param className Class Name
     *
     * @return String or null if string length == 0
     */
    private String getText(Class<? extends GenericDetail> className) {
        EditText editText = detailToTextField.get(className);
        return ViewUtil.toString(editText);
    }

    /**
     * A helper method to decide whether to add new
     * <code>ServerStoredDetails</code> or to replace an old one.
     *
     * @param oldDetail the detail to be replaced.
     * @param newDetail the replacement.
     */
    private void changeDetail(GenericDetail oldDetail, GenericDetail newDetail) {
        try {
            if (newDetail == null) {
                accountInfoOpSet.removeDetail(oldDetail);
            }
            else if (oldDetail == null) {
                accountInfoOpSet.addDetail(newDetail);
            }
            else {
                accountInfoOpSet.replaceDetail(oldDetail, newDetail);
            }
        } catch (ArrayIndexOutOfBoundsException | OperationFailedException e1) {
            Timber.d("Failed to update account details.%s %s", mAccount.getAccountName(), e1.getMessage());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);
        getMenuInflater().inflate(R.menu.presence_status_menu, menu);
        return true;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        if (id == R.id.remove) {
            AccountDeleteDialog.create(this, mAccount, accID -> {
                // Prevent from submitting status
                hasStatusChanges = false;
                hasChanges = false;
                finish();
            });
            return true;
        }
        else if (id == R.id.account_settings) {
            Intent preferences = AccountPreferenceActivity.getIntent(this, mAccount.getAccountID());
            startActivity(preferences);
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    public void onCreateContextMenu(ContextMenu menu, View v, ContextMenu.ContextMenuInfo menuInfo) {
        super.onCreateContextMenu(menu, v, menuInfo);
        if (v.getId() == R.id.accountAvatar) {
            getMenuInflater().inflate(R.menu.avatar_menu, menu);
        }
    }

    @Override
    public boolean onContextItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.avatar_ChoosePicture:
                onAvatarClicked(avatarView);
                return true;

            case R.id.avatar_RemovePicture:
                imageUrlField.setText(AVATAR_ICON_REMOVE);
                avatarView.setImageResource(R.drawable.person_photo);
                hasChanges = true;
                return true;

            case R.id.avatar_Cancel:
                return true;

            default:
                return super.onContextItemSelected(item);
        }
    }

    /**
     * Method mapped to the avatar image clicked event. It starts the select image {@link Intent}
     *
     * @param avatarView the {@link View} that has been clicked
     */
    public void onAvatarClicked(View avatarView) {
        if (mAccount.getAvatarOpSet() == null) {
            Timber.w("Avatar operation set is not supported by %s", mAccount.getAccountName());
            showAvatarChangeError();
            return;
        }
        mGetContent.launch("image/*");
    }

    /**
     * A contract specifying that an activity can be called with an input of type I
     * and produce an output of type O
     *
     * @return an instant of ActivityResultLauncher<String>
     *
     * @see ActivityResultCaller
     */
    private ActivityResultLauncher<String> getAvatarContent() {
        return registerForActivityResult(new ActivityResultContracts.GetContent(), uri -> {
            if (uri == null) {
                Timber.e("No image data selected for avatar!");
                showAvatarChangeError();
            }
            else {
                String fileName = "cropImage";
                File tmpFile = new File(this.getCacheDir(), fileName);
                Uri destinationUri = Uri.fromFile(tmpFile);

                UCrop.of(uri, destinationUri)
                        .withAspectRatio(1, 1)
                        .withMaxResultSize(CROP_MAX_SIZE, CROP_MAX_SIZE)
                        .start(this);
            }
        });
    }

    /**
     * Method handles callbacks from external {@link Intent} that retrieve avatar image
     *
     * @param requestCode the request code
     * @param resultCode the result code
     * @param data the source {@link Intent} that returns the result
     */
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode != RESULT_OK)
            return;

        switch (requestCode) {
            case UCrop.REQUEST_CROP:
                final Uri resultUri = UCrop.getOutput(data);
                if (resultUri == null)
                    break;
                try {
                    Bitmap bmp = AppImageUtil.scaledBitmapFromContentUri(this, resultUri,
                            AVATAR_PREFERRED_SIZE, AVATAR_PREFERRED_SIZE);
                    if (bmp == null) {
                        Timber.e("Failed to obtain bitmap from: %s", data);
                        showAvatarChangeError();
                    }
                    else {
                        avatarView.setImageBitmap(bmp);
                        imageUrlField.setText(resultUri.toString());
                        hasChanges = true;
                    }
                } catch (IOException e) {
                    Timber.e(e, "%s", e.getMessage());
                    showAvatarChangeError();
                }
                break;

            case UCrop.RESULT_ERROR:
                final Throwable cropError = UCrop.getError(data);
                String errMsg = "Image crop error: ";
                if (cropError != null)
                    errMsg += cropError.getMessage();
                Timber.e("%s", errMsg);
                showAvatarChangeError();
                break;
        }
    }

    private void showAvatarChangeError() {
        DialogActivity.showDialog(this,
                R.string.error, R.string.avatar_set_error, mAccount.getAccountName());
    }

    /**
     * Method starts a new Thread and publishes the status
     *
     * @param status {@link PresenceStatus} to be set
     * @param text the status message
     */
    private void publishStatus(final PresenceStatus status, final String text) {
        new Thread(() -> {
            try {
                // Try to publish selected status
                Timber.d("Publishing status %s msg: %s", status, text);
                GlobalStatusService globalStatus
                        = ServiceUtils.getService(AppGUIActivator.bundleContext, GlobalStatusService.class);

                ProtocolProviderService pps = mAccount.getProtocolProvider();
                // cmeng: set state to false to force it to execute offline->online
                if (globalStatus != null)
                    globalStatus.publishStatus(pps, status, false);
                if (pps.isRegistered())
                    accountPresence.publishPresenceStatus(status, text);
            } catch (Exception e) {
                Timber.e(e);
            }
        }).start();
    }

    /**
     * Fired when the {@link #mAccount} has changed and the UI need to be updated
     *
     * @param eventObject the instance that has been changed
     * cmeng: may not be required anymore with new implementation
     */
    @Override
    public void onChangeEvent(final AccountEvent eventObject) {
        if (eventObject.getEventType() != AccountEvent.AVATAR_CHANGE) {
            return;
        }
        runOnUiThread(() -> {
            Account account = eventObject.getSource();
            avatarView.setImageDrawable(account.getAvatarIcon());
        });
    }

    /**
     * Checks if there are any uncommitted changes and applies them eventually
     */
    private void commitStatusChanges() {
        if (hasStatusChanges) {
            Spinner statusSpinner = findViewById(R.id.presenceStatusSpinner);

            PresenceStatus selectedStatus = (PresenceStatus) statusSpinner.getSelectedItem();
            String statusMessageText = ViewUtil.toString(findViewById(R.id.statusMessage));

            if ((selectedStatus.getStatus() == PresenceStatus.OFFLINE) && (hasChanges)) {
                // abort all account info changes if user goes offline
                hasChanges = false;

                if (ProgressDialog.isShowing(pDialogId)) {
                    ProgressDialog.setMessage(pDialogId, getString(R.string.accountinfo_discard_change));
                }
            }
            // Publish status in new thread
            publishStatus(selectedStatus, statusMessageText);
        }
    }

    /**
     * Progressing dialog while applying changes to account info/status
     * Auto cancel the dialog at end of applying cycle
     */
    public void launchApplyProgressDialog() {
        pDialogId = ProgressDialog.show(this, getString(R.string.please_wait),
                getString(R.string.apply_changes), true);
        new Thread(() -> {
            try {
                commitStatusChanges();
                SubmitChangesAction();
                // too fast to be viewed by user at times - so pause for 2.0 seconds
                Thread.sleep(2000);
            } catch (Exception ex) {
                Timber.w("Progress Dialog: %s", ex.getMessage());
            }
            ProgressDialog.dismiss(pDialogId);
            finish();
        }).start();
    }

    /*
     * cmeng 20191118 - manipulate android softKeyboard may cause problem in >= android-9 (API-28)
     * all view Dimensions are incorrectly init when soffKeyboard is auto launched.
     * aboutMeArea.setEnabled(false); cause softKeyboard to auto-launch
     *
     * SoftKeyboard event handler to show/hide view buttons to give more space for fields' text entry.
     * # init to handle when softKeyboard is hided/shown
     */
    private void initSoftKeyboard() {
        LinearLayout mainLayout = findViewById(R.id.accountInfo_layout);
        InputMethodManager imm = (InputMethodManager) getSystemService(Service.INPUT_METHOD_SERVICE);

        /*  Instantiate and pass a callback */
        softKeyboard = new SoftKeyboard(mainLayout, imm);
        softKeyboard.setSoftKeyboardCallback(this);
    }

    // Events to show or hide buttons for bigger view space for text entry
    @Override
    public void onSoftKeyboardHide() {
        runOnUiThread(() -> mButtonContainer.setVisibility(View.VISIBLE));
    }

    @Override
    public void onSoftKeyboardShow() {
        runOnUiThread(() -> mButtonContainer.setVisibility(View.GONE));
    }

    private class EditTextWatcher implements TextWatcher {

        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            // Ignore
        }

        public void onTextChanged(CharSequence s, int start, int before, int count) {
        }

        public void afterTextChanged(Editable s) {
            hasChanges = true;
        }
    }
}