/*
 * Jitsi, the OpenSource Java VoIP and Instant Messaging client.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */
package net.java.sip.communicator.service.protocol.event;

import net.java.sip.communicator.service.protocol.Contact;
import net.java.sip.communicator.service.protocol.ContactGroup;
import net.java.sip.communicator.service.protocol.ProtocolProviderService;

import java.util.EventObject;

/**
 * Events of this class represent the fact that a server stored subscription/contact has been moved
 * from one server stored group to another. Such events are only generated by implementations of
 * OperationSetPersistentPresence as non persistent presence operation sets do not support the
 * notion of server stored groups.
 *
 * @author Emil Ivov
 */
public class SubscriptionMovedEvent extends EventObject
{
	/**
	 * Serial version UID.
	 */
	private static final long serialVersionUID = 0L;

	private ContactGroup oldParent = null;
	private ContactGroup newParent = null;
	private ProtocolProviderService sourceProvider = null;

	/**
	 * Creates an event instance with the specified source contact and old and new parent.
	 *
	 * @param sourceContact
	 *        the <code>Contact</code> that has been moved.
	 * @param sourceProvider
	 *        a reference to the <code>ProtocolProviderService</code> that the source <code>Contact</code>
	 *        belongs to.
	 * @param oldParent
	 *        the <code>ContactGroup</code> that has previously been the parent
	 * @param newParent
	 *        the new <code>ContactGroup</code> parent of <code>sourceContact</code>
	 */
	public SubscriptionMovedEvent(Contact sourceContact, ProtocolProviderService sourceProvider,
		ContactGroup oldParent, ContactGroup newParent)
	{
		super(sourceContact);
		this.oldParent = oldParent;
		this.newParent = newParent;
		this.sourceProvider = sourceProvider;
	}

	/**
	 * Returns a reference to the contact that has been moved.
	 * 
	 * @return a reference to the <code>Contact</code> that has been moved.
	 */
	public Contact getSourceContact()
	{
		return (Contact) getSource();
	}

	/**
	 * Returns a reference to the ContactGroup that contained the source contact before it was
	 * moved.
	 * 
	 * @return a reference to the previous <code>ContactGroup</code> parent of the source
	 *         <code>Contact</code>.
	 */
	public ContactGroup getOldParentGroup()
	{
		return oldParent;
	}

	/**
	 * Returns a reference to the ContactGroup that currently contains the source contact.
	 *
	 * @return a reference to the current <code>ContactGroup</code> parent of the source
	 *         <code>Contact</code>.
	 */
	public ContactGroup getNewParentGroup()
	{
		return newParent;
	}

	/**
	 * Returns the provider that the source contact belongs to.
	 * 
	 * @return the provider that the source contact belongs to.
	 */
	public ProtocolProviderService getSourceProvider()
	{
		return sourceProvider;
	}

	/**
	 * Returns a String representation of this ContactPresenceStatusChangeEvent
	 *
	 * @return A a String representation of this SubscriptionMovedEvent.
	 */
	@Override
	public String toString()
	{
		StringBuffer buff = new StringBuffer("SubscriptionMovedEvent-[ ContactID=");
		buff.append(getSourceContact().getAddress());
		buff.append(", OldParentGroup=").append(getOldParentGroup().getGroupName());
		buff.append(", NewParentGroup=").append(getNewParentGroup().getGroupName());
		return buff.toString();
	}

}
