/*
 * Jitsi, the OpenSource Java VoIP and Instant Messaging client.
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package net.java.sip.communicator.service.gui;

import net.java.sip.communicator.service.protocol.OperationSet;
import net.java.sip.communicator.service.protocol.PresenceStatus;
import net.java.sip.communicator.service.protocol.ProtocolProviderService;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * The <code>UIContactDetail</code> corresponds to a particular contact detail,
 * phone number, IM identifier, email, etc. which has it's preferred mode of
 * transport <code>ProtocolProviderService</code>.
 *
 * @author Yana Stamcheva
 * @author Eng Chong Meng
 */
public abstract class UIContactDetail
{
    /**
     * The prefix to be used when calling this contact detail.
     */
    private String prefix;

    /**
     * The address of this detail.
     */
    private final String address;

    /**
     * The display name of this detail.
     */
    private final String displayName;

    /**
     * The <code>ProtocolProviderService</code> corresponding to this detail.
     */
    private Map<Class<? extends OperationSet>, ProtocolProviderService> preferredProviders;

    /**
     * The protocol to be used for this contact detail if no protocol provider is set.
     */
    private Map<Class<? extends OperationSet>, String> preferredProtocols;

    /**
     * The collection of labels associated with this detail.
     */
    private final Collection<String> labels;

    /**
     * The category of the underlying contact detail.
     */
    private final String category;

    /**
     * The underlying object that this class is wrapping
     */
    private final Object descriptor;

    /**
     * Creates a <code>UIContactDetail</code> by specifying the contact
     * <code>address</code>, the <code>displayName</code> and <code>preferredProvider</code>.
     *
     * @param address the contact address
     * @param displayName the contact display name
     * @param descriptor the underlying object that this class is wrapping
     */
    public UIContactDetail(
            String address,
            String displayName,
            Object descriptor)
    {
        this(address,
                displayName,
                null,
                null,
                null,
                null,
                descriptor);
    }

    /**
     * Creates a <code>UIContactDetail</code> by specifying the contact
     * <code>address</code>, the <code>displayName</code> and <code>preferredProvider</code>.
     *
     * @param address the contact address
     * @param displayName the contact display name
     * @param category the category of the underlying contact detail
     * @param labels the collection of labels associated with this detail
     * @param preferredProviders the preferred protocol provider
     * @param preferredProtocols the preferred protocol if no protocol provider is set
     * @param descriptor the underlying object that this class is wrapping
     */
    public UIContactDetail(
            String address,
            String displayName,
            String category,
            Collection<String> labels,
            Map<Class<? extends OperationSet>, ProtocolProviderService> preferredProviders,
            Map<Class<? extends OperationSet>, String> preferredProtocols,
            Object descriptor)
    {
        this.address = address;
        this.displayName = displayName;
        this.category = category;
        this.labels = labels;
        this.preferredProviders = preferredProviders;
        this.preferredProtocols = preferredProtocols;
        this.descriptor = descriptor;
    }

    /**
     * Returns the display name of this detail.
     *
     * @return the display name of this detail
     */
    public String getDisplayName()
    {
        return displayName;
    }

    /**
     * Returns the address of this detail.
     *
     * @return the address of this detail
     */
    public String getAddress()
    {
        if (prefix != null && prefix.trim().length() >= 0)
            return prefix + address;

        return address;
    }

    /**
     * Returns the category of the underlying detail.
     *
     * @return the category of the underlying detail
     */
    public String getCategory()
    {
        return category;
    }

    /**
     * Returns an iterator over the collection of labels associated with this detail.
     *
     * @return an iterator over the collection of labels associated with this detail
     */
    public Iterator<String> getLabels()
    {
        if (labels != null)
            return labels.iterator();

        return null;
    }

    /**
     * Returns the protocol provider preferred for contacting this detail for
     * the given <code>OperationSet</code> class.
     *
     * @param opSetClass the <code>OperationSet</code> class for which we're looking for provider
     * @return the protocol provider preferred for contacting this detail
     */
    public ProtocolProviderService getPreferredProtocolProvider(Class<? extends OperationSet> opSetClass)
    {
        if (preferredProviders != null)
            return preferredProviders.get(opSetClass);
        return null;
    }

    /**
     * Adds a preferred protocol provider for a given OperationSet class.
     *
     * @param opSetClass the <code>OperationSet</code> class for which we're looking for protocol
     * @param protocolProvider the preferred protocol provider to add
     */
    public void addPreferredProtocolProvider(
            Class<? extends OperationSet> opSetClass, ProtocolProviderService protocolProvider)
    {
        if (preferredProviders == null)
            preferredProviders = new HashMap<>();

        preferredProviders.put(opSetClass, protocolProvider);
    }

    /**
     * Returns the name of the protocol preferred for contacting this detail for
     * the given <code>OperationSet</code> class if no preferred protocol provider
     * is set.
     *
     * @param opSetClass the <code>OperationSet</code> class for which we're looking
     * for protocol
     * @return the name of the protocol preferred for contacting this detail
     */
    public String getPreferredProtocol(Class<? extends OperationSet> opSetClass)
    {
        if (preferredProtocols != null)
            return preferredProtocols.get(opSetClass);
        return null;
    }

    /**
     * Adds a preferred protocol for a given OperationSet class.
     *
     * @param opSetClass the <code>OperationSet</code> class for which we're looking for protocol
     * @param protocol the preferred protocol to add
     */
    public void addPreferredProtocol(Class<? extends OperationSet> opSetClass, String protocol)
    {
        if (preferredProtocols == null)
            preferredProtocols = new HashMap<>();

        preferredProtocols.put(opSetClass, protocol);
    }

    /**
     * Returns the prefix to be used when calling this contact detail.
     *
     * @return the prefix to be used when calling this contact detail
     */
    public String getPrefix()
    {
        return prefix;
    }

    /**
     * Sets the prefix to be used when calling this contact detail.
     *
     * @param prefix the prefix to be used when calling this contact detail
     */
    public void setPrefix(String prefix)
    {
        this.prefix = prefix;
    }

    /**
     * Returns the underlying object that this class is wrapping
     *
     * @return the underlying object that this class is wrapping
     */
    public Object getDescriptor()
    {
        return descriptor;
    }

    /**
     * Returns the <code>PresenceStatus</code> of this <code>ContactDetail</code> or
     * null if the detail doesn't support presence.
     *
     * @return the <code>PresenceStatus</code> of this <code>ContactDetail</code> or
     * null if the detail doesn't support presence
     */
    public abstract PresenceStatus getPresenceStatus();
}
