@file:OptIn(ExperimentalMaterial3Api::class)

package org.application.shikiapp.screens

import androidx.activity.compose.BackHandler
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.Crossfade
import androidx.compose.animation.slideInHorizontally
import androidx.compose.animation.slideOutHorizontally
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.combinedClickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.ListItem
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.FilterQuality
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.paging.compose.LazyPagingItems
import androidx.paging.compose.collectAsLazyPagingItems
import coil3.compose.AsyncImage
import org.application.shikiapp.R
import org.application.shikiapp.di.Preferences
import org.application.shikiapp.events.ContentDetailEvent
import org.application.shikiapp.models.states.UserMessagesState
import org.application.shikiapp.models.states.UserState
import org.application.shikiapp.models.states.showClubs
import org.application.shikiapp.models.states.showDialogDelete
import org.application.shikiapp.models.states.showFavourite
import org.application.shikiapp.models.states.showFriends
import org.application.shikiapp.models.states.showHistory
import org.application.shikiapp.models.ui.Comment
import org.application.shikiapp.models.ui.User
import org.application.shikiapp.models.ui.list.Dialog
import org.application.shikiapp.models.viewModels.UserMessagesViewModel
import org.application.shikiapp.models.viewModels.UserViewModel
import org.application.shikiapp.network.response.Response
import org.application.shikiapp.ui.templates.AnimatedScreen
import org.application.shikiapp.ui.templates.Comments
import org.application.shikiapp.ui.templates.DialogClubs
import org.application.shikiapp.ui.templates.DialogFavourites
import org.application.shikiapp.ui.templates.DialogFriends
import org.application.shikiapp.ui.templates.DialogHistory
import org.application.shikiapp.ui.templates.ErrorScreen
import org.application.shikiapp.ui.templates.IconComment
import org.application.shikiapp.ui.templates.LoadingScreen
import org.application.shikiapp.ui.templates.NavigationIcon
import org.application.shikiapp.ui.templates.Statistics
import org.application.shikiapp.ui.templates.UserBriefItem
import org.application.shikiapp.ui.templates.UserMenuItems
import org.application.shikiapp.ui.templates.VectorIcon
import org.application.shikiapp.utils.BLANK
import org.application.shikiapp.utils.HtmlComment
import org.application.shikiapp.utils.enums.FavouriteItem
import org.application.shikiapp.utils.extensions.getLastMessage
import org.application.shikiapp.utils.navigation.NavigationBarVisibility
import org.application.shikiapp.utils.navigation.Screen

@Composable
fun UserScreen(onNavigate: (Screen) -> Unit, back: () -> Unit) {
    val model = viewModel<UserViewModel>()
    val response by model.response.collectAsStateWithLifecycle()
    val state by model.state.collectAsStateWithLifecycle()

    AnimatedScreen(response, model::loadData) { user ->
        UserView(user, state, model::onEvent, onNavigate, back)
    }
}

@Composable
fun UserView(
    user: User,
    state: UserState,
    onEvent: (ContentDetailEvent) -> Unit,
    onNavigate: (Screen) -> Unit,
    back: () -> Unit,
    visibility: NavigationBarVisibility? = null
) {
    val listState = rememberLazyListState()
    val friends = user.friends.collectAsLazyPagingItems()
    val history = user.history.collectAsLazyPagingItems()
    val comments = user.comments.collectAsLazyPagingItems()

    val listStates = FavouriteItem.entries.map { rememberLazyListState() }

    Scaffold(
        topBar = {
            TopAppBar(
                title = {},
                navigationIcon = {
                    if (Preferences.userId == user.id) {
                        IconButton(back) {
                            VectorIcon(R.drawable.vector_exit_app)
                        }
                    } else {
                        NavigationIcon(back)
                    }
                },
                actions = {
                    TopBarActions(user, onEvent, comments)
                }
            )
        }
    ) { values ->
        LazyColumn(
            contentPadding = PaddingValues(8.dp, values.calculateTopPadding()),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            item {
                UserBriefItem(user)
            }

            item {
                UserMenuItems { onEvent(ContentDetailEvent.User.PickMenu(it)) }
            }

            if (Preferences.userId != user.id) {
                item {
                    Statistics(
                        id = user.id,
                        statistics = user.stats,
                        onNavigate = onNavigate
                    )
                }
            }

            user.about.let {
                if (it.isNotEmpty()) {
                    item {
                        HorizontalDivider()
                    }

                    item {
                        Column(Modifier.wrapContentHeight(), Arrangement.spacedBy(6.dp)) {
                            Text(
                                text = stringResource(R.string.text_about_me),
                                style = MaterialTheme.typography.titleLarge
                            )

                            Text(it)
                        }
                    }
                }
            }
        }
    }

    LaunchedEffect(state.showSettings, state.menu, state.showDialogs) {
        if (state.showSettings || state.menu != null || state.showDialogs) {
            visibility?.hide()
        } else {
            visibility?.show()
        }
    }

    Comments(
        list = comments,
        listState = listState,
        visible = state.showComments,
        hide = { onEvent(ContentDetailEvent.ShowComments) },
        onNavigate = onNavigate
    )

    DialogUserDialogs(
        visible = state.showDialogs,
        onNavigate = onNavigate,
        dialogId = if (Preferences.userId == user.id) null else user.nickname,
        userId = if (Preferences.userId == user.id) null else user.id,
        hide = { onEvent(ContentDetailEvent.User.ShowDialogs) }
    )

    DialogFriends(
        friends = friends,
        visible = state.showFriends,
        onNavigate = onNavigate,
        hide = { onEvent(ContentDetailEvent.User.PickMenu()) },
    )

    DialogClubs(
        clubs = user.clubs,
        visible = state.showClubs,
        onNavigate = onNavigate,
        hide = { onEvent(ContentDetailEvent.User.PickMenu()) },
    )

    DialogFavourites(
        favourites = user.favourites,
        visible = state.showFavourite,
        listStates = listStates,
        hide = { onEvent(ContentDetailEvent.User.PickMenu()) },
        onNavigate = onNavigate,
    )

    DialogHistory(
        history = history,
        visible = state.showHistory,
        hide = { onEvent(ContentDetailEvent.User.PickMenu()) },
        onNavigate = onNavigate,
    )

    if (state.showDialogToggleFriend) {
        DialogToggleFriend(user.inFriends, onEvent)
    }
}

@Composable
private fun TopBarActions(
    user: User,
    onEvent: (ContentDetailEvent) -> Unit,
    comments: LazyPagingItems<Comment>
) {
    when {
        Preferences.userId == user.id -> {
            IconButton(
                onClick = { onEvent(ContentDetailEvent.User.ShowDialogs) },
                content = { VectorIcon(R.drawable.vector_mail) }
            )
            IconButton(
                onClick = { onEvent(ContentDetailEvent.User.ShowSettings) },
                content = { VectorIcon(R.drawable.vector_settings) }
            )
        }

        else -> {
            IconComment(
                comments = comments,
                onEvent = { onEvent(ContentDetailEvent.ShowComments) }
            )

            if (Preferences.token != null && Preferences.userId != user.id) {
                IconButton(
                    onClick = { onEvent(ContentDetailEvent.User.ShowDialogToggleFriend) }
                ) {
                    Icon(
                        contentDescription = null,
                        painter = painterResource(
                            if (user.inFriends) R.drawable.vector_remove_friend
                            else R.drawable.vector_add_friend
                        )
                    )
                }
            }

            IconButton(
                onClick = { onEvent(ContentDetailEvent.User.ShowDialogs) },
                content = { VectorIcon(R.drawable.vector_mail) }
            )
        }
    }
}

@Composable
private fun DialogUserDialogs(
    hide: () -> Unit,
    visible: Boolean,
    onNavigate: (Screen) -> Unit,
    dialogId: String? = null,
    userId: Long? = null
) = AnimatedVisibility(
    visible = visible,
    enter = slideInHorizontally(initialOffsetX = { it }),
    exit = slideOutHorizontally(targetOffsetX = { it })
) {
    val model = viewModel<UserMessagesViewModel>()
    val state by model.state.collectAsStateWithLifecycle()
    val response by model.response.collectAsStateWithLifecycle()
    val dialogMessages by model.messages.collectAsStateWithLifecycle()

    LaunchedEffect(Unit) {
        if (dialogId != null && userId != null) {
            model.getDialog(userId, dialogId)
        }
    }

    BackHandler(visible) {
        if (state.dialogId == null || userId != null) hide()
        else model.showDialogs()
    }
    Scaffold(
        topBar = {
            TopAppBar(
                title = {
                    Text(
                        maxLines = 1,
                        text = state.dialogId ?: stringResource(R.string.text_dialogs),
                        overflow = TextOverflow.Ellipsis
                    )
                },
                navigationIcon = {
                    NavigationIcon {
                        if (state.dialogId == null || userId != null) hide()
                        else model.showDialogs()
                    }
                }
            )
        }
    ) { values ->
        Crossfade(state.dialogId) {
            when (it) {
                null -> AllUserDialogs(
                    values = values,
                    response = response,
                    reload = model::loadData,
                    getDialog = model::getDialog,
                    showDelete = model::showDialogDelete
                )

                else -> UserDialog(
                    values = values,
                    state = state,
                    dialogMessages = dialogMessages,
                    onNavigate = onNavigate,
                    reload = model::getDialog,
                    setText = model::setText,
                    sendMessage = model::sendMessage
                )
            }
        }
    }

    if (state.showDialogDelete) {
        DialogRemoveUserDialog(
            nickname = state.toDeleteId ?: BLANK,
            hide = model::showDialogDelete,
            remove = model::removeDialog
        )
    }
}

@Composable
private fun UserDialog(
    dialogMessages: Response<List<Dialog>, Throwable>,
    state: UserMessagesState,
    values: PaddingValues,
    reload: () -> Unit,
    setText: (String) -> Unit,
    sendMessage: () -> Unit,
    onNavigate: (Screen) -> Unit
) = when (val messages = dialogMessages) {
    Response.Loading -> LoadingScreen()
    is Response.Error -> ErrorScreen(reload)
    is Response.Success -> Column(Modifier.imePadding()) {
        LazyColumn(
            modifier = Modifier.weight(1f),
            verticalArrangement = Arrangement.Top,
            reverseLayout = true,
            contentPadding = values
        ) {
            items(messages.data) { message ->
                Column(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(12.dp, 10.dp)
                ) {
                    Row(
                        verticalAlignment = Alignment.CenterVertically,
                        modifier = Modifier
                            .fillMaxWidth()
                            .clickable { onNavigate(Screen.User(message.userId)) }
                    ) {
                        AsyncImage(
                            model = message.userAvatar,
                            contentDescription = null,
                            contentScale = ContentScale.Crop,
                            filterQuality = FilterQuality.High,
                            modifier = Modifier
                                .size(56.dp)
                                .clip(CircleShape)
                                .border(1.dp, MaterialTheme.colorScheme.outlineVariant, CircleShape)
                        )

                        Spacer(Modifier.width(12.dp))

                        Column(Modifier.weight(1f)) {
                            Text(
                                text = message.userNickname,
                                style = MaterialTheme.typography.titleSmall.copy(
                                    color = MaterialTheme.colorScheme.onSurface,
                                    fontSize = 16.sp,
                                    fontWeight = FontWeight.SemiBold
                                )
                            )
                            Text(
                                text = message.lastDate,
                                style = MaterialTheme.typography.bodySmall.copy(
                                    color = MaterialTheme.colorScheme.onSurfaceVariant,
                                    fontSize = 13.sp
                                )
                            )
                        }
                    }

                    Spacer(Modifier.height(8.dp))

                    HtmlComment(message.lastMessage)
                }
            }
        }

        Row(
            verticalAlignment = Alignment.CenterVertically,
            modifier = Modifier
                .fillMaxWidth()
                .padding(vertical = 8.dp)
        ) {
            Spacer(Modifier.width(8.dp))

            OutlinedTextField(
                maxLines = 1,
                value = state.text,
                onValueChange = setText,
                modifier = Modifier.weight(1f),
                placeholder = { Text(stringResource(R.string.text_enter_message)) },
            )

            IconButton(
                onClick = sendMessage,
                enabled = state.text.isNotEmpty(),
                content = { VectorIcon(R.drawable.vector_send) }
            )
        }
    }

    else -> Unit
}

@Composable
private fun AllUserDialogs(
    response: Response<List<Dialog>, Throwable>,
    reload: () -> Unit,
    getDialog: (Long, String) -> Unit,
    showDelete: (String) -> Unit,
    values: PaddingValues
) = when (val dialogs = response) {
    Response.Loading -> LoadingScreen()
    is Response.Error -> ErrorScreen(reload)
    is Response.Success -> LazyColumn(contentPadding = values) {
        items(dialogs.data) { dialog ->
            ListItem(
                headlineContent = { Text(dialog.userNickname) },
                trailingContent = { Text(dialog.lastDate) },
                modifier = Modifier
                    .combinedClickable(
                        onClick = { getDialog(dialog.userId, dialog.userNickname) },
                        onLongClick = { showDelete(dialog.id) }
                    ),
                leadingContent = {
                    AsyncImage(
                        model = dialog.userAvatar,
                        contentDescription = null,
                        modifier = Modifier
                            .size(56.dp)
                            .clip(CircleShape)
                    )
                },
                supportingContent = {
                    Text(
                        minLines = 2,
                        maxLines = 2,
                        text = dialog.lastMessage.getLastMessage().asString(),
                        overflow = TextOverflow.Ellipsis
                    )
                }
            )
        }
    }

    else -> Unit
}

@Composable
private fun DialogRemoveUserDialog(nickname: String, hide: () -> Unit, remove: () -> Unit) =
    AlertDialog(
        onDismissRequest = hide,
        dismissButton = {
            TextButton(hide) {
                Text(stringResource(R.string.text_cancel))
            }
        },
        confirmButton = {
            TextButton(remove) {
                Text(stringResource(R.string.text_confirm))
            }
        },
        text = {
            Text(
                text = stringResource(R.string.text_sure_to_delete_dialog, nickname),
                style = MaterialTheme.typography.bodyLarge
            )
        }
    )

@Composable
private fun DialogToggleFriend(inFriends: Boolean, onEvent: (ContentDetailEvent) -> Unit) =
    AlertDialog(
        onDismissRequest = { onEvent(ContentDetailEvent.User.ShowDialogToggleFriend) },
        dismissButton = {
            TextButton(
                onClick = { onEvent(ContentDetailEvent.User.ShowDialogToggleFriend) }
            ) { Text(stringResource(R.string.text_cancel)) }
        },
        confirmButton = {
            TextButton(
                onClick = { onEvent(ContentDetailEvent.User.ToggleFriend) }
            ) { Text(stringResource(R.string.text_confirm)) }
        },
        title = {
            Text(
                text = stringResource(
                    if (inFriends) R.string.text_remove_friend
                    else R.string.text_add_friend
                )
            )
        },
        text = {
            Text(
                style = MaterialTheme.typography.bodyLarge,
                text = stringResource(
                    if (inFriends) R.string.text_confirm_remove_friend
                    else R.string.text_confirm_add_friend
                )
            )
        }
    )