import 'package:matomo_tracker/src/assert.dart';

/// Describes a campaign.
///
/// Multiple `track...` methods in [MatomoTracker] can take a campaign as argument.
///
/// When using campaigns, it should be noted that each visit can only have at most
/// one campaign associated with it. It does not matter if the first `track...`
/// call or a subsequent `track...` call has the campaign attached, it will be treated
/// as the campaign for the whole visit. Calling `track...` methods with the same
/// campaign (in respect to the objects field values) will not start a new visit,
/// nor will calling `track...` without a campaign after setting a campaign in a
/// previous `track...` call start a new visit. On the other hand, calling a
/// `track...` method with a different campaign will start a new visit.
///
/// Read more about [Campaign Tracking](https://matomo.org/faq/reports/what-is-campaign-tracking-and-why-it-is-important/).
class Campaign {
  /// Creates a campaign description.
  ///
  /// Note: Strings filled with whitespace will be considered as (invalid) empty
  /// values.
  factory Campaign({
    required String name,
    String? keyword,
    String? source,
    String? medium,
    String? content,
    String? id,
    String? group,
    String? placement,
  }) {
    assertStringIsFilled(value: name, name: 'name');
    assertStringIsFilled(value: keyword, name: 'keyword');
    assertStringIsFilled(value: source, name: 'source');
    assertStringIsFilled(value: medium, name: 'medium');
    assertStringIsFilled(value: content, name: 'content');
    assertStringIsFilled(value: id, name: 'id');
    assertStringIsFilled(value: group, name: 'group');
    assertStringIsFilled(value: placement, name: 'placement');

    return Campaign._(
      name: name,
      keyword: keyword,
      source: source,
      medium: medium,
      content: content,
      id: id,
      group: group,
      placement: placement,
    );
  }

  const Campaign._({
    required this.name,
    this.keyword,
    this.source,
    this.medium,
    this.content,
    this.id,
    this.group,
    this.placement,
  });

  /// A descriptive name for the campaign, e.g. a blog post title or email campaign name.
  ///
  /// Corresponds with `mtm_campaign`.
  final String name;

  /// The specific keyword that someone searched for, or category of interest.
  ///
  /// Corresponds with `mtm_keyword`.
  final String? keyword;

  /// The actual source of the traffic, e.g. newsletter, twitter, ebay, etc.
  ///
  /// Requires Matomo Cloud or Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_source`.
  final String? source;

  /// The type of marketing channel, e.g. email, social, paid, etc.
  ///
  /// Requires Matomo Cloud or Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_medium`.
  final String? medium;

  /// This is a specific link or content that somebody clicked. e.g. banner, big-green-button.
  ///
  /// Requires Matomo Cloud or Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_content`.
  final String? content;

  /// A unique identifier for your specific ad. This parameter is often used with the numeric IDs automatically generated by advertising platforms.
  ///
  /// Requires Matomo Cloud or Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_cid`.
  final String? id;

  /// The audience your campaign is targeting e.g. customers, retargeting, etc.
  ///
  /// Requires Matomo Cloud or Matomo 4 or above with Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_group`.
  final String? group;

  ///The placement on an advertising network e.g. newsfeed, sidebar, home-banner, etc.
  ///
  /// Requires Matomo Cloud or Matomo 4 or above with Marketing Campaigns Reporting Plugin.
  /// Corresponds with `mtm_placement`.
  final String? placement;

  Map<String, String> toMap() {
    final mtmKeyword = keyword;
    final mtmSource = source;
    final mtmMedium = medium;
    final mtmContent = content;
    final mtmCid = id;
    final mtmGroup = group;
    final mtmPlacement = placement;

    return {
      'mtm_campaign': name,
      if (mtmKeyword != null) 'mtm_keyword': mtmKeyword,
      if (mtmSource != null) 'mtm_source': mtmSource,
      if (mtmMedium != null) 'mtm_medium': mtmMedium,
      if (mtmContent != null) 'mtm_content': mtmContent,
      if (mtmCid != null) 'mtm_cid': mtmCid,
      if (mtmGroup != null) 'mtm_group': mtmGroup,
      if (mtmPlacement != null) 'mtm_placement': mtmPlacement,
    };
  }
}
