// Copyright 2013 The Flutter Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Autogenerated from Pigeon (v25.5.0), do not edit directly.
// See also: https://pub.dev/packages/pigeon

import Foundation
import WebKit

#if !os(macOS)
  import UIKit
#endif

#if os(iOS)
  import Flutter
#elseif os(macOS)
  import FlutterMacOS
#else
  #error("Unsupported platform.")
#endif

/// Error class for passing custom error details to Dart side.
final class PigeonError: Error {
  let code: String
  let message: String?
  let details: Sendable?

  init(code: String, message: String?, details: Sendable?) {
    self.code = code
    self.message = message
    self.details = details
  }

  var localizedDescription: String {
    return
      "PigeonError(code: \(code), message: \(message ?? "<nil>"), details: \(details ?? "<nil>")"
  }
}

private func wrapResult(_ result: Any?) -> [Any?] {
  return [result]
}

private func wrapError(_ error: Any) -> [Any?] {
  if let pigeonError = error as? PigeonError {
    return [
      pigeonError.code,
      pigeonError.message,
      pigeonError.details,
    ]
  }
  if let flutterError = error as? FlutterError {
    return [
      flutterError.code,
      flutterError.message,
      flutterError.details,
    ]
  }
  return [
    "\(error)",
    "\(type(of: error))",
    "Stacktrace: \(Thread.callStackSymbols)",
  ]
}

private func createConnectionError(withChannelName channelName: String) -> PigeonError {
  return PigeonError(
    code: "channel-error", message: "Unable to establish connection on channel: '\(channelName)'.",
    details: "")
}

private func isNullish(_ value: Any?) -> Bool {
  return value is NSNull || value == nil
}

private func nilOrValue<T>(_ value: Any?) -> T? {
  if value is NSNull { return nil }
  return value as! T?
}

/// Handles the callback when an object is deallocated.
protocol WebKitLibraryPigeonInternalFinalizerDelegate: AnyObject {
  /// Invoked when the strong reference of an object is deallocated in an `InstanceManager`.
  func onDeinit(identifier: Int64)
}

// Attaches to an object to receive a callback when the object is deallocated.
internal final class WebKitLibraryPigeonInternalFinalizer {
  internal static let associatedObjectKey = malloc(1)!

  private let identifier: Int64
  // Reference to the delegate is weak because the callback should be ignored if the
  // `InstanceManager` is deallocated.
  internal weak var delegate: WebKitLibraryPigeonInternalFinalizerDelegate?

  private init(identifier: Int64, delegate: WebKitLibraryPigeonInternalFinalizerDelegate) {
    self.identifier = identifier
    self.delegate = delegate
  }

  internal static func attach(
    to instance: AnyObject, identifier: Int64,
    delegate: WebKitLibraryPigeonInternalFinalizerDelegate
  ) {
    let finalizer = WebKitLibraryPigeonInternalFinalizer(identifier: identifier, delegate: delegate)
    objc_setAssociatedObject(instance, associatedObjectKey, finalizer, .OBJC_ASSOCIATION_RETAIN)
  }

  static func detach(from instance: AnyObject) {
    let finalizer =
      objc_getAssociatedObject(instance, associatedObjectKey)
      as? WebKitLibraryPigeonInternalFinalizer
    if let finalizer = finalizer {
      finalizer.delegate = nil
      objc_setAssociatedObject(instance, associatedObjectKey, nil, .OBJC_ASSOCIATION_ASSIGN)
    }
  }

  deinit {
    delegate?.onDeinit(identifier: identifier)
  }
}

/// Maintains instances used to communicate with the corresponding objects in Dart.
///
/// Objects stored in this container are represented by an object in Dart that is also stored in
/// an InstanceManager with the same identifier.
///
/// When an instance is added with an identifier, either can be used to retrieve the other.
///
/// Added instances are added as a weak reference and a strong reference. When the strong
/// reference is removed and the weak reference is deallocated,`WebKitLibraryPigeonInternalFinalizerDelegate.onDeinit`
/// is called with the instance's identifier. However, if the strong reference is removed and then the identifier is
/// retrieved with the intention to pass the identifier to Dart (e.g. by calling `identifierWithStrongReference`),
/// the strong reference to the instance is re-added. The strong reference will then need to be removed manually
/// again.
///
/// Accessing and inserting to an InstanceManager is thread safe.
final class WebKitLibraryPigeonInstanceManager {
  // Identifiers are locked to a specific range to avoid collisions with objects
  // created simultaneously from Dart.
  // Host uses identifiers >= 2^16 and Dart is expected to use values n where,
  // 0 <= n < 2^16.
  private static let minHostCreatedIdentifier: Int64 = 65536

  private let lockQueue = DispatchQueue(label: "WebKitLibraryPigeonInstanceManager")
  private let identifiers: NSMapTable<AnyObject, NSNumber> = NSMapTable(
    keyOptions: [.weakMemory, .objectPointerPersonality], valueOptions: .strongMemory)
  private let weakInstances: NSMapTable<NSNumber, AnyObject> = NSMapTable(
    keyOptions: .strongMemory, valueOptions: [.weakMemory, .objectPointerPersonality])
  private let strongInstances: NSMapTable<NSNumber, AnyObject> = NSMapTable(
    keyOptions: .strongMemory, valueOptions: [.strongMemory, .objectPointerPersonality])
  private let finalizerDelegate: WebKitLibraryPigeonInternalFinalizerDelegate
  private var nextIdentifier: Int64 = minHostCreatedIdentifier

  public init(finalizerDelegate: WebKitLibraryPigeonInternalFinalizerDelegate) {
    self.finalizerDelegate = finalizerDelegate
  }

  /// Adds a new instance that was instantiated from Dart.
  ///
  /// The same instance can be added multiple times, but each identifier must be unique. This allows
  /// two objects that are equivalent (e.g. conforms to `Equatable`)  to both be added.
  ///
  /// - Parameters:
  ///   - instance: the instance to be stored
  ///   - identifier: the identifier to be paired with instance. This value must be >= 0 and unique
  func addDartCreatedInstance(_ instance: AnyObject, withIdentifier identifier: Int64) {
    lockQueue.async {
      self.addInstance(instance, withIdentifier: identifier)
    }
  }

  /// Adds a new instance that was instantiated from the host platform.
  ///
  /// - Parameters:
  ///   - instance: the instance to be stored. This must be unique to all other added instances.
  /// - Returns: the unique identifier (>= 0) stored with instance
  func addHostCreatedInstance(_ instance: AnyObject) -> Int64 {
    assert(!containsInstance(instance), "Instance of \(instance) has already been added.")
    var identifier: Int64 = -1
    lockQueue.sync {
      identifier = nextIdentifier
      nextIdentifier += 1
      self.addInstance(instance, withIdentifier: identifier)
    }
    return identifier
  }

  /// Removes `instanceIdentifier` and its associated strongly referenced instance, if present, from the manager.
  ///
  /// - Parameters:
  ///   - instanceIdentifier: the identifier paired to an instance.
  /// - Returns: removed instance if the manager contains the given identifier, otherwise `nil` if
  ///   the manager doesn't contain the value
  func removeInstance<T: AnyObject>(withIdentifier instanceIdentifier: Int64) throws -> T? {
    var instance: AnyObject? = nil
    lockQueue.sync {
      instance = strongInstances.object(forKey: NSNumber(value: instanceIdentifier))
      strongInstances.removeObject(forKey: NSNumber(value: instanceIdentifier))
    }
    return instance as? T
  }

  /// Retrieves the instance associated with identifier.
  ///
  /// - Parameters:
  ///   - instanceIdentifier: the identifier associated with an instance
  /// - Returns: the instance associated with `instanceIdentifier` if the manager contains the value, otherwise
  ///   `nil` if the manager doesn't contain the value
  func instance<T: AnyObject>(forIdentifier instanceIdentifier: Int64) -> T? {
    var instance: AnyObject? = nil
    lockQueue.sync {
      instance = weakInstances.object(forKey: NSNumber(value: instanceIdentifier))
    }
    return instance as? T
  }

  private func addInstance(_ instance: AnyObject, withIdentifier identifier: Int64) {
    assert(identifier >= 0)
    assert(
      weakInstances.object(forKey: identifier as NSNumber) == nil,
      "Identifier has already been added: \(identifier)")
    identifiers.setObject(NSNumber(value: identifier), forKey: instance)
    weakInstances.setObject(instance, forKey: NSNumber(value: identifier))
    strongInstances.setObject(instance, forKey: NSNumber(value: identifier))
    WebKitLibraryPigeonInternalFinalizer.attach(
      to: instance, identifier: identifier, delegate: finalizerDelegate)
  }

  /// Retrieves the identifier paired with an instance.
  ///
  /// If the manager contains a strong reference to `instance`, it will return the identifier
  /// associated with `instance`. If the manager contains only a weak reference to `instance`, a new
  /// strong reference to `instance` will be added and will need to be removed again with `removeInstance`.
  ///
  /// If this method returns a nonnull identifier, this method also expects the Dart
  /// `WebKitLibraryPigeonInstanceManager` to have, or recreate, a weak reference to the Dart instance the
  /// identifier is associated with.
  ///
  /// - Parameters:
  ///   - instance: an instance that may be stored in the manager
  /// - Returns: the identifier associated with `instance` if the manager contains the value, otherwise
  ///   `nil` if the manager doesn't contain the value
  func identifierWithStrongReference(forInstance instance: AnyObject) -> Int64? {
    var identifier: Int64? = nil
    lockQueue.sync {
      if let existingIdentifier = identifiers.object(forKey: instance)?.int64Value {
        strongInstances.setObject(instance, forKey: NSNumber(value: existingIdentifier))
        identifier = existingIdentifier
      }
    }
    return identifier
  }

  /// Whether this manager contains the given `instance`.
  ///
  /// - Parameters:
  ///   - instance: the instance whose presence in this manager is to be tested
  /// - Returns: whether this manager contains the given `instance`
  func containsInstance(_ instance: AnyObject) -> Bool {
    var containsInstance = false
    lockQueue.sync {
      containsInstance = identifiers.object(forKey: instance) != nil
    }
    return containsInstance
  }

  /// Removes all of the instances from this manager.
  ///
  /// The manager will be empty after this call returns.
  func removeAllObjects() throws {
    lockQueue.sync {
      let weakInstancesEnumerator = weakInstances.objectEnumerator()!
      while let instance = weakInstancesEnumerator.nextObject() {
        WebKitLibraryPigeonInternalFinalizer.detach(from: instance as AnyObject)
      }
      identifiers.removeAllObjects()
      weakInstances.removeAllObjects()
      strongInstances.removeAllObjects()
      nextIdentifier = WebKitLibraryPigeonInstanceManager.minHostCreatedIdentifier
    }
  }

  /// The number of instances stored as a strong reference.
  ///
  /// For debugging and testing purposes.
  internal var strongInstanceCount: Int {
    var count: Int = 0
    lockQueue.sync {
      count = strongInstances.count
    }
    return count
  }

  /// The number of instances stored as a weak reference.
  ///
  /// For debugging and testing purposes. NSMapTables that store keys or objects as weak
  /// reference will be reclaimed non-deterministically.
  internal var weakInstanceCount: Int {
    var count: Int = 0
    lockQueue.sync {
      count = weakInstances.count
    }
    return count
  }
}

private class WebKitLibraryPigeonInstanceManagerApi {
  /// The codec used for serializing messages.
  var codec: FlutterStandardMessageCodec { WebKitLibraryPigeonCodec.shared }

  /// Handles sending and receiving messages with Dart.
  unowned let binaryMessenger: FlutterBinaryMessenger

  init(binaryMessenger: FlutterBinaryMessenger) {
    self.binaryMessenger = binaryMessenger
  }

  /// Sets up an instance of `WebKitLibraryPigeonInstanceManagerApi` to handle messages through the `binaryMessenger`.
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, instanceManager: WebKitLibraryPigeonInstanceManager?
  ) {
    let codec = WebKitLibraryPigeonCodec.shared
    let removeStrongReferenceChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.PigeonInternalInstanceManager.removeStrongReference",
      binaryMessenger: binaryMessenger, codec: codec)
    if let instanceManager = instanceManager {
      removeStrongReferenceChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let identifierArg = args[0] as! Int64
        do {
          let _: AnyObject? = try instanceManager.removeInstance(withIdentifier: identifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      removeStrongReferenceChannel.setMessageHandler(nil)
    }
    let clearChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.PigeonInternalInstanceManager.clear",
      binaryMessenger: binaryMessenger, codec: codec)
    if let instanceManager = instanceManager {
      clearChannel.setMessageHandler { _, reply in
        do {
          try instanceManager.removeAllObjects()
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      clearChannel.setMessageHandler(nil)
    }
  }

  /// Sends a message to the Dart `InstanceManager` to remove the strong reference of the instance associated with `identifier`.
  func removeStrongReference(
    identifier identifierArg: Int64, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.PigeonInternalInstanceManager.removeStrongReference"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([identifierArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }
}
protocol WebKitLibraryPigeonProxyApiDelegate {
  /// An implementation of [PigeonApiURLRequest] used to add a new Dart instance of
  /// `URLRequest` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURLRequest(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiURLRequest
  /// An implementation of [PigeonApiHTTPURLResponse] used to add a new Dart instance of
  /// `HTTPURLResponse` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiHTTPURLResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiHTTPURLResponse
  /// An implementation of [PigeonApiURLResponse] used to add a new Dart instance of
  /// `URLResponse` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURLResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiURLResponse
  /// An implementation of [PigeonApiWKUserScript] used to add a new Dart instance of
  /// `WKUserScript` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKUserScript(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKUserScript
  /// An implementation of [PigeonApiWKNavigationAction] used to add a new Dart instance of
  /// `WKNavigationAction` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKNavigationAction(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKNavigationAction
  /// An implementation of [PigeonApiWKNavigationResponse] used to add a new Dart instance of
  /// `WKNavigationResponse` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKNavigationResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKNavigationResponse
  /// An implementation of [PigeonApiWKFrameInfo] used to add a new Dart instance of
  /// `WKFrameInfo` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKFrameInfo(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKFrameInfo
  /// An implementation of [PigeonApiNSError] used to add a new Dart instance of
  /// `NSError` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiNSError(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiNSError
  /// An implementation of [PigeonApiWKScriptMessage] used to add a new Dart instance of
  /// `WKScriptMessage` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKScriptMessage(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKScriptMessage
  /// An implementation of [PigeonApiWKSecurityOrigin] used to add a new Dart instance of
  /// `WKSecurityOrigin` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKSecurityOrigin(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKSecurityOrigin
  /// An implementation of [PigeonApiHTTPCookie] used to add a new Dart instance of
  /// `HTTPCookie` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiHTTPCookie(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiHTTPCookie
  /// An implementation of [PigeonApiAuthenticationChallengeResponse] used to add a new Dart instance of
  /// `AuthenticationChallengeResponse` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiAuthenticationChallengeResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiAuthenticationChallengeResponse
  /// An implementation of [PigeonApiWKWebsiteDataStore] used to add a new Dart instance of
  /// `WKWebsiteDataStore` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKWebsiteDataStore(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKWebsiteDataStore
  /// An implementation of [PigeonApiUIView] used to add a new Dart instance of
  /// `UIView` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiUIView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiUIView
  /// An implementation of [PigeonApiUIScrollView] used to add a new Dart instance of
  /// `UIScrollView` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiUIScrollView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiUIScrollView
  /// An implementation of [PigeonApiWKWebViewConfiguration] used to add a new Dart instance of
  /// `WKWebViewConfiguration` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKWebViewConfiguration(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKWebViewConfiguration
  /// An implementation of [PigeonApiWKUserContentController] used to add a new Dart instance of
  /// `WKUserContentController` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKUserContentController(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKUserContentController
  /// An implementation of [PigeonApiWKPreferences] used to add a new Dart instance of
  /// `WKPreferences` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKPreferences(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKPreferences
  /// An implementation of [PigeonApiWKScriptMessageHandler] used to add a new Dart instance of
  /// `WKScriptMessageHandler` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKScriptMessageHandler(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKScriptMessageHandler
  /// An implementation of [PigeonApiWKNavigationDelegate] used to add a new Dart instance of
  /// `WKNavigationDelegate` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKNavigationDelegate(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKNavigationDelegate
  /// An implementation of [PigeonApiNSObject] used to add a new Dart instance of
  /// `NSObject` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiNSObject(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiNSObject
  /// An implementation of [PigeonApiUIViewWKWebView] used to add a new Dart instance of
  /// `UIViewWKWebView` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiUIViewWKWebView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiUIViewWKWebView
  /// An implementation of [PigeonApiNSViewWKWebView] used to add a new Dart instance of
  /// `NSViewWKWebView` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiNSViewWKWebView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiNSViewWKWebView
  /// An implementation of [PigeonApiWKWebView] used to add a new Dart instance of
  /// `WKWebView` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKWebView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiWKWebView
  /// An implementation of [PigeonApiWKUIDelegate] used to add a new Dart instance of
  /// `WKUIDelegate` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKUIDelegate(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKUIDelegate
  /// An implementation of [PigeonApiWKHTTPCookieStore] used to add a new Dart instance of
  /// `WKHTTPCookieStore` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKHTTPCookieStore(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKHTTPCookieStore
  /// An implementation of [PigeonApiUIScrollViewDelegate] used to add a new Dart instance of
  /// `UIScrollViewDelegate` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiUIScrollViewDelegate(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiUIScrollViewDelegate
  /// An implementation of [PigeonApiURLCredential] used to add a new Dart instance of
  /// `URLCredential` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURLCredential(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiURLCredential
  /// An implementation of [PigeonApiURLProtectionSpace] used to add a new Dart instance of
  /// `URLProtectionSpace` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURLProtectionSpace(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiURLProtectionSpace
  /// An implementation of [PigeonApiURLAuthenticationChallenge] used to add a new Dart instance of
  /// `URLAuthenticationChallenge` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURLAuthenticationChallenge(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiURLAuthenticationChallenge
  /// An implementation of [PigeonApiURL] used to add a new Dart instance of
  /// `URL` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiURL(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiURL
  /// An implementation of [PigeonApiWKWebpagePreferences] used to add a new Dart instance of
  /// `WKWebpagePreferences` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiWKWebpagePreferences(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiWKWebpagePreferences
  /// An implementation of [PigeonApiGetTrustResultResponse] used to add a new Dart instance of
  /// `GetTrustResultResponse` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiGetTrustResultResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiGetTrustResultResponse
  /// An implementation of [PigeonApiSecTrust] used to add a new Dart instance of
  /// `SecTrust` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiSecTrust(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiSecTrust
  /// An implementation of [PigeonApiSecCertificate] used to add a new Dart instance of
  /// `SecCertificate` to the Dart `InstanceManager` and make calls to Dart.
  func pigeonApiSecCertificate(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiSecCertificate
}

extension WebKitLibraryPigeonProxyApiDelegate {
  func pigeonApiURLResponse(_ registrar: WebKitLibraryPigeonProxyApiRegistrar)
    -> PigeonApiURLResponse
  {
    return PigeonApiURLResponse(
      pigeonRegistrar: registrar, delegate: PigeonApiDelegateURLResponse())
  }
  func pigeonApiWKWebView(_ registrar: WebKitLibraryPigeonProxyApiRegistrar) -> PigeonApiWKWebView {
    return PigeonApiWKWebView(pigeonRegistrar: registrar, delegate: PigeonApiDelegateWKWebView())
  }
}

open class WebKitLibraryPigeonProxyApiRegistrar {
  let binaryMessenger: FlutterBinaryMessenger
  let apiDelegate: WebKitLibraryPigeonProxyApiDelegate
  let instanceManager: WebKitLibraryPigeonInstanceManager
  /// Whether APIs should ignore calling to Dart.
  public var ignoreCallsToDart = false
  private var _codec: FlutterStandardMessageCodec?
  var codec: FlutterStandardMessageCodec {
    if _codec == nil {
      _codec = FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(pigeonRegistrar: self))
    }
    return _codec!
  }

  private class InstanceManagerApiFinalizerDelegate: WebKitLibraryPigeonInternalFinalizerDelegate {
    let api: WebKitLibraryPigeonInstanceManagerApi

    init(_ api: WebKitLibraryPigeonInstanceManagerApi) {
      self.api = api
    }

    public func onDeinit(identifier: Int64) {
      api.removeStrongReference(identifier: identifier) {
        _ in
      }
    }
  }

  init(binaryMessenger: FlutterBinaryMessenger, apiDelegate: WebKitLibraryPigeonProxyApiDelegate) {
    self.binaryMessenger = binaryMessenger
    self.apiDelegate = apiDelegate
    self.instanceManager = WebKitLibraryPigeonInstanceManager(
      finalizerDelegate: InstanceManagerApiFinalizerDelegate(
        WebKitLibraryPigeonInstanceManagerApi(binaryMessenger: binaryMessenger)))
  }

  func setUp() {
    WebKitLibraryPigeonInstanceManagerApi.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, instanceManager: instanceManager)
    PigeonApiURLRequest.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiURLRequest(self))
    PigeonApiWKUserScript.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKUserScript(self))
    PigeonApiHTTPCookie.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiHTTPCookie(self))
    PigeonApiAuthenticationChallengeResponse.setUpMessageHandlers(
      binaryMessenger: binaryMessenger,
      api: apiDelegate.pigeonApiAuthenticationChallengeResponse(self))
    PigeonApiWKWebsiteDataStore.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKWebsiteDataStore(self))
    PigeonApiUIView.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiUIView(self))
    PigeonApiUIScrollView.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiUIScrollView(self))
    PigeonApiWKWebViewConfiguration.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKWebViewConfiguration(self))
    PigeonApiWKUserContentController.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKUserContentController(self))
    PigeonApiWKPreferences.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKPreferences(self))
    PigeonApiWKScriptMessageHandler.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKScriptMessageHandler(self))
    PigeonApiWKNavigationDelegate.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKNavigationDelegate(self))
    PigeonApiNSObject.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiNSObject(self))
    PigeonApiUIViewWKWebView.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiUIViewWKWebView(self))
    PigeonApiNSViewWKWebView.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiNSViewWKWebView(self))
    PigeonApiWKUIDelegate.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKUIDelegate(self))
    PigeonApiWKHTTPCookieStore.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKHTTPCookieStore(self))
    PigeonApiUIScrollViewDelegate.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiUIScrollViewDelegate(self))
    PigeonApiURLCredential.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiURLCredential(self))
    PigeonApiURLProtectionSpace.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiURLProtectionSpace(self))
    PigeonApiURLAuthenticationChallenge.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiURLAuthenticationChallenge(self))
    PigeonApiURL.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiURL(self))
    PigeonApiWKWebpagePreferences.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiWKWebpagePreferences(self))
    PigeonApiSecTrust.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiSecTrust(self))
    PigeonApiSecCertificate.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: apiDelegate.pigeonApiSecCertificate(self))
  }
  func tearDown() {
    WebKitLibraryPigeonInstanceManagerApi.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, instanceManager: nil)
    PigeonApiURLRequest.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKUserScript.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiHTTPCookie.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiAuthenticationChallengeResponse.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKWebsiteDataStore.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiUIView.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiUIScrollView.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKWebViewConfiguration.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKUserContentController.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKPreferences.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKScriptMessageHandler.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKNavigationDelegate.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiNSObject.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiUIViewWKWebView.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiNSViewWKWebView.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKUIDelegate.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKHTTPCookieStore.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiUIScrollViewDelegate.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiURLCredential.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiURLProtectionSpace.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiURLAuthenticationChallenge.setUpMessageHandlers(
      binaryMessenger: binaryMessenger, api: nil)
    PigeonApiURL.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiWKWebpagePreferences.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiSecTrust.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
    PigeonApiSecCertificate.setUpMessageHandlers(binaryMessenger: binaryMessenger, api: nil)
  }
}
private class WebKitLibraryPigeonInternalProxyApiCodecReaderWriter: FlutterStandardReaderWriter {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar

  private class WebKitLibraryPigeonInternalProxyApiCodecReader: WebKitLibraryPigeonCodecReader {
    unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar

    init(data: Data, pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar) {
      self.pigeonRegistrar = pigeonRegistrar
      super.init(data: data)
    }

    override func readValue(ofType type: UInt8) -> Any? {
      switch type {
      case 128:
        let identifier = self.readValue()
        let instance: AnyObject? = pigeonRegistrar.instanceManager.instance(
          forIdentifier: identifier is Int64 ? identifier as! Int64 : Int64(identifier as! Int32))
        if instance == nil {
          print("Failed to find instance with identifier: \(identifier!)")
        }
        return instance
      default:
        return super.readValue(ofType: type)
      }
    }
  }

  private class WebKitLibraryPigeonInternalProxyApiCodecWriter: WebKitLibraryPigeonCodecWriter {
    unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar

    init(data: NSMutableData, pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar) {
      self.pigeonRegistrar = pigeonRegistrar
      super.init(data: data)
    }

    override func writeValue(_ value: Any) {
      if value is [Any] || value is Bool || value is Data || value is [AnyHashable: Any]
        || value is Double || value is FlutterStandardTypedData || value is Int64 || value is String
        || value is KeyValueObservingOptions || value is KeyValueChange
        || value is KeyValueChangeKey || value is UserScriptInjectionTime
        || value is AudiovisualMediaType || value is WebsiteDataType
        || value is NavigationActionPolicy || value is NavigationResponsePolicy
        || value is HttpCookiePropertyKey || value is NavigationType || value is PermissionDecision
        || value is MediaCaptureType || value is UrlSessionAuthChallengeDisposition
        || value is UrlCredentialPersistence || value is DartSecTrustResultType
      {
        super.writeValue(value)
        return
      }

      if let instance = value as? URLRequestWrapper {
        pigeonRegistrar.apiDelegate.pigeonApiURLRequest(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? HTTPURLResponse {
        pigeonRegistrar.apiDelegate.pigeonApiHTTPURLResponse(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? URLResponse {
        pigeonRegistrar.apiDelegate.pigeonApiURLResponse(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKUserScript {
        pigeonRegistrar.apiDelegate.pigeonApiWKUserScript(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKNavigationAction {
        pigeonRegistrar.apiDelegate.pigeonApiWKNavigationAction(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKNavigationResponse {
        pigeonRegistrar.apiDelegate.pigeonApiWKNavigationResponse(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKFrameInfo {
        pigeonRegistrar.apiDelegate.pigeonApiWKFrameInfo(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? NSError {
        pigeonRegistrar.apiDelegate.pigeonApiNSError(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKScriptMessage {
        pigeonRegistrar.apiDelegate.pigeonApiWKScriptMessage(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKSecurityOrigin {
        pigeonRegistrar.apiDelegate.pigeonApiWKSecurityOrigin(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? HTTPCookie {
        pigeonRegistrar.apiDelegate.pigeonApiHTTPCookie(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? AuthenticationChallengeResponse {
        pigeonRegistrar.apiDelegate.pigeonApiAuthenticationChallengeResponse(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKWebsiteDataStore {
        pigeonRegistrar.apiDelegate.pigeonApiWKWebsiteDataStore(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      #if !os(macOS)
        if let instance = value as? UIScrollView {
          pigeonRegistrar.apiDelegate.pigeonApiUIScrollView(pigeonRegistrar).pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
          super.writeByte(128)
          super.writeValue(
            pigeonRegistrar.instanceManager.identifierWithStrongReference(
              forInstance: instance as AnyObject)!)
          return
        }
      #endif

      if let instance = value as? WKWebViewConfiguration {
        pigeonRegistrar.apiDelegate.pigeonApiWKWebViewConfiguration(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKUserContentController {
        pigeonRegistrar.apiDelegate.pigeonApiWKUserContentController(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKPreferences {
        pigeonRegistrar.apiDelegate.pigeonApiWKPreferences(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKScriptMessageHandler {
        pigeonRegistrar.apiDelegate.pigeonApiWKScriptMessageHandler(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKNavigationDelegate {
        pigeonRegistrar.apiDelegate.pigeonApiWKNavigationDelegate(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      #if !os(macOS)
        if let instance = value as? WKWebView {
          pigeonRegistrar.apiDelegate.pigeonApiUIViewWKWebView(pigeonRegistrar).pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
          super.writeByte(128)
          super.writeValue(
            pigeonRegistrar.instanceManager.identifierWithStrongReference(
              forInstance: instance as AnyObject)!)
          return
        }
      #endif
      #if !os(macOS)
        if let instance = value as? UIView {
          pigeonRegistrar.apiDelegate.pigeonApiUIView(pigeonRegistrar).pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
          super.writeByte(128)
          super.writeValue(
            pigeonRegistrar.instanceManager.identifierWithStrongReference(
              forInstance: instance as AnyObject)!)
          return
        }
      #endif
      #if !os(iOS)
        if let instance = value as? WKWebView {
          pigeonRegistrar.apiDelegate.pigeonApiNSViewWKWebView(pigeonRegistrar).pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
          super.writeByte(128)
          super.writeValue(
            pigeonRegistrar.instanceManager.identifierWithStrongReference(
              forInstance: instance as AnyObject)!)
          return
        }
      #endif

      if let instance = value as? WKWebView {
        pigeonRegistrar.apiDelegate.pigeonApiWKWebView(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKUIDelegate {
        pigeonRegistrar.apiDelegate.pigeonApiWKUIDelegate(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKHTTPCookieStore {
        pigeonRegistrar.apiDelegate.pigeonApiWKHTTPCookieStore(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      #if !os(macOS)
        if let instance = value as? UIScrollViewDelegate {
          pigeonRegistrar.apiDelegate.pigeonApiUIScrollViewDelegate(pigeonRegistrar)
            .pigeonNewInstance(
              pigeonInstance: instance
            ) { _ in }
          super.writeByte(128)
          super.writeValue(
            pigeonRegistrar.instanceManager.identifierWithStrongReference(
              forInstance: instance as AnyObject)!)
          return
        }
      #endif

      if let instance = value as? URLCredential {
        pigeonRegistrar.apiDelegate.pigeonApiURLCredential(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? URLProtectionSpace {
        pigeonRegistrar.apiDelegate.pigeonApiURLProtectionSpace(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? URLAuthenticationChallenge {
        pigeonRegistrar.apiDelegate.pigeonApiURLAuthenticationChallenge(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? URL {
        pigeonRegistrar.apiDelegate.pigeonApiURL(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? WKWebpagePreferences {
        pigeonRegistrar.apiDelegate.pigeonApiWKWebpagePreferences(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? GetTrustResultResponse {
        pigeonRegistrar.apiDelegate.pigeonApiGetTrustResultResponse(pigeonRegistrar)
          .pigeonNewInstance(
            pigeonInstance: instance
          ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? SecTrustWrapper {
        pigeonRegistrar.apiDelegate.pigeonApiSecTrust(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? SecCertificateWrapper {
        pigeonRegistrar.apiDelegate.pigeonApiSecCertificate(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as? NSObject {
        pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar).pigeonNewInstance(
          pigeonInstance: instance
        ) { _ in }
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(
            forInstance: instance as AnyObject)!)
        return
      }

      if let instance = value as AnyObject?,
        pigeonRegistrar.instanceManager.containsInstance(instance)
      {
        super.writeByte(128)
        super.writeValue(
          pigeonRegistrar.instanceManager.identifierWithStrongReference(forInstance: instance)!)
      } else {
        print("Unsupported value: \(value) of \(type(of: value))")
        assert(false, "Unsupported value for WebKitLibraryPigeonInternalProxyApiCodecWriter")
      }

    }
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar) {
    self.pigeonRegistrar = pigeonRegistrar
  }

  override func reader(with data: Data) -> FlutterStandardReader {
    return WebKitLibraryPigeonInternalProxyApiCodecReader(
      data: data, pigeonRegistrar: pigeonRegistrar)
  }

  override func writer(with data: NSMutableData) -> FlutterStandardWriter {
    return WebKitLibraryPigeonInternalProxyApiCodecWriter(
      data: data, pigeonRegistrar: pigeonRegistrar)
  }
}

/// The values that can be returned in a change dictionary.
///
/// See https://developer.apple.com/documentation/foundation/nskeyvalueobservingoptions.
enum KeyValueObservingOptions: Int {
  /// Indicates that the change dictionary should provide the new attribute
  /// value, if applicable.
  case newValue = 0
  /// Indicates that the change dictionary should contain the old attribute
  /// value, if applicable.
  case oldValue = 1
  /// If specified, a notification should be sent to the observer immediately,
  /// before the observer registration method even returns.
  case initialValue = 2
  /// Whether separate notifications should be sent to the observer before and
  /// after each change, instead of a single notification after the change.
  case priorNotification = 3
}

/// The kinds of changes that can be observed.
///
/// See https://developer.apple.com/documentation/foundation/nskeyvaluechange.
enum KeyValueChange: Int {
  /// Indicates that the value of the observed key path was set to a new value.
  case setting = 0
  /// Indicates that an object has been inserted into the to-many relationship
  /// that is being observed.
  case insertion = 1
  /// Indicates that an object has been removed from the to-many relationship
  /// that is being observed.
  case removal = 2
  /// Indicates that an object has been replaced in the to-many relationship
  /// that is being observed.
  case replacement = 3
  /// The value is not recognized by the wrapper.
  case unknown = 4
}

/// The keys that can appear in the change dictionary.
///
/// See https://developer.apple.com/documentation/foundation/nskeyvaluechangekey.
enum KeyValueChangeKey: Int {
  /// If the value of the `KeyValueChangeKey.kind` entry is
  /// `KeyValueChange.insertion`, `KeyValueChange.removal`, or
  /// `KeyValueChange.replacement`, the value of this key is a Set object that
  /// contains the indexes of the inserted, removed, or replaced objects.
  case indexes = 0
  /// An object that contains a value corresponding to one of the
  /// `KeyValueChange` enum, indicating what sort of change has occurred.
  case kind = 1
  /// If the value of the `KeyValueChange.kind` entry is
  /// `KeyValueChange.setting, and `KeyValueObservingOptions.newValue` was
  /// specified when the observer was registered, the value of this key is the
  /// new value for the attribute.
  case newValue = 2
  /// If the `KeyValueObservingOptions.priorNotification` option was specified
  /// when the observer was registered this notification is sent prior to a
  /// change.
  case notificationIsPrior = 3
  /// If the value of the `KeyValueChange.kind` entry is
  /// `KeyValueChange.setting`, and `KeyValueObservingOptions.old` was specified
  /// when the observer was registered, the value of this key is the value
  /// before the attribute was changed.
  case oldValue = 4
  /// The value is not recognized by the wrapper.
  case unknown = 5
}

/// Constants for the times at which to inject script content into a webpage.
///
/// See https://developer.apple.com/documentation/webkit/wkuserscriptinjectiontime.
enum UserScriptInjectionTime: Int {
  /// A constant to inject the script after the creation of the webpage’s
  /// document element, but before loading any other content.
  case atDocumentStart = 0
  /// A constant to inject the script after the document finishes loading, but
  /// before loading any other subresources.
  case atDocumentEnd = 1
  /// The value is not recognized by the wrapper.
  case unknown = 2
}

/// The media types that require a user gesture to begin playing.
///
/// See https://developer.apple.com/documentation/webkit/wkaudiovisualmediatypes.
enum AudiovisualMediaType: Int {
  /// No media types require a user gesture to begin playing.
  case none = 0
  /// Media types that contain audio require a user gesture to begin playing.
  case audio = 1
  /// Media types that contain video require a user gesture to begin playing.
  case video = 2
  /// All media types require a user gesture to begin playing.
  case all = 3
}

/// A `WKWebsiteDataRecord` object includes these constants in its dataTypes
/// property.
///
/// See https://developer.apple.com/documentation/webkit/wkwebsitedatarecord/data_store_record_types.
enum WebsiteDataType: Int {
  /// Cookies.
  case cookies = 0
  /// In-memory caches.
  case memoryCache = 1
  /// On-disk caches.
  case diskCache = 2
  /// HTML offline web app caches.
  case offlineWebApplicationCache = 3
  /// HTML local storage.
  case localStorage = 4
  /// HTML session storage.
  case sessionStorage = 5
  /// WebSQL databases.
  case webSQLDatabases = 6
  /// IndexedDB databases.
  case indexedDBDatabases = 7
}

/// Constants that indicate whether to allow or cancel navigation to a webpage
/// from an action.
///
/// See https://developer.apple.com/documentation/webkit/wknavigationactionpolicy.
enum NavigationActionPolicy: Int {
  /// Allow the navigation to continue.
  case allow = 0
  /// Cancel the navigation.
  case cancel = 1
  /// Allow the download to proceed.
  case download = 2
}

/// Constants that indicate whether to allow or cancel navigation to a webpage
/// from a response.
///
/// See https://developer.apple.com/documentation/webkit/wknavigationresponsepolicy.
enum NavigationResponsePolicy: Int {
  /// Allow the navigation to continue.
  case allow = 0
  /// Cancel the navigation.
  case cancel = 1
  /// Allow the download to proceed.
  case download = 2
}

/// Constants that define the supported keys in a cookie attributes dictionary.
///
/// See https://developer.apple.com/documentation/foundation/httpcookiepropertykey.
enum HttpCookiePropertyKey: Int {
  /// A String object containing the comment for the cookie.
  case comment = 0
  /// An Uri object or String object containing the comment URL for the cookie.
  case commentUrl = 1
  /// Aa String object stating whether the cookie should be discarded at the end
  /// of the session.
  case discard = 2
  /// An String object containing the domain for the cookie.
  case domain = 3
  /// An Date object or String object specifying the expiration date for the
  /// cookie.
  case expires = 4
  /// An String object containing an integer value stating how long in seconds
  /// the cookie should be kept, at most.
  case maximumAge = 5
  /// An String object containing the name of the cookie (required).
  case name = 6
  /// A URL or String object containing the URL that set this cookie.
  case originUrl = 7
  /// A String object containing the path for the cookie.
  case path = 8
  /// An String object containing comma-separated integer values specifying the
  /// ports for the cookie.
  case port = 9
  /// A string indicating the same-site policy for the cookie.
  case sameSitePolicy = 10
  /// A String object indicating that the cookie should be transmitted only over
  /// secure channels.
  case secure = 11
  /// A String object containing the value of the cookie.
  case value = 12
  /// A String object that specifies the version of the cookie.
  case version = 13
  /// The value is not recognized by the wrapper.
  case unknown = 14
}

/// The type of action that triggered the navigation.
///
/// See https://developer.apple.com/documentation/webkit/wknavigationtype.
enum NavigationType: Int {
  /// A link activation.
  case linkActivated = 0
  /// A request to submit a form.
  case formSubmitted = 1
  /// A request for the frame’s next or previous item.
  case backForward = 2
  /// A request to reload the webpage.
  case reload = 3
  /// A request to resubmit a form.
  case formResubmitted = 4
  /// A navigation request that originates for some other reason.
  case other = 5
  /// The value is not recognized by the wrapper.
  case unknown = 6
}

/// Possible permission decisions for device resource access.
///
/// See https://developer.apple.com/documentation/webkit/wkpermissiondecision.
enum PermissionDecision: Int {
  /// Deny permission for the requested resource.
  case deny = 0
  /// Deny permission for the requested resource.
  case grant = 1
  /// Prompt the user for permission for the requested resource.
  case prompt = 2
}

/// List of the types of media devices that can capture audio, video, or both.
///
/// See https://developer.apple.com/documentation/webkit/wkmediacapturetype.
enum MediaCaptureType: Int {
  /// A media device that can capture video.
  case camera = 0
  /// A media device or devices that can capture audio and video.
  case cameraAndMicrophone = 1
  /// A media device that can capture audio.
  case microphone = 2
  /// The value is not recognized by the wrapper.
  case unknown = 3
}

/// Responses to an authentication challenge.
///
/// See https://developer.apple.com/documentation/foundation/urlsession/authchallengedisposition.
enum UrlSessionAuthChallengeDisposition: Int {
  /// Use the specified credential, which may be nil.
  case useCredential = 0
  /// Use the default handling for the challenge as though this delegate method
  /// were not implemented.
  case performDefaultHandling = 1
  /// Cancel the entire request.
  case cancelAuthenticationChallenge = 2
  /// Reject this challenge, and call the authentication delegate method again
  /// with the next authentication protection space.
  case rejectProtectionSpace = 3
  /// The value is not recognized by the wrapper.
  case unknown = 4
}

/// Specifies how long a credential will be kept.
///
/// See https://developer.apple.com/documentation/foundation/nsurlcredentialpersistence.
enum UrlCredentialPersistence: Int {
  /// The credential should not be stored.
  case none = 0
  /// The credential should be stored only for this session.
  case forSession = 1
  /// The credential should be stored in the keychain.
  case permanent = 2
  /// The credential should be stored permanently in the keychain, and in
  /// addition should be distributed to other devices based on the owning Apple
  /// ID.
  case synchronizable = 3
}

/// Trust evaluation result codes.
///
/// See https://developer.apple.com/documentation/security/sectrustresulttype?language=objc.
enum DartSecTrustResultType: Int {
  /// The user did not specify a trust setting.
  case unspecified = 0
  /// The user granted permission to trust the certificate for the purposes
  /// designated in the specified policies.
  case proceed = 1
  /// The user specified that the certificate should not be trusted.
  case deny = 2
  /// Trust is denied, but recovery may be possible.
  case recoverableTrustFailure = 3
  /// Trust is denied and no simple fix is available.
  case fatalTrustFailure = 4
  /// A value that indicates a failure other than trust evaluation.
  case otherError = 5
  /// An indication of an invalid setting or result.
  case invalid = 6
  /// User confirmation is required before proceeding.
  case confirm = 7
  /// The type is not recognized by this wrapper.
  case unknown = 8
}

private class WebKitLibraryPigeonCodecReader: FlutterStandardReader {
  override func readValue(ofType type: UInt8) -> Any? {
    switch type {
    case 129:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return KeyValueObservingOptions(rawValue: enumResultAsInt)
      }
      return nil
    case 130:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return KeyValueChange(rawValue: enumResultAsInt)
      }
      return nil
    case 131:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return KeyValueChangeKey(rawValue: enumResultAsInt)
      }
      return nil
    case 132:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return UserScriptInjectionTime(rawValue: enumResultAsInt)
      }
      return nil
    case 133:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return AudiovisualMediaType(rawValue: enumResultAsInt)
      }
      return nil
    case 134:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return WebsiteDataType(rawValue: enumResultAsInt)
      }
      return nil
    case 135:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return NavigationActionPolicy(rawValue: enumResultAsInt)
      }
      return nil
    case 136:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return NavigationResponsePolicy(rawValue: enumResultAsInt)
      }
      return nil
    case 137:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return HttpCookiePropertyKey(rawValue: enumResultAsInt)
      }
      return nil
    case 138:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return NavigationType(rawValue: enumResultAsInt)
      }
      return nil
    case 139:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return PermissionDecision(rawValue: enumResultAsInt)
      }
      return nil
    case 140:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return MediaCaptureType(rawValue: enumResultAsInt)
      }
      return nil
    case 141:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return UrlSessionAuthChallengeDisposition(rawValue: enumResultAsInt)
      }
      return nil
    case 142:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return UrlCredentialPersistence(rawValue: enumResultAsInt)
      }
      return nil
    case 143:
      let enumResultAsInt: Int? = nilOrValue(self.readValue() as! Int?)
      if let enumResultAsInt = enumResultAsInt {
        return DartSecTrustResultType(rawValue: enumResultAsInt)
      }
      return nil
    default:
      return super.readValue(ofType: type)
    }
  }
}

private class WebKitLibraryPigeonCodecWriter: FlutterStandardWriter {
  override func writeValue(_ value: Any) {
    if let value = value as? KeyValueObservingOptions {
      super.writeByte(129)
      super.writeValue(value.rawValue)
    } else if let value = value as? KeyValueChange {
      super.writeByte(130)
      super.writeValue(value.rawValue)
    } else if let value = value as? KeyValueChangeKey {
      super.writeByte(131)
      super.writeValue(value.rawValue)
    } else if let value = value as? UserScriptInjectionTime {
      super.writeByte(132)
      super.writeValue(value.rawValue)
    } else if let value = value as? AudiovisualMediaType {
      super.writeByte(133)
      super.writeValue(value.rawValue)
    } else if let value = value as? WebsiteDataType {
      super.writeByte(134)
      super.writeValue(value.rawValue)
    } else if let value = value as? NavigationActionPolicy {
      super.writeByte(135)
      super.writeValue(value.rawValue)
    } else if let value = value as? NavigationResponsePolicy {
      super.writeByte(136)
      super.writeValue(value.rawValue)
    } else if let value = value as? HttpCookiePropertyKey {
      super.writeByte(137)
      super.writeValue(value.rawValue)
    } else if let value = value as? NavigationType {
      super.writeByte(138)
      super.writeValue(value.rawValue)
    } else if let value = value as? PermissionDecision {
      super.writeByte(139)
      super.writeValue(value.rawValue)
    } else if let value = value as? MediaCaptureType {
      super.writeByte(140)
      super.writeValue(value.rawValue)
    } else if let value = value as? UrlSessionAuthChallengeDisposition {
      super.writeByte(141)
      super.writeValue(value.rawValue)
    } else if let value = value as? UrlCredentialPersistence {
      super.writeByte(142)
      super.writeValue(value.rawValue)
    } else if let value = value as? DartSecTrustResultType {
      super.writeByte(143)
      super.writeValue(value.rawValue)
    } else {
      super.writeValue(value)
    }
  }
}

private class WebKitLibraryPigeonCodecReaderWriter: FlutterStandardReaderWriter {
  override func reader(with data: Data) -> FlutterStandardReader {
    return WebKitLibraryPigeonCodecReader(data: data)
  }

  override func writer(with data: NSMutableData) -> FlutterStandardWriter {
    return WebKitLibraryPigeonCodecWriter(data: data)
  }
}

class WebKitLibraryPigeonCodec: FlutterStandardMessageCodec, @unchecked Sendable {
  static let shared = WebKitLibraryPigeonCodec(readerWriter: WebKitLibraryPigeonCodecReaderWriter())
}

protocol PigeonApiDelegateURLRequest {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiURLRequest, url: String) throws
    -> URLRequestWrapper
  /// The URL being requested.
  func getUrl(pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper) throws -> String?
  /// The HTTP request method.
  func setHttpMethod(
    pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper, method: String?) throws
  /// The HTTP request method.
  func getHttpMethod(pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper) throws
    -> String?
  /// The request body.
  func setHttpBody(
    pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper,
    body: FlutterStandardTypedData?) throws
  /// The request body.
  func getHttpBody(pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper) throws
    -> FlutterStandardTypedData?
  /// A dictionary containing all of the HTTP header fields for a request.
  func setAllHttpHeaderFields(
    pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper, fields: [String: String]?)
    throws
  /// A dictionary containing all of the HTTP header fields for a request.
  func getAllHttpHeaderFields(pigeonApi: PigeonApiURLRequest, pigeonInstance: URLRequestWrapper)
    throws -> [String: String]?
}

protocol PigeonApiProtocolURLRequest {
}

final class PigeonApiURLRequest: PigeonApiProtocolURLRequest {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURLRequest
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateURLRequest)
  {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiURLRequest?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        let urlArg = args[1] as! String
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api, url: urlArg),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
    let getUrlChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.getUrl",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getUrlChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        do {
          let result = try api.pigeonDelegate.getUrl(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getUrlChannel.setMessageHandler(nil)
    }
    let setHttpMethodChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.setHttpMethod",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setHttpMethodChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        let methodArg: String? = nilOrValue(args[1])
        do {
          try api.pigeonDelegate.setHttpMethod(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, method: methodArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setHttpMethodChannel.setMessageHandler(nil)
    }
    let getHttpMethodChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.getHttpMethod",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getHttpMethodChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        do {
          let result = try api.pigeonDelegate.getHttpMethod(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getHttpMethodChannel.setMessageHandler(nil)
    }
    let setHttpBodyChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.setHttpBody",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setHttpBodyChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        let bodyArg: FlutterStandardTypedData? = nilOrValue(args[1])
        do {
          try api.pigeonDelegate.setHttpBody(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, body: bodyArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setHttpBodyChannel.setMessageHandler(nil)
    }
    let getHttpBodyChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.getHttpBody",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getHttpBodyChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        do {
          let result = try api.pigeonDelegate.getHttpBody(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getHttpBodyChannel.setMessageHandler(nil)
    }
    let setAllHttpHeaderFieldsChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.setAllHttpHeaderFields",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setAllHttpHeaderFieldsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        let fieldsArg: [String: String]? = nilOrValue(args[1])
        do {
          try api.pigeonDelegate.setAllHttpHeaderFields(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, fields: fieldsArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setAllHttpHeaderFieldsChannel.setMessageHandler(nil)
    }
    let getAllHttpHeaderFieldsChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.getAllHttpHeaderFields",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getAllHttpHeaderFieldsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLRequestWrapper
        do {
          let result = try api.pigeonDelegate.getAllHttpHeaderFields(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getAllHttpHeaderFieldsChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of URLRequest and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URLRequestWrapper, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLRequest.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateHTTPURLResponse {
  /// The response’s HTTP status code.
  func statusCode(pigeonApi: PigeonApiHTTPURLResponse, pigeonInstance: HTTPURLResponse) throws
    -> Int64
}

protocol PigeonApiProtocolHTTPURLResponse {
}

final class PigeonApiHTTPURLResponse: PigeonApiProtocolHTTPURLResponse {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateHTTPURLResponse
  ///An implementation of [URLResponse] used to access callback methods
  var pigeonApiURLResponse: PigeonApiURLResponse {
    return pigeonRegistrar.apiDelegate.pigeonApiURLResponse(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateHTTPURLResponse
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of HTTPURLResponse and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: HTTPURLResponse, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let statusCodeArg = try! pigeonDelegate.statusCode(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.HTTPURLResponse.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, statusCodeArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
open class PigeonApiDelegateURLResponse {
}

protocol PigeonApiProtocolURLResponse {
}

final class PigeonApiURLResponse: PigeonApiProtocolURLResponse {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURLResponse
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateURLResponse
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of URLResponse and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URLResponse, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLResponse.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKUserScript {
  /// Creates a user script object that contains the specified source code and
  /// attributes.
  func pigeonDefaultConstructor(
    pigeonApi: PigeonApiWKUserScript, source: String, injectionTime: UserScriptInjectionTime,
    isForMainFrameOnly: Bool
  ) throws -> WKUserScript
  /// The script’s source code.
  func source(pigeonApi: PigeonApiWKUserScript, pigeonInstance: WKUserScript) throws -> String
  /// The time at which to inject the script into the webpage.
  func injectionTime(pigeonApi: PigeonApiWKUserScript, pigeonInstance: WKUserScript) throws
    -> UserScriptInjectionTime
  /// A Boolean value that indicates whether to inject the script into the main
  /// frame or all frames.
  func isForMainFrameOnly(pigeonApi: PigeonApiWKUserScript, pigeonInstance: WKUserScript) throws
    -> Bool
}

protocol PigeonApiProtocolWKUserScript {
}

final class PigeonApiWKUserScript: PigeonApiProtocolWKUserScript {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKUserScript
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateWKUserScript
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKUserScript?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserScript.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        let sourceArg = args[1] as! String
        let injectionTimeArg = args[2] as! UserScriptInjectionTime
        let isForMainFrameOnlyArg = args[3] as! Bool
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(
              pigeonApi: api, source: sourceArg, injectionTime: injectionTimeArg,
              isForMainFrameOnly: isForMainFrameOnlyArg),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKUserScript and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKUserScript, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let sourceArg = try! pigeonDelegate.source(pigeonApi: self, pigeonInstance: pigeonInstance)
      let injectionTimeArg = try! pigeonDelegate.injectionTime(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let isForMainFrameOnlyArg = try! pigeonDelegate.isForMainFrameOnly(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserScript.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage(
        [pigeonIdentifierArg, sourceArg, injectionTimeArg, isForMainFrameOnlyArg] as [Any?]
      ) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKNavigationAction {
  /// The URL request object associated with the navigation action.
  func request(pigeonApi: PigeonApiWKNavigationAction, pigeonInstance: WKNavigationAction) throws
    -> URLRequestWrapper
  /// The frame in which to display the new content.
  ///
  /// If the target of the navigation is a new window, this property is nil.
  func targetFrame(pigeonApi: PigeonApiWKNavigationAction, pigeonInstance: WKNavigationAction)
    throws -> WKFrameInfo?
  /// The type of action that triggered the navigation.
  func navigationType(pigeonApi: PigeonApiWKNavigationAction, pigeonInstance: WKNavigationAction)
    throws -> NavigationType
}

protocol PigeonApiProtocolWKNavigationAction {
}

final class PigeonApiWKNavigationAction: PigeonApiProtocolWKNavigationAction {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKNavigationAction
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKNavigationAction
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKNavigationAction and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKNavigationAction, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let requestArg = try! pigeonDelegate.request(pigeonApi: self, pigeonInstance: pigeonInstance)
      let targetFrameArg = try! pigeonDelegate.targetFrame(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let navigationTypeArg = try! pigeonDelegate.navigationType(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationAction.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage(
        [pigeonIdentifierArg, requestArg, targetFrameArg, navigationTypeArg] as [Any?]
      ) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKNavigationResponse {
  /// The frame’s response.
  func response(pigeonApi: PigeonApiWKNavigationResponse, pigeonInstance: WKNavigationResponse)
    throws -> URLResponse
  /// A Boolean value that indicates whether the response targets the web view’s
  /// main frame.
  func isForMainFrame(
    pigeonApi: PigeonApiWKNavigationResponse, pigeonInstance: WKNavigationResponse
  ) throws -> Bool
}

protocol PigeonApiProtocolWKNavigationResponse {
}

final class PigeonApiWKNavigationResponse: PigeonApiProtocolWKNavigationResponse {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKNavigationResponse
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKNavigationResponse
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKNavigationResponse and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKNavigationResponse, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let responseArg = try! pigeonDelegate.response(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let isForMainFrameArg = try! pigeonDelegate.isForMainFrame(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationResponse.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, responseArg, isForMainFrameArg] as [Any?]) {
        response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKFrameInfo {
  /// A Boolean value indicating whether the frame is the web site's main frame
  /// or a subframe.
  func isMainFrame(pigeonApi: PigeonApiWKFrameInfo, pigeonInstance: WKFrameInfo) throws -> Bool
  /// The frame’s current request.
  func request(pigeonApi: PigeonApiWKFrameInfo, pigeonInstance: WKFrameInfo) throws
    -> URLRequestWrapper?
}

protocol PigeonApiProtocolWKFrameInfo {
}

final class PigeonApiWKFrameInfo: PigeonApiProtocolWKFrameInfo {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKFrameInfo
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateWKFrameInfo
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKFrameInfo and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKFrameInfo, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let isMainFrameArg = try! pigeonDelegate.isMainFrame(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let requestArg = try! pigeonDelegate.request(pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKFrameInfo.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, isMainFrameArg, requestArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateNSError {
  /// The error code.
  func code(pigeonApi: PigeonApiNSError, pigeonInstance: NSError) throws -> Int64
  /// A string containing the error domain.
  func domain(pigeonApi: PigeonApiNSError, pigeonInstance: NSError) throws -> String
  /// The user info dictionary.
  func userInfo(pigeonApi: PigeonApiNSError, pigeonInstance: NSError) throws -> [String: Any?]
}

protocol PigeonApiProtocolNSError {
}

final class PigeonApiNSError: PigeonApiProtocolNSError {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateNSError
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateNSError) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of NSError and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: NSError, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let codeArg = try! pigeonDelegate.code(pigeonApi: self, pigeonInstance: pigeonInstance)
      let domainArg = try! pigeonDelegate.domain(pigeonApi: self, pigeonInstance: pigeonInstance)
      let userInfoArg = try! pigeonDelegate.userInfo(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.NSError.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, codeArg, domainArg, userInfoArg] as [Any?]) {
        response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKScriptMessage {
  /// The name of the message handler to which the message is sent.
  func name(pigeonApi: PigeonApiWKScriptMessage, pigeonInstance: WKScriptMessage) throws -> String
  /// The body of the message.
  func body(pigeonApi: PigeonApiWKScriptMessage, pigeonInstance: WKScriptMessage) throws -> Any?
}

protocol PigeonApiProtocolWKScriptMessage {
}

final class PigeonApiWKScriptMessage: PigeonApiProtocolWKScriptMessage {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKScriptMessage
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKScriptMessage
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKScriptMessage and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKScriptMessage, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let nameArg = try! pigeonDelegate.name(pigeonApi: self, pigeonInstance: pigeonInstance)
      let bodyArg = try! pigeonDelegate.body(pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKScriptMessage.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, nameArg, bodyArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKSecurityOrigin {
  /// The security origin’s host.
  func host(pigeonApi: PigeonApiWKSecurityOrigin, pigeonInstance: WKSecurityOrigin) throws -> String
  /// The security origin's port.
  func port(pigeonApi: PigeonApiWKSecurityOrigin, pigeonInstance: WKSecurityOrigin) throws -> Int64
  /// The security origin's protocol.
  func securityProtocol(pigeonApi: PigeonApiWKSecurityOrigin, pigeonInstance: WKSecurityOrigin)
    throws -> String
}

protocol PigeonApiProtocolWKSecurityOrigin {
}

final class PigeonApiWKSecurityOrigin: PigeonApiProtocolWKSecurityOrigin {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKSecurityOrigin
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKSecurityOrigin
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKSecurityOrigin and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKSecurityOrigin, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let hostArg = try! pigeonDelegate.host(pigeonApi: self, pigeonInstance: pigeonInstance)
      let portArg = try! pigeonDelegate.port(pigeonApi: self, pigeonInstance: pigeonInstance)
      let securityProtocolArg = try! pigeonDelegate.securityProtocol(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKSecurityOrigin.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, hostArg, portArg, securityProtocolArg] as [Any?]) {
        response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateHTTPCookie {
  func pigeonDefaultConstructor(
    pigeonApi: PigeonApiHTTPCookie, properties: [HttpCookiePropertyKey: Any]
  ) throws -> HTTPCookie
  /// The cookie’s properties.
  func getProperties(pigeonApi: PigeonApiHTTPCookie, pigeonInstance: HTTPCookie) throws
    -> [HttpCookiePropertyKey: Any]?
}

protocol PigeonApiProtocolHTTPCookie {
}

final class PigeonApiHTTPCookie: PigeonApiProtocolHTTPCookie {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateHTTPCookie
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateHTTPCookie)
  {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiHTTPCookie?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.HTTPCookie.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        let propertiesArg = args[1] as? [HttpCookiePropertyKey: Any]
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(
              pigeonApi: api, properties: propertiesArg!),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
    let getPropertiesChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.HTTPCookie.getProperties",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getPropertiesChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! HTTPCookie
        do {
          let result = try api.pigeonDelegate.getProperties(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getPropertiesChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of HTTPCookie and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: HTTPCookie, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.HTTPCookie.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateAuthenticationChallengeResponse {
  /// Creates an [AuthenticationChallengeResponse].
  ///
  /// Due to https://github.com/flutter/flutter/issues/162437, this should only
  /// be used for testing.
  func pigeonDefaultConstructor(
    pigeonApi: PigeonApiAuthenticationChallengeResponse,
    disposition: UrlSessionAuthChallengeDisposition, credential: URLCredential?
  ) throws -> AuthenticationChallengeResponse
  /// The option to use to handle the challenge.
  func disposition(
    pigeonApi: PigeonApiAuthenticationChallengeResponse,
    pigeonInstance: AuthenticationChallengeResponse
  ) throws -> UrlSessionAuthChallengeDisposition
  /// The credential to use for authentication when the disposition parameter
  /// contains the value URLSession.AuthChallengeDisposition.useCredential.
  func credential(
    pigeonApi: PigeonApiAuthenticationChallengeResponse,
    pigeonInstance: AuthenticationChallengeResponse
  ) throws -> URLCredential?
  /// Creates an [AuthenticationChallengeResponse]
  ///
  /// This provides the native `AuthenticationChallengeResponse()` constructor
  /// as an async method to ensure the class is added to the InstanceManager.
  /// See https://github.com/flutter/flutter/issues/162437.
  func createAsync(
    pigeonApi: PigeonApiAuthenticationChallengeResponse,
    disposition: UrlSessionAuthChallengeDisposition, credential: URLCredential?,
    completion: @escaping (Result<AuthenticationChallengeResponse, Error>) -> Void)
}

protocol PigeonApiProtocolAuthenticationChallengeResponse {
}

final class PigeonApiAuthenticationChallengeResponse:
  PigeonApiProtocolAuthenticationChallengeResponse
{
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateAuthenticationChallengeResponse
  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateAuthenticationChallengeResponse
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiAuthenticationChallengeResponse?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.AuthenticationChallengeResponse.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        let dispositionArg = args[1] as! UrlSessionAuthChallengeDisposition
        let credentialArg: URLCredential? = nilOrValue(args[2])
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(
              pigeonApi: api, disposition: dispositionArg, credential: credentialArg),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
    let createAsyncChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.AuthenticationChallengeResponse.createAsync",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      createAsyncChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let dispositionArg = args[0] as! UrlSessionAuthChallengeDisposition
        let credentialArg: URLCredential? = nilOrValue(args[1])
        api.pigeonDelegate.createAsync(
          pigeonApi: api, disposition: dispositionArg, credential: credentialArg
        ) { result in
          switch result {
          case .success(let res):
            reply(wrapResult(res))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      createAsyncChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of AuthenticationChallengeResponse and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: AuthenticationChallengeResponse,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let dispositionArg = try! pigeonDelegate.disposition(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let credentialArg = try! pigeonDelegate.credential(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.AuthenticationChallengeResponse.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, dispositionArg, credentialArg] as [Any?]) {
        response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKWebsiteDataStore {
  /// The default data store, which stores data persistently to disk.
  func defaultDataStore(pigeonApi: PigeonApiWKWebsiteDataStore) throws -> WKWebsiteDataStore
  /// The object that manages the HTTP cookies for your website.
  func httpCookieStore(pigeonApi: PigeonApiWKWebsiteDataStore, pigeonInstance: WKWebsiteDataStore)
    throws -> WKHTTPCookieStore
  /// Removes the specified types of website data from one or more data records.
  func removeDataOfTypes(
    pigeonApi: PigeonApiWKWebsiteDataStore, pigeonInstance: WKWebsiteDataStore,
    dataTypes: [WebsiteDataType], modificationTimeInSecondsSinceEpoch: Double,
    completion: @escaping (Result<Bool, Error>) -> Void)
}

protocol PigeonApiProtocolWKWebsiteDataStore {
}

final class PigeonApiWKWebsiteDataStore: PigeonApiProtocolWKWebsiteDataStore {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKWebsiteDataStore
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKWebsiteDataStore
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKWebsiteDataStore?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let defaultDataStoreChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebsiteDataStore.defaultDataStore",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      defaultDataStoreChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.defaultDataStore(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      defaultDataStoreChannel.setMessageHandler(nil)
    }
    let httpCookieStoreChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebsiteDataStore.httpCookieStore",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      httpCookieStoreChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebsiteDataStore
        let pigeonIdentifierArg = args[1] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.httpCookieStore(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      httpCookieStoreChannel.setMessageHandler(nil)
    }
    let removeDataOfTypesChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebsiteDataStore.removeDataOfTypes",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      removeDataOfTypesChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebsiteDataStore
        let dataTypesArg = args[1] as! [WebsiteDataType]
        let modificationTimeInSecondsSinceEpochArg = args[2] as! Double
        api.pigeonDelegate.removeDataOfTypes(
          pigeonApi: api, pigeonInstance: pigeonInstanceArg, dataTypes: dataTypesArg,
          modificationTimeInSecondsSinceEpoch: modificationTimeInSecondsSinceEpochArg
        ) { result in
          switch result {
          case .success(let res):
            reply(wrapResult(res))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      removeDataOfTypesChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKWebsiteDataStore and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKWebsiteDataStore, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebsiteDataStore.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateUIView {
  #if !os(macOS)
    /// The view’s background color.
    func setBackgroundColor(pigeonApi: PigeonApiUIView, pigeonInstance: UIView, value: Int64?)
      throws
  #endif
  #if !os(macOS)
    /// A Boolean value that determines whether the view is opaque.
    func setOpaque(pigeonApi: PigeonApiUIView, pigeonInstance: UIView, opaque: Bool) throws
  #endif
}

protocol PigeonApiProtocolUIView {
}

final class PigeonApiUIView: PigeonApiProtocolUIView {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateUIView
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateUIView) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(binaryMessenger: FlutterBinaryMessenger, api: PigeonApiUIView?) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    #if !os(macOS)
      let setBackgroundColorChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIView.setBackgroundColor",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setBackgroundColorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIView
          let valueArg: Int64? = nilOrValue(args[1])
          do {
            try api.pigeonDelegate.setBackgroundColor(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setBackgroundColorChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setOpaqueChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIView.setOpaque",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setOpaqueChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIView
          let opaqueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setOpaque(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, opaque: opaqueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setOpaqueChannel.setMessageHandler(nil)
      }
    #endif
  }

  #if !os(macOS)
    ///Creates a Dart instance of UIView and attaches it to [pigeonInstance].
    func pigeonNewInstance(
      pigeonInstance: UIView, completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
      } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
        completion(.success(()))
      } else {
        let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
          pigeonInstance as AnyObject)
        let binaryMessenger = pigeonRegistrar.binaryMessenger
        let codec = pigeonRegistrar.codec
        let channelName: String =
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIView.pigeon_newInstance"
        let channel = FlutterBasicMessageChannel(
          name: channelName, binaryMessenger: binaryMessenger, codec: codec)
        channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
          guard let listResponse = response as? [Any?] else {
            completion(.failure(createConnectionError(withChannelName: channelName)))
            return
          }
          if listResponse.count > 1 {
            let code: String = listResponse[0] as! String
            let message: String? = nilOrValue(listResponse[1])
            let details: String? = nilOrValue(listResponse[2])
            completion(.failure(PigeonError(code: code, message: message, details: details)))
          } else {
            completion(.success(()))
          }
        }
      }
    }
  #endif
}
protocol PigeonApiDelegateUIScrollView {
  #if !os(macOS)
    /// The point at which the origin of the content view is offset from the
    /// origin of the scroll view.
    func getContentOffset(pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView) throws
      -> [Double]
  #endif
  #if !os(macOS)
    /// Move the scrolled position of your view.
    ///
    /// Convenience method to synchronize change to the x and y scroll position.
    func scrollBy(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, x: Double, y: Double) throws
  #endif
  #if !os(macOS)
    /// The point at which the origin of the content view is offset from the
    /// origin of the scroll view.
    func setContentOffset(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, x: Double, y: Double) throws
  #endif
  #if !os(macOS)
    /// The delegate of the scroll view.
    func setDelegate(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView,
      delegate: UIScrollViewDelegate?) throws
  #endif
  #if !os(macOS)
    /// Whether the scroll view bounces past the edge of content and back again.
    func setBounces(pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool)
      throws
  #endif
  #if !os(macOS)
    /// Whether the scroll view bounces when it reaches the ends of its horizontal
    /// axis.
    func setBouncesHorizontally(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
  #if !os(macOS)
    /// Whether the scroll view bounces when it reaches the ends of its vertical
    /// axis.
    func setBouncesVertically(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
  #if !os(macOS)
    /// Whether bouncing always occurs when vertical scrolling reaches the end of
    /// the content.
    ///
    /// If the value of this property is true and `bouncesVertically` is true, the
    /// scroll view allows vertical dragging even if the content is smaller than
    /// the bounds of the scroll view.
    func setAlwaysBounceVertical(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
  #if !os(macOS)
    /// Whether bouncing always occurs when horizontal scrolling reaches the end
    /// of the content view.
    ///
    /// If the value of this property is true and `bouncesHorizontally` is true,
    /// the scroll view allows horizontal dragging even if the content is smaller
    /// than the bounds of the scroll view.
    func setAlwaysBounceHorizontal(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
  #if !os(macOS)
    /// Whether the vertical scroll indicator is visible.
    ///
    /// The default value is true.
    func setShowsVerticalScrollIndicator(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
  #if !os(macOS)
    /// Whether the horizontal scroll indicator is visible.
    ///
    /// The default value is true.
    func setShowsHorizontalScrollIndicator(
      pigeonApi: PigeonApiUIScrollView, pigeonInstance: UIScrollView, value: Bool) throws
  #endif
}

protocol PigeonApiProtocolUIScrollView {
}

final class PigeonApiUIScrollView: PigeonApiProtocolUIScrollView {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateUIScrollView
  ///An implementation of [UIView] used to access callback methods
  var pigeonApiUIView: PigeonApiUIView {
    return pigeonRegistrar.apiDelegate.pigeonApiUIView(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateUIScrollView
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiUIScrollView?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    #if !os(macOS)
      let getContentOffsetChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.getContentOffset",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getContentOffsetChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          do {
            let result = try api.pigeonDelegate.getContentOffset(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getContentOffsetChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let scrollByChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.scrollBy",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        scrollByChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let xArg = args[1] as! Double
          let yArg = args[2] as! Double
          do {
            try api.pigeonDelegate.scrollBy(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, x: xArg, y: yArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        scrollByChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setContentOffsetChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setContentOffset",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setContentOffsetChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let xArg = args[1] as! Double
          let yArg = args[2] as! Double
          do {
            try api.pigeonDelegate.setContentOffset(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, x: xArg, y: yArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setContentOffsetChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setDelegateChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setDelegate",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setDelegateChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let delegateArg: UIScrollViewDelegate? = nilOrValue(args[1])
          do {
            try api.pigeonDelegate.setDelegate(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, delegate: delegateArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setDelegateChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setBouncesChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setBounces",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setBouncesChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setBounces(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setBouncesChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setBouncesHorizontallyChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setBouncesHorizontally",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setBouncesHorizontallyChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setBouncesHorizontally(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setBouncesHorizontallyChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setBouncesVerticallyChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setBouncesVertically",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setBouncesVerticallyChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setBouncesVertically(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setBouncesVerticallyChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setAlwaysBounceVerticalChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setAlwaysBounceVertical",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAlwaysBounceVerticalChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAlwaysBounceVertical(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAlwaysBounceVerticalChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setAlwaysBounceHorizontalChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setAlwaysBounceHorizontal",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAlwaysBounceHorizontalChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAlwaysBounceHorizontal(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAlwaysBounceHorizontalChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setShowsVerticalScrollIndicatorChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setShowsVerticalScrollIndicator",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setShowsVerticalScrollIndicatorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setShowsVerticalScrollIndicator(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setShowsVerticalScrollIndicatorChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setShowsHorizontalScrollIndicatorChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.setShowsHorizontalScrollIndicator",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setShowsHorizontalScrollIndicatorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! UIScrollView
          let valueArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setShowsHorizontalScrollIndicator(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, value: valueArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setShowsHorizontalScrollIndicatorChannel.setMessageHandler(nil)
      }
    #endif
  }

  #if !os(macOS)
    ///Creates a Dart instance of UIScrollView and attaches it to [pigeonInstance].
    func pigeonNewInstance(
      pigeonInstance: UIScrollView, completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
      } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
        completion(.success(()))
      } else {
        let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
          pigeonInstance as AnyObject)
        let binaryMessenger = pigeonRegistrar.binaryMessenger
        let codec = pigeonRegistrar.codec
        let channelName: String =
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollView.pigeon_newInstance"
        let channel = FlutterBasicMessageChannel(
          name: channelName, binaryMessenger: binaryMessenger, codec: codec)
        channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
          guard let listResponse = response as? [Any?] else {
            completion(.failure(createConnectionError(withChannelName: channelName)))
            return
          }
          if listResponse.count > 1 {
            let code: String = listResponse[0] as! String
            let message: String? = nilOrValue(listResponse[1])
            let details: String? = nilOrValue(listResponse[2])
            completion(.failure(PigeonError(code: code, message: message, details: details)))
          } else {
            completion(.success(()))
          }
        }
      }
    }
  #endif
}
protocol PigeonApiDelegateWKWebViewConfiguration {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiWKWebViewConfiguration) throws
    -> WKWebViewConfiguration
  /// The object that coordinates interactions between your app’s native code
  /// and the webpage’s scripts and other content.
  func setUserContentController(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration,
    controller: WKUserContentController) throws
  /// The object that coordinates interactions between your app’s native code
  /// and the webpage’s scripts and other content.
  func getUserContentController(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration
  ) throws -> WKUserContentController
  /// The object you use to get and set the site’s cookies and to track the
  /// cached data objects.
  func setWebsiteDataStore(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration,
    dataStore: WKWebsiteDataStore) throws
  /// The object you use to get and set the site’s cookies and to track the
  /// cached data objects.
  func getWebsiteDataStore(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration
  ) throws -> WKWebsiteDataStore
  /// The object that manages the preference-related settings for the web view.
  func setPreferences(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration,
    preferences: WKPreferences) throws
  /// The object that manages the preference-related settings for the web view.
  func getPreferences(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration
  ) throws -> WKPreferences
  /// A Boolean value that indicates whether HTML5 videos play inline or use the
  /// native full-screen controller.
  func setAllowsInlineMediaPlayback(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration, allow: Bool)
    throws
  /// A Boolean value that indicates whether the web view limits navigation to
  /// pages within the app’s domain.
  func setLimitsNavigationsToAppBoundDomains(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration, limit: Bool)
    throws
  /// The media types that require a user gesture to begin playing.
  func setMediaTypesRequiringUserActionForPlayback(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration,
    type: AudiovisualMediaType) throws
  /// The default preferences to use when loading and rendering content.
  func getDefaultWebpagePreferences(
    pigeonApi: PigeonApiWKWebViewConfiguration, pigeonInstance: WKWebViewConfiguration
  ) throws -> WKWebpagePreferences
}

protocol PigeonApiProtocolWKWebViewConfiguration {
}

final class PigeonApiWKWebViewConfiguration: PigeonApiProtocolWKWebViewConfiguration {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKWebViewConfiguration
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKWebViewConfiguration
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKWebViewConfiguration?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
    let setUserContentControllerChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setUserContentController",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setUserContentControllerChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let controllerArg = args[1] as! WKUserContentController
        do {
          try api.pigeonDelegate.setUserContentController(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, controller: controllerArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setUserContentControllerChannel.setMessageHandler(nil)
    }
    let getUserContentControllerChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.getUserContentController",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getUserContentControllerChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        do {
          let result = try api.pigeonDelegate.getUserContentController(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getUserContentControllerChannel.setMessageHandler(nil)
    }
    let setWebsiteDataStoreChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setWebsiteDataStore",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setWebsiteDataStoreChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let dataStoreArg = args[1] as! WKWebsiteDataStore
        do {
          try api.pigeonDelegate.setWebsiteDataStore(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, dataStore: dataStoreArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setWebsiteDataStoreChannel.setMessageHandler(nil)
    }
    let getWebsiteDataStoreChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.getWebsiteDataStore",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getWebsiteDataStoreChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        do {
          let result = try api.pigeonDelegate.getWebsiteDataStore(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getWebsiteDataStoreChannel.setMessageHandler(nil)
    }
    let setPreferencesChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setPreferences",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setPreferencesChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let preferencesArg = args[1] as! WKPreferences
        do {
          try api.pigeonDelegate.setPreferences(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, preferences: preferencesArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setPreferencesChannel.setMessageHandler(nil)
    }
    let getPreferencesChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.getPreferences",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getPreferencesChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        do {
          let result = try api.pigeonDelegate.getPreferences(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getPreferencesChannel.setMessageHandler(nil)
    }
    let setAllowsInlineMediaPlaybackChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setAllowsInlineMediaPlayback",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setAllowsInlineMediaPlaybackChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let allowArg = args[1] as! Bool
        do {
          try api.pigeonDelegate.setAllowsInlineMediaPlayback(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setAllowsInlineMediaPlaybackChannel.setMessageHandler(nil)
    }
    let setLimitsNavigationsToAppBoundDomainsChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setLimitsNavigationsToAppBoundDomains",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setLimitsNavigationsToAppBoundDomainsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let limitArg = args[1] as! Bool
        do {
          try api.pigeonDelegate.setLimitsNavigationsToAppBoundDomains(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, limit: limitArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setLimitsNavigationsToAppBoundDomainsChannel.setMessageHandler(nil)
    }
    let setMediaTypesRequiringUserActionForPlaybackChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.setMediaTypesRequiringUserActionForPlayback",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setMediaTypesRequiringUserActionForPlaybackChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        let typeArg = args[1] as! AudiovisualMediaType
        do {
          try api.pigeonDelegate.setMediaTypesRequiringUserActionForPlayback(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, type: typeArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setMediaTypesRequiringUserActionForPlaybackChannel.setMessageHandler(nil)
    }
    let getDefaultWebpagePreferencesChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.getDefaultWebpagePreferences",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getDefaultWebpagePreferencesChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebViewConfiguration
        do {
          let result = try api.pigeonDelegate.getDefaultWebpagePreferences(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getDefaultWebpagePreferencesChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKWebViewConfiguration and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKWebViewConfiguration,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebViewConfiguration.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKUserContentController {
  /// Installs a message handler that you can call from your JavaScript code.
  func addScriptMessageHandler(
    pigeonApi: PigeonApiWKUserContentController, pigeonInstance: WKUserContentController,
    handler: WKScriptMessageHandler, name: String) throws
  /// Uninstalls the custom message handler with the specified name from your
  /// JavaScript code.
  func removeScriptMessageHandler(
    pigeonApi: PigeonApiWKUserContentController, pigeonInstance: WKUserContentController,
    name: String) throws
  /// Uninstalls all custom message handlers associated with the user content
  /// controller.
  func removeAllScriptMessageHandlers(
    pigeonApi: PigeonApiWKUserContentController, pigeonInstance: WKUserContentController) throws
  /// Injects the specified script into the webpage’s content.
  func addUserScript(
    pigeonApi: PigeonApiWKUserContentController, pigeonInstance: WKUserContentController,
    userScript: WKUserScript) throws
  /// Removes all user scripts from the web view.
  func removeAllUserScripts(
    pigeonApi: PigeonApiWKUserContentController, pigeonInstance: WKUserContentController) throws
}

protocol PigeonApiProtocolWKUserContentController {
}

final class PigeonApiWKUserContentController: PigeonApiProtocolWKUserContentController {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKUserContentController
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKUserContentController
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKUserContentController?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let addScriptMessageHandlerChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.addScriptMessageHandler",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      addScriptMessageHandlerChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKUserContentController
        let handlerArg = args[1] as! WKScriptMessageHandler
        let nameArg = args[2] as! String
        do {
          try api.pigeonDelegate.addScriptMessageHandler(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, handler: handlerArg, name: nameArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      addScriptMessageHandlerChannel.setMessageHandler(nil)
    }
    let removeScriptMessageHandlerChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.removeScriptMessageHandler",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      removeScriptMessageHandlerChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKUserContentController
        let nameArg = args[1] as! String
        do {
          try api.pigeonDelegate.removeScriptMessageHandler(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, name: nameArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      removeScriptMessageHandlerChannel.setMessageHandler(nil)
    }
    let removeAllScriptMessageHandlersChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.removeAllScriptMessageHandlers",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      removeAllScriptMessageHandlersChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKUserContentController
        do {
          try api.pigeonDelegate.removeAllScriptMessageHandlers(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      removeAllScriptMessageHandlersChannel.setMessageHandler(nil)
    }
    let addUserScriptChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.addUserScript",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      addUserScriptChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKUserContentController
        let userScriptArg = args[1] as! WKUserScript
        do {
          try api.pigeonDelegate.addUserScript(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, userScript: userScriptArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      addUserScriptChannel.setMessageHandler(nil)
    }
    let removeAllUserScriptsChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.removeAllUserScripts",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      removeAllUserScriptsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKUserContentController
        do {
          try api.pigeonDelegate.removeAllUserScripts(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      removeAllUserScriptsChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKUserContentController and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKUserContentController,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKUserContentController.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKPreferences {
  /// A Boolean value that indicates whether JavaScript is enabled.
  func setJavaScriptEnabled(
    pigeonApi: PigeonApiWKPreferences, pigeonInstance: WKPreferences, enabled: Bool) throws
}

protocol PigeonApiProtocolWKPreferences {
}

final class PigeonApiWKPreferences: PigeonApiProtocolWKPreferences {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKPreferences
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateWKPreferences
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKPreferences?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let setJavaScriptEnabledChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKPreferences.setJavaScriptEnabled",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setJavaScriptEnabledChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKPreferences
        let enabledArg = args[1] as! Bool
        do {
          try api.pigeonDelegate.setJavaScriptEnabled(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, enabled: enabledArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setJavaScriptEnabledChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKPreferences and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKPreferences, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKPreferences.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKScriptMessageHandler {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiWKScriptMessageHandler) throws
    -> WKScriptMessageHandler
}

protocol PigeonApiProtocolWKScriptMessageHandler {
  /// Tells the handler that a webpage sent a script message.
  func didReceiveScriptMessage(
    pigeonInstance pigeonInstanceArg: WKScriptMessageHandler,
    controller controllerArg: WKUserContentController, message messageArg: WKScriptMessage,
    completion: @escaping (Result<Void, PigeonError>) -> Void)
}

final class PigeonApiWKScriptMessageHandler: PigeonApiProtocolWKScriptMessageHandler {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKScriptMessageHandler
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKScriptMessageHandler
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKScriptMessageHandler?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKScriptMessageHandler.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKScriptMessageHandler and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKScriptMessageHandler,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      completion(
        .failure(
          PigeonError(
            code: "new-instance-error",
            message:
              "Error: Attempting to create a new Dart instance of WKScriptMessageHandler, but the class has a nonnull callback method.",
            details: "")))
    }
  }
  /// Tells the handler that a webpage sent a script message.
  func didReceiveScriptMessage(
    pigeonInstance pigeonInstanceArg: WKScriptMessageHandler,
    controller controllerArg: WKUserContentController, message messageArg: WKScriptMessage,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKScriptMessageHandler.didReceiveScriptMessage"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, controllerArg, messageArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

}
protocol PigeonApiDelegateWKNavigationDelegate {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiWKNavigationDelegate) throws
    -> WKNavigationDelegate
}

protocol PigeonApiProtocolWKNavigationDelegate {
  /// Tells the delegate that navigation is complete.
  func didFinishNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    url urlArg: String?, completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Tells the delegate that navigation from the main frame has started.
  func didStartProvisionalNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    url urlArg: String?, completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Asks the delegate for permission to navigate to new content based on the
  /// specified action information.
  func decidePolicyForNavigationAction(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    navigationAction navigationActionArg: WKNavigationAction,
    completion: @escaping (Result<NavigationActionPolicy, PigeonError>) -> Void)
  /// Asks the delegate for permission to navigate to new content after the
  /// response to the navigation request is known.
  func decidePolicyForNavigationResponse(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    navigationResponse navigationResponseArg: WKNavigationResponse,
    completion: @escaping (Result<NavigationResponsePolicy, PigeonError>) -> Void)
  /// Tells the delegate that an error occurred during navigation.
  func didFailNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    error errorArg: NSError, completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Tells the delegate that an error occurred during the early navigation
  /// process.
  func didFailProvisionalNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    error errorArg: NSError, completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Tells the delegate that the web view’s content process was terminated.
  func webViewWebContentProcessDidTerminate(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Asks the delegate to respond to an authentication challenge.
  func didReceiveAuthenticationChallenge(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    challenge challengeArg: URLAuthenticationChallenge,
    completion: @escaping (Result<AuthenticationChallengeResponse, PigeonError>) -> Void)
}

final class PigeonApiWKNavigationDelegate: PigeonApiProtocolWKNavigationDelegate {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKNavigationDelegate
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKNavigationDelegate
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKNavigationDelegate?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKNavigationDelegate and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKNavigationDelegate, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      completion(
        .failure(
          PigeonError(
            code: "new-instance-error",
            message:
              "Error: Attempting to create a new Dart instance of WKNavigationDelegate, but the class has a nonnull callback method.",
            details: "")))
    }
  }
  /// Tells the delegate that navigation is complete.
  func didFinishNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    url urlArg: String?, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.didFinishNavigation"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, urlArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Tells the delegate that navigation from the main frame has started.
  func didStartProvisionalNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    url urlArg: String?, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.didStartProvisionalNavigation"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, urlArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Asks the delegate for permission to navigate to new content based on the
  /// specified action information.
  func decidePolicyForNavigationAction(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    navigationAction navigationActionArg: WKNavigationAction,
    completion: @escaping (Result<NavigationActionPolicy, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.decidePolicyForNavigationAction"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, navigationActionArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else if listResponse[0] == nil {
        completion(
          .failure(
            PigeonError(
              code: "null-error",
              message: "Flutter api returned null value for non-null return value.", details: "")))
      } else {
        let result = listResponse[0] as! NavigationActionPolicy
        completion(.success(result))
      }
    }
  }

  /// Asks the delegate for permission to navigate to new content after the
  /// response to the navigation request is known.
  func decidePolicyForNavigationResponse(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    navigationResponse navigationResponseArg: WKNavigationResponse,
    completion: @escaping (Result<NavigationResponsePolicy, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.decidePolicyForNavigationResponse"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, navigationResponseArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else if listResponse[0] == nil {
        completion(
          .failure(
            PigeonError(
              code: "null-error",
              message: "Flutter api returned null value for non-null return value.", details: "")))
      } else {
        let result = listResponse[0] as! NavigationResponsePolicy
        completion(.success(result))
      }
    }
  }

  /// Tells the delegate that an error occurred during navigation.
  func didFailNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    error errorArg: NSError, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.didFailNavigation"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, errorArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Tells the delegate that an error occurred during the early navigation
  /// process.
  func didFailProvisionalNavigation(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    error errorArg: NSError, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.didFailProvisionalNavigation"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, errorArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Tells the delegate that the web view’s content process was terminated.
  func webViewWebContentProcessDidTerminate(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.webViewWebContentProcessDidTerminate"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Asks the delegate to respond to an authentication challenge.
  func didReceiveAuthenticationChallenge(
    pigeonInstance pigeonInstanceArg: WKNavigationDelegate, webView webViewArg: WKWebView,
    challenge challengeArg: URLAuthenticationChallenge,
    completion: @escaping (Result<AuthenticationChallengeResponse, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKNavigationDelegate.didReceiveAuthenticationChallenge"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, challengeArg] as [Any?]) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else if listResponse[0] == nil {
        completion(
          .failure(
            PigeonError(
              code: "null-error",
              message: "Flutter api returned null value for non-null return value.", details: "")))
      } else {
        let result = listResponse[0] as! AuthenticationChallengeResponse
        completion(.success(result))
      }
    }
  }

}
protocol PigeonApiDelegateNSObject {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiNSObject) throws -> NSObject
  /// Registers the observer object to receive KVO notifications for the key
  /// path relative to the object receiving this message.
  func addObserver(
    pigeonApi: PigeonApiNSObject, pigeonInstance: NSObject, observer: NSObject, keyPath: String,
    options: [KeyValueObservingOptions]) throws
  /// Stops the observer object from receiving change notifications for the
  /// property specified by the key path relative to the object receiving this
  /// message.
  func removeObserver(
    pigeonApi: PigeonApiNSObject, pigeonInstance: NSObject, observer: NSObject, keyPath: String)
    throws
}

protocol PigeonApiProtocolNSObject {
  /// Informs the observing object when the value at the specified key path
  /// relative to the observed object has changed.
  func observeValue(
    pigeonInstance pigeonInstanceArg: NSObject, keyPath keyPathArg: String?,
    object objectArg: NSObject?, change changeArg: [KeyValueChangeKey: Any?]?,
    completion: @escaping (Result<Void, PigeonError>) -> Void)
}

final class PigeonApiNSObject: PigeonApiProtocolNSObject {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateNSObject
  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateNSObject) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(binaryMessenger: FlutterBinaryMessenger, api: PigeonApiNSObject?)
  {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSObject.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
    let addObserverChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSObject.addObserver",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      addObserverChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! NSObject
        let observerArg = args[1] as! NSObject
        let keyPathArg = args[2] as! String
        let optionsArg = args[3] as! [KeyValueObservingOptions]
        do {
          try api.pigeonDelegate.addObserver(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, observer: observerArg,
            keyPath: keyPathArg, options: optionsArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      addObserverChannel.setMessageHandler(nil)
    }
    let removeObserverChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSObject.removeObserver",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      removeObserverChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! NSObject
        let observerArg = args[1] as! NSObject
        let keyPathArg = args[2] as! String
        do {
          try api.pigeonDelegate.removeObserver(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, observer: observerArg,
            keyPath: keyPathArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      removeObserverChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of NSObject and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: NSObject, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.NSObject.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
  /// Informs the observing object when the value at the specified key path
  /// relative to the observed object has changed.
  func observeValue(
    pigeonInstance pigeonInstanceArg: NSObject, keyPath keyPathArg: String?,
    object objectArg: NSObject?, change changeArg: [KeyValueChangeKey: Any?]?,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String = "dev.flutter.pigeon.webview_flutter_wkwebview.NSObject.observeValue"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, keyPathArg, objectArg, changeArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

}
protocol PigeonApiDelegateUIViewWKWebView {
  #if !os(macOS)
    func pigeonDefaultConstructor(
      pigeonApi: PigeonApiUIViewWKWebView, initialConfiguration: WKWebViewConfiguration
    ) throws -> WKWebView
  #endif
  #if !os(macOS)
    /// The object that contains the configuration details for the web view.
    func configuration(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
      -> WKWebViewConfiguration
  #endif
  #if !os(macOS)
    /// The scroll view associated with the web view.
    func scrollView(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
      -> UIScrollView
  #endif
  #if !os(macOS)
    /// The object you use to integrate custom user interface elements, such as
    /// contextual menus or panels, into web view interactions.
    func setUIDelegate(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, delegate: WKUIDelegate) throws
  #endif
  #if !os(macOS)
    /// The object you use to manage navigation behavior for the web view.
    func setNavigationDelegate(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, delegate: WKNavigationDelegate
    ) throws
  #endif
  #if !os(macOS)
    /// The URL for the current webpage.
    func getUrl(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws -> String?
  #endif
  #if !os(macOS)
    /// An estimate of what fraction of the current navigation has been loaded.
    func getEstimatedProgress(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
      -> Double
  #endif
  #if !os(macOS)
    /// Loads the web content that the specified URL request object references and
    /// navigates to that content.
    func load(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, request: URLRequestWrapper)
      throws
  #endif
  #if !os(macOS)
    /// Loads the contents of the specified HTML string and navigates to it.
    func loadHtmlString(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, string: String,
      baseUrl: String?) throws
  #endif
  #if !os(macOS)
    /// Loads the web content from the specified file and navigates to it.
    func loadFileUrl(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, url: String,
      readAccessUrl: String) throws
  #endif
  #if !os(macOS)
    /// Convenience method to load a Flutter asset.
    func loadFlutterAsset(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, key: String) throws
  #endif
  #if !os(macOS)
    /// A Boolean value that indicates whether there is a valid back item in the
    /// back-forward list.
    func canGoBack(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws -> Bool
  #endif
  #if !os(macOS)
    /// A Boolean value that indicates whether there is a valid forward item in
    /// the back-forward list.
    func canGoForward(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws -> Bool
  #endif
  #if !os(macOS)
    /// Navigates to the back item in the back-forward list.
    func goBack(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(macOS)
    /// Navigates to the forward item in the back-forward list.
    func goForward(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(macOS)
    /// Reloads the current webpage.
    func reload(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(macOS)
    /// The page title.
    func getTitle(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws -> String?
  #endif
  #if !os(macOS)
    /// A Boolean value that indicates whether horizontal swipe gestures trigger
    /// backward and forward page navigation.
    func setAllowsBackForwardNavigationGestures(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, allow: Bool) throws
  #endif
  #if !os(macOS)
    /// The custom user agent string.
    func setCustomUserAgent(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, userAgent: String?) throws
  #endif
  #if !os(macOS)
    /// Evaluates the specified JavaScript string.
    func evaluateJavaScript(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, javaScriptString: String,
      completion: @escaping (Result<Any?, Error>) -> Void)
  #endif
  #if !os(macOS)
    /// A Boolean value that indicates whether you can inspect the view with
    /// Safari Web Inspector.
    func setInspectable(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, inspectable: Bool) throws
  #endif
  #if !os(macOS)
    /// The custom user agent string.
    func getCustomUserAgent(pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView) throws
      -> String?
  #endif
  #if !os(macOS)
    /// Whether to allow previews for link destinations and detected data such as
    /// addresses and phone numbers.
    ///
    /// Defaults to true.
    func setAllowsLinkPreview(
      pigeonApi: PigeonApiUIViewWKWebView, pigeonInstance: WKWebView, allow: Bool) throws
  #endif
}

protocol PigeonApiProtocolUIViewWKWebView {
}

final class PigeonApiUIViewWKWebView: PigeonApiProtocolUIViewWKWebView {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateUIViewWKWebView
  ///An implementation of [UIView] used to access callback methods
  var pigeonApiUIView: PigeonApiUIView {
    return pigeonRegistrar.apiDelegate.pigeonApiUIView(pigeonRegistrar)
  }

  ///An implementation of [WKWebView] used to access callback methods
  var pigeonApiWKWebView: PigeonApiWKWebView {
    return pigeonRegistrar.apiDelegate.pigeonApiWKWebView(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateUIViewWKWebView
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiUIViewWKWebView?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    #if !os(macOS)
      let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.pigeon_defaultConstructor",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonIdentifierArg = args[0] as! Int64
          let initialConfigurationArg = args[1] as! WKWebViewConfiguration
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.pigeonDefaultConstructor(
                pigeonApi: api, initialConfiguration: initialConfigurationArg),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        pigeonDefaultConstructorChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let configurationChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.configuration",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        configurationChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let pigeonIdentifierArg = args[1] as! Int64
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.configuration(
                pigeonApi: api, pigeonInstance: pigeonInstanceArg),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        configurationChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let scrollViewChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.scrollView",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        scrollViewChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let pigeonIdentifierArg = args[1] as! Int64
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.scrollView(pigeonApi: api, pigeonInstance: pigeonInstanceArg),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        scrollViewChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setUIDelegateChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setUIDelegate",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setUIDelegateChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let delegateArg = args[1] as! WKUIDelegate
          do {
            try api.pigeonDelegate.setUIDelegate(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, delegate: delegateArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setUIDelegateChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setNavigationDelegateChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setNavigationDelegate",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setNavigationDelegateChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let delegateArg = args[1] as! WKNavigationDelegate
          do {
            try api.pigeonDelegate.setNavigationDelegate(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, delegate: delegateArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setNavigationDelegateChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let getUrlChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.getUrl",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getUrlChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getUrl(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getUrlChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let getEstimatedProgressChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.getEstimatedProgress",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getEstimatedProgressChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getEstimatedProgress(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getEstimatedProgressChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let loadChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.load",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let requestArg = args[1] as! URLRequestWrapper
          do {
            try api.pigeonDelegate.load(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, request: requestArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let loadHtmlStringChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.loadHtmlString",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadHtmlStringChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let stringArg = args[1] as! String
          let baseUrlArg: String? = nilOrValue(args[2])
          do {
            try api.pigeonDelegate.loadHtmlString(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, string: stringArg,
              baseUrl: baseUrlArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadHtmlStringChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let loadFileUrlChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.loadFileUrl",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadFileUrlChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let urlArg = args[1] as! String
          let readAccessUrlArg = args[2] as! String
          do {
            try api.pigeonDelegate.loadFileUrl(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, url: urlArg,
              readAccessUrl: readAccessUrlArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadFileUrlChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let loadFlutterAssetChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.loadFlutterAsset",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadFlutterAssetChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let keyArg = args[1] as! String
          do {
            try api.pigeonDelegate.loadFlutterAsset(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, key: keyArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadFlutterAssetChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let canGoBackChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.canGoBack",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        canGoBackChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.canGoBack(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        canGoBackChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let canGoForwardChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.canGoForward",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        canGoForwardChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.canGoForward(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        canGoForwardChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let goBackChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.goBack",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        goBackChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.goBack(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        goBackChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let goForwardChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.goForward",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        goForwardChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.goForward(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        goForwardChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let reloadChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.reload",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        reloadChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.reload(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        reloadChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let getTitleChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.getTitle",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getTitleChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getTitle(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getTitleChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setAllowsBackForwardNavigationGesturesChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setAllowsBackForwardNavigationGestures",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAllowsBackForwardNavigationGesturesChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let allowArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAllowsBackForwardNavigationGestures(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAllowsBackForwardNavigationGesturesChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setCustomUserAgentChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setCustomUserAgent",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setCustomUserAgentChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let userAgentArg: String? = nilOrValue(args[1])
          do {
            try api.pigeonDelegate.setCustomUserAgent(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, userAgent: userAgentArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setCustomUserAgentChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let evaluateJavaScriptChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.evaluateJavaScript",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        evaluateJavaScriptChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let javaScriptStringArg = args[1] as! String
          api.pigeonDelegate.evaluateJavaScript(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, javaScriptString: javaScriptStringArg
          ) { result in
            switch result {
            case .success(let res):
              reply(wrapResult(res))
            case .failure(let error):
              reply(wrapError(error))
            }
          }
        }
      } else {
        evaluateJavaScriptChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setInspectableChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setInspectable",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setInspectableChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let inspectableArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setInspectable(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, inspectable: inspectableArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setInspectableChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let getCustomUserAgentChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.getCustomUserAgent",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getCustomUserAgentChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getCustomUserAgent(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getCustomUserAgentChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(macOS)
      let setAllowsLinkPreviewChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.setAllowsLinkPreview",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAllowsLinkPreviewChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let allowArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAllowsLinkPreview(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAllowsLinkPreviewChannel.setMessageHandler(nil)
      }
    #endif
  }

  #if !os(macOS)
    ///Creates a Dart instance of UIViewWKWebView and attaches it to [pigeonInstance].
    func pigeonNewInstance(
      pigeonInstance: WKWebView, completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
      } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
        completion(.success(()))
      } else {
        let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
          pigeonInstance as AnyObject)
        let binaryMessenger = pigeonRegistrar.binaryMessenger
        let codec = pigeonRegistrar.codec
        let channelName: String =
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIViewWKWebView.pigeon_newInstance"
        let channel = FlutterBasicMessageChannel(
          name: channelName, binaryMessenger: binaryMessenger, codec: codec)
        channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
          guard let listResponse = response as? [Any?] else {
            completion(.failure(createConnectionError(withChannelName: channelName)))
            return
          }
          if listResponse.count > 1 {
            let code: String = listResponse[0] as! String
            let message: String? = nilOrValue(listResponse[1])
            let details: String? = nilOrValue(listResponse[2])
            completion(.failure(PigeonError(code: code, message: message, details: details)))
          } else {
            completion(.success(()))
          }
        }
      }
    }
  #endif
}
protocol PigeonApiDelegateNSViewWKWebView {
  #if !os(iOS)
    func pigeonDefaultConstructor(
      pigeonApi: PigeonApiNSViewWKWebView, initialConfiguration: WKWebViewConfiguration
    ) throws -> WKWebView
  #endif
  #if !os(iOS)
    /// The object that contains the configuration details for the web view.
    func configuration(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
      -> WKWebViewConfiguration
  #endif
  #if !os(iOS)
    /// The object you use to integrate custom user interface elements, such as
    /// contextual menus or panels, into web view interactions.
    func setUIDelegate(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, delegate: WKUIDelegate) throws
  #endif
  #if !os(iOS)
    /// The object you use to manage navigation behavior for the web view.
    func setNavigationDelegate(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, delegate: WKNavigationDelegate
    ) throws
  #endif
  #if !os(iOS)
    /// The URL for the current webpage.
    func getUrl(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws -> String?
  #endif
  #if !os(iOS)
    /// An estimate of what fraction of the current navigation has been loaded.
    func getEstimatedProgress(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
      -> Double
  #endif
  #if !os(iOS)
    /// Loads the web content that the specified URL request object references and
    /// navigates to that content.
    func load(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, request: URLRequestWrapper)
      throws
  #endif
  #if !os(iOS)
    /// Loads the contents of the specified HTML string and navigates to it.
    func loadHtmlString(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, string: String,
      baseUrl: String?) throws
  #endif
  #if !os(iOS)
    /// Loads the web content from the specified file and navigates to it.
    func loadFileUrl(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, url: String,
      readAccessUrl: String) throws
  #endif
  #if !os(iOS)
    /// Convenience method to load a Flutter asset.
    func loadFlutterAsset(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, key: String) throws
  #endif
  #if !os(iOS)
    /// A Boolean value that indicates whether there is a valid back item in the
    /// back-forward list.
    func canGoBack(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws -> Bool
  #endif
  #if !os(iOS)
    /// A Boolean value that indicates whether there is a valid forward item in
    /// the back-forward list.
    func canGoForward(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws -> Bool
  #endif
  #if !os(iOS)
    /// Navigates to the back item in the back-forward list.
    func goBack(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(iOS)
    /// Navigates to the forward item in the back-forward list.
    func goForward(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(iOS)
    /// Reloads the current webpage.
    func reload(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
  #endif
  #if !os(iOS)
    /// The page title.
    func getTitle(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws -> String?
  #endif
  #if !os(iOS)
    /// A Boolean value that indicates whether horizontal swipe gestures trigger
    /// backward and forward page navigation.
    func setAllowsBackForwardNavigationGestures(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, allow: Bool) throws
  #endif
  #if !os(iOS)
    /// The custom user agent string.
    func setCustomUserAgent(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, userAgent: String?) throws
  #endif
  #if !os(iOS)
    /// Evaluates the specified JavaScript string.
    func evaluateJavaScript(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, javaScriptString: String,
      completion: @escaping (Result<Any?, Error>) -> Void)
  #endif
  #if !os(iOS)
    /// A Boolean value that indicates whether you can inspect the view with
    /// Safari Web Inspector.
    func setInspectable(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, inspectable: Bool) throws
  #endif
  #if !os(iOS)
    /// The custom user agent string.
    func getCustomUserAgent(pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView) throws
      -> String?
  #endif
  #if !os(iOS)
    /// Whether to allow previews for link destinations and detected data such as
    /// addresses and phone numbers.
    ///
    /// Defaults to true.
    func setAllowsLinkPreview(
      pigeonApi: PigeonApiNSViewWKWebView, pigeonInstance: WKWebView, allow: Bool) throws
  #endif
}

protocol PigeonApiProtocolNSViewWKWebView {
}

final class PigeonApiNSViewWKWebView: PigeonApiProtocolNSViewWKWebView {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateNSViewWKWebView
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  ///An implementation of [WKWebView] used to access callback methods
  var pigeonApiWKWebView: PigeonApiWKWebView {
    return pigeonRegistrar.apiDelegate.pigeonApiWKWebView(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateNSViewWKWebView
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiNSViewWKWebView?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    #if !os(iOS)
      let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.pigeon_defaultConstructor",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonIdentifierArg = args[0] as! Int64
          let initialConfigurationArg = args[1] as! WKWebViewConfiguration
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.pigeonDefaultConstructor(
                pigeonApi: api, initialConfiguration: initialConfigurationArg),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        pigeonDefaultConstructorChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let configurationChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.configuration",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        configurationChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let pigeonIdentifierArg = args[1] as! Int64
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.configuration(
                pigeonApi: api, pigeonInstance: pigeonInstanceArg),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        configurationChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setUIDelegateChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setUIDelegate",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setUIDelegateChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let delegateArg = args[1] as! WKUIDelegate
          do {
            try api.pigeonDelegate.setUIDelegate(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, delegate: delegateArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setUIDelegateChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setNavigationDelegateChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setNavigationDelegate",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setNavigationDelegateChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let delegateArg = args[1] as! WKNavigationDelegate
          do {
            try api.pigeonDelegate.setNavigationDelegate(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, delegate: delegateArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setNavigationDelegateChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let getUrlChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.getUrl",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getUrlChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getUrl(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getUrlChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let getEstimatedProgressChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.getEstimatedProgress",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getEstimatedProgressChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getEstimatedProgress(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getEstimatedProgressChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let loadChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.load",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let requestArg = args[1] as! URLRequestWrapper
          do {
            try api.pigeonDelegate.load(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, request: requestArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let loadHtmlStringChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.loadHtmlString",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadHtmlStringChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let stringArg = args[1] as! String
          let baseUrlArg: String? = nilOrValue(args[2])
          do {
            try api.pigeonDelegate.loadHtmlString(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, string: stringArg,
              baseUrl: baseUrlArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadHtmlStringChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let loadFileUrlChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.loadFileUrl",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadFileUrlChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let urlArg = args[1] as! String
          let readAccessUrlArg = args[2] as! String
          do {
            try api.pigeonDelegate.loadFileUrl(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, url: urlArg,
              readAccessUrl: readAccessUrlArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadFileUrlChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let loadFlutterAssetChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.loadFlutterAsset",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        loadFlutterAssetChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let keyArg = args[1] as! String
          do {
            try api.pigeonDelegate.loadFlutterAsset(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, key: keyArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        loadFlutterAssetChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let canGoBackChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.canGoBack",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        canGoBackChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.canGoBack(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        canGoBackChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let canGoForwardChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.canGoForward",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        canGoForwardChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.canGoForward(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        canGoForwardChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let goBackChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.goBack",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        goBackChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.goBack(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        goBackChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let goForwardChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.goForward",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        goForwardChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.goForward(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        goForwardChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let reloadChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.reload",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        reloadChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            try api.pigeonDelegate.reload(pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        reloadChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let getTitleChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.getTitle",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getTitleChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getTitle(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getTitleChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setAllowsBackForwardNavigationGesturesChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setAllowsBackForwardNavigationGestures",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAllowsBackForwardNavigationGesturesChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let allowArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAllowsBackForwardNavigationGestures(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAllowsBackForwardNavigationGesturesChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setCustomUserAgentChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setCustomUserAgent",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setCustomUserAgentChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let userAgentArg: String? = nilOrValue(args[1])
          do {
            try api.pigeonDelegate.setCustomUserAgent(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, userAgent: userAgentArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setCustomUserAgentChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let evaluateJavaScriptChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.evaluateJavaScript",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        evaluateJavaScriptChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let javaScriptStringArg = args[1] as! String
          api.pigeonDelegate.evaluateJavaScript(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, javaScriptString: javaScriptStringArg
          ) { result in
            switch result {
            case .success(let res):
              reply(wrapResult(res))
            case .failure(let error):
              reply(wrapError(error))
            }
          }
        }
      } else {
        evaluateJavaScriptChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setInspectableChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setInspectable",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setInspectableChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let inspectableArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setInspectable(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, inspectable: inspectableArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setInspectableChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let getCustomUserAgentChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.getCustomUserAgent",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        getCustomUserAgentChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          do {
            let result = try api.pigeonDelegate.getCustomUserAgent(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg)
            reply(wrapResult(result))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        getCustomUserAgentChannel.setMessageHandler(nil)
      }
    #endif
    #if !os(iOS)
      let setAllowsLinkPreviewChannel = FlutterBasicMessageChannel(
        name: "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.setAllowsLinkPreview",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        setAllowsLinkPreviewChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonInstanceArg = args[0] as! WKWebView
          let allowArg = args[1] as! Bool
          do {
            try api.pigeonDelegate.setAllowsLinkPreview(
              pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        setAllowsLinkPreviewChannel.setMessageHandler(nil)
      }
    #endif
  }

  #if !os(iOS)
    ///Creates a Dart instance of NSViewWKWebView and attaches it to [pigeonInstance].
    func pigeonNewInstance(
      pigeonInstance: WKWebView, completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
      } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
        completion(.success(()))
      } else {
        let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
          pigeonInstance as AnyObject)
        let binaryMessenger = pigeonRegistrar.binaryMessenger
        let codec = pigeonRegistrar.codec
        let channelName: String =
          "dev.flutter.pigeon.webview_flutter_wkwebview.NSViewWKWebView.pigeon_newInstance"
        let channel = FlutterBasicMessageChannel(
          name: channelName, binaryMessenger: binaryMessenger, codec: codec)
        channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
          guard let listResponse = response as? [Any?] else {
            completion(.failure(createConnectionError(withChannelName: channelName)))
            return
          }
          if listResponse.count > 1 {
            let code: String = listResponse[0] as! String
            let message: String? = nilOrValue(listResponse[1])
            let details: String? = nilOrValue(listResponse[2])
            completion(.failure(PigeonError(code: code, message: message, details: details)))
          } else {
            completion(.success(()))
          }
        }
      }
    }
  #endif
}
open class PigeonApiDelegateWKWebView {
}

protocol PigeonApiProtocolWKWebView {
}

final class PigeonApiWKWebView: PigeonApiProtocolWKWebView {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKWebView
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateWKWebView)
  {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of WKWebView and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKWebView, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebView.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKUIDelegate {
  func pigeonDefaultConstructor(pigeonApi: PigeonApiWKUIDelegate) throws -> WKUIDelegate
}

protocol PigeonApiProtocolWKUIDelegate {
  /// Creates a new web view.
  func onCreateWebView(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    configuration configurationArg: WKWebViewConfiguration,
    navigationAction navigationActionArg: WKNavigationAction,
    completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Determines whether a web resource, which the security origin object
  /// describes, can access to the device’s microphone audio and camera video.
  func requestMediaCapturePermission(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    origin originArg: WKSecurityOrigin, frame frameArg: WKFrameInfo, type typeArg: MediaCaptureType,
    completion: @escaping (Result<PermissionDecision, PigeonError>) -> Void)
  /// Displays a JavaScript alert panel.
  func runJavaScriptAlertPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    message messageArg: String, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<Void, PigeonError>) -> Void)
  /// Displays a JavaScript confirm panel.
  func runJavaScriptConfirmPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    message messageArg: String, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<Bool, PigeonError>) -> Void)
  /// Displays a JavaScript text input panel.
  func runJavaScriptTextInputPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    prompt promptArg: String, defaultText defaultTextArg: String?, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<String?, PigeonError>) -> Void)
}

final class PigeonApiWKUIDelegate: PigeonApiProtocolWKUIDelegate {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKUIDelegate
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateWKUIDelegate
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKUIDelegate?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.pigeon_defaultConstructor",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      pigeonDefaultConstructorChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKUIDelegate and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKUIDelegate, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      completion(
        .failure(
          PigeonError(
            code: "new-instance-error",
            message:
              "Error: Attempting to create a new Dart instance of WKUIDelegate, but the class has a nonnull callback method.",
            details: "")))
    }
  }
  /// Creates a new web view.
  func onCreateWebView(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    configuration configurationArg: WKWebViewConfiguration,
    navigationAction navigationActionArg: WKNavigationAction,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.onCreateWebView"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage(
      [pigeonInstanceArg, webViewArg, configurationArg, navigationActionArg] as [Any?]
    ) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Determines whether a web resource, which the security origin object
  /// describes, can access to the device’s microphone audio and camera video.
  func requestMediaCapturePermission(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    origin originArg: WKSecurityOrigin, frame frameArg: WKFrameInfo, type typeArg: MediaCaptureType,
    completion: @escaping (Result<PermissionDecision, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.requestMediaCapturePermission"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, originArg, frameArg, typeArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else if listResponse[0] == nil {
        completion(
          .failure(
            PigeonError(
              code: "null-error",
              message: "Flutter api returned null value for non-null return value.", details: "")))
      } else {
        let result = listResponse[0] as! PermissionDecision
        completion(.success(result))
      }
    }
  }

  /// Displays a JavaScript alert panel.
  func runJavaScriptAlertPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    message messageArg: String, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.runJavaScriptAlertPanel"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, messageArg, frameArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        completion(.success(()))
      }
    }
  }

  /// Displays a JavaScript confirm panel.
  func runJavaScriptConfirmPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    message messageArg: String, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<Bool, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.runJavaScriptConfirmPanel"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage([pigeonInstanceArg, webViewArg, messageArg, frameArg] as [Any?]) {
      response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else if listResponse[0] == nil {
        completion(
          .failure(
            PigeonError(
              code: "null-error",
              message: "Flutter api returned null value for non-null return value.", details: "")))
      } else {
        let result = listResponse[0] as! Bool
        completion(.success(result))
      }
    }
  }

  /// Displays a JavaScript text input panel.
  func runJavaScriptTextInputPanel(
    pigeonInstance pigeonInstanceArg: WKUIDelegate, webView webViewArg: WKWebView,
    prompt promptArg: String, defaultText defaultTextArg: String?, frame frameArg: WKFrameInfo,
    completion: @escaping (Result<String?, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
      return
    }
    let binaryMessenger = pigeonRegistrar.binaryMessenger
    let codec = pigeonRegistrar.codec
    let channelName: String =
      "dev.flutter.pigeon.webview_flutter_wkwebview.WKUIDelegate.runJavaScriptTextInputPanel"
    let channel = FlutterBasicMessageChannel(
      name: channelName, binaryMessenger: binaryMessenger, codec: codec)
    channel.sendMessage(
      [pigeonInstanceArg, webViewArg, promptArg, defaultTextArg, frameArg] as [Any?]
    ) { response in
      guard let listResponse = response as? [Any?] else {
        completion(.failure(createConnectionError(withChannelName: channelName)))
        return
      }
      if listResponse.count > 1 {
        let code: String = listResponse[0] as! String
        let message: String? = nilOrValue(listResponse[1])
        let details: String? = nilOrValue(listResponse[2])
        completion(.failure(PigeonError(code: code, message: message, details: details)))
      } else {
        let result: String? = nilOrValue(listResponse[0])
        completion(.success(result))
      }
    }
  }

}
protocol PigeonApiDelegateWKHTTPCookieStore {
  /// Sets a cookie policy that indicates whether the cookie store allows cookie
  /// storage.
  func setCookie(
    pigeonApi: PigeonApiWKHTTPCookieStore, pigeonInstance: WKHTTPCookieStore, cookie: HTTPCookie,
    completion: @escaping (Result<Void, Error>) -> Void)
}

protocol PigeonApiProtocolWKHTTPCookieStore {
}

final class PigeonApiWKHTTPCookieStore: PigeonApiProtocolWKHTTPCookieStore {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKHTTPCookieStore
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKHTTPCookieStore
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKHTTPCookieStore?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let setCookieChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.WKHTTPCookieStore.setCookie",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setCookieChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKHTTPCookieStore
        let cookieArg = args[1] as! HTTPCookie
        api.pigeonDelegate.setCookie(
          pigeonApi: api, pigeonInstance: pigeonInstanceArg, cookie: cookieArg
        ) { result in
          switch result {
          case .success:
            reply(wrapResult(nil))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      setCookieChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKHTTPCookieStore and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKHTTPCookieStore, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKHTTPCookieStore.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateUIScrollViewDelegate {
  #if !os(macOS)
    func pigeonDefaultConstructor(pigeonApi: PigeonApiUIScrollViewDelegate) throws
      -> UIScrollViewDelegate
  #endif
}

protocol PigeonApiProtocolUIScrollViewDelegate {
  #if !os(macOS)
    /// Tells the delegate when the user scrolls the content view within the
    /// scroll view.
    ///
    /// Note that this is a convenient method that includes the `contentOffset` of
    /// the `scrollView`.
    func scrollViewDidScroll(
      pigeonInstance pigeonInstanceArg: UIScrollViewDelegate,
      scrollView scrollViewArg: UIScrollView, x xArg: Double, y yArg: Double,
      completion: @escaping (Result<Void, PigeonError>) -> Void)
  #endif

}

final class PigeonApiUIScrollViewDelegate: PigeonApiProtocolUIScrollViewDelegate {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateUIScrollViewDelegate
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateUIScrollViewDelegate
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiUIScrollViewDelegate?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    #if !os(macOS)
      let pigeonDefaultConstructorChannel = FlutterBasicMessageChannel(
        name:
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollViewDelegate.pigeon_defaultConstructor",
        binaryMessenger: binaryMessenger, codec: codec)
      if let api = api {
        pigeonDefaultConstructorChannel.setMessageHandler { message, reply in
          let args = message as! [Any?]
          let pigeonIdentifierArg = args[0] as! Int64
          do {
            api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
              try api.pigeonDelegate.pigeonDefaultConstructor(pigeonApi: api),
              withIdentifier: pigeonIdentifierArg)
            reply(wrapResult(nil))
          } catch {
            reply(wrapError(error))
          }
        }
      } else {
        pigeonDefaultConstructorChannel.setMessageHandler(nil)
      }
    #endif
  }

  #if !os(macOS)
    ///Creates a Dart instance of UIScrollViewDelegate and attaches it to [pigeonInstance].
    func pigeonNewInstance(
      pigeonInstance: UIScrollViewDelegate,
      completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
      } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
        completion(.success(()))
      } else {
        let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
          pigeonInstance as AnyObject)
        let binaryMessenger = pigeonRegistrar.binaryMessenger
        let codec = pigeonRegistrar.codec
        let channelName: String =
          "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollViewDelegate.pigeon_newInstance"
        let channel = FlutterBasicMessageChannel(
          name: channelName, binaryMessenger: binaryMessenger, codec: codec)
        channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
          guard let listResponse = response as? [Any?] else {
            completion(.failure(createConnectionError(withChannelName: channelName)))
            return
          }
          if listResponse.count > 1 {
            let code: String = listResponse[0] as! String
            let message: String? = nilOrValue(listResponse[1])
            let details: String? = nilOrValue(listResponse[2])
            completion(.failure(PigeonError(code: code, message: message, details: details)))
          } else {
            completion(.success(()))
          }
        }
      }
    }
  #endif
  #if !os(macOS)
    /// Tells the delegate when the user scrolls the content view within the
    /// scroll view.
    ///
    /// Note that this is a convenient method that includes the `contentOffset` of
    /// the `scrollView`.
    func scrollViewDidScroll(
      pigeonInstance pigeonInstanceArg: UIScrollViewDelegate,
      scrollView scrollViewArg: UIScrollView, x xArg: Double, y yArg: Double,
      completion: @escaping (Result<Void, PigeonError>) -> Void
    ) {
      if pigeonRegistrar.ignoreCallsToDart {
        completion(
          .failure(
            PigeonError(
              code: "ignore-calls-error",
              message: "Calls to Dart are being ignored.", details: "")))
        return
      }
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.UIScrollViewDelegate.scrollViewDidScroll"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonInstanceArg, scrollViewArg, xArg, yArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  #endif

}
protocol PigeonApiDelegateURLCredential {
  /// Creates a URL credential instance for internet password authentication
  /// with a given user name and password, using a given persistence setting.
  func withUser(
    pigeonApi: PigeonApiURLCredential, user: String, password: String,
    persistence: UrlCredentialPersistence
  ) throws -> URLCredential
  /// Creates a URL credential instance for internet password authentication
  /// with a given user name and password, using a given persistence setting.
  ///
  /// This provides the native `UrlCredential(user:password:persistence)`
  /// constructor as an async method to ensure the class is added to the
  /// InstanceManager. See https://github.com/flutter/flutter/issues/162437.
  func withUserAsync(
    pigeonApi: PigeonApiURLCredential, user: String, password: String,
    persistence: UrlCredentialPersistence,
    completion: @escaping (Result<URLCredential, Error>) -> Void)
  /// Creates a URL credential instance for server trust authentication,
  /// initialized with a accepted trust.
  ///
  /// This provides the native `UrlCredential(forTrust:)` constructor as an
  /// async method to ensure the class is added to the InstanceManager. See
  /// https://github.com/flutter/flutter/issues/162437.
  func serverTrustAsync(
    pigeonApi: PigeonApiURLCredential, trust: SecTrustWrapper,
    completion: @escaping (Result<URLCredential, Error>) -> Void)
}

protocol PigeonApiProtocolURLCredential {
}

final class PigeonApiURLCredential: PigeonApiProtocolURLCredential {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURLCredential
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateURLCredential
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiURLCredential?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let withUserChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLCredential.withUser",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      withUserChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonIdentifierArg = args[0] as! Int64
        let userArg = args[1] as! String
        let passwordArg = args[2] as! String
        let persistenceArg = args[3] as! UrlCredentialPersistence
        do {
          api.pigeonRegistrar.instanceManager.addDartCreatedInstance(
            try api.pigeonDelegate.withUser(
              pigeonApi: api, user: userArg, password: passwordArg, persistence: persistenceArg),
            withIdentifier: pigeonIdentifierArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      withUserChannel.setMessageHandler(nil)
    }
    let withUserAsyncChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLCredential.withUserAsync",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      withUserAsyncChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let userArg = args[0] as! String
        let passwordArg = args[1] as! String
        let persistenceArg = args[2] as! UrlCredentialPersistence
        api.pigeonDelegate.withUserAsync(
          pigeonApi: api, user: userArg, password: passwordArg, persistence: persistenceArg
        ) { result in
          switch result {
          case .success(let res):
            reply(wrapResult(res))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      withUserAsyncChannel.setMessageHandler(nil)
    }
    let serverTrustAsyncChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLCredential.serverTrustAsync",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      serverTrustAsyncChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        api.pigeonDelegate.serverTrustAsync(pigeonApi: api, trust: trustArg) { result in
          switch result {
          case .success(let res):
            reply(wrapResult(res))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      serverTrustAsyncChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of URLCredential and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URLCredential, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLCredential.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateURLProtectionSpace {
  /// The receiver’s host.
  func host(pigeonApi: PigeonApiURLProtectionSpace, pigeonInstance: URLProtectionSpace) throws
    -> String
  /// The receiver’s port.
  func port(pigeonApi: PigeonApiURLProtectionSpace, pigeonInstance: URLProtectionSpace) throws
    -> Int64
  /// The receiver’s authentication realm.
  func realm(pigeonApi: PigeonApiURLProtectionSpace, pigeonInstance: URLProtectionSpace) throws
    -> String?
  /// The authentication method used by the receiver.
  func authenticationMethod(
    pigeonApi: PigeonApiURLProtectionSpace, pigeonInstance: URLProtectionSpace
  ) throws -> String?
  /// A representation of the server’s SSL transaction state.
  func getServerTrust(pigeonApi: PigeonApiURLProtectionSpace, pigeonInstance: URLProtectionSpace)
    throws -> SecTrustWrapper?
}

protocol PigeonApiProtocolURLProtectionSpace {
}

final class PigeonApiURLProtectionSpace: PigeonApiProtocolURLProtectionSpace {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURLProtectionSpace
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateURLProtectionSpace
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiURLProtectionSpace?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let getServerTrustChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URLProtectionSpace.getServerTrust",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getServerTrustChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLProtectionSpace
        do {
          let result = try api.pigeonDelegate.getServerTrust(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getServerTrustChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of URLProtectionSpace and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URLProtectionSpace, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let hostArg = try! pigeonDelegate.host(pigeonApi: self, pigeonInstance: pigeonInstance)
      let portArg = try! pigeonDelegate.port(pigeonApi: self, pigeonInstance: pigeonInstance)
      let realmArg = try! pigeonDelegate.realm(pigeonApi: self, pigeonInstance: pigeonInstance)
      let authenticationMethodArg = try! pigeonDelegate.authenticationMethod(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLProtectionSpace.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage(
        [pigeonIdentifierArg, hostArg, portArg, realmArg, authenticationMethodArg] as [Any?]
      ) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateURLAuthenticationChallenge {
  /// The receiver’s protection space.
  func getProtectionSpace(
    pigeonApi: PigeonApiURLAuthenticationChallenge, pigeonInstance: URLAuthenticationChallenge
  ) throws -> URLProtectionSpace
}

protocol PigeonApiProtocolURLAuthenticationChallenge {
}

final class PigeonApiURLAuthenticationChallenge: PigeonApiProtocolURLAuthenticationChallenge {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURLAuthenticationChallenge
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateURLAuthenticationChallenge
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiURLAuthenticationChallenge?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let getProtectionSpaceChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLAuthenticationChallenge.getProtectionSpace",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getProtectionSpaceChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URLAuthenticationChallenge
        do {
          let result = try api.pigeonDelegate.getProtectionSpace(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getProtectionSpaceChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of URLAuthenticationChallenge and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URLAuthenticationChallenge,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URLAuthenticationChallenge.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateURL {
  /// The absolute string for the URL.
  func getAbsoluteString(pigeonApi: PigeonApiURL, pigeonInstance: URL) throws -> String
}

protocol PigeonApiProtocolURL {
}

final class PigeonApiURL: PigeonApiProtocolURL {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateURL
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateURL) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(binaryMessenger: FlutterBinaryMessenger, api: PigeonApiURL?) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let getAbsoluteStringChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.URL.getAbsoluteString",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getAbsoluteStringChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! URL
        do {
          let result = try api.pigeonDelegate.getAbsoluteString(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getAbsoluteStringChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of URL and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: URL, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.URL.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateWKWebpagePreferences {
  /// A Boolean value that indicates whether JavaScript from web content is
  /// allowed to run.
  func setAllowsContentJavaScript(
    pigeonApi: PigeonApiWKWebpagePreferences, pigeonInstance: WKWebpagePreferences, allow: Bool)
    throws
}

protocol PigeonApiProtocolWKWebpagePreferences {
}

final class PigeonApiWKWebpagePreferences: PigeonApiProtocolWKWebpagePreferences {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateWKWebpagePreferences
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateWKWebpagePreferences
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiWKWebpagePreferences?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let setAllowsContentJavaScriptChannel = FlutterBasicMessageChannel(
      name:
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebpagePreferences.setAllowsContentJavaScript",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setAllowsContentJavaScriptChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let pigeonInstanceArg = args[0] as! WKWebpagePreferences
        let allowArg = args[1] as! Bool
        do {
          try api.pigeonDelegate.setAllowsContentJavaScript(
            pigeonApi: api, pigeonInstance: pigeonInstanceArg, allow: allowArg)
          reply(wrapResult(nil))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setAllowsContentJavaScriptChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of WKWebpagePreferences and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: WKWebpagePreferences, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.WKWebpagePreferences.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateGetTrustResultResponse {
  /// The result code from the most recent trust evaluation.
  func result(pigeonApi: PigeonApiGetTrustResultResponse, pigeonInstance: GetTrustResultResponse)
    throws -> DartSecTrustResultType
  /// A result code.
  ///
  /// See https://developer.apple.com/documentation/security/security-framework-result-codes?language=objc.
  func resultCode(
    pigeonApi: PigeonApiGetTrustResultResponse, pigeonInstance: GetTrustResultResponse
  ) throws -> Int64
}

protocol PigeonApiProtocolGetTrustResultResponse {
}

final class PigeonApiGetTrustResultResponse: PigeonApiProtocolGetTrustResultResponse {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateGetTrustResultResponse
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar,
    delegate: PigeonApiDelegateGetTrustResultResponse
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  ///Creates a Dart instance of GetTrustResultResponse and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: GetTrustResultResponse,
    completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let resultArg = try! pigeonDelegate.result(pigeonApi: self, pigeonInstance: pigeonInstance)
      let resultCodeArg = try! pigeonDelegate.resultCode(
        pigeonApi: self, pigeonInstance: pigeonInstance)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.GetTrustResultResponse.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg, resultArg, resultCodeArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateSecTrust {
  /// Evaluates trust for the specified certificate and policies.
  func evaluateWithError(
    pigeonApi: PigeonApiSecTrust, trust: SecTrustWrapper,
    completion: @escaping (Result<Bool, Error>) -> Void)
  /// Returns an opaque cookie containing exceptions to trust policies that will
  /// allow future evaluations of the current certificate to succeed.
  func copyExceptions(pigeonApi: PigeonApiSecTrust, trust: SecTrustWrapper) throws
    -> FlutterStandardTypedData?
  /// Sets a list of exceptions that should be ignored when the certificate is
  /// evaluated.
  func setExceptions(
    pigeonApi: PigeonApiSecTrust, trust: SecTrustWrapper, exceptions: FlutterStandardTypedData?
  ) throws -> Bool
  /// Returns the result code from the most recent trust evaluation.
  func getTrustResult(pigeonApi: PigeonApiSecTrust, trust: SecTrustWrapper) throws
    -> GetTrustResultResponse
  /// Certificates used to evaluate trust.
  func copyCertificateChain(pigeonApi: PigeonApiSecTrust, trust: SecTrustWrapper) throws
    -> [SecCertificateWrapper]?
}

protocol PigeonApiProtocolSecTrust {
}

final class PigeonApiSecTrust: PigeonApiProtocolSecTrust {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateSecTrust
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateSecTrust) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(binaryMessenger: FlutterBinaryMessenger, api: PigeonApiSecTrust?)
  {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let evaluateWithErrorChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.evaluateWithError",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      evaluateWithErrorChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        api.pigeonDelegate.evaluateWithError(pigeonApi: api, trust: trustArg) { result in
          switch result {
          case .success(let res):
            reply(wrapResult(res))
          case .failure(let error):
            reply(wrapError(error))
          }
        }
      }
    } else {
      evaluateWithErrorChannel.setMessageHandler(nil)
    }
    let copyExceptionsChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.copyExceptions",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      copyExceptionsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        do {
          let result = try api.pigeonDelegate.copyExceptions(pigeonApi: api, trust: trustArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      copyExceptionsChannel.setMessageHandler(nil)
    }
    let setExceptionsChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.setExceptions",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      setExceptionsChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        let exceptionsArg: FlutterStandardTypedData? = nilOrValue(args[1])
        do {
          let result = try api.pigeonDelegate.setExceptions(
            pigeonApi: api, trust: trustArg, exceptions: exceptionsArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      setExceptionsChannel.setMessageHandler(nil)
    }
    let getTrustResultChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.getTrustResult",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      getTrustResultChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        do {
          let result = try api.pigeonDelegate.getTrustResult(pigeonApi: api, trust: trustArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      getTrustResultChannel.setMessageHandler(nil)
    }
    let copyCertificateChainChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.copyCertificateChain",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      copyCertificateChainChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let trustArg = args[0] as! SecTrustWrapper
        do {
          let result = try api.pigeonDelegate.copyCertificateChain(pigeonApi: api, trust: trustArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      copyCertificateChainChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of SecTrust and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: SecTrustWrapper, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.SecTrust.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
protocol PigeonApiDelegateSecCertificate {
  /// Returns a DER representation of a certificate given a certificate object.
  func copyData(pigeonApi: PigeonApiSecCertificate, certificate: SecCertificateWrapper) throws
    -> FlutterStandardTypedData
}

protocol PigeonApiProtocolSecCertificate {
}

final class PigeonApiSecCertificate: PigeonApiProtocolSecCertificate {
  unowned let pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar
  let pigeonDelegate: PigeonApiDelegateSecCertificate
  ///An implementation of [NSObject] used to access callback methods
  var pigeonApiNSObject: PigeonApiNSObject {
    return pigeonRegistrar.apiDelegate.pigeonApiNSObject(pigeonRegistrar)
  }

  init(
    pigeonRegistrar: WebKitLibraryPigeonProxyApiRegistrar, delegate: PigeonApiDelegateSecCertificate
  ) {
    self.pigeonRegistrar = pigeonRegistrar
    self.pigeonDelegate = delegate
  }
  static func setUpMessageHandlers(
    binaryMessenger: FlutterBinaryMessenger, api: PigeonApiSecCertificate?
  ) {
    let codec: FlutterStandardMessageCodec =
      api != nil
      ? FlutterStandardMessageCodec(
        readerWriter: WebKitLibraryPigeonInternalProxyApiCodecReaderWriter(
          pigeonRegistrar: api!.pigeonRegistrar))
      : FlutterStandardMessageCodec.sharedInstance()
    let copyDataChannel = FlutterBasicMessageChannel(
      name: "dev.flutter.pigeon.webview_flutter_wkwebview.SecCertificate.copyData",
      binaryMessenger: binaryMessenger, codec: codec)
    if let api = api {
      copyDataChannel.setMessageHandler { message, reply in
        let args = message as! [Any?]
        let certificateArg = args[0] as! SecCertificateWrapper
        do {
          let result = try api.pigeonDelegate.copyData(pigeonApi: api, certificate: certificateArg)
          reply(wrapResult(result))
        } catch {
          reply(wrapError(error))
        }
      }
    } else {
      copyDataChannel.setMessageHandler(nil)
    }
  }

  ///Creates a Dart instance of SecCertificate and attaches it to [pigeonInstance].
  func pigeonNewInstance(
    pigeonInstance: SecCertificateWrapper, completion: @escaping (Result<Void, PigeonError>) -> Void
  ) {
    if pigeonRegistrar.ignoreCallsToDart {
      completion(
        .failure(
          PigeonError(
            code: "ignore-calls-error",
            message: "Calls to Dart are being ignored.", details: "")))
    } else if pigeonRegistrar.instanceManager.containsInstance(pigeonInstance as AnyObject) {
      completion(.success(()))
    } else {
      let pigeonIdentifierArg = pigeonRegistrar.instanceManager.addHostCreatedInstance(
        pigeonInstance as AnyObject)
      let binaryMessenger = pigeonRegistrar.binaryMessenger
      let codec = pigeonRegistrar.codec
      let channelName: String =
        "dev.flutter.pigeon.webview_flutter_wkwebview.SecCertificate.pigeon_newInstance"
      let channel = FlutterBasicMessageChannel(
        name: channelName, binaryMessenger: binaryMessenger, codec: codec)
      channel.sendMessage([pigeonIdentifierArg] as [Any?]) { response in
        guard let listResponse = response as? [Any?] else {
          completion(.failure(createConnectionError(withChannelName: channelName)))
          return
        }
        if listResponse.count > 1 {
          let code: String = listResponse[0] as! String
          let message: String? = nilOrValue(listResponse[1])
          let details: String? = nilOrValue(listResponse[2])
          completion(.failure(PigeonError(code: code, message: message, details: details)))
        } else {
          completion(.success(()))
        }
      }
    }
  }
}
