import 'dart:io';
import 'dart:typed_data';

import 'package:analyzer/dart/analysis/results.dart';
import 'package:analyzer/dart/analysis/session.dart';
import 'package:analyzer/dart/ast/ast.dart';
import 'package:analyzer/dart/ast/visitor.dart';
import 'package:analyzer/diagnostic/diagnostic.dart';
import 'package:analyzer/file_system/file_system.dart';
import 'package:analyzer/file_system/overlay_file_system.dart';
import 'package:analyzer/file_system/physical_file_system.dart';
import 'package:analyzer/src/dart/analysis/analysis_context_collection.dart';
import 'package:analyzer/src/dart/analysis/byte_store.dart';
import 'package:analyzer/src/error/codes.dart';
import 'package:analyzer/src/util/file_paths.dart' as file_paths;
import 'package:analyzer/src/util/performance/operation_performance.dart';
import 'package:collection/collection.dart';

Future<void> main(List<String> arguments) async {
  // arguments = [
  //   'generate-conservative',
  //   '--mutations-directory=/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer_cli/lib',
  //   '--output-directory=/Users/scheglov/tmp/2025/2025-08-05',
  //   '--analysis-directories=/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer_cli',
  // ];
  arguments = [
    'generate-conservative',
    '--mutations-directory=/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer/lib/src/fine',
    '--output-directory=/Users/scheglov/tmp/2025/2025-08-05',
    // '--analysis-directories=/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer/test/src/dart/resolution',
    '--analysis-directories=/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer',
  ];

  switch (arguments.firstOrNull) {
    case 'generate-conservative':
      await _generateConservative(arguments.skip(1).toList());
    case 'verify-fine':
      break;
    default:
      _printUsage();
      exit(1);
  }
}

final withFineDependencies = true;

Future<void> _generateConservative(List<String> arguments) async {
  var resourceProvider = PhysicalResourceProvider.INSTANCE;
  var pathContext = resourceProvider.pathContext;

  const mutationsDirectoryOption = '--mutations-directory=';
  Folder? mutationsDirectory;

  const outputDirectoryOption = '--output-directory=';
  Folder? outputDirectory;

  const analysisDirectoriesOption = '--analysis-directories=';
  var analysisDirectories = <Folder>[];

  for (var argument in arguments) {
    if (argument.startsWith(mutationsDirectoryOption)) {
      mutationsDirectory = resourceProvider.getFolder(
        argument.substring(mutationsDirectoryOption.length),
      );
      if (!mutationsDirectory.exists) {
        print('Directory does not exist: ${mutationsDirectory.path}');
        exit(1);
      }
    } else if (argument.startsWith(outputDirectoryOption)) {
      outputDirectory = resourceProvider.getFolder(
        argument.substring(outputDirectoryOption.length),
      );
      if (!outputDirectory.exists) {
        print('Directory does not exist: ${outputDirectory.path}');
        exit(1);
      }
    } else if (argument.startsWith(analysisDirectoriesOption)) {
      for (var analysisPath in argument
          .substring(analysisDirectoriesOption.length)
          .split(',')) {
        var analysisDirectory = resourceProvider.getFolder(analysisPath);
        if (!analysisDirectory.exists) {
          print('Directory does not exist: ${analysisDirectory.path}');
          exit(1);
        }
        analysisDirectories.add(analysisDirectory);
      }
    } else {
      print('Unknown argument: $argument');
      exit(1);
    }
  }

  if (outputDirectory == null) {
    print('Missing $outputDirectoryOption');
    exit(1);
  }

  if (mutationsDirectory == null) {
    print('Missing $mutationsDirectoryOption');
    exit(1);
  }

  if (analysisDirectories.isEmpty) {
    print('Missing $analysisDirectoriesOption');
    exit(1);
  }

  print('Generate mutations and analysis results after each mutation.');
  print('  Output directory: ${outputDirectory.path}');
  print('  Mutations directory: ${mutationsDirectory.path}');
  print('  Analysis directories:');
  for (var analysisDirectory in analysisDirectories) {
    print('    ${analysisDirectory.path}');
  }
  print('');

  var initByteStore = MemoryByteStore();

  List<File> mutationFiles;
  {
    var collection = AnalysisContextCollectionImpl(
      byteStore: initByteStore,
      resourceProvider: resourceProvider,
      includedPaths: [mutationsDirectory.path],
      withFineDependencies: withFineDependencies,
    );

    mutationFiles =
        collection.contexts
            .expand((context) => context.contextRoot.analyzedFiles())
            .where((path) => file_paths.isDart(pathContext, path))
            .sorted()
            .map((path) => resourceProvider.getFile(path))
            .toList();
  }

  List<File> analysisFiles;
  {
    var collection = AnalysisContextCollectionImpl(
      byteStore: initByteStore,
      resourceProvider: resourceProvider,
      includedPaths: [...analysisDirectories.map((e) => e.path)],
      withFineDependencies: withFineDependencies,
    );

    analysisFiles =
        collection.contexts
            .expand((context) => context.contextRoot.analyzedFiles())
            .where((path) => file_paths.isDart(pathContext, path))
            .sorted()
            .map((path) => resourceProvider.getFile(path))
            .toList();

    print('Analyzed files:');
    print('  ${analysisFiles.join('\n  ')}');

    for (var file in analysisFiles) {
      var session = collection.contextFor(file.path).currentSession;
      var diagnostics = await session.computeDiagnostics(file.path);
      if (diagnostics.isNotEmpty) {
        throw StateError(
          'Must have no diagnostics initially:\n'
          '${diagnostics.join('\n')}',
        );
      }
    }
  }

  var allMutations = <Mutation>[];
  for (var file in mutationFiles) {
    var collection = AnalysisContextCollectionImpl(
      byteStore: _CascadingByteStore(initByteStore, MemoryByteStore()),
      resourceProvider: resourceProvider,
      includedPaths: [file.path],
    );

    var session = collection.contextFor(file.path).currentSession;
    var parseResult = session.getParsedUnit(file.path);
    parseResult as ParsedUnitResult;

    var mutationsBuilder = MutationsBuilder(unitResult: parseResult);
    parseResult.unit.accept(mutationsBuilder);
    var fileMutations = mutationsBuilder.mutations;
    allMutations.addAll(fileMutations);
  }

  var totalTimer = Stopwatch()..start();
  for (var mutation in allMutations.take(25)) {
    print(mutation.path);
    print('  ${mutation.description}  ${mutation.replacement}');

    var overlayResourceProvider = OverlayResourceProvider(resourceProvider);
    var file = overlayResourceProvider.getFile(mutation.path);
    var fileInitialContent = file.readAsStringSync();

    var collection = AnalysisContextCollectionImpl(
      byteStore: _CascadingByteStore(initByteStore, MemoryByteStore()),
      resourceProvider: overlayResourceProvider,
      includedPaths: [
        '/Users/scheglov/Source/Dart/sdk.git/sdk/pkg/analyzer',
        // mutationsDirectory.path,
        // ...analysisDirectories.map((e) => e.path),
      ],
      withFineDependencies: withFineDependencies,
    );

    // Initial analysis.
    for (var file in analysisFiles) {
      var session = collection.contextFor(file.path).currentSession;
      var diagnostics = await session.computeDiagnostics(file.path);
      if (diagnostics.isNotEmpty) {
        throw StateError(
          'Must have no diagnostics initially:\n'
          '${diagnostics.join('\n')}',
        );
      }
    }

    {
      var mutatedContent = mutation.getUpdatedContent(fileInitialContent);
      overlayResourceProvider.setOverlay(
        file.path,
        content: mutatedContent,
        modificationStamp: 0,
      );
      var analysisContext = collection.contextFor(file.path);
      analysisContext.changeFile(file.path);
      await analysisContext.applyPendingFileChanges();
    }

    collection.discardPerformance();
    var timer = Stopwatch()..start();
    for (var file in analysisFiles) {
      var session = collection.contextFor(file.path).currentSession;
      var diagnostics = await session.computeDiagnostics(file.path);
      if (diagnostics.isNotEmpty) {
        print('  ${diagnostics.join('\n  ')}');
      }
    }
    print('  [timer: ${timer.elapsedMilliseconds} ms]');
    collection.printPerformance();

    print('');
  }

  print('\n[totalTime: ${totalTimer.elapsedMilliseconds} ms]');
}

void _printUsage() {
  print('''
Usage: dart fine_testing.dart <command> [options]

A tool for testing fine-grained dependencies.

Available commands:
  generate-conservative   Generate mutations, and compute results using conservative dependencies.
  verify-fine             Apply the generated mutations, and verify that results are the same.
''');
}

class Diagnostic2 {
  final int offset;
  final int length;
  final String message;

  Diagnostic2({
    required this.offset,
    required this.length,
    required this.message,
  });
}

class FileDiagnostics {
  final String path;
  final List<Diagnostic2> diagnostics;

  FileDiagnostics({required this.path, required this.diagnostics});
}

class Mutation {
  final String path;
  final String description;
  final Replacement replacement;

  Mutation({
    required this.path,
    required this.description,
    required this.replacement,
  });

  factory Mutation.fromJson(Map<String, Object?> json) {
    return Mutation(
      path: json['path'] as String,
      description: json['description'] as String,
      replacement: Replacement.fromJson(
        json['replacement'] as Map<String, Object?>,
      ),
    );
  }

  String getUpdatedContent(String content) {
    return content.replaceRange(
      replacement.offset,
      replacement.end,
      replacement.text,
    );
  }

  Map<String, Object?> toJson() {
    return {
      'path': path,
      'description': description,
      'replacement': replacement.toJson(),
    };
  }

  @override
  String toString() {
    return 'Mutation(path: $path, description: $description, '
        'replacement: $replacement)';
  }
}

class MutationsBuilder extends RecursiveAstVisitor<void> {
  final ParsedUnitResult unitResult;
  final List<Mutation> mutations = [];

  String? instanceContainerName;

  MutationsBuilder({required this.unitResult});

  @override
  void visitClassDeclaration(ClassDeclaration node) {
    instanceContainerName = 'class: ${node.name.lexeme}';
    super.visitClassDeclaration(node);
    instanceContainerName = null;
  }

  @override
  void visitMethodDeclaration(MethodDeclaration node) {
    var returnType = node.returnType;
    if (returnType != null) {
      if ('$returnType' != 'void') {
        var offset = returnType.offset;
        var location = unitResult.lineInfo.getLocation(offset);
        mutations.add(
          Mutation(
            path: unitResult.path,
            description:
                '[${location.lineNumber}:${location.columnNumber}]'
                '[$instanceContainerName]'
                '[method: ${node.name.lexeme}]'
                '[Replace return type with "void"]',
            replacement: Replacement(offset, returnType.end, 'void'),
          ),
        );
      }
    }

    super.visitMethodDeclaration(node);
  }
}

class MutationWithResults {
  final Mutation mutation;
  final String fileDiagnostics;

  MutationWithResults({required this.mutation, required this.fileDiagnostics});
}

class Replacement {
  final int offset;
  final int end;
  final String text;

  Replacement(this.offset, this.end, this.text);

  factory Replacement.fromJson(Map<String, Object?> json) {
    return Replacement(
      json['offset'] as int,
      json['end'] as int,
      json['text'] as String,
    );
  }

  Map<String, Object?> toJson() {
    return {'offset': offset, 'end': end, 'text': text};
  }

  @override
  String toString() {
    return 'Replacement(offset: $offset, length: ${end - offset}, '
        'end: $end, text: "$text")';
  }
}

class _CascadingByteStore implements ByteStore {
  final ByteStore parent;
  final ByteStore local;

  _CascadingByteStore(this.parent, this.local);

  @override
  Uint8List? get(String key) {
    return local.get(key) ?? parent.get(key);
  }

  @override
  Uint8List putGet(String key, Uint8List bytes) {
    return local.putGet(key, bytes);
  }

  @override
  void release(Iterable<String> keys) {}
}

extension on AnalysisSession {
  Future<List<Diagnostic>> computeDiagnostics(String path) async {
    var errorsResult = await getErrors(path);
    errorsResult as ErrorsResult;
    return errorsResult.diagnostics.withoutTodo.withoutIgnored;
  }
}

extension on List<Diagnostic> {
  List<Diagnostic> get withoutIgnored {
    return whereNot(
          (d) =>
              d.diagnosticCode ==
              WarningCode.INFERENCE_FAILURE_ON_FUNCTION_RETURN_TYPE,
        )
        .whereNot(
          (d) =>
              d.diagnosticCode ==
              WarningCode.INFERENCE_FAILURE_ON_UNTYPED_PARAMETER,
        )
        .toList();
  }

  List<Diagnostic> get withoutTodo {
    return where((d) => d.diagnosticCode is! TodoCode).toList();
  }
}

extension on AnalysisContextCollectionImpl {
  void discardPerformance() {
    scheduler.accumulatedPerformance = OperationPerformanceImpl('<scheduler>');
  }

  void printPerformance() {
    var buffer = StringBuffer();
    scheduler.accumulatedPerformance.write(buffer: buffer);
    print(buffer);
    discardPerformance();
  }
}
