package nz.eloque.foss_wallet.ui.screens.pass

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.OutlinedCard
import androidx.compose.material3.Switch
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.TopAppBarScrollBehavior
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import nz.eloque.foss_wallet.R
import nz.eloque.foss_wallet.model.LocalizedPassWithTags
import nz.eloque.foss_wallet.model.Pass
import nz.eloque.foss_wallet.model.PassType
import nz.eloque.foss_wallet.model.Tag
import nz.eloque.foss_wallet.model.field.PassContent
import nz.eloque.foss_wallet.model.field.PassField
import nz.eloque.foss_wallet.persistence.BarcodePosition
import nz.eloque.foss_wallet.ui.card.PassCard
import nz.eloque.foss_wallet.ui.effects.ForceOrientation
import nz.eloque.foss_wallet.ui.effects.Orientation
import java.time.Instant

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PassView(
    localizedPass: LocalizedPassWithTags,
    allTags: Set<Tag>,
    onTagClick: (Tag) -> Unit,
    onTagAdd: (Tag) -> Unit,
    onTagCreate: (Tag) -> Unit,
    barcodePosition: BarcodePosition,
    increaseBrightness: Boolean,
    onRenderingChange: () -> Unit,
    modifier: Modifier = Modifier,
    scrollBehavior: TopAppBarScrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(),
) {
    val pass = localizedPass.pass

    val barCode = pass.barCodes.firstOrNull()
    val hasLegacyRepresentation = barCode?.hasLegacyRepresentation() ?: false
    val context = LocalContext.current
    ForceOrientation(Orientation.Locked)
    Column(
        verticalArrangement = Arrangement.spacedBy(10.dp),
        modifier = modifier
            .nestedScroll(scrollBehavior.nestedScrollConnection)
            .verticalScroll(rememberScrollState())
    ) {
        PassCard(
            localizedPass = localizedPass,
            allTags = allTags,
            onTagClick = onTagClick,
            onTagAdd = onTagAdd,
            onTagCreate = onTagCreate
        ) {
            Column(
                verticalArrangement = Arrangement.spacedBy(25.dp)
            ) {
                barCode?.let {
                    AsyncPassImage(model = pass.footerFile(context))
                    BarcodesView(
                        legacyRendering = pass.renderLegacy && hasLegacyRepresentation,
                        barcode = it,
                        barcodePosition = barcodePosition,
                        increaseBrightness = increaseBrightness
                    )
                }
            }
        }
        Column(
            verticalArrangement = Arrangement.spacedBy(25.dp),
            modifier = Modifier
                .padding(10.dp)
        ) {
            if (hasLegacyRepresentation) {
                OutlinedCard(
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Row(
                        modifier = Modifier.padding(16.dp),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Switch(
                            checked = pass.renderLegacy,
                            onCheckedChange = {
                                onRenderingChange.invoke()
                            }
                        )
                        Spacer(Modifier.width(16.dp))
                        Text(
                            text = stringResource(R.string.compatibility_mode),
                        )
                    }
                }
            }
            BackFields(pass.backFields)
            Spacer(modifier = Modifier.padding(4.dp))
        }
        Spacer(Modifier.imePadding())
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Preview
@Composable
private fun PassPreview() {
    val kscTag = Tag("KSC", Color.Blue)
    val gameTag = Tag("Spiel", Color.Red)
    val allTags = setOf(kscTag, gameTag)

    val pass = Pass(
        "",
        "KSC - SV Elversberg",
        1,
        "KSC",
        "serial",
        PassType.Generic,
        HashSet(),
        Instant.ofEpochSecond(0),
        hasLogo = false,
        hasStrip = false,
        hasThumbnail = false,
        hasFooter = false,
        headerFields = mutableListOf(
            PassField("block", "Block", PassContent.Plain("S1")),
            PassField("seat", "Seat", PassContent.Plain("47")),
        ),
        primaryFields = mutableListOf(
            PassField("name", "Name", PassContent.Plain("Max Mustermann")),
            PassField("seat", "Seat", PassContent.Plain("47")),
        ),
        auxiliaryFields = mutableListOf(
            PassField("block", "Block", PassContent.Plain("S1 | Gegengerade")),
            PassField("seat", "Seat", PassContent.Plain("36E")),
        ),
        secondaryFields = mutableListOf(
            PassField("data1", "data1", PassContent.Plain("Longer Value here i guess")),
            PassField("data2", "data2", PassContent.Plain("Shorter Value")),
        ),
    )
    PassView(
        localizedPass = LocalizedPassWithTags(pass, allTags),
        allTags = allTags,
        onTagClick = {},
        onTagAdd = {},
        onTagCreate = {},
        barcodePosition = BarcodePosition.Center,
        increaseBrightness = false,
        onRenderingChange = {}
    )
}
