extends Node

enum LevelSource {
	BUILT_IN,
	LOCAL,
	ONLINE
}

var level_source := LevelSource.BUILT_IN
var level_id = null
var level_data: Dictionary = {}
var editing := false
var verifying := false


func _ready() -> void:
	DirAccess.make_dir_absolute("user://levels/")
	DirAccess.make_dir_absolute("user://stats/")


func get_local_levels() -> Array[Dictionary]:
	var levels: Array[Dictionary] = []
	var dir := DirAccess.open("user://levels/")
	dir.include_hidden = false
	dir.list_dir_begin()
	for file_name: String in dir.get_files():
		var data = get_local_level_data(file_name.split(".")[0])
		if data:
			levels.push_back(data)
	levels.sort_custom(func(a, b): return a.get("creation_time", 0) > b.get("creation_time", 0))
	return levels


func get_local_level_data(id: String):
	var level_meta := ConfigFile.new()
	level_meta.load("user://levels/%s.cfg" % id)
	return level_meta.get_value("level", "data", null)


func new_local_level() -> Dictionary:
	var id = "%d" % randi_range(1_000_000_000, 9_999_999_999)
	return {
		"id": id
	}


func save_local_level(level_data: Dictionary) -> void:
	var level_meta := ConfigFile.new()
	level_meta.set_value("level", "data", level_data)
	level_meta.save("user://levels/%s.cfg" % level_data.get("id"))


func get_built_in_level_data(id: String) -> Dictionary:
	var level_meta := ConfigFile.new()
	level_meta.load("res://builtin_levels/%s.cfg" % id)
	return level_meta.get_value("level", "data", {})


func get_level_data(id: String) -> Dictionary:
	if level_source == LevelSource.BUILT_IN:
		return get_built_in_level_data(id)
	elif level_source == LevelSource.LOCAL:
		return get_local_level_data(id)
	return {}


func get_level_stats(id: String) -> Dictionary:
	var stats := ConfigFile.new()
	stats.load("user://stats/%s.cfg" % id)
	return {
		"id": stats.get_value("stats", "id", id),
		"completed": stats.get_value("stats", "completed", false),
		"best_time": stats.get_value("stats", "best_time", INF),
		"most_coins": stats.get_value("stats", "most_coins", 0),
		"least_jumps": stats.get_value("stats", "least_jumps", INF),
		"beat_secret_ending": stats.get_value("stats", "beat_secret_ending", false)
	}


func save_level_stats(stats: Dictionary) -> void:
	var stats_cfg := ConfigFile.new()
	stats_cfg.set_value("stats", "id", stats.get("id"))
	stats_cfg.set_value("stats", "completed", stats.get("completed"))
	stats_cfg.set_value("stats", "best_time", stats.get("best_time"))
	stats_cfg.set_value("stats", "most_coins", stats.get("most_coins"))
	stats_cfg.set_value("stats", "least_jumps", stats.get("least_jumps"))
	stats_cfg.set_value("stats", "beat_secret_ending", stats.get("beat_secret_ending"))
	stats_cfg.save("user://stats/%s.cfg" % stats.get("id"))
