extends CharacterBody2D

const SPEED = 1000.0
const FLOATY_GROUND_ACCEL = 4000.0
const FLOATY_AIR_ACCEL = 2000.0
const STRICT_GROUND_ACCEL = 6000.0
const STRICT_AIR_ACCEL = 2500.0
const JUMP_VELOCITY = -2010.0

const EDIT_SPEED = 2000.0
const EDIT_ACCEL = 10_000.0


@onready var game = get_tree().get_first_node_in_group("game")
var jumps := 0
var ice_mult := 1.0
var force_move_dir = null


func _ready() -> void:
	%Camera.position = game.camera_offset * 256
	%Camera.reset_smoothing()
	%Camera.reset_physics_interpolation()
	


func _physics_process(delta: float) -> void:
	if get_tree().paused:
		return
		
	if game.first_load_major_version >= 12:
		%Collision.shape.size.y = 224
		%Collision.position.y = 16
	elif game.physics_mode == game.PhysicsMode.FLOATY:
		%Collision.shape.size.y = 256
		%Collision.position.y = 0
	else:
		%Collision.shape.size.y = 240
		%Collision.position.y = 8
	
	%Collision.disabled = game.health <= 0
	
	set_collision_layer_value(1, game.game_mode == game.GameMode.PLAYING)
	set_collision_mask_value(2, game.game_mode == game.GameMode.PLAYING)
	set_collision_mask_value(3, game.game_mode == game.GameMode.PLAYING)
	
	var cam_last_pos = %Camera.position
	%Camera.position = game.camera_offset * 256
	if %Camera.position != cam_last_pos and game.game_mode == game.GameMode.PLAYING:
		# TODO: Doesn't seem to actually reset the smoothing?
		%Camera.reset_smoothing()
		%Camera.reset_physics_interpolation()
	
	var is_flying: bool = game.game_mode == game.GameMode.EDITING or game.physics_mode == game.PhysicsMode.FLYING
	
	if not is_flying:
		if not is_on_floor():
			velocity += get_gravity() * delta
			
		if Input.is_action_just_pressed("jump"):
			%QueuedJumpTimer.start()
		
		if is_on_floor() or game.air_jumping:
			%CoyoteTimer.start()

		# Handle jump.
		if not %QueuedJumpTimer.is_stopped() and not %CoyoteTimer.is_stopped():
			if game.max_jumps < 0 or jumps < game.max_jumps:
				jumps += 1
				%QueuedJumpTimer.stop()
				%CoyoteTimer.stop()
				velocity.y = JUMP_VELOCITY
				%JumpAudio.play()
		elif not Input.is_action_pressed("jump"):
			if velocity.y < 0:
				if game.physics_mode == game.PhysicsMode.STRICT_V2:
					velocity.y *= pow(0.8, delta * 60)
				else:
					velocity.y *= pow(0.94, delta * 60)
		
		if velocity.y > game.get_max_y_velocity():
			#velocity.y = game.get_max_y_velocity()
			velocity.y *= pow(0.95, delta * 60)

	var speed = SPEED
	if game.physics_mode == game.PhysicsMode.STRICT_V2:
		speed = 800
	
	var accel = FLOATY_GROUND_ACCEL if is_on_floor() else FLOATY_AIR_ACCEL
	if game.physics_mode != game.PhysicsMode.FLOATY and game.physics_mode != game.PhysicsMode.INSTANT:
		accel = STRICT_GROUND_ACCEL if is_on_floor() else STRICT_AIR_ACCEL
	elif game.physics_mode == game.PhysicsMode.INSTANT:
		accel = 10_000_000
	
	if ice_mult != 1.0:
		accel *= ice_mult
		ice_mult = 1.0
	
	if game.game_mode == game.GameMode.EDITING:
		speed = EDIT_SPEED
		accel = EDIT_ACCEL
	elif is_flying:
		speed = 1250
		accel = 7500
		
	if (game.game_mode == game.GameMode.EDITING and Input.is_action_pressed("run"))\
	or (game.game_mode == game.GameMode.PLAYING and Input.is_action_pressed("run") and game.sprint_mode == game.SprintMode.NORMAL) \
	or (game.game_mode == game.GameMode.PLAYING and game.sprint_mode == game.SprintMode.ALWAYS):
		speed *= 1.5
		
	if is_flying:
		var vert_direction := Input.get_axis("move_up", "move_down")
		if vert_direction:
			velocity.y = move_toward(velocity.y, speed * vert_direction, accel * delta)
		else:
			velocity.y = move_toward(velocity.y, 0, accel * delta)
	
	var direction := Input.get_axis("move_left", "move_right")
	if not is_flying:
		if force_move_dir == game.ForceMove.LEFT:
			direction = -1.0
		elif force_move_dir == game.ForceMove.RIGHT:
			direction = 1.0
	
	if direction and game.health > 0:
		velocity.x = move_toward(velocity.x, speed * direction, accel * delta)
	else:
		velocity.x = move_toward(velocity.x, 0, accel * delta)
		
	if game.game_mode == game.GameMode.PLAYING:
		if not is_on_floor():
			if velocity.y < 0 and game.health > 0:
				%Sprite.play("%s_jumping" % OptionsManager.character)
			else:
				%Sprite.play("%s_falling" % OptionsManager.character)
		elif direction:
			%Sprite.play("%s_walking" % OptionsManager.character)
		else:
			%Sprite.play("%s_idle" % OptionsManager.character)

		%Smoke.emitting = is_on_floor() and abs(velocity.x) > SPEED
	else:
		%Sprite.play("%s_idle" % OptionsManager.character)
	
	
	if direction > 0:
		%Sprite.flip_h = false
	elif direction < 0:
		%Sprite.flip_h = true

	move_and_slide()
	
	if game.background == game.Background.RED_SKY\
	and LevelManager.level_id == "the_world_of_play_maker"\
	and game.game_mode == game.GameMode.PLAYING:
		if position.y > -58.25*256.0 or position.x < 164*256.0:
			game.restart()


func damage(body: Node2D, amount: int = 1) -> void:
	if game.health <= 0:
		return
	
	if %InvulTimer.is_stopped() or amount >= 1_000_000:
		%InvulTimer.start()
		game.health -= amount
		%CoyoteTimer.stop()
		if body:
			var dir = 1
			if global_position.x < body.global_position.x:
				dir = -1
			if game.physics_mode == game.PhysicsMode.FLYING:
				velocity = body.position.direction_to(position) * 1500
			else:
				velocity.x = 1000 * dir
				velocity.y = -1000
		%HurtAudio.play()
		if game.health <= 0:
			%AnimationPlayer.play("death")


func bounce(body: Node, amount: float = 1.0) -> void:
	%CoyoteTimer.stop()
	velocity.y = -2250 * amount


func _on_animation_player_animation_finished(anim_name: StringName) -> void:
	if anim_name == "death":
		game.restart()


func is_player() -> bool:
	return true
