extends Control

@onready var game = get_tree().get_first_node_in_group("game")
var built_in_levels_scene: PackedScene = load("res://scenes/built_in_levels_menu.tscn")
var local_levels_scene: PackedScene = load("res://scenes/local_levels_menu.tscn")
var online_levels_scene: PackedScene = load("res://scenes/online_levels_menu.tscn")
var info_menu_scene: PackedScene = load("res://scenes/info_menu.tscn")

enum Actions {
	CONTINUE,
	RESTART
}

var action := Actions.CONTINUE


func _ready() -> void:
	get_tree().paused = true
	%ContinueButton.grab_focus.call_deferred()
	%GameModeButton.visible = LevelManager.editing and DisplayServer.is_touchscreen_available()
	%EditKeysContainer.visible = LevelManager.editing and not DisplayServer.is_touchscreen_available()

	# Level skipping is disabled for now since the point for locked items was to 
	# prevent too many people from having things like shader effects
	# since they shouldn't be (ab)used too much
	return

	var skippable_levels := [
		"the_world_of_play_maker", # For the easy path secret
		"rude_cave",
		"tower_of_saws",
		"precision_hill",
		"conveyor_climb",
		#"precision_mountain", # Don't let them skip this level, as the Expert status should only be for experts
		"blocky_descent",
		"checkpoint_rush",
		"underground_race"
	]
	
	if LevelManager.level_id in skippable_levels:
		var stats := LevelManager.get_level_stats(LevelManager.level_id)
		if LevelManager.level_id == "the_world_of_play_maker":
			if stats.get("completed", false) and not LevelManager.get_level_stats("the_world_of_play_maker").get("beat_secret_ending", false):
				%SkipLabel.text = "If you can't find or\nbeat the secret ending,\nyou can skip it."
				%SkipContainer.show()
		elif LevelManager.level_id == "rude_cave":
			if stats.get("completed", false) and LevelManager.get_level_stats("rude_cave").get("least_jumps", INF) > 0:
				%SkipLabel.text = "If you find the \"no\njump\" challenge too\nhard, you can skip it."
				%SkipContainer.show()
		elif not stats.get("completed", false):
			%SkipContainer.show()


func _process(delta: float) -> void:	
	if Input.is_action_just_pressed("back"):
		if %MainMenu.visible:
			_on_continue_button_pressed()
		else:
			_on_restart_back_button_pressed()


func _on_continue_button_pressed() -> void:
	action = Actions.CONTINUE
	%AnimationPlayer.play("close")


func _on_restart_button_pressed() -> void:
	if game.checkpoint_pos:
		%MainMenu.hide()
		%RestartMenu.show()
		%RestartCheckpointButton.grab_focus.call_deferred()
	else:
		action = Actions.RESTART
		%AnimationPlayer.play("close")
		game.restart()


func _on_leave_button_pressed() -> void:
	game.save_level()
	# If we set pased to false here, the audio will play for a split second
	# so we do it in each scene that it can switch to instead
	if LevelManager.level_source == LevelManager.LevelSource.BUILT_IN:
		if game.is_info_menu_level:
			get_tree().change_scene_to_packed(info_menu_scene)
		else:
			get_tree().change_scene_to_packed(built_in_levels_scene)
	elif LevelManager.level_source == LevelManager.LevelSource.LOCAL:
		get_tree().change_scene_to_packed(local_levels_scene)
	elif LevelManager.level_source == LevelManager.LevelSource.ONLINE:
		get_tree().change_scene_to_packed(online_levels_scene)


func _on_load_level_button_pressed() -> void:
	game.load_level_data(JSON.to_native(JSON.parse_string(%LevelData.text)))


func _on_animation_player_animation_finished(anim_name: StringName) -> void:
	if anim_name == "close":
		get_tree().paused = false
		queue_free()


func _on_game_mode_button_pressed() -> void:
	game.set_game_mode(game.GameMode.EDITING if game.game_mode == game.GameMode.PLAYING else game.GameMode.PLAYING)
	_on_continue_button_pressed()


func _on_restart_checkpoint_button_pressed() -> void:
	action = Actions.RESTART
	%AnimationPlayer.play("close")
	game.restart()


func _on_restart_beginning_button_pressed() -> void:
	action = Actions.RESTART
	%AnimationPlayer.play("close")
	game.remove_checkpoint()
	game.restart()


func _on_restart_back_button_pressed() -> void:
	%MainMenu.show()
	%RestartMenu.hide()
	%ContinueButton.grab_focus.call_deferred()


func _on_skip_button_pressed() -> void:
	if %SkipButton.text != "Confirm Skip?":
		%SkipButton.text = "Confirm Skip?"
		return
	
	var stats := LevelManager.get_level_stats(LevelManager.level_id)
	stats.set("completed", true)
	
	var best_time := 999.99
	if best_time < stats.get("best_time", INF):
		stats.set("best_time", best_time)
		
	var coins := 0
	if coins > stats.get("most_coins", -1):
		stats.set("most_coins", coins)
	
	var jumps := 999
	if LevelManager.level_id == "rude_cave":
		jumps = 0
	if jumps < stats.get("least_jumps", INF):
		stats.set("least_jumps", jumps)
	
	if not stats.get("beat_secret_ending", INF):
		stats.set("beat_secret_ending", LevelManager.level_id == "the_world_of_play_maker")
	
	LevelManager.save_level_stats(stats)
	
	_on_leave_button_pressed()
