extends Control

var game_scene: PackedScene = load("res://scenes/game.tscn")
var built_in_levels_scene: PackedScene = load("res://scenes/built_in_levels_menu.tscn")
var uploaded_levels_scene: PackedScene = load("res://scenes/uploaded_levels_menu.tscn")
var level_item_scene: PackedScene = load("res://scenes/level_item.tscn")
var current_level_id := ""
static var offset := 0


func _ready() -> void:
	get_tree().paused = false
	%ColorOverlay.modulate = Color.TRANSPARENT
	%ColorOverlay.hide()
	
	%MyLevelsButton.disabled = OptionsManager.login_token == null

	%BackButton.grab_focus.call_deferred()
	do_req()


func _process(delta: float) -> void:
	if Input.is_action_just_pressed("back"):
		_on_back_button_pressed()


func do_req() -> void:
	%Status.text = "Loading..."
	for child in %Levels.get_children():
		child.queue_free()
	%PrevPageButton.disabled = true
	%NextPageButton.disabled = true
	var headers = []
	if OptionsManager.login_token:
		headers.push_back("Authorization: Bearer %s" % OptionsManager.login_token)
	%LevelListReq.request("%s/levels?limit=%d&offset=%d" % [OptionsManager.server_base_url, OptionsManager.levels_per_page, offset], headers)


func _on_back_button_pressed() -> void:
	get_tree().change_scene_to_packed(built_in_levels_scene)


func _on_level_list_req_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	var err = Globals.get_req_error(result, response_code, body)
	if err:
		%Status.text = err
		return
	%Status.text = "Success"
		
	var json = JSON.parse_string(body.get_string_from_utf8())
	%PrevPageButton.disabled = offset == 0
	%NextPageButton.disabled = json.get("last_page")
	
	for level_meta in json.get("levels"):
		var level_item = level_item_scene.instantiate()
		level_item.title = level_meta.get("title")
		level_item.author = level_meta.get("author_username")
		level_item.show_author = true
		level_item.built_in_level_id = level_meta.get("id")
		level_item.upload_time = level_meta.get("created_at")
		level_item.show_comments = true
		
		level_item.get_node("%PlayButton").connect("pressed", play.bind(level_meta.get("id")))
		
		level_item.supporter = level_meta.get("supporter", level_item.supporter)
		level_item.supporter_colors = level_meta.get("supporter_colors", level_item.supporter_colors)
		level_item.supporter_emphasis = level_meta.get("supporter_emphasis", level_item.supporter_emphasis)
		level_item.bg_style = level_meta.get("bg_style", level_item.bg_style)
		level_item.author_suffix = level_meta.get("author_suffix", level_item.author_suffix)
		level_item.rainbow_speed = level_meta.get("rainbow_speed", level_item.rainbow_speed)
		level_item.rainbow_title = level_meta.get("rainbow_title", level_item.rainbow_title)
		level_item.rainbow_author = level_meta.get("rainbow_author", level_item.rainbow_author)
		level_item.rainbow_upload_date = level_meta.get("rainbow_upload_date", level_item.rainbow_upload_date)
		level_item.rainbow_best_time = level_meta.get("rainbow_best_time", level_item.rainbow_best_time)
		level_item.rainbow_play_button = level_meta.get("rainbow_play_button", level_item.rainbow_play_button)
		level_item.rainbow_play_text = level_meta.get("rainbow_play_text", level_item.rainbow_play_text)
		
		%Levels.add_child(level_item)


func play(level_id: String) -> void:
	DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_CAPTURED)
	%ColorOverlay.show()
	%AnimationPlayer.play("fade_exit")
	current_level_id = level_id


func _on_get_level_data_req_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	var err = Globals.get_req_error(result, response_code, body)
	if err:
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_VISIBLE)
		%Status.text = err
		%ColorOverlay.hide()
		return
	
	var json = JSON.parse_string(body.get_string_from_utf8())
	LevelManager.level_source = LevelManager.LevelSource.ONLINE
	LevelManager.level_id = current_level_id
	var level_meta := ConfigFile.new()
	level_meta.parse(json.get("level_data_native"))
	LevelManager.level_data = level_meta.get_value("level", "data", {})
	LevelManager.level_data.set("id", json.get("id"))
	LevelManager.editing = false
	LevelManager.verifying = false
	DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_CAPTURED)
	get_tree().change_scene_to_packed(game_scene)


func _on_prev_page_button_pressed() -> void:
	offset -= OptionsManager.levels_per_page
	offset = max(offset, 0)
	do_req()


func _on_next_page_button_pressed() -> void:
	offset += OptionsManager.levels_per_page
	do_req()


func _on_my_levels_button_pressed() -> void:
	get_tree().change_scene_to_packed(uploaded_levels_scene)


func _on_animation_player_animation_finished(anim_name: StringName) -> void:
	if anim_name == "fade_exit":
		%GetLevelDataReq.request("%s/levels/%s" % [OptionsManager.server_base_url, current_level_id])
