extends Control

var game_scene: PackedScene = load("res://scenes/game.tscn")
var main_menu_scene: PackedScene = load("res://scenes/main_menu.tscn")
var level_item_scene: PackedScene = load("res://scenes/level_item.tscn")

# Called when the node enters the scene tree for the first time.
func _ready() -> void:
	get_tree().paused = false
	%ColorOverlay.modulate = Color.TRANSPARENT
	%ColorOverlay.hide()
	%PublishOverlay.hide()
	
	%BackButton.grab_focus.call_deferred()
	for level_meta in LevelManager.get_local_levels():
		var level_item = level_item_scene.instantiate()
		level_item.title = level_meta.get("title", level_item.title)
		level_item.author = level_meta.get("author", level_item.author)
		level_item.show_stats = false
		level_item.show_edit = true
		level_item.show_publish = true
		level_item.get_node("%PublishButton").disabled = OptionsManager.login_token == null
		
		level_item.get_node("%EditButton").connect("pressed", open_level.bind(level_meta.id, true))
		level_item.get_node("%PlayButton").connect("pressed", open_level.bind(level_meta.id, false))
		level_item.get_node("%PublishButton").connect("pressed", publish.bind(level_meta.id))
		%Levels.add_child(level_item)


# Called every frame. 'delta' is the elapsed time since the previous frame.
func _process(delta: float) -> void:
	if Input.is_action_just_pressed("back"):
		_on_back_button_pressed()


func open_level(level_id: String, editing: bool) -> void:
	LevelManager.level_source = LevelManager.LevelSource.LOCAL
	LevelManager.level_id = level_id
	LevelManager.level_data = LevelManager.get_local_level_data(LevelManager.level_id)
	LevelManager.editing = editing
	LevelManager.verifying = false
	if not editing:
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_CAPTURED)
	%ColorOverlay.show()
	%AnimationPlayer.play("fade_exit")


func _on_new_button_pressed() -> void:
	var level = LevelManager.new_local_level()
	LevelManager.level_source = LevelManager.LevelSource.LOCAL
	LevelManager.level_id = level.get("id")
	LevelManager.level_data = {}
	LevelManager.editing = true
	LevelManager.verifying = false
	%ColorOverlay.show()
	%AnimationPlayer.play("fade_exit")


func _on_back_button_pressed() -> void:
	get_tree().change_scene_to_packed(main_menu_scene)


func _on_animation_player_animation_finished(anim_name: StringName) -> void:
	if anim_name == "fade_exit":
		get_tree().change_scene_to_packed(game_scene)


func publish(level_id: String) -> void:
	LevelManager.level_source = LevelManager.LevelSource.LOCAL
	LevelManager.level_id = level_id
	LevelManager.editing = false
	LevelManager.verifying = true
	var meta = LevelManager.get_local_level_data(level_id)
	if meta.title == "Untitled":
		%Title.text = "No Title"
		%Description.text = "You must name your\n level to upload it."
		%PublishConfirmButton.hide()
	else:
		%Title.text = "Upload"
		%Description.text = "You must play and win\nyour level to upload it."
		%PublishConfirmButton.show()
	%PublishOverlay.show()
	%PublishCancelButton.grab_focus.call_deferred()

func _on_publish_request_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	if result != HTTPRequest.RESULT_SUCCESS:
		printerr("Got error when publishing level: %d" % result)
		return
	elif response_code < 200 or response_code >= 300:
		printerr("Got response code error when publishing level: %d" % response_code)
		printerr("Response: " + body.get_string_from_utf8())
		return
	
	var json = JSON.parse_string(body.get_string_from_utf8())
	if not json:
		printerr("Failed to parse as JSON: " + body.get_string_from_utf8())
		return


func _on_publish_cancel_button_pressed() -> void:
	%PublishOverlay.hide()


func _on_publish_confirm_button_pressed() -> void:
	LevelManager.level_data = LevelManager.get_local_level_data(LevelManager.level_id)
	if not LevelManager.level_data.has("music") and %Title.text == "Upload":
		%Title.text = "No Music"
		%Description.text = "This level does not have\nany music. Are you sure\nyou want to continue?"
		return
	
	DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_CAPTURED)
	%ColorOverlay.show()
	%AnimationPlayer.play("fade_exit")
