extends Node2D

const LEVEL_MAJOR_VERSION := 15
var main_menu_scene: PackedScene = load("res://scenes/main_menu.tscn")
var player_scene: PackedScene = load("res://scenes/player.tscn")
var pause_menu_scene: PackedScene = load("res://scenes/pause_menu.tscn")
var win_menu_scene: PackedScene = load("res://scenes/win_menu.tscn")
var publish_menu_scene: PackedScene = load("res://scenes/publish_menu.tscn")
var entity_props_scene: PackedScene = load("res://scenes/entity_props_menu.tscn")
var exit_when_possible := false

var tile_types := {
	# The number is the source
	"grass": 16,
	"sand": 44,
	"bridge": 51,
}

var terrain := {
	"grass": 0,
	"sand": 1
}

var entity_types := {
	"goal": load("res://scenes/goal.tscn"),
	"checkpoint": load("res://scenes/checkpoint.tscn"),
	"coin": load("res://scenes/coin.tscn"),
	"heart": load("res://scenes/heart.tscn"),
	"mouse": load("res://scenes/mouse.tscn"),
	"snail": load("res://scenes/snail.tscn"),
	"bee": load("res://scenes/bee.tscn"),
	"fly": load("res://scenes/fly.tscn"),
	"saw": load("res://scenes/saw.tscn"),
	"falling_block": load("res://scenes/falling_block.tscn"),
	"blue_switch": load("res://scenes/blue_switch.tscn"),
	"red_switch": load("res://scenes/red_switch.tscn"),
	"purple_switch": load("res://scenes/purple_switch.tscn"),
	"blue_block": load("res://scenes/blue_block.tscn"),
	"red_block": load("res://scenes/red_block.tscn"),
	"purple_block": load("res://scenes/purple_block.tscn"),
	"spring": load("res://scenes/spring.tscn"),
	"conveyor": load("res://scenes/conveyor.tscn"),
	"brick": load("res://scenes/brick.tscn"),
	"ice": load("res://scenes/ice.tscn"),
	"sfx_positive": load("res://scenes/sfx_positive.tscn"),
	"sfx_negative": load("res://scenes/sfx_negative.tscn"),
	"sfx_alarm": load("res://scenes/sfx_alarm.tscn"),
	"sfx_magic": load("res://scenes/sfx_magic.tscn"),
	"clear_effects": load("res://scenes/clear_effects.tscn"),
	"invert": load("res://scenes/invert.tscn"),
	"pixelate": load("res://scenes/pixelate.tscn"),
	"flip_vertically": load("res://scenes/flip_vertically.tscn"),
	"flip_horizontally": load("res://scenes/flip_horizontally.tscn"),
	"grayscale": load("res://scenes/grayscale.tscn"),
	"hide_screen": load("res://scenes/hide_screen.tscn"),
	"gravestone": load("res://scenes/gravestone.tscn"),
	"background_changer": load("res://scenes/background_changer.tscn"),
	"music_changer": load("res://scenes/music_changer.tscn"),
	"red_mushroom": load("res://scenes/red_mushroom.tscn"),
	"brown_mushroom": load("res://scenes/brown_mushroom.tscn"),
	"grass_decoration_a": load("res://scenes/grass_decoration_a.tscn"),
	"sign": load("res://scenes/sign.tscn")
}

var allowed_to_rotate := [
	"goal",
	"checkpoint",
	"falling_block",
	"blue_switch",
	"red_switch",
	"purple_switch",
	"spring",
	"conveyor",
	"gravestone",
	"brown_mushroom",
	"red_mushroom",
	"grass_decoration_a",
	"sign"
]

var allowed_to_flip := [
	"goal",
	"checkpoint",
	"mouse",
	"snail",
	"bee",
	"fly",
	"conveyor",
	"grass_decoration_a",
	"sign"
]

var entity_properties := {
	"goal": {
		"secret_ending": {
			"title": "Secret Ending",
			"type": "bool"
		},
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"checkpoint": {
		"respawn_health": {
			"title": "Respawn Health\n(0 = Level health)",
			"type": "int",
			"min": 0,
			"max": 100
		},
		"immediate_health": {
			"title": "Set health\non collect",
			"type": "bool"
		},
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"coin": {
		"amount": {
			"title": "Amount",
			"type": "int",
			"min": 1,
			"max": 5
		}
	},
	"heart": {
		"amount": {
			"title": "Amount",
			"type": "int",
			"min": 1,
			"max": 5
		}
	},
	"saw": {
		"damage": {
			"title": "Damage (-1 = Insta-kill)",
			"type": "int",
			"min": -1,
			"max": 100
		},
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"falling_block": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"blue_switch": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"red_switch": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"purple_switch": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"brick": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"spring": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"conveyor": {
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"ice": {
		"ice_mult": {
			"title": "Accel Multiplier",
			"type": "float",
			"min": 0,
			"max": 5.0,
			"step": 0.05
		},
		"fake": {
			"title": "Fake",
			"type": "bool"
		},
		"invisible": {
			"title": "Invisible",
			"type": "bool"
		}
	},
	"sfx_positive": {
		"only_play_once": {
			"title": "Only Play Once",
			"type": "bool"
		}
	},
	"sfx_negative": {
		"only_play_once": {
			"title": "Only Play Once",
			"type": "bool"
		}
	},
	"sfx_alarm": {
		"only_play_once": {
			"title": "Only Play Once",
			"type": "bool"
		}
	},
	"sfx_magic": {
		"only_play_once": {
			"title": "Only Play Once",
			"type": "bool"
		}
	},
	"background_changer": {
		"background": {
			"title": "Background",
			"type": "int",
			"min": 0,
			"max": 7
		}
	},
	"music_changer": {
		"music": {
			"title": "Music",
			"type": "int",
			"min": 0,
			"max": 14
		}
	}
}

enum GameMode {
	PLAYING,
	EDITING
}

enum PhysicsMode {
	FLOATY,
	STRICT,
	INSTANT,
	STRICT_V2,
	SLOW_FALL,
	LOW_GRAV,
	FLYING
}

var music := {
	"none": null,
	"determined_start": "res://assets/music/DeterminedStart.ogg",
	"staring_at_reflections": "res://assets/music/StaringAtReflections.ogg",
	"sweet_talk": "res://assets/music/SweetTalk.ogg",
	"bugmintide": "res://assets/music/Bugmintide.ogg",
	"sewer_city": "res://assets/music/SewerCity.ogg",
	"firefly_stream": "res://assets/music/1.1 Firefly Stream - Loop1 - Blue Lava.ogg",
	"falling_leaf_path": "res://assets/music/2.2 Falling Leaf Path_Autumnly Town - Loop1 - Blue Lava.ogg",
	"bamboo_battle": "res://assets/music/3.3 Bamboo Battle - Loop1 - Blue Lava.ogg",
	"autumn_valley": "res://assets/music/4.2 Autumn Valley - Loop1 - Blue Lava.ogg",
	"evil_incoming": "res://assets/music/Evil Incoming.ogg",
	"improv_for_evil": "res://assets/music/Improv for Evil.ogg",
	"desert_fox": "res://assets/music/Desert Fox.ogg",
	"finally_see_the_light": "res://assets/music/Finally See The Light.ogg"
}

enum Background {
	SKY,
	NIGHT_SKY,
	DIRT,
	SAND,
	DESERT,
	STARS,
	NEBULA,
	RED_SKY
}

enum ProgressSource {
	NONE,
	HORIZONTAL,
	VERTICAL,
	COINS,
	TIME
}

enum SprintMode {
	NORMAL,
	ALWAYS,
	NEVER
}

enum ForceMove {
	OFF,
	LEFT,
	RIGHT
}

var game_mode := GameMode.PLAYING
var switch_color := Globals.SwitchColor.BLUE
var in_bounds := false

var editor_selected_item = "grass"
var editor_position = null
var level_title := "Untitled"
var level_creation_time := 0
var start_pos = Vector2i(5, -4)
var checkpoint_pos = null
var checkpoint_health = null
var collected := []
var checkpoint_collected := []
var start_health := 5
var show_health := true
var max_time := 0
var show_time_left := true
var time_spent := 0.0
var added_time := 0.0
var checkpoint_time := 0.0
var checkpoint_jumps := 0
var added_jumps := 0
var health := 5
var max_jumps := -1
var min_coins := 0
var coins := 0
var show_coin_count := true
var camera_offset := Vector2i(0, 0)
var physics_mode := PhysicsMode.FLOATY
var air_jumping := false
var music_selected = null
var background := Background.SKY
var progress_source := ProgressSource.NONE
var sprint_mode := SprintMode.NORMAL
var inverted := false
var grayscale := false
var screen_hidden := false
var force_move := ForceMove.OFF
var first_load := true
var first_load_major_version = -1
var is_info_menu_level := false


func _ready() -> void:
	%ColorOverlay.show()
	load_level_data(LevelManager.level_data)
	set_game_mode(GameMode.EDITING if LevelManager.editing else GameMode.PLAYING)
	
	# For mobile
	var unpress_actions = ["move_left", "move_right", "move_up", "move_down"]
	for action in unpress_actions:
		var unpress_event = InputEventAction.new()
		unpress_event.action = action
		unpress_event.pressed = false
		Input.parse_input_event(unpress_event)
	
	%Joystick.joystick_mode = OptionsManager.joystick_mode
	is_info_menu_level = Globals.next_is_info_menu_level
	Globals.next_is_info_menu_level = false


func _process(delta: float) -> void:
	if exit_when_possible:
		Globals.show_old_version_warning = true
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_VISIBLE)
		get_tree().change_scene_to_packed(main_menu_scene)
		return
	
	# Uses the control and not joystickd directly because otherwise it will show on non-touch devices
	%JoystickControl.visible = game_mode == GameMode.EDITING\
	or get_tree().get_first_node_in_group("player").force_move_dir == ForceMove.OFF
	
	if Input.is_action_just_pressed("next_game_mode") and LevelManager.editing:
		set_game_mode(GameMode.EDITING if game_mode == GameMode.PLAYING else GameMode.PLAYING)
	
	if game_mode == GameMode.EDITING:
		if Input.is_action_just_pressed("place") or Input.is_action_just_pressed("remove"):
			in_bounds = get_viewport().get_mouse_position().y > 448.0
			if DisplayServer.is_touchscreen_available():
				var viewport_height = float(ProjectSettings.get_setting("display/window/size/viewport_height"))
				if %Joystick.get_global_rect().has_point(get_viewport().get_mouse_position()):
					in_bounds = false
				
				if OptionsManager.pause_visible:
					# TouchScreenButton isn't a control so we make the rect manually
					var pause_btn = get_tree().get_first_node_in_group("editor_overlay").get_node("%PauseButton")
					var pause_rect = Rect2(pause_btn.global_position.x, pause_btn.global_position.y, %Joystick.size.x * %Joystick.scale.x, %Joystick.size.y * %Joystick.scale.y)
					if pause_rect.has_point(get_viewport().get_mouse_position()):
						in_bounds = false
		
		if Input.is_action_just_pressed("place") and (editor_selected_item == "rotate" or editor_selected_item == "flip" or editor_selected_item == "props"):
			in_bounds = false
		
		if (Input.is_action_pressed("place") or Input.is_action_pressed("remove")) and in_bounds:
			var item = editor_selected_item
			if Input.is_action_pressed("remove"):
				item = null
			
			var tilemap_pos = %Blocks.local_to_map(%Blocks.to_local(get_global_mouse_position()))
			
			# On the web, they can draw blocks outside the game space
			if tilemap_pos.x < 0:
				tilemap_pos.x = 0
			if tilemap_pos.y > -1:
				tilemap_pos.y = -1
			
			for entity: Node2D in %Entities.get_children():
				if entity.scene_file_path == player_scene.resource_path:
					continue
					
				if %Blocks.local_to_map(%Blocks.to_local(entity.global_position)) == tilemap_pos:
					entity.queue_free()
			
			%Blocks.set_cell(tilemap_pos, -1)
			
			# Source ID can be zero so we have != null: and not just :
			if tile_types.get(item) != null:
				if terrain.has(item):
					%Blocks.set_cells_terrain_connect([tilemap_pos], 0, terrain.get(item))
				else:
					%Blocks.set_cell(tilemap_pos, tile_types.get(item), Vector2i(0, 0), 0)
			elif item:
				var scene: PackedScene = entity_types.get(item)
				var entity = scene.instantiate()
				entity.position = (Vector2(tilemap_pos) + Vector2(0.5, 0.5)) * 256.0
				%Entities.add_child(entity)
			
			var edge_item = terrain.get(item)
			if tilemap_pos.x == 0:
				if edge_item != null:
					%Blocks.set_cells_terrain_connect([tilemap_pos + Vector2i(-1, 0)], 0, edge_item)
				else:
					%Blocks.set_cell(tilemap_pos + Vector2i(-1, 0), -1)
			if tilemap_pos.y == -1:
				if edge_item != null:
					%Blocks.set_cells_terrain_connect([tilemap_pos + Vector2i(0, 1)], 0, edge_item)
				else:
					%Blocks.set_cell(tilemap_pos + Vector2i(0, 1), -1)
			if tilemap_pos == Vector2i(0, -1):
				if edge_item != null:
					%Blocks.set_cells_terrain_connect([tilemap_pos + Vector2i(-1, 1)], 0, edge_item)
				else:
					%Blocks.set_cell(tilemap_pos + Vector2i(-1, 1), -1)
		
			for surrounding_pos in %Blocks.get_surrounding_cells(tilemap_pos):
				var tile_data = %Blocks.get_cell_tile_data(surrounding_pos)
				if tile_data:
					var tile_id = tile_data.get_custom_data("id")
					if terrain.has(tile_id):
						%Blocks.set_cell(surrounding_pos, -1)
						%Blocks.set_cells_terrain_connect([surrounding_pos], 0, terrain.get(tile_id))
		
		elif Input.is_action_just_pressed("rotate") or (editor_selected_item == "rotate" and Input.is_action_just_pressed("place")):
			var tilemap_pos = %Blocks.local_to_map(%Blocks.to_local(get_global_mouse_position()))
			for entity: Node2D in %Entities.get_children():
				if entity.scene_file_path == player_scene.resource_path:
					continue
					
				if %Blocks.local_to_map(%Blocks.to_local(entity.global_position)) == tilemap_pos:
					for allowed_id in allowed_to_rotate:
						if entity.scene_file_path == entity_types.get(allowed_id).resource_path:
							entity.rotation_degrees += 90
							entity.reset_physics_interpolation()
							continue
		elif Input.is_action_just_pressed("flip") or (editor_selected_item == "flip" and Input.is_action_just_pressed("place")):
			var tilemap_pos = %Blocks.local_to_map(%Blocks.to_local(get_global_mouse_position()))
			for entity: Node2D in %Entities.get_children():
				if entity.scene_file_path == player_scene.resource_path:
					continue
					
				if %Blocks.local_to_map(%Blocks.to_local(entity.global_position)) == tilemap_pos:
					for allowed_id in allowed_to_flip:
						if entity.scene_file_path == entity_types.get(allowed_id).resource_path:
							entity.get_node("%Sprite").flip_h = not entity.get_node("%Sprite").flip_h 
							if entity.has_node("%Sprite2"):
								entity.get_node("%Sprite2").flip_h = not entity.get_node("%Sprite2").flip_h 
							if entity.has_node("%Sprite3"):
								entity.get_node("%Sprite3").flip_h = not entity.get_node("%Sprite3").flip_h 
							continue
		
		elif Input.is_action_just_pressed("props") or (editor_selected_item == "props" and Input.is_action_just_pressed("place")):
			var tilemap_pos = %Blocks.local_to_map(%Blocks.to_local(get_global_mouse_position()))
			for entity: Node2D in %Entities.get_children():
				if entity.scene_file_path == player_scene.resource_path:
					continue
					
				if %Blocks.local_to_map(%Blocks.to_local(entity.global_position)) == tilemap_pos:
					if entity_properties.get(get_id_from_node(entity)):
						var scene = entity_props_scene.instantiate()
						scene.entity = entity
						%Menu.add_child(scene)
	
	
	if Input.is_action_just_pressed("restart"):
		restart()
			
	if Input.is_action_just_pressed("back"):
		%Menu.add_child(pause_menu_scene.instantiate())
		
	if get_tree().paused or game_mode == GameMode.EDITING:
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_VISIBLE)
	elif not OS.has_feature("web"):
		# On the web, this makes their cursor get stuck as captured
		# Ideally we wouldn't have this loop here
		# For now, the cursor may not always get captured when it should be on the web
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_CAPTURED)

	%Invert.modulate = %Invert.modulate.lerp(Color.WHITE if inverted else Color.TRANSPARENT, 1 - exp(-5 * delta))
	%Invert.visible = %Invert.modulate.a > 0.01
	
	%Grayscale.modulate = %Grayscale.modulate.lerp(Color.WHITE if grayscale else Color.TRANSPARENT, 1 - exp(-5 * delta))
	%Grayscale.visible = %Grayscale.modulate.a > 0.01
	
	%HideScreen.modulate = %HideScreen.modulate.lerp(Color.WHITE if screen_hidden else Color.TRANSPARENT, 1 - exp(-5 * delta))
	%HideScreen.visible = %HideScreen.modulate.a > 0.01


func _physics_process(delta: float) -> void:
	time_spent += delta
	
	if max_time > 0 and time_spent > max_time and game_mode == GameMode.PLAYING:
		get_tree().get_first_node_in_group("player").damage(null, 100_000_000)

	if physics_mode == PhysicsMode.STRICT_V2:
		PhysicsServer2D.area_set_param(get_world_2d().space, PhysicsServer2D.AREA_PARAM_GRAVITY, 1960 * 1.35)
	elif physics_mode == PhysicsMode.LOW_GRAV:
		PhysicsServer2D.area_set_param(get_world_2d().space, PhysicsServer2D.AREA_PARAM_GRAVITY, 1960 / 1.5)
	else:
		PhysicsServer2D.area_set_param(get_world_2d().space, PhysicsServer2D.AREA_PARAM_GRAVITY, 1960)


func get_max_y_velocity() -> float:
	if physics_mode == PhysicsMode.STRICT_V2:
		return 2000
	elif physics_mode == PhysicsMode.SLOW_FALL:
		return 750
	elif first_load_major_version >= 12:
		return 5000
	return INF
		

func set_game_mode(game_mode: GameMode) -> void:
	self.game_mode = game_mode
	%BottomCollision.disabled = game_mode == GameMode.PLAYING
	
	if game_mode == GameMode.EDITING:
		remove_checkpoint()
		%EditorOverlay.show()
		%GameOverlay.hide()
		load_level_data(LevelManager.level_data, true)
		%Entities.process_mode = Node.PROCESS_MODE_DISABLED
		set_music("none")
		remove_effects()
	elif game_mode == GameMode.PLAYING:
		%GameOverlay.show()
		%EditorOverlay.hide()
		editor_position = get_tree().get_first_node_in_group("player").position
		LevelManager.level_data = get_level_data()
		%Entities.process_mode = Node.PROCESS_MODE_INHERIT
		time_spent = 0
		get_tree().get_first_node_in_group("player").force_move_dir = force_move
		
		
		if music_selected:
			set_music(music_selected)


var music_playing = null
func set_music(music_id) -> void:
	if music_id != music_playing:
		var music_res_path = music.get(music_id)
		if music_res_path:
			%MusicPlayer.stream = load(music_res_path)
			%MusicPlayer.play()
		else:
			%MusicPlayer.stop()
	music_playing = music_id


func restart() -> void:
	if game_mode == GameMode.EDITING:
		var player = get_tree().get_first_node_in_group("player")
		player.velocity = Vector2.ZERO
		player.position = (Vector2(start_pos) + Vector2(0.5, 0.5)) * 256.0
		player.reset_physics_interpolation()
	elif game_mode == GameMode.PLAYING:
		added_time = 0.0
		added_jumps = 0
		clear_effects()
		load_level_data(LevelManager.level_data, true)


func win(goal) -> void:
	if %Menu.get_children().size() != 0:
		return
	
	if coins < min_coins:
		return
	var stats := LevelManager.get_level_stats(LevelManager.level_id)
	stats.set("completed", true)
	if checkpoint_time + time_spent + added_time < stats.get("best_time", INF):
		stats.set("best_time", checkpoint_time + time_spent + added_time)
	if coins > stats.get("most_coins", -1):
		stats.set("most_coins", coins)
	
	var total_jumps = checkpoint_jumps + get_tree().get_first_node_in_group("player").jumps + added_jumps
	if total_jumps < stats.get("least_jumps", INF):
		stats.set("least_jumps", total_jumps)
	if not stats.get("beat_secret_ending", false):
		stats.set("beat_secret_ending", goal.secret_ending)
	LevelManager.save_level_stats(stats)
	if LevelManager.verifying:
		%Menu.add_child(publish_menu_scene.instantiate())
	else:
		%Menu.add_child(win_menu_scene.instantiate())
	DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_VISIBLE)


func set_checkpoint(entity: Node2D) -> void:
	var new_checkpoint_pos = Vector2i(floor(entity.position / 256))
	if checkpoint_pos == new_checkpoint_pos:
		return
	
	checkpoint_time += time_spent + added_time
	added_time = -time_spent
	
	checkpoint_pos = new_checkpoint_pos
	if entity.respawn_health > 0:
		checkpoint_health = entity.respawn_health
	else:
		checkpoint_health = null
	
	checkpoint_collected = collected.duplicate()
	
	if entity.immediate_health:
		health = checkpoint_health if checkpoint_health else start_health
	
	var player_jumps = get_tree().get_first_node_in_group("player").jumps
	checkpoint_jumps += player_jumps + added_jumps
	added_jumps = -player_jumps


func remove_checkpoint() -> void:
	checkpoint_pos = null
	checkpoint_time = 0.0
	added_time = 0.0
	checkpoint_jumps = 0
	added_jumps = 0
	checkpoint_health = null
	checkpoint_collected.clear()


func get_id_from_node(node: Node):
	for entity_id in entity_types:
		if node.scene_file_path == entity_types.get(entity_id).resource_path:
			return entity_id
	return null


func get_level_data() -> Dictionary:
	var data = {}
	data.set("version", {
		"major": LEVEL_MAJOR_VERSION,
		"minor": 0
	})
	
	data.set("id", LevelManager.level_id)
	data.set("title", level_title)
	data.set("creation_time", level_creation_time)
	data.set("start_pos", start_pos)
	data.set("camera_offset", camera_offset)
	data.set("health", start_health)
	data.set("show_health", show_health)
	data.set("max_time", max_time)
	data.set("show_time_left", show_time_left)
	data.set("max_jumps", max_jumps)
	data.set("min_coins", min_coins)
	data.set("show_coin_count", show_coin_count)
	data.set("physics_mode", physics_mode)
	data.set("air_jumping", air_jumping)
	data.set("background", background)
	data.set("progress_source", progress_source)
	data.set("sprint_mode", sprint_mode)
	data.set("force_move", force_move)
	if music_selected:
		data.set("music", music_selected)
	
	var blocks: Array[Dictionary] = []
	for block_pos in %Blocks.get_used_cells():
		if block_pos.x < -1 or block_pos.y > 0:
			continue
		
		var tile_data = %Blocks.get_cell_tile_data(block_pos)
		if tile_data:
			var id = tile_data.get_custom_data("id")
			if id:
				blocks.push_back({
					"pos": block_pos,
					"id": tile_data.get_custom_data("id"),
					"source": %Blocks.get_cell_source_id(block_pos)
				})
			else:
				printerr("MISSING TILE ID")
		else:
			printerr("MISSING TILE DATA")
	data.set("blocks", blocks)
	
	var entities: Array[Dictionary] = []
	for entity in %Entities.get_children():
		var id = get_id_from_node(entity)
		if id:
			var flip = false
			if id in allowed_to_flip:
				flip = entity.get_node("%Sprite").flip_h
			
			var props = null
			var entity_props = entity_properties.get(id)
			if entity_props:
				props = {}
				for prop_id in entity_props.keys():
					var prop = entity_props.get(prop_id)
					props.set(prop_id, entity[prop_id])
			
			entities.push_back({
				"pos": Vector2i(floor(entity.position / 256)),
				"rot": roundi(entity.rotation_degrees / 90),
				"flip": flip,
				"id": id,
				"props": props
			})
		elif entity.scene_file_path != player_scene.resource_path:
			printerr("MISSING ENTITY ID")
	data.set("entities", entities)
		
	return data


func load_level_data(level_data: Dictionary, fast: bool = false) -> void:
	LevelManager.level_data = level_data
	
	if first_load:
		first_load = false
		first_load_major_version = level_data.get("version", {"major": -1}).get("major")
	
	# Edited levels always get upgraded
	if LevelManager.editing:
		first_load_major_version = LEVEL_MAJOR_VERSION
		
	# Defaults to -1 because a new level doesn't have data yet
	if level_data.get("version", {"major": -1}).get("major") > LEVEL_MAJOR_VERSION:
		exit_when_possible = true
		return
	
	# Update values shown on the GUI right away
	coins = 0
	switch_color = Globals.SwitchColor.BLUE
	level_title = level_data.get("title", level_title)
	level_creation_time = level_data.get("creation_time", int(floor(Time.get_unix_time_from_system())))
	start_pos = level_data.get("start_pos", start_pos)
	camera_offset = level_data.get("camera_offset", camera_offset)
	start_health = level_data.get("health", start_health)
	health = checkpoint_health if checkpoint_health else start_health
	if health <= 0:
		health = 10_000_000
	show_health = level_data.get("show_health", true)
	max_time = level_data.get("max_time", 0)
	show_time_left = level_data.get("show_time_left", true)
	max_jumps = level_data.get("max_jumps", -1)
	min_coins = level_data.get("min_coins", 0)
	show_coin_count = level_data.get("show_coin_count", true)
	physics_mode = level_data.get("physics_mode", PhysicsMode.FLOATY)
	air_jumping = level_data.get("air_jumping", air_jumping)
	music_selected = level_data.get("music")
	set_background(level_data.get("background", Background.SKY))
	progress_source = level_data.get("progress_source", ProgressSource.NONE)
	sprint_mode = level_data.get("sprint_mode", SprintMode.NORMAL)
	force_move = level_data.get("force_move", force_move)
	
	if not fast:
		%Blocks.clear()
		var block_ids = {}
		
		var has_block_without_source := false
		for block in level_data.get("blocks", {}):
			var block_pos = block.get("pos")
			if terrain.has(block.id):
				# Currently only grass has all of the tile texture combinations
				if block.has("source") and block.id == "grass":
					%Blocks.set_cell(block_pos, block.get("source"), Vector2i(0, 0), 0)
				else:
					has_block_without_source = true
					%Blocks.set_cells_terrain_connect([block_pos], 0, terrain.get(block.id))
					if block_pos.x == 0:
						%Blocks.set_cells_terrain_connect([block_pos + Vector2i(-1, 0)], 0, terrain.get(block.id))
					if block_pos.y == -1:
						%Blocks.set_cells_terrain_connect([block_pos + Vector2i(0, 1)], 0, terrain.get(block.id))
					if block_pos == Vector2i(0, -1):
						%Blocks.set_cells_terrain_connect([block_pos + Vector2i(-1, 1)], 0, terrain.get(block.id))
			elif tile_types.get(block.id):
				%Blocks.set_cell(block_pos, tile_types.get(block.id), Vector2i(0, 0), 0)
			else:
				# Block ID isn't known to the client
				exit_when_possible = true
				return
			block_ids.set(block_pos, block.id)
		
		if has_block_without_source:
			# Fixed connections in some cases
			for block_pos in %Blocks.get_used_cells():
				var block_id = block_ids.get(block_pos)
				if terrain.has(block_id) and block_id == "sand":
					%Blocks.set_cells_terrain_connect([block_pos], 0, terrain.get(block_id))
		
	for child in %Entities.get_children():
		child.queue_free()
	
	var entity_id := -1
	for entity_data in level_data.get("entities", {}):
		entity_id += 1
		var scene: PackedScene = entity_types.get(entity_data.get("id"))
		if scene:
			var entity = scene.instantiate()
			if "entity_id" in entity:
				entity.entity_id = entity_id
			entity.position = (Vector2(entity_data.get("pos")) + Vector2(0.5, 0.5)) * 256.0
			entity.rotation_degrees = entity_data.get("rot", 0) * 90.0
			if entity_data.get("flip"):
				entity.get_node("%Sprite").flip_h = true
				if entity.has_node("%Sprite2"):
					entity.get_node("%Sprite2").flip_h = true
				if entity.has_node("%Sprite3"):
					entity.get_node("%Sprite3").flip_h = true
			
			var entity_props = entity_properties.get(entity_data.get("id"))
			if entity_props:
				var props = entity_data.get("props")
				if props:
					for prop_id in props.keys():
						if props.get(prop_id) != null:
							entity[prop_id] = props.get(prop_id)
			
			if entity_id in checkpoint_collected:
				entity.collect(self)
				entity.queue_free()
			else:
				%Entities.add_child(entity)
		else:
			# Entity ID isn't known to the client
			exit_when_possible = true
			return

	var player: Node2D = player_scene.instantiate()
	var spawn_pos = checkpoint_pos if checkpoint_pos else start_pos
	player.position = (Vector2(spawn_pos) + Vector2(0.5, 0.5)) * 256.0
	if game_mode == GameMode.EDITING and editor_position:
		player.position = editor_position
	%Entities.add_child(player)
	player.force_move_dir = force_move
	player.get_node("Camera").make_current()
	
	collected = checkpoint_collected.duplicate()
	time_spent = 0


func set_background(bg: Background) -> void:
	background = bg
	%SkyBackground.hide()
	%NightSkyBackground.hide()
	%DirtBackground.hide()
	%SandBackground.hide()
	%DesertBackground.hide()
	%StarsBackground.hide()
	%NebulaBackground.hide()
	%RedSkyBackground.hide()
	if background == Background.SKY:
		%SkyBackground.show()
	elif background == Background.NIGHT_SKY:
		%NightSkyBackground.show()
	elif background == Background.DIRT:
		%DirtBackground.show()
	elif background == Background.SAND:
		%SandBackground.show()
	elif background == Background.DESERT:
		%DesertBackground.show()
	elif background == Background.STARS:
		%StarsBackground.show()
	elif background == Background.NEBULA:
		%NebulaBackground.show()
	elif background == Background.RED_SKY:
		%RedSkyBackground.show()
	else:
		# This should never be possible, but some people edited their level
		# file or something, so we just fall back to the default background
		background = Background.SKY
		%SkyBackground.show()
	
	
	if background == Background.RED_SKY\
	and LevelManager.level_id == "the_world_of_play_maker":
		set_music("none")
		%RedOverlay.show()
		checkpoint_pos = null
		LevelManager.level_data.start_pos = Vector2i(170, -59)
		LevelManager.level_data.background = Background.RED_SKY
		show_health = false
		show_coin_count = false
		health = 2
		start_health = 2
		for entity: Node2D in %Entities.get_children():
			if entity.scene_file_path == player_scene.resource_path:
				continue
			
			if entity.position.y > -58*256.0:
				entity.queue_free()
			else:
				if "stop" in entity.get_node("%Sprite"):
					entity.get_node("%Sprite").stop()
		
		for x in 20:
			for y in 10:
				%Blocks.set_cell(Vector2i(172+x, -54+y), -1, Vector2i(0, 0), 0)


func get_progress() -> float:
	# called every frame in the game overlay, and loops over all entities
	# premature optimisation is the root of all evil and this game would not have as many features as it does if I felt the need to optimise things before they became an issue
	if progress_source == ProgressSource.HORIZONTAL:
		# We choose the closest goal vertically to the start pos x
		# (y axis has no influence on the choice)
		var start_x = (start_pos.x + 0.5) * 256.0
		var closest_goal_x := INF
		for entity: Node2D in %Entities.get_children():
			if get_id_from_node(entity) == "goal":
				if abs(entity.position.x - start_x) < abs(closest_goal_x - start_x):
					closest_goal_x = entity.position.x
		if closest_goal_x == INF:
			return 0.0
		var player_x = get_tree().get_first_node_in_group("player").position.x
		return (player_x - start_x) / (closest_goal_x - start_x)
	elif progress_source == ProgressSource.VERTICAL:
		# We choose the closest goal vertically to the start pos y
		# (x axis has no influence on the choice)
		var start_y = (start_pos.y + 0.5) * 256.0	
		var closest_goal_y := INF
		for entity: Node2D in %Entities.get_children():
			if get_id_from_node(entity) == "goal":
				if abs(entity.position.y - start_y) < abs(closest_goal_y - start_y):
					closest_goal_y = entity.position.y
		if closest_goal_y == INF:
			return 0.0
		var player_y = get_tree().get_first_node_in_group("player").position.y
		return (player_y - start_y) / (closest_goal_y - start_y)
	elif progress_source == ProgressSource.COINS:
		if min_coins <= 0:
			return 1.0
		return float(coins) / float(min_coins)
	elif progress_source == ProgressSource.TIME:
		return float(time_spent) / float(max_time)
	return 0.0


func clear_effects() -> void:
	set_inverted(false, false)
	set_pixelation(1, false)
	set_flipped_vertically(false, false)
	set_flipped_horizontally(false, false)
	set_grayscale(false, false)
	set_screen_hidden(false, false)


func remove_effects() -> void:
	%Invert.modulate = Color.TRANSPARENT
	%Pixelate.material.set_shader_parameter("pixelSize", 1)
	%Grayscale.modulate = Color.TRANSPARENT
	%HideScreen.modulate = Color.TRANSPARENT
	clear_effects()


func set_inverted(screen_inverted: bool, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	inverted = screen_inverted


func set_pixelation(pixel_size: int, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	var prev_pixel_size = %Pixelate.material.get_shader_parameter("pixelSize")
	%Pixelate.visible = pixel_size > 1
	%Pixelate.material.set_shader_parameter("pixelSize", pixel_size)

	# Removed the animation for now as it caused issues if triggered twice
	#%Pixelate.visible = prev_pixel_size > 1 or pixel_size > 1
	#if %Pixelate.visible:
		#var set_pixel_size = func(pxl_size):
			#%Pixelate.material.set_shader_parameter("pixelSize", pxl_size)
		#var tween = create_tween()
		#await tween.tween_method(set_pixel_size, %Pixelate.material.get_shader_parameter("pixelSize"), pixel_size, 1).set_trans(Tween.TRANS_QUAD).finished
		#if pixel_size <= 1:
			#%Pixelate.visible = false


func set_flipped_vertically(flipped: bool, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	%FlipVertically.visible = flipped


func set_flipped_horizontally(flipped: bool, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	%FlipHorizontally.visible = flipped


var did_hard_path_effect := false
func set_grayscale(screen_grayscale: bool, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	grayscale = screen_grayscale
	
	if screen_grayscale\
	and background == Background.RED_SKY\
	and LevelManager.level_id == "the_world_of_play_maker"\
	and not did_hard_path_effect:
		did_hard_path_effect = true
		
		OptionsManager.got_first_level_extra_secret = true
		OptionsManager.save_options()
		
		for x in 7:
			for y in 6:
				%Blocks.set_cell(Vector2i(195+x, -68+y), -1, Vector2i(0, 0), 0)
		
		for x in 7:
			var scene: PackedScene = entity_types.get("saw")
			var entity = scene.instantiate()
			entity.position = (Vector2(Vector2i(195+x, -63)) + Vector2(0.5, 0.5)) * 256.0
			entity.fake = true
			%Entities.add_child.call_deferred(entity)
		
		await get_tree().create_timer(2.0, false).timeout
		DisplayServer.mouse_set_mode(DisplayServer.MOUSE_MODE_VISIBLE)
		get_tree().change_scene_to_packed(main_menu_scene)


func set_screen_hidden(hidden: bool, clear_effects: bool = true) -> void:
	if clear_effects:
		clear_effects()
	screen_hidden = hidden


func _on_lose_area_body_entered(body: Node2D) -> void:
	if body.has_method("damage"):
		body.damage(null, 100_000_000)


func save_level() -> void:
	if LevelManager.level_source == LevelManager.LevelSource.LOCAL:
		if game_mode == GameMode.PLAYING:
			LevelManager.save_local_level(LevelManager.level_data)
		else:
			LevelManager.save_local_level(get_level_data())


func _on_auto_save_timer_timeout() -> void:
	save_level()


func _notification(what):
	if what == NOTIFICATION_WM_CLOSE_REQUEST:
		save_level()


func _on_animation_player_animation_finished(anim_name: StringName) -> void:
	if anim_name == "fade_enter":
		%ColorOverlay.hide()
