extends CharacterBody2D

enum State {
	IDLE,
	FALLING,
	RISING
}

@onready var game = get_tree().get_first_node_in_group("game")
const FALL_SPEED = 1500.0
const RISE_SPEED = 300.0
var state := State.IDLE
@onready var start_x := position.x
@onready var start_y := position.y

var fake := false
var invisible := false


func _ready() -> void:
	update_props()


func update_props() -> void:
	%Collision.disabled = fake

# The collision is a bit smaller than expected to prevent ghost collisions with blocks side by side
# However, this does cause issues when trying to walk from them onto a full block

func _process(delta: float) -> void:
	%Sprite.visible = not invisible
	if state == State.IDLE:
		%Sprite.play("idle")
	elif state == State.FALLING:
		%Sprite.play("falling")
	elif state == State.RISING:
		%Sprite.play("rising")


func _physics_process(delta: float) -> void:
	if game.first_load_major_version < 11 or round(rotation_degrees) != 0.0:
		%Collision.shape.size.y = 248
		do_legacy_physics_process(delta)
		return

	# v11 broke at some point, don't know what changed, so just use v12 for those levels
	%Collision.shape.size.y = 256
	do_v12_physics_process(delta)


func do_v12_physics_process(delta: float):
	var rot_deg = round(rotation_degrees)
	if rot_deg == 0.0 or rot_deg == -180.0:
		velocity.x = 0
		position.x = start_x
	else:
		velocity.y = 0
		position.y = start_y
	
	# Check
	if state == State.IDLE and %FallTimer.is_stopped():
		for below in %BelowArea.get_overlapping_bodies():
			if below == self:
				continue
			%FallTimer.start()
			state = State.FALLING
			break
	
	# Apply
	var gravity_direction = Vector2.DOWN.rotated(deg_to_rad(rot_deg))
	if state == State.FALLING:
		velocity = gravity_direction * FALL_SPEED
	elif state == State.RISING:
		velocity = -gravity_direction * RISE_SPEED


	if state == State.RISING:
		for hit in %AboveArea.get_overlapping_bodies():
			if hit == self:
				continue
			if hit is CharacterBody2D:
				if rot_deg == 0.0:
					if hit.velocity.y > velocity.y:
						hit.velocity.y = velocity.y
					if hit.has_node("%CoyoteTimer"):
						hit.get_node("%CoyoteTimer").start()
	elif state == State.FALLING:
		for i in get_slide_collision_count():
			var hit = get_slide_collision(i).get_collider()
			if hit is CharacterBody2D and hit in %BelowArea.get_overlapping_bodies():
				if not hit.has_method("is_falling_block") or not hit.is_falling_block():
					hit.velocity.y = velocity.y

	var pre_pos = position
	move_and_slide()
	
	if state == State.FALLING and pre_pos.distance_to(position) < 0.2:
		for i in get_slide_collision_count():
			var hit = get_slide_collision(i).get_collider()
			if hit.has_method("damage"):
				hit.damage(self)
			state = State.RISING
	
	
	if state == State.IDLE or state == State.RISING:
		var margin := 16
		if (rot_deg == 0.0 and position.y <= start_y + margin) or (rot_deg == -180.0 and position.y >= start_y - margin):
			state = State.IDLE
			velocity.y = 0
			position.y = start_y
		elif (rot_deg == 90.0 and position.x >= start_x - margin) or (rot_deg == -90.0 and position.x <= start_x + margin):
			state = State.IDLE
			velocity.x = 0
			position.x = start_x


func do_legacy_physics_process(delta: float):
	var rot_deg = round(rotation_degrees)
	if rot_deg == 0.0 or rot_deg == -180.0:
		velocity.x = 0
	else:
		velocity.y = 0
	
	# Check
	if state == State.IDLE and %BelowArea.get_overlapping_bodies().size() > 0 and %FallTimer.is_stopped():
		%FallTimer.start()
		state = State.FALLING
	
	# Apply
	if state == State.FALLING:
		if rot_deg == 0.0:
			velocity.y = FALL_SPEED
		elif rot_deg == -180.0:
			velocity.y = -FALL_SPEED
		elif rot_deg == 90.0:
			velocity.x = -FALL_SPEED
		elif rot_deg == -90.0:
			velocity.x = FALL_SPEED
	elif state == State.RISING:
		if rot_deg == 0.0:
			velocity.y = -RISE_SPEED
		elif rot_deg == -180.0:
			velocity.y = RISE_SPEED
		elif rot_deg == 90.0:
			velocity.x = RISE_SPEED
		elif rot_deg == -90.0:
			velocity.x = -RISE_SPEED
	
	if state == State.FALLING or state == State.RISING:
		for i in get_slide_collision_count():
			var hit = get_slide_collision(i).get_collider()
			if hit is CharacterBody2D:
				if rot_deg == 0.0 or rot_deg == -180:
					hit.velocity.y = velocity.y
					#hit.position.y += velocity.y * delta
				else:
					hit.velocity.x = velocity.x
					#hit.position.x += velocity.x * delta

	var pre_pos = position
	move_and_slide()
	
	if state == State.FALLING and pre_pos.distance_to(position) < 0.5:
		for i in get_slide_collision_count():
			var hit = get_slide_collision(i).get_collider()
			if hit.has_method("damage"):
				hit.damage(self)
			state = State.RISING
	
	if (rot_deg == 0.0 and position.y <= start_y) or (rot_deg == -180.0 and position.y >= start_y):
		state = State.IDLE
		velocity.y = 0
		position.y = start_y
	elif (rot_deg == 90.0 and position.x >= start_x) or (rot_deg == -90.0 and position.x <= start_x):
		state = State.IDLE
		velocity.x = 0
		position.x = start_x
	

func is_falling_block() -> bool:
	return true
