extends Control

var online_levels_scene: PackedScene = load("res://scenes/online_levels_menu.tscn")
var uploaded_levels_scene: PackedScene = load("res://scenes/uploaded_levels_menu.tscn")
var comment_item_scene: PackedScene = load("res://scenes/comment_item.tscn")
static var offset := 0
var needs_refresh := false
var needs_exit := false


func _ready() -> void:
	var level_stats := LevelManager.get_level_stats(LevelManager.level_id)
	if level_stats.get("completed", false):
		%StatsNotCompletedLabel.hide()
		%StatsBestTimeLabel.text = "Best Time: %.2fs" % level_stats.get("best_time", INF)
		%StatsBestTimeLabel.show()
		%StatsLeastJumpsLabel.text = "Least Jumps: %d" % level_stats.get("least_jumps", INF)
		%StatsLeastJumpsLabel.show()
		%StatsMostCoinsLabel.text = "Most Coins: %d" % level_stats.get("most_coins", 0)
		%StatsMostCoinsLabel.show()
		if level_stats.get("beat_secret_ending", false):
			%StatsSecretEndingLabel.show()
	
	%BackButton.grab_focus.call_deferred()
	%MakeCommentButton.disabled = OptionsManager.login_token == null
	do_req()


func do_req() -> void:
	%Status.text = "Loading..."
	for child in %Comments.get_children():
		child.queue_free()
	%PrevPageButton.disabled = true
	%NextPageButton.disabled = true
	var headers = []
	if OptionsManager.login_token:
		headers.push_back("Authorization: Bearer %s" % OptionsManager.login_token)
	%CommentListReq.request("%s/comments?limit=%d&offset=%d&level_id=%s" % [OptionsManager.server_base_url, OptionsManager.comments_per_page, offset, LevelManager.level_id], headers)


func _process(delta: float) -> void:
	if Input.is_action_just_pressed("back"):
		if %MakeCommentMenu.visible:
			if not %CancelCommentButton.disabled:
				_on_cancel_comment_button_pressed()
		elif %ConfirmDelete.visible:
			if not %CancelDeleteButton.disabled:
				_on_cancel_delete_button_pressed()
		else:
			_on_back_button_pressed()


func _on_back_button_pressed() -> void:
	if Globals.return_to_uploaded_levels:
		get_tree().change_scene_to_packed(uploaded_levels_scene)
	else:
		get_tree().change_scene_to_packed(online_levels_scene)


func _on_comment_list_req_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	var err = Globals.get_req_error(result, response_code, body)
	if err:
		%Status.text = err
		return
	%Status.text = "Success"
		
	var json = JSON.parse_string(body.get_string_from_utf8())
	%PrevPageButton.disabled = offset == 0
	%NextPageButton.disabled = json.get("last_page")
	
	for comment_meta in json.get("comments"):
		var comment_item = comment_item_scene.instantiate()
		comment_item.author = comment_meta.author_username
		comment_item.author_suffix = comment_meta.author_suffix
		comment_item.message = comment_meta.message
		comment_item.upload_time = comment_meta.created_at
		comment_item.best_time = comment_meta.best_time
		comment_item.beat_secret_ending = comment_meta.beat_secret_ending
		%Comments.add_child(comment_item)
	
	%DeleteLevelButton.visible = json.get("can_delete_level", false)


func _on_make_comment_button_pressed() -> void:
	%MakeCommentMenu.show()
	%CommentText.grab_focus.call_deferred()


func _on_cancel_comment_button_pressed() -> void:
	%MakeCommentMenu.hide()
	if needs_refresh:
		get_tree().reload_current_scene()
	else:
		# Doesn't seem to work...
		%PostCommentButton.grab_focus.call_deferred()


func _on_post_comment_button_pressed() -> void:
	%CommentText.editable = false
	%CancelCommentButton.disabled = true
	%PostCommentButton.disabled = true
	%CommentStatus.text = "Loading..."
	
	var level_stats := LevelManager.get_level_stats(LevelManager.level_id)
	
	# We can't just change the default to null becuase other places set the default too
	var best_time = level_stats.get("best_time", INF)
	if best_time == INF:
		best_time = null
	
	var least_jumps = level_stats.get("least_jumps", INF)
	if least_jumps == INF:
		least_jumps = null
	
	var most_coins = level_stats.get("most_coins", -1)
	if most_coins == -1:
		most_coins = null
	if not level_stats.get("completed", false):
		most_coins = null
	
	var beat_secret_ending = level_stats.get("beat_secret_ending", false)
	
	var json = JSON.stringify({
		"level_id": LevelManager.level_id,
		"message": %CommentText.text,
		"best_time": best_time,
		"least_jumps": least_jumps,
		"most_coins": most_coins,
		"beat_secret_ending": beat_secret_ending
	})
	
	var headers = ["Content-Type: application/json", "Authorization: Bearer %s" % OptionsManager.login_token]
	%PostCommentReq.request("%s/comments" % OptionsManager.server_base_url, headers, HTTPClient.METHOD_POST, json)


func _on_post_comment_req_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	var err = Globals.get_req_error(result, response_code, body)
	if err:
		%CommentStatus.text = err
		%CommentText.editable = true
		%CancelCommentButton.disabled = false
		%PostCommentButton.disabled = false
		return
	
	needs_refresh = true
	%CommentStatus.text = "Comment has been posted."
	%PostCommentButton.hide()
	%CancelCommentButton.text = "Done"
	%CancelCommentButton.disabled = false
	%CancelCommentButton.grab_focus.call_deferred()


func _on_prev_page_button_pressed() -> void:
	offset -= OptionsManager.comments_per_page
	offset = max(offset, 0)
	do_req()


func _on_next_page_button_pressed() -> void:
	offset += OptionsManager.comments_per_page
	do_req()


func _on_stats_button_pressed() -> void:
	%StatsMenu.show()
	%CloseStatsButton.grab_focus.call_deferred()


func _on_close_stats_button_pressed() -> void:
	%StatsMenu.hide()
	%StatsButton.grab_focus.call_deferred()


func _on_delete_level_button_pressed() -> void:
	%ConfirmDelete.show()
	%CancelDeleteButton.grab_focus.call_deferred()


func _on_delete_level_req_request_completed(result: int, response_code: int, headers: PackedStringArray, body: PackedByteArray) -> void:
	var err = Globals.get_req_error(result, response_code, body)
	if err:
		%DeleteStatus.text = err
		%CancelDeleteButton.disabled = false
		%ConfirmDeleteButton.disabled = false
		return
	
	needs_exit = true
	%DeleteStatus.text = "Level has been deleted."
	%ConfirmDeleteButton.hide()
	%CancelDeleteButton.text = "Done"
	%CancelDeleteButton.disabled = false


func _on_cancel_delete_button_pressed() -> void:
	%ConfirmDelete.hide()
	if needs_exit:
		_on_back_button_pressed()


func _on_confirm_delete_button_pressed() -> void:
	%CancelDeleteButton.disabled = true
	%ConfirmDeleteButton.disabled = true
	%DeleteStatus.text = "Loading..."
	var headers = ["Authorization: Bearer %s" % OptionsManager.login_token]
	%DeleteLevelReq.request("%s/levels/%s" % [OptionsManager.server_base_url, LevelManager.level_id], headers, HTTPClient.METHOD_DELETE)
