package nl.sonck.streamplayer

import androidx.annotation.StringRes
import androidx.compose.animation.AnimatedContentTransitionScope
import androidx.compose.animation.core.tween
import androidx.compose.foundation.layout.padding
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.navigation.NavHostController
import androidx.navigation.NavType
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.currentBackStackEntryAsState
import androidx.navigation.compose.rememberNavController
import androidx.navigation.navArgument
import nl.sonck.streamplayer.radioknop.radios
import nl.sonck.streamplayer.settings.rememberStringSettingState
import nl.sonck.streamplayer.ui.MainScreen
import nl.sonck.streamplayer.ui.RadioknopGenreScreen
import nl.sonck.streamplayer.ui.RadioknopRadioScreen

enum class PlayerScreen(@StringRes val title: Int) {
    // Main page
    Start(title = R.string.app_name),

    // Radioknop functionality
    Radioknop(title = R.string.radioknop_title),
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PlayerAppBar(
    currentScreen: PlayerScreen,
    canNavigateBack: Boolean,
    navigateUp: () -> Unit,
    modifier: Modifier = Modifier
) {
    TopAppBar(
        title = { Text(stringResource(currentScreen.title)) },
        colors = TopAppBarDefaults.mediumTopAppBarColors(
            containerColor = MaterialTheme.colorScheme.primaryContainer
        ),
        modifier = modifier,
        navigationIcon = {
            if (canNavigateBack) {
                IconButton(onClick = navigateUp) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                        contentDescription = stringResource(id = R.string.back_button)
                    )
                }
            }
        }
    )
}

@Composable
fun PlayerApp(
    navController: NavHostController = rememberNavController(),
) {
    val backStackEntry by navController.currentBackStackEntryAsState()
    val currentScreen =
        PlayerScreen.valueOf(
            backStackEntry?.destination?.route?.split("/")?.first() ?: PlayerScreen.Start.name
        )

    val radios = radios()
    var uri by rememberStringSettingState("stream_uri")
    val currentRadio = radios.urls[uri]
    var radioknopGenre by remember { mutableStateOf("") }

    Scaffold(
        topBar = {
            PlayerAppBar(
                currentScreen = currentScreen,
                canNavigateBack = navController.previousBackStackEntry != null,
                navigateUp = { navController.navigateUp() },
            )
        }
    ) { innerPadding ->
        NavHost(
            navController = navController,
            startDestination = PlayerScreen.Start.name,
            modifier = Modifier.padding(innerPadding)
        ) {
            composable(route = PlayerScreen.Start.name) {
                MainScreen(
                    onRadioKnop = {
                        navController.navigate(PlayerScreen.Radioknop.name)
                    },
                    uri = uri ?: "",
                    onUriChange = {
                        uri = it
                    },
                    currentRadio = currentRadio
                )
            }
            composable(
                route = PlayerScreen.Radioknop.name,
                enterTransition = {
                    when {
                        initialState.destination.route == PlayerScreen.Start.name ->
                            slideIntoContainer(
                                AnimatedContentTransitionScope.SlideDirection.Up,
                                animationSpec = tween(700)
                            )

                        initialState.destination.route?.startsWith("${PlayerScreen.Radioknop.name}/") ?: false ->
                            slideIntoContainer(
                                AnimatedContentTransitionScope.SlideDirection.Right,
                                animationSpec = tween(700)
                            )

                        else -> null
                    }
                },
                exitTransition = {
                    when {
                        targetState.destination.route == PlayerScreen.Start.name ->
                            slideOutOfContainer(
                                AnimatedContentTransitionScope.SlideDirection.Down,
                                animationSpec = tween(700)
                            )

                        targetState.destination.route?.startsWith("${PlayerScreen.Radioknop.name}/") ?: false ->
                            slideOutOfContainer(
                                AnimatedContentTransitionScope.SlideDirection.Left,
                                animationSpec = tween(700)
                            )

                        else -> null
                    }
                }
            ) {
                RadioknopGenreScreen(onGenreSelect = {
                    radioknopGenre = it
                    navController.navigate("${PlayerScreen.Radioknop.name}/${it}")
                }, radios = radios.radios)
            }
            composable(
                route = "${PlayerScreen.Radioknop.name}/{genre}",
                enterTransition = {
                    slideIntoContainer(
                        AnimatedContentTransitionScope.SlideDirection.Left,
                        animationSpec = tween(700)
                    )
                },
                exitTransition = {
                    when (targetState.destination.route) {
                        PlayerScreen.Start.name ->
                            slideOutOfContainer(
                                AnimatedContentTransitionScope.SlideDirection.Down,
                                animationSpec = tween(700)
                            )

                        PlayerScreen.Radioknop.name ->
                            slideOutOfContainer(
                                AnimatedContentTransitionScope.SlideDirection.Right,
                                animationSpec = tween(700)
                            )

                        else -> null
                    }
                },
                arguments = listOf(navArgument("genre") {
                    type = NavType.StringType
                })
            ) {
                RadioknopRadioScreen(
                    genre = it.arguments?.getString("genre") ?: "",
                    onRadioSelect = {
                        uri = it
                        navController.popBackStack(PlayerScreen.Start.name, inclusive = false)
                    },
                    radios = radios.radios
                )
            }
        }
    }
}