package nl.sonck.streamplayer.ui

import androidx.compose.foundation.border
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Pause
import androidx.compose.material.icons.filled.PlayArrow
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.FilledIconButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.shadow
import androidx.compose.ui.semantics.contentDescription
import androidx.compose.ui.semantics.semantics
import androidx.compose.ui.unit.dp
import androidx.media3.common.Player

private class IsPlayingListener(
    val onIsPlaying: (isPlaying: Boolean) -> Unit = {},
    val onIsLoading: (isLoading: Boolean) -> Unit = {},
    val onPlaybackState: (playbackState: Int) -> Unit = {},
) : Player.Listener {
    override fun onIsPlayingChanged(isPlaying: Boolean) {
        onIsPlaying(isPlaying)
    }

    override fun onIsLoadingChanged(isLoading: Boolean) {
        onIsLoading(isLoading)
    }

    override fun onPlaybackStateChanged(playbackState: Int) {
        onPlaybackState(playbackState)
    }
}

@Composable
fun MediaButton(
    controller: Player?,
    modifier: Modifier = Modifier,
    interactionSource: MutableInteractionSource = remember { MutableInteractionSource() },
) {
    var playing by remember { mutableStateOf(false) }
    var state by remember { mutableIntStateOf(Player.STATE_ENDED) }

    val running = playing || state == Player.STATE_BUFFERING

    LaunchedEffect(controller) {
        controller?.let {
            it.addListener(
                IsPlayingListener(
                    onIsPlaying = { isPlaying ->
                        playing = isPlaying
                    },
                    onPlaybackState = { playbackState ->
                        state = playbackState
                        when(state) {
                            Player.STATE_IDLE -> playing = false
                        }
                    }
                )
            )
            playing = it.isPlaying
        }
    }

    FilledIconButton(
        modifier = modifier
            .semantics { contentDescription = if(playing) "Stop playback" else "Start playback" }
            .size(128.dp)
            .let {
                if (state != Player.STATE_BUFFERING) {
                    it.border(4.dp, MaterialTheme.colorScheme.onPrimary, shape = CircleShape)
                } else {
                    it
                }
            }
            .shadow(8.dp, shape = CircleShape),
        onClick = {
            controller?.let {
                if (it.isPlaying) {
                    it.stop()
                } else {
                    it.play()
                }
            }
        },
        interactionSource = interactionSource,
    ) {
        Box(contentAlignment = Alignment.Center) {
            Icon(
                if (running) Icons.Filled.Pause else Icons.Filled.PlayArrow,
                contentDescription = null,
                Modifier.size(64.dp),
                tint = MaterialTheme.colorScheme.onPrimary
            )

            if (state == Player.STATE_BUFFERING) {
                CircularProgressIndicator(
                    modifier = Modifier.size(128.dp),
                    color = MaterialTheme.colorScheme.onPrimary
                )
            }
        }
    }
}

