package nl.sonck.streamplayer.ui

import android.content.ComponentName
import android.widget.Toast
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.MarqueeAnimationMode
import androidx.compose.foundation.basicMarquee
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Button
import androidx.compose.material3.Icon
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.material3.VerticalDivider
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import androidx.media3.common.MediaItem
import androidx.media3.session.MediaController
import androidx.media3.session.SessionToken
import kotlinx.coroutines.guava.await
import nl.sonck.streamplayer.PlaybackService
import nl.sonck.streamplayer.R
import nl.sonck.streamplayer.radioknop.Radio
import nl.sonck.streamplayer.ui.components.LabeledText
import nl.sonck.streamplayer.util.listenToError
import nl.sonck.streamplayer.util.listenToMetadata

@OptIn(ExperimentalFoundationApi::class)
@Composable
fun MainScreen(
    modifier: Modifier = Modifier,
    uri: String,
    onUriChange: (String) -> Unit = {},
    onRadioKnop: () -> Unit = {},
    currentRadio: Radio?
) {
    val context = LocalContext.current
    var controller by remember { mutableStateOf<MediaController?>(null) }
    var title by remember { mutableStateOf("…") }

    LaunchedEffect(Unit) {
        val sessionToken =
            SessionToken(context, ComponentName(context, PlaybackService::class.java))
        controller = MediaController.Builder(context, sessionToken).buildAsync().await().also {
            uri.let { uri ->
                it.setMediaItems(
                    listOf(
                        MediaItem.fromUri(uri)
                    )
                )
            }
            it.listenToMetadata { meta ->
                meta.title?.let { title = it.toString() }
            }
            it.listenToError { error ->
                error?.errorCodeName?.let { errorCodeName ->
                    Toast.makeText(
                        context,
                        context.getString(R.string.error, errorCodeName), Toast.LENGTH_LONG
                    ).show()
                }
            }
        }
    }

    LaunchedEffect(uri) {
        uri.let { uri ->
            controller?.setMediaItems(
                listOf(
                    MediaItem.fromUri(uri)
                )
            )
        }
    }

    DisposableEffect(Unit) {
        onDispose {
            controller?.release()
            controller = null
        }
    }

    Column(modifier = modifier, horizontalAlignment = Alignment.CenterHorizontally) {
        TextField(
            modifier = Modifier.fillMaxWidth(),
            value = uri,
            onValueChange = onUriChange,
            label = { Text(text = "URL") })

        Surface(
            modifier = Modifier.padding(12.dp),
        ) {
            LabeledText(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(12.dp),
                label = stringResource(R.string.now_playing),
                text = title,
            )
        }

        Box(
            modifier = modifier.weight(1f),
            contentAlignment = Alignment.Center,
        ) {
            MediaButton(controller = controller)
        }

        Button(modifier = Modifier.padding(8.dp), onClick = onRadioKnop) {
            Row(
                modifier = Modifier
                    .height(IntrinsicSize.Min)
                    .padding(0.dp, 8.dp), verticalAlignment = Alignment.CenterVertically
            ) {
                Icon(
                    painter = painterResource(id = R.drawable.ic_radioknop),
                    contentDescription = "Radioknop logo",
                )
                Text(
                    text =
                        currentRadio.let {
                            if (it != null) {
                                stringResource(id = R.string.playing_from_radioknop)
                            } else {
                                stringResource(id = R.string.play_from_radioknop)
                            }
                        })
                currentRadio.let {
                    if (it != null) {
                        VerticalDivider(
                            thickness = Dp.Hairline,
                            modifier = Modifier.padding(8.dp, 0.dp)
                        )
                        Text(
                            modifier = Modifier
                                .basicMarquee(
                                    Int.MAX_VALUE,
                                    animationMode = MarqueeAnimationMode.Immediately,
                                )
                                .weight(1f), text = it.name
                        )
                    }
                }
            }
        }
    }
}

@Preview
@Composable
fun MainScreenPreview() {
    var uri by remember { mutableStateOf("https://example.com/stream.mp3") }

    MainScreen(uri = uri, onUriChange = { uri = it }, currentRadio = null)
}

@Preview
@Composable
fun MainScreenRadioknopPlaying() {
    var uri by remember { mutableStateOf("https://example.com/stream.mp3") }

    MainScreen(
        uri = uri,
        onUriChange = { uri = it },
        currentRadio = Radio("Test", "The best test", uri)
    )
}
