/* SPDX-License-Identifier: Apache-2.0
 *
 * Copyright © 2017-2021 Jason A. Donenfeld <Jason@zx2c4.com>. All Rights Reserved.
 */

#include <jni.h>
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>

static JavaVM *globalVM;
static jclass globalClazz;
struct go_string { const char *str; long n; };
typedef void (*ProxySetupCB)(int fd);
extern int wgTurnOn(struct go_string ifname, int tun_fd, struct go_string settings, ProxySetupCB);
extern void wgTurnOff(int handle);
extern int wgGetSocketV4(int handle);
extern int wgGetSocketV6(int handle);
extern char *wgGetConfig(int handle);
extern char *wgVersion();

// Adapted from: https://codeberg.org/eduVPN/android/src/commit/ef97f5a917dcb037687bfccbd05ae6b401dd2fcb/common/src/main/cpp/jni.cpp
bool GetJniEnv(JavaVM *vm, JNIEnv **env)
{
	bool did_attach_thread = false;
	*env = NULL;
	// Check if the current thread is attached to the VM
	jint get_env_result = (*vm)->GetEnv(vm, (void **)env, JNI_VERSION_1_6);
	if (get_env_result == JNI_EDETACHED) {
		if ((*vm)->AttachCurrentThread(vm, env, NULL) == JNI_OK) {
			did_attach_thread = true;
		}
		else {
			// Failed to attach thread
		}
	}
	else if (get_env_result == JNI_EVERSION) {
		// Unsupported JNI version
	}
	return did_attach_thread;
}

void proxySetup(int fd)
{
	if (!globalVM) {
		return;
	}
	JNIEnv *env;
	bool didAttach = GetJniEnv(globalVM, &env);
	jmethodID methodID = (*env)->GetStaticMethodID(env, globalClazz, "onProxySetup", "(I)V");
	(*env)->CallStaticVoidMethod(env, globalClazz, methodID, fd);

	if (didAttach) {
		(*globalVM)->DetachCurrentThread(globalVM);
	}
}

JNIEXPORT jint JNICALL Java_com_wireguard_android_backend_GoBackend_wgTurnOn(JNIEnv *env, jclass c, jstring ifname,
									     jint tun_fd, jstring settings)
{
	(*env)->GetJavaVM(env, &globalVM);
	jclass backendCls = (*env)->FindClass(env, "com/wireguard/android/backend/GoBackend");
	globalClazz = (jclass)(*env)->NewGlobalRef(env, backendCls);
	const char *ifname_str = (*env)->GetStringUTFChars(env, ifname, 0);
	size_t ifname_len = (*env)->GetStringUTFLength(env, ifname);
	const char *settings_str = (*env)->GetStringUTFChars(env, settings, 0);
	size_t settings_len = (*env)->GetStringUTFLength(env, settings);
	int ret = wgTurnOn((struct go_string){
		.str = ifname_str,
		.n = ifname_len
	}, tun_fd, (struct go_string){
		.str = settings_str,
		.n = settings_len
	}, proxySetup);
	(*env)->ReleaseStringUTFChars(env, ifname, ifname_str);
	(*env)->ReleaseStringUTFChars(env, settings, settings_str);
	return ret;
}

JNIEXPORT void JNICALL Java_com_wireguard_android_backend_GoBackend_wgTurnOff(JNIEnv *env, jclass c, jint handle)
{
	wgTurnOff(handle);
}

JNIEXPORT jint JNICALL Java_com_wireguard_android_backend_GoBackend_wgGetSocketV4(JNIEnv *env, jclass c, jint handle)
{
	return wgGetSocketV4(handle);
}

JNIEXPORT jint JNICALL Java_com_wireguard_android_backend_GoBackend_wgGetSocketV6(JNIEnv *env, jclass c, jint handle)
{
	return wgGetSocketV6(handle);
}

JNIEXPORT jstring JNICALL Java_com_wireguard_android_backend_GoBackend_wgGetConfig(JNIEnv *env, jclass c, jint handle)
{
	jstring ret;
	char *config = wgGetConfig(handle);
	if (!config)
		return NULL;
	ret = (*env)->NewStringUTF(env, config);
	free(config);
	return ret;
}

JNIEXPORT jstring JNICALL Java_com_wireguard_android_backend_GoBackend_wgVersion(JNIEnv *env, jclass c)
{
	jstring ret;
	char *version = wgVersion();
	if (!version)
		return NULL;
	ret = (*env)->NewStringUTF(env, version);
	free(version);
	return ret;
}
