package nl.eduvpn.app.viewmodel

import android.content.Context
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import nl.eduvpn.app.service.BackendService
import nl.eduvpn.app.service.HistoryService
import nl.eduvpn.app.service.VPNConnectionService
import nl.eduvpn.app.service.VPNService
import nl.eduvpn.app.utils.Log
import java.util.Optional
import javax.inject.Inject

class SettingsViewModel @Inject constructor(
    private val historyService: HistoryService,
    private val backendService: BackendService,
    private val optionalVPNService: Optional<VPNService>,
    private val vpnConnectionService: VPNConnectionService,
): ViewModel() {

    companion object {
        private val TAG = SettingsViewModel::class.simpleName
    }

    val apiLogFile get() = backendService.getLogFile()

    val hasAddedServers get() = historyService.addedServers?.hasServers() == true

    /**
     * The user might have opened settings from the status fragment.
     * In this case resetting the organization data should disconnect the VPN to avoid
     * an incomplete cleanup.
     */
    private fun disconnect(context: Context?) {
        val vpnService = optionalVPNService.orElse(null) ?: return // Maybe the VPN service is not initialized yet, in that case, no need to disconnect
        if (context != null && vpnService.getStatus() != VPNService.VPNStatus.DISCONNECTED) {
            Log.i(TAG, "Disconnecting VPN before removing organization data")
            backendService.notifyDisconnecting()
            vpnConnectionService.disconnect(context, vpnService)
            backendService.notifyDisconnected()
        }
    }

    fun removeOrganizationData(context: Context?) {
        disconnect(context)
        viewModelScope.launch(Dispatchers.IO) {
            historyService.removeOrganizationData()
        }
    }
}