/*
 *  This file is part of eduVPN.
 *
 *     eduVPN is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     eduVPN is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with eduVPN.  If not, see <http://www.gnu.org/licenses/>.
 */
package nl.eduvpn.app

import android.app.NotificationChannel
import android.app.NotificationManager
import android.content.ComponentName
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.service.quicksettings.TileService
import android.view.MenuItem
import android.view.View
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.activity.viewModels
import androidx.appcompat.app.AlertDialog
import androidx.browser.customtabs.CustomTabsIntent
import androidx.fragment.app.Fragment
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nl.eduvpn.app.base.BaseActivity
import nl.eduvpn.app.databinding.ActivityMainBinding
import nl.eduvpn.app.entity.AddedServer
import nl.eduvpn.app.entity.exception.CommonException
import nl.eduvpn.app.fragment.AddServerFragment
import nl.eduvpn.app.fragment.ConnectionStatusFragment
import nl.eduvpn.app.fragment.OrganizationSelectionFragment
import nl.eduvpn.app.fragment.ProfileSelectionFragment
import nl.eduvpn.app.fragment.ServerSelectionFragment
import nl.eduvpn.app.service.EduVPNOpenVPNService
import nl.eduvpn.app.service.VPNService
import nl.eduvpn.app.service.VpnTileService
import nl.eduvpn.app.service.WireGuardService
import nl.eduvpn.app.utils.ErrorDialog.show
import nl.eduvpn.app.utils.countryCodeToCountryNameAndFlag
import nl.eduvpn.app.viewmodel.MainViewModel
import nl.eduvpn.app.viewmodel.ViewModelFactory
import java.util.*
import javax.inject.Inject


class MainActivity : BaseActivity<ActivityMainBinding>() {

    companion object {
        private const val REQUEST_CODE_SETTINGS = 1001
        private const val KEY_BACK_NAVIGATION_ENABLED = "back_navigation_enabled"
    }

    @Inject
    lateinit var vpnService: Optional<VPNService>

    @Inject
    lateinit var eduVPNOpenVPNService: EduVPNOpenVPNService

    @Inject
    lateinit var wireGuardService: WireGuardService

    @Inject
    lateinit var viewModelFactory: ViewModelFactory

    protected val viewModel by viewModels<MainViewModel> { viewModelFactory }

    private var _backNavigationEnabled = false
    private var _parseIntentOnStart = true

    override val layout: Int = R.layout.activity_main

    private val notificationPermissionLauncher =
        registerForActivityResult(ActivityResultContracts.RequestPermission()) { isGranted ->
            if (!isGranted) {
                if (Build.VERSION.SDK_INT >= 33) {
                    if (shouldShowRequestPermissionRationale(android.Manifest.permission.POST_NOTIFICATIONS)) {
                        showNotificationPermissionRationale()
                    }
                }
            }
        }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        EduVPNApplication.get(this).component().inject(this)
        setSupportActionBar(binding.toolbar.toolbar)
        viewModel.mainParentAction.observe(this) { parentAction ->
            when (parentAction) {
                is MainViewModel.MainParentAction.OpenLink -> {
                    openLink(parentAction.oAuthUrl)
                }

                is MainViewModel.MainParentAction.SelectCountry -> {
                    selectCountry(parentAction.cookie)
                }

                is MainViewModel.MainParentAction.SelectProfiles -> {
                    val currentFragment = supportFragmentManager.findFragmentById(R.id.content_frame)
                    if (currentFragment is ConnectionStatusFragment) {
                        if (currentFragment.canRenewSessionWithProfileList(parentAction.profileList)) {
                            return@observe
                        }
                    }
                    openFragment(
                        ProfileSelectionFragment.newInstance(parentAction.profileList),
                        false
                    )
                }

                is MainViewModel.MainParentAction.ConnectWithConfig -> {
                    viewModel.parseConfigAndStartConnection(this, parentAction.config, parentAction.preferTcp)
                    // Restart the tile service to make sure it uses the correct VPN service type
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        TileService.requestListeningState(this, ComponentName(this, VpnTileService::class.java))
                    }
                    openFragment(ConnectionStatusFragment(), false)
                }

                is MainViewModel.MainParentAction.ShowCountriesDialog -> {
                    showCountriesDialog(parentAction.serverWithCountries, parentAction.cookie)
                }

                is MainViewModel.MainParentAction.ShowError -> {
                    show(this, parentAction.throwable)
                }
            }
        }
        eduVPNOpenVPNService.onCreate(this)
        if (savedInstanceState == null) {
            // If there's an ongoing VPN connection, open the status screen.
            if (vpnService.isPresent && vpnService.get()
                    .getStatus() != VPNService.VPNStatus.DISCONNECTED
            ) {
                openFragment(ConnectionStatusFragment(), false)
            } else if (viewModel.hasServers()) {
                openFragment(ServerSelectionFragment.newInstance(false), false)
            } else if (BuildConfig.API_DISCOVERY_ENABLED) {
                openFragment(OrganizationSelectionFragment(), false)
            } else {
                openFragment(AddServerFragment(), false)
            }
        } else if (savedInstanceState.containsKey(KEY_BACK_NAVIGATION_ENABLED)) {
            _backNavigationEnabled = savedInstanceState.getBoolean(KEY_BACK_NAVIGATION_ENABLED)
        }
        _parseIntentOnStart = true
        binding.toolbar.settingsButton.setOnClickListener { _: View? -> onSettingsButtonClicked() }
        binding.toolbar.helpButton.setOnClickListener {
            startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(BuildConfig.HELP_URI)))
        }
        createCertExpiryNotificationChannel()
        createVPNConnectionNotificationChannel()
        requestPushPermissionIfNeeded()
    }

    private fun requestPushPermissionIfNeeded() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            notificationPermissionLauncher.launch(android.Manifest.permission.POST_NOTIFICATIONS)
        }
    }

    private fun showCountriesDialog(
        instancesWithNames: List<Pair<AddedServer, String>>,
        cookie: Int?
    ) {
        AlertDialog.Builder(this)
            .setItems(instancesWithNames.map { it.second.countryCodeToCountryNameAndFlag() }.toTypedArray()) { _, which ->
                val selectedInstance = instancesWithNames[which]
                viewModel.onCountrySelected(cookie, selectedInstance.first.identifier, selectedInstance.second)
            }.setOnCancelListener {
                val selectedInstance = instancesWithNames[0]
                viewModel.onCountrySelected(cookie, selectedInstance.first.identifier, null)
            }.show()
    }

    fun selectCountry(cookie: Int? = null) {
        viewModel.getCountryList(cookie)
    }

    private fun openLink(oAuthUrl: String) {
        if (isFinishing) {
            return
        }
        val oAuthUri: Uri
        try {
            oAuthUri = Uri.parse(oAuthUrl)
        } catch (ex: Exception) {
            show(this, ex)
            return
        }
        val intent = CustomTabsIntent.Builder().build()
        intent.launchUrl(this@MainActivity, oAuthUri)
    }

    override fun onResume() {
        super.onResume()
        viewModel.onResume()
    }

    override fun onStart() {
        super.onStart()
        if (_parseIntentOnStart) {
            // The app might have been reopened from a URL.
            _parseIntentOnStart = false
            onNewIntent(intent)
        }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        outState.putBoolean(KEY_BACK_NAVIGATION_ENABLED, _backNavigationEnabled)
    }

    override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        this.lifecycleScope.launch(Dispatchers.IO) {
            try {
                if (viewModel.handleRedirection(intent.data)) {
                    // Remove it so we don't parse it again.
                    intent.data = null
                    val currentFragment =
                        supportFragmentManager.findFragmentById(R.id.content_frame)
                    withContext(Dispatchers.Main) {
                        when (currentFragment) {
                            is ServerSelectionFragment -> currentFragment.connectToSelectedInstance()
                            is OrganizationSelectionFragment -> {
                                Toast.makeText(
                                    this@MainActivity,
                                    R.string.provider_added_new_configs_available,
                                    Toast.LENGTH_LONG
                                ).show()
                            }

                            is ConnectionStatusFragment -> currentFragment.reconnectToInstance()
                        }
                    }
                    if (currentFragment !is ConnectionStatusFragment
                        && currentFragment !is ServerSelectionFragment
                    ) {
                        openFragment(ServerSelectionFragment.newInstance(true), false)
                    }
                }
            } catch (ex: CommonException) {
                withContext(Dispatchers.Main) {
                    show(
                        this@MainActivity, R.string.authorization_error_title, getString(
                            R.string.authorization_error_message,
                            ex.javaClass.simpleName,
                            0,
                            ex.message
                        )
                    )
                }
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        eduVPNOpenVPNService.onDestroy(this)
    }

    fun openFragment(fragment: Fragment?, openOnTop: Boolean) {
        if (openOnTop) {
            supportFragmentManager.beginTransaction()
                .setCustomAnimations(R.anim.fade_in, R.anim.fade_out)
                .addToBackStack(null)
                .add(R.id.content_frame, fragment!!)
                .commitAllowingStateLoss()
        } else {
            supportFragmentManager.beginTransaction()
                .setCustomAnimations(R.anim.fade_in, R.anim.fade_out)
                .replace(R.id.content_frame, fragment!!)
                .commitAllowingStateLoss()
            // Clean the back stack
            for (i in 0 until supportFragmentManager.backStackEntryCount) {
                if (!supportFragmentManager.isStateSaved && !isFinishing) {
                    supportFragmentManager.popBackStack()
                }
            }
        }
    }

    private fun onSettingsButtonClicked() {
        val intent = Intent(this, SettingsActivity::class.java)
        @Suppress("DEPRECATION") //todo
        startActivityForResult(intent, REQUEST_CODE_SETTINGS)
    }

    fun setBackNavigationEnabled(enabled: Boolean) {
        _backNavigationEnabled = enabled
        val actionBar = supportActionBar
        if (actionBar != null) {
            actionBar.setDisplayHomeAsUpEnabled(enabled)
            actionBar.setHomeButtonEnabled(enabled)
        }
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        if (item.itemId == android.R.id.home) {
            onBackPressedDispatcher.onBackPressed()
            return true
        }
        return super.onOptionsItemSelected(item)
    }

    private fun showNotificationPermissionRationale() {
        AlertDialog.Builder(this)
            .setTitle(R.string.notification_permission_required_title)
            .setMessage(R.string.notification_permission_required_message)
            .setPositiveButton(R.string.ok) { _, _ ->
                if (Build.VERSION.SDK_INT >= 33) {
                    notificationPermissionLauncher.launch(android.Manifest.permission.POST_NOTIFICATIONS)
                }
            }
            .setNegativeButton(R.string.cancel) { _, _ ->
                // Do nothing
            }
            .show()

    }

    private fun createCertExpiryNotificationChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val name: CharSequence = getString(R.string.cert_expiry_channel_name)
            val importance = NotificationManager.IMPORTANCE_DEFAULT
            val channelID = Constants.CERT_EXPIRY_NOTIFICATION_CHANNEL_ID
            val channel = NotificationChannel(channelID, name, importance)
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.createNotificationChannel(channel)
        }
    }

    private fun createVPNConnectionNotificationChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val name: CharSequence = getString(R.string.vpn_connection_channel_name)
            val importance = NotificationManager.IMPORTANCE_LOW
            val channelID = Constants.VPN_CONNECTION_NOTIFICATION_CHANNEL_ID
            val channel = NotificationChannel(channelID, name, importance)
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.createNotificationChannel(channel)
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (requestCode == REQUEST_CODE_SETTINGS) {
            if (resultCode == SettingsActivity.RESULT_APP_DATA_CLEARED) {
                if (vpnService.isPresent
                    && vpnService.get().getStatus() != VPNService.VPNStatus.DISCONNECTED
                ) {
                    vpnService.get().disconnect()
                }
                openFragment(OrganizationSelectionFragment(), false)
            }
        } else if (requestCode == 0) {
            // Probably VPN permission granted. Can't really check for it
            wireGuardService.tryResumeConnecting(this)
        }
        super.onActivityResult(requestCode, resultCode, data)
    }
}
