package nl.eduvpn.app.service

import android.os.Build
import android.service.quicksettings.Tile
import android.service.quicksettings.TileService
import android.widget.Toast
import androidx.annotation.RequiresApi
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import nl.eduvpn.app.EduVPNApplication
import nl.eduvpn.app.R
import nl.eduvpn.app.viewmodel.BaseConnectionViewModel
import nl.eduvpn.app.viewmodel.ConnectionStatusViewModel
import nl.eduvpn.app.viewmodel.MainViewModel
import java.util.Optional
import javax.inject.Inject


/**
 * Derived from the OpenVPNTileService from the ics-openvpn library.
 * With our modifications, we can also toggle the VPN connection if connected to a WireGuard server.
 */
@RequiresApi(Build.VERSION_CODES.N)
class VpnTileService : TileService() {

    @Inject
    lateinit var optionalVPNService: Optional<VPNService>

    @Inject
    lateinit var eduVpnOpenVpnService: EduVPNOpenVPNService

    @Inject
    lateinit var vpnConnectionService: VPNConnectionService

    @Inject
    lateinit var preferencesService: PreferencesService

    @Inject
    lateinit var historyService: HistoryService

    @Inject
    lateinit var backendService: BackendService

    private var observer: Observer<VPNService.VPNStatus>? = null
    private var viewModelParentActionObserver: Observer<BaseConnectionViewModel.ParentAction?>? = null
    private var temporaryViewModel: ConnectionStatusViewModel? = null


    override fun onStartListening() {
        super.onStartListening()
        EduVPNApplication.get(this).component().inject(this)
        observer = Observer {
            updateTile(it)
        }
        optionalVPNService.get().observeForever(observer!!)
        if (!EduVPNApplication.get(this).hasOpenWindows()) {
            eduVpnOpenVpnService.onCreate(this)
            backendService.register(
                startOAuth = {
                    handleConnectionError(getString(R.string.quick_settings_tile_notification_authentication_required_open_app))
                },
                selectProfiles = {
                    // Select the first one
                    it.firstOrNull()?.let {
                        GlobalScope.launch {
                            temporaryViewModel?.selectProfileToConnectTo(it, false)
                        }
                    }
                },
                selectCountry = {
                    // Ignore, we don't support this from the tile
                },
                connectWithConfig = { config, preferTcp ->
                    MainScope().launch {
                        MainViewModel(
                            this@VpnTileService,
                            historyService,
                            backendService,
                            preferencesService,
                            eduVpnOpenVpnService,
                            vpnConnectionService
                        ).parseConfigAndStartConnection(this@VpnTileService, config, preferTcp)
                    }
                },
                showError = {
                    handleConnectionError(it.message ?: it.toString())
                }
            )
        } else {
            // The activity will bind instead.
        }
        updateTile(optionalVPNService.get().getStatus())
    }

    private fun updateTile(status: VPNService.VPNStatus) {
        val tile = qsTile ?: return
        if (status == VPNService.VPNStatus.DISCONNECTED || status == VPNService.VPNStatus.FAILED) {
            // No VPN connected, try to reconnect
            val vpnInstance = backendService.getCurrentServer()?.asInstance() ?: preferencesService.getCurrentInstance()
            if (vpnInstance == null) {
                tile.label = getString(R.string.quick_settings_tile_no_vpn_selected)
                tile.state = Tile.STATE_UNAVAILABLE
            } else {
                tile.label = getString(R.string.quick_settings_tile_connect_vpn)
                tile.state = Tile.STATE_INACTIVE
            }
        } else {
            if (status == VPNService.VPNStatus.CONNECTING) {
                tile.label = getString(R.string.quick_settings_tile_connecting)
            } else {
                tile.label = getString(R.string.quick_settings_tile_disconnect_vpn)
            }
            tile.state = Tile.STATE_ACTIVE
        }
        tile.updateTile()
    }

    override fun onStopListening() {
        super.onStopListening()
        viewModelParentActionObserver?.let {
            temporaryViewModel?.parentAction?.removeObserver(it)
        }
        observer?.let {
            optionalVPNService.get().removeObserver(it)
        }
        temporaryViewModel = null
        observer = null
    }

    override fun onTileAdded() {
        // No action needed
    }


    override fun onClick() {
        val vpnService = optionalVPNService.get()
        if (vpnService.getStatus() == VPNService.VPNStatus.CONNECTED ||
            vpnService.getStatus() == VPNService.VPNStatus.CONNECTING ||
            vpnService.getStatus() == VPNService.VPNStatus.PAUSED
        ) {
            vpnConnectionService.disconnect(this, vpnService)
        } else {
            // Try to connect to the last used server, or if not available, to the first server in the list of added servers.
            // Use a temporary ViewModel for this, just for code reuse
            updateTile(VPNService.VPNStatus.CONNECTING)
            val viewModel = ConnectionStatusViewModel(
                this,
                preferencesService,
                vpnService,
                MutableLiveData(),
                MutableLiveData(),
                backendService,
                historyService,
                vpnConnectionService
            )
            viewModelParentActionObserver = Observer {
                when (it) {
                    is BaseConnectionViewModel.ParentAction.ShowContextCanceledToast -> {
                        handleConnectionError(it.message)

                    }
                    is BaseConnectionViewModel.ParentAction.DisplayError -> {
                        handleConnectionError(it.message)
                    }
                    else -> {
                        // Ignore
                    }
                }
            }
            viewModel.parentAction.observeForever(viewModelParentActionObserver!!)
            viewModel.reconnectWithCurrentProfile(false)
            temporaryViewModel = viewModel
        }
    }

    private fun handleConnectionError(message: String) {
        Toast.makeText(
            this,
            message,
            Toast.LENGTH_LONG
        ).show()
        qsTile?.let { tile ->
            tile.label = getString(R.string.quick_settings_tile_connect_vpn)
            tile.state = Tile.STATE_INACTIVE
            tile.updateTile()
        }
        viewModelParentActionObserver?.let { observer ->
            temporaryViewModel?.parentAction?.removeObserver(observer)
        }
        temporaryViewModel = null
    }
}