package org.thoughtcrime.securesms.ui.components

import android.content.ContentResolver
import android.content.ContentValues
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.provider.MediaStore
import android.widget.Toast
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.lifecycleScope
import com.squareup.phrase.Phrase
import dagger.hilt.android.qualifiers.ApplicationContext
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import network.loki.messenger.BuildConfig
import network.loki.messenger.R
import org.session.libsession.utilities.StringSubstitutionConstants.APP_NAME_KEY
import org.session.libsignal.utilities.ExternalStorageUtil
import org.session.libsignal.utilities.Log
import org.thoughtcrime.securesms.logging.PersistentLogger
import org.thoughtcrime.securesms.ui.AlertDialog
import org.thoughtcrime.securesms.ui.DialogButtonData
import org.thoughtcrime.securesms.ui.GetString
import org.thoughtcrime.securesms.ui.LoadingDialog
import org.thoughtcrime.securesms.util.FileProviderUtil
import java.io.File
import java.io.IOException
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import javax.inject.Singleton

@Composable
fun ExportLogsDialog(
    logExporter: LogExporter,
    onDismissRequest: () -> Unit,
){
    val context = LocalContext.current
    val scope = rememberCoroutineScope()

    //todo use retain once in the dev branch
    var exportingLogs by remember { mutableStateOf(false) }

    if(exportingLogs){
        LoadingDialog()
    } else {
        AlertDialog(
            onDismissRequest = onDismissRequest,
            title = stringResource(R.string.helpReportABugExportLogs),
            text = Phrase.from(context, R.string.helpReportABugDescription)
                .put(APP_NAME_KEY, stringResource(R.string.app_name))
                .format().toString(),
            buttons = listOf(
                DialogButtonData(
                    text = GetString(stringResource(R.string.share)),
                    dismissOnClick = false,
                    onClick = {
                        logExporter.runShareLogsJob(
                            scope = scope,
                            updateCallback = {
                                exportingLogs = it
                            },
                            onShareReady = { intent ->
                                context.startActivity(intent)
                            },
                            onComplete = onDismissRequest
                        )
                    }
                ),
                DialogButtonData(
                    text = GetString(stringResource(R.string.cancel)),
                ),
            )
        )
    }
}

@Singleton
class LogExporter @Inject constructor(
    @ApplicationContext private val context: Context,
    private val persistentLogger: PersistentLogger
){
    private val TAG = "LogExporter"

    private var shareJob: Job? = null

    fun runShareLogsJob(
        scope: CoroutineScope,
        updateCallback: (Boolean)->Unit,
        onShareReady: (Intent) -> Unit,
        onComplete: ()->Unit
    ) {
        // Cancel any existing share job that might already be running to start anew
        shareJob?.cancel()

        updateCallback(true)

        shareJob = scope.launch {
            try {
                Log.d(TAG, "Starting share logs job...")
                val mediaUri = withContext(Dispatchers.IO) {
                    withExternalFile(persistentLogger::readAllLogsCompressed)
                } ?: return@launch

                val shareIntent = Intent().apply {
                    action = Intent.ACTION_SEND
                    putExtra(Intent.EXTRA_STREAM, mediaUri)
                    type = "application/zip"
                    addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
                }

                val chooserIntent = Intent.createChooser(shareIntent, context.getString(R.string.share))

                onShareReady(chooserIntent)
            } catch (e: Exception) {
                if (e !is CancellationException) {
                    Log.e("Loki", "Error saving logs", e)
                    Toast.makeText(context, context.getString(R.string.errorUnknown), Toast.LENGTH_LONG)
                        .show()
                }
            }
        }.also { shareJob ->
            shareJob.invokeOnCompletion { handler ->
                // Note: Don't show Toasts here directly - use `withContext(Main)` or such if req'd
                handler?.message.let { msg ->
                    if (shareJob.isCancelled) {
                        if (msg.isNullOrBlank()) {
                            Log.w(TAG, "Share logs job was cancelled.")
                        } else {
                            Log.d(TAG, "Share logs job was cancelled. Reason: $msg")
                        }

                    }
                    else if (shareJob.isCompleted) {
                        Log.d(TAG, "Share logs job completed. Msg: $msg")
                    }
                    else {
                        Log.w(TAG, "Share logs job finished while still Active. Msg: $msg")
                    }
                }

                onComplete()
            }
        }
    }

    private fun withExternalFile(action: (Uri) -> Unit): Uri? {
        val base = "${Build.MANUFACTURER}-${Build.DEVICE}-API${Build.VERSION.SDK_INT}-v${BuildConfig.VERSION_NAME}-${System.currentTimeMillis()}"
        val extension = "zip"
        val fileName = "$base.$extension"
        val outputUri: Uri = ExternalStorageUtil.getDownloadUri()
        if (outputUri.scheme == ContentResolver.SCHEME_FILE) {
            val outputDirectory = File(outputUri.path)
            var outputFile = File(outputDirectory, "$base.$extension")
            var i = 0
            while (outputFile.exists()) {
                outputFile = File(outputDirectory, base + "-" + ++i + "." + extension)
            }
            if (outputFile.isHidden) {
                throw IOException("Specified name would not be visible")
            }
            try {
                return FileProviderUtil.getUriFor(context, outputFile).also(action)
            } catch (e: Exception) {
                outputFile.delete()
                throw e
            }
        } else {
            val contentValues = ContentValues()
            contentValues.put(MediaStore.MediaColumns.DISPLAY_NAME, fileName)
            contentValues.put(MediaStore.MediaColumns.MIME_TYPE, "application/zip")
            contentValues.put(MediaStore.MediaColumns.DATE_ADDED, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()))
            contentValues.put(MediaStore.MediaColumns.DATE_MODIFIED, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()))
            contentValues.put(MediaStore.MediaColumns.IS_PENDING, 1)
            val uri = context.contentResolver.insert(outputUri, contentValues) ?: return null
            try {
                action(uri)

                // Remove the pending flag to make the file available
                contentValues.clear()
                contentValues.put(MediaStore.MediaColumns.IS_PENDING, 0)
                context.contentResolver.update(uri, contentValues, null, null)
            } catch (e: Exception) {
                context.contentResolver.delete(uri, null, null)
                throw e
            }

            return uri
        }
    }
}