package org.thoughtcrime.securesms.reactions;

import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import org.thoughtcrime.securesms.components.emoji.EmojiImageView;
import org.thoughtcrime.securesms.database.model.MessageId;
import org.thoughtcrime.securesms.util.AvatarUtils;

import java.util.Collections;
import java.util.List;

import network.loki.messenger.R;

public final class ReactionRecipientsAdapter extends RecyclerView.Adapter<ReactionRecipientsAdapter.ViewHolder> {

  private static final int MAX_REACTORS = 5;
  private static final int HEADER_COUNT = 1;
  private static final int HEADER_POSITION = 0;

  private static final int FOOTER_COUNT = 1;
  private static final int FOOTER_POSITION = 6;

  private static final int HEADER_TYPE = 0;
  private static final int RECIPIENT_TYPE = 1;
  private static final int FOOTER_TYPE = 2;

  private ReactionViewPagerAdapter.Listener callback;
  private final AvatarUtils avatarUtils;
  private List<ReactionDetails> data = Collections.emptyList();
  private MessageId messageId;
  private boolean isUserModerator;
  private EmojiCount emojiData;
  private final boolean canRemove;

  public ReactionRecipientsAdapter(ReactionViewPagerAdapter.Listener callback, AvatarUtils avatarUtils, boolean canRemove) {
    this.callback = callback;
      this.avatarUtils = avatarUtils;
      this.canRemove = canRemove;
  }

  public void updateData(MessageId messageId, EmojiCount newData, boolean isUserModerator) {
    this.messageId = messageId;
    emojiData = newData;
    data = newData.getReactions();
    this.isUserModerator = isUserModerator;
    notifyDataSetChanged();
  }

  @Override
  public int getItemViewType(int position) {
    switch (position) {
      case HEADER_POSITION:
        return HEADER_TYPE;
      case FOOTER_POSITION:
        return FOOTER_TYPE;
      default:
        return RECIPIENT_TYPE;
    }
  }

  @Override
  public @NonNull
  ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
    switch (viewType) {
      case HEADER_TYPE:
        return new HeaderViewHolder(callback, LayoutInflater.from(parent.getContext()).inflate(R.layout.reactions_bottom_sheet_dialog_fragment_recycler_header, parent, false));
      case FOOTER_TYPE:
        return new FooterViewHolder(LayoutInflater.from(parent.getContext()).inflate(R.layout.reactions_bottom_sheet_dialog_fragment_recycler_footer, parent, false));
      default:
        return new RecipientViewHolder(
                avatarUtils,
                callback,
                network.loki.messenger.databinding.ReactionsBottomSheetDialogFragmentRecipientItemBinding.inflate(LayoutInflater.from(parent.getContext()), parent, false),
                canRemove
        );
    }
  }

  @Override
  public void onBindViewHolder(@NonNull ViewHolder holder, int position) {
    if (holder instanceof RecipientViewHolder) {
      ((RecipientViewHolder) holder).bind(data.get(position - HEADER_COUNT));
    } else if (holder instanceof HeaderViewHolder) {
      ((HeaderViewHolder) holder).bind(emojiData, messageId, isUserModerator);
    } else if (holder instanceof FooterViewHolder) {
      ((FooterViewHolder) holder).bind(emojiData);
    }
  }

  @Override
  public void onViewRecycled(@NonNull ViewHolder holder) {
    if (holder instanceof RecipientViewHolder) {
      ((RecipientViewHolder) holder).unbind();
    }
  }

  @Override
  public int getItemCount() {
    if (data.isEmpty()) {
      return 0;
    } else if (emojiData.getCount() <= MAX_REACTORS) {
      return data.size() + HEADER_COUNT;
    } else {
      return MAX_REACTORS + HEADER_COUNT + FOOTER_COUNT;
    }
  }

  static public class ViewHolder extends RecyclerView.ViewHolder {
    public ViewHolder(@NonNull View itemView) {
      super(itemView);
    }
  }

  static class HeaderViewHolder extends ViewHolder {

    private final ReactionViewPagerAdapter.Listener callback;

    public HeaderViewHolder(ReactionViewPagerAdapter.Listener callback, @NonNull View itemView) {
      super(itemView);
      this.callback = callback;
    }

    private void bind(@NonNull final EmojiCount emoji, final MessageId messageId, boolean isUserModerator) {
      View clearAll = itemView.findViewById(R.id.header_view_clear_all);
      clearAll.setVisibility(isUserModerator ? View.VISIBLE : View.GONE);
      clearAll.setOnClickListener(isUserModerator ? (View.OnClickListener) v -> {
        callback.onClearAll(emoji.getBaseEmoji(), messageId);
      } : null);
      EmojiImageView emojiView = itemView.findViewById(R.id.header_view_emoji);
      emojiView.setImageEmoji(emoji.getDisplayEmoji());
      TextView count = itemView.findViewById(R.id.header_view_emoji_count);
      count.setText(String.format(" •  %s", emoji.getCount()));
    }
  }

  static class FooterViewHolder extends ViewHolder {

    public FooterViewHolder(@NonNull View itemView) {
      super(itemView);
    }

    private void bind(@NonNull final EmojiCount emoji) {
      if (emoji.getCount() > 5) {
        TextView count = itemView.findViewById(R.id.footer_view_emoji_count);

        // We display the first 5 people to react w/ a given emoji so we'll subtract that to get the 'others' count
        int othersCount = emoji.getCount() - 5;
        String s = itemView.getResources().getQuantityString(R.plurals.emojiReactsCountOthers, othersCount, othersCount, emoji.getBaseEmoji());
        count.setText(s);

        itemView.setVisibility(View.VISIBLE);
      } else {
        itemView.setVisibility(View.GONE);
      }
    }
  }

}
