package org.session.libsession.messaging.jobs

import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import okhttp3.HttpUrl.Companion.toHttpUrl
import okhttp3.HttpUrl.Companion.toHttpUrlOrNull
import org.session.libsession.database.MessageDataProvider
import org.session.libsession.database.StorageProtocol
import org.session.libsession.messaging.file_server.FileServerApi
import org.session.libsession.messaging.open_groups.OpenGroupApi
import org.session.libsession.messaging.sending_receiving.attachments.AttachmentId
import org.session.libsession.messaging.sending_receiving.attachments.AttachmentState
import org.session.libsession.messaging.sending_receiving.attachments.DatabaseAttachment
import org.session.libsession.messaging.utilities.Data
import org.session.libsession.snode.OnionRequestAPI
import org.session.libsession.utilities.Address
import org.session.libsession.utilities.DecodedAudio
import org.session.libsession.utilities.InputStreamMediaDataSource
import org.session.libsignal.exceptions.NonRetryableException
import org.session.libsignal.utilities.Base64
import org.session.libsignal.utilities.HTTP
import org.session.libsignal.utilities.Log
import org.thoughtcrime.securesms.attachments.AttachmentProcessor
import org.thoughtcrime.securesms.database.model.MessageId

class AttachmentDownloadJob @AssistedInject constructor(
    @Assisted("attachmentID") val attachmentID: Long,
    @Assisted val mmsMessageId: Long,
    private val storage: StorageProtocol,
    private val messageDataProvider: MessageDataProvider,
    private val attachmentProcessor: AttachmentProcessor,
    private val fileServerApi: FileServerApi,
) : Job {
    override var delegate: JobDelegate? = null
    override var id: String? = null
    override var failureCount: Int = 0

    // Error
    internal sealed class Error(val description: String) : Exception(description) {
        object NoAttachment : Error("No such attachment.")
        object NoThread: Error("Thread no longer exists")
        object NoSender: Error("Thread recipient or sender does not exist")
        object DuplicateData: Error("Attachment already downloaded")
    }

    // Settings
    override val maxFailureCount: Int = 2

    companion object {
        const val KEY: String = "AttachmentDownloadJob"

        // Keys used for database storage
        private val ATTACHMENT_ID_KEY = "attachment_id"
        private val TS_INCOMING_MESSAGE_ID_KEY = "tsIncoming_message_id"

        /**
         * Check if the attachment in the given message is eligible for download.
         *
         * Note that this function only checks for the eligibility of the attachment in the sense
         * of whether the download is allowed, it does not check if the download has already taken
         * place.
         */
        fun eligibleForDownload(
            threadID: Long,
            storage: StorageProtocol,
            messageDataProvider: MessageDataProvider,
            mmsId: Long
        ): Boolean {
            val threadRecipient = storage.getRecipientForThread(threadID) ?: return false

            // if we are the sender we are always eligible
            val selfSend = messageDataProvider.isOutgoingMessage(MessageId(mmsId, true))
            if (selfSend) {
                return true
            }

            return threadRecipient.autoDownloadAttachments == true
        }
    }

    override suspend fun execute(dispatcherName: String) {
        val threadID = storage.getThreadIdForMms(mmsMessageId)

        val handleFailure: (java.lang.Exception, attachmentId: AttachmentId?) -> Unit = { exception, attachment ->
            if(exception is HTTP.HTTPRequestFailedException && exception.statusCode == 404){
                attachment?.let { id ->
                    Log.d("AttachmentDownloadJob", "Setting attachment state = failed, have attachment")
                    messageDataProvider.setAttachmentState(AttachmentState.EXPIRED, id, mmsMessageId)
                } ?: run {
                    Log.d("AttachmentDownloadJob", "Setting attachment state = failed, don't have attachment")
                    messageDataProvider.setAttachmentState(AttachmentState.EXPIRED, AttachmentId(attachmentID,0), mmsMessageId)
                }
            } else if (exception == Error.NoAttachment
                    || exception == Error.NoThread
                    || exception == Error.NoSender
                    || (exception is OnionRequestAPI.HTTPRequestFailedAtDestinationException && exception.statusCode == 400)
                    || exception is NonRetryableException) {
                attachment?.let { id ->
                    Log.d("AttachmentDownloadJob", "Setting attachment state = failed, have attachment")
                    messageDataProvider.setAttachmentState(AttachmentState.FAILED, id, mmsMessageId)
                } ?: run {
                    Log.d("AttachmentDownloadJob", "Setting attachment state = failed, don't have attachment")
                    messageDataProvider.setAttachmentState(AttachmentState.FAILED, AttachmentId(attachmentID,0), mmsMessageId)
                }
                this.handlePermanentFailure(dispatcherName, exception)
            } else if (exception == Error.DuplicateData) {
                attachment?.let { id ->
                    Log.d("AttachmentDownloadJob", "Setting attachment state = done from duplicate data")
                    messageDataProvider.setAttachmentState(AttachmentState.DONE, id, mmsMessageId)
                } ?: run {
                    Log.d("AttachmentDownloadJob", "Setting attachment state = done from duplicate data")
                    messageDataProvider.setAttachmentState(AttachmentState.DONE, AttachmentId(attachmentID,0), mmsMessageId)
                }
                this.handleSuccess(dispatcherName)
            } else {
                if (failureCount + 1 >= maxFailureCount) {
                    attachment?.let { id ->
                        Log.d("AttachmentDownloadJob", "Setting attachment state = failed from max failure count, have attachment")
                        messageDataProvider.setAttachmentState(AttachmentState.FAILED, id, mmsMessageId)
                    } ?: run {
                        Log.d("AttachmentDownloadJob", "Setting attachment state = failed from max failure count, don't have attachment")
                        messageDataProvider.setAttachmentState(AttachmentState.FAILED, AttachmentId(attachmentID,0), mmsMessageId)
                    }
                }
                this.handleFailure(dispatcherName, exception)
            }
        }

        if (threadID < 0) {
            handleFailure(Error.NoThread, null)
            return
        }

        if (!eligibleForDownload(
                threadID = threadID,
                storage = storage,
                messageDataProvider = messageDataProvider,
                mmsId = mmsMessageId
            )) {
            handleFailure(Error.NoSender, null)
            return
        }

        val threadRecipient = storage.getRecipientForThread(threadID)

        var attachment: DatabaseAttachment? = null

        try {
            attachment = messageDataProvider.getDatabaseAttachment(attachmentID)
                ?: return handleFailure(Error.NoAttachment, null)
            if (attachment.hasData()) {
                handleFailure(Error.DuplicateData, attachment.attachmentId)
                return
            }
            messageDataProvider.setAttachmentState(AttachmentState.DOWNLOADING, attachment.attachmentId, this.mmsMessageId)

            val decrypted = if (threadRecipient?.address !is Address.Community) {
                Log.d("AttachmentDownloadJob", "downloading normal attachment")
                val r = runCatching { fileServerApi.parseAttachmentUrl(attachment.url.toHttpUrl()) }
                    .recover { throw NonRetryableException("Invalid file server URL", it) }
                    .getOrThrow()

                val key = requireNotNull(attachment.key) {
                    throw NonRetryableException("Missing attachment key")
                }.let(Base64::decode)

                val cipherText = fileServerApi.download(
                    fileId = r.fileId,
                    fileServer = r.fileServer
                ).body

                runCatching {
                    if (r.usesDeterministicEncryption) {
                        attachmentProcessor.decryptDeterministically(
                            ciphertext = cipherText,
                            key = key
                        )
                    } else {
                        attachmentProcessor.decryptAttachmentLegacy(
                            ciphertext = cipherText,
                            key = key,
                            digest = attachment.digest
                        )
                    }
                }.recover { throw NonRetryableException("Decryption failed", it) }
                    .getOrThrow()
            } else {
                Log.d("AttachmentDownloadJob", "downloading open group attachment")
                val url = attachment.url.toHttpUrlOrNull()!!
                val fileID = url.pathSegments.last()
                OpenGroupApi.download(fileID, room = threadRecipient.address.room, server = threadRecipient.address.serverUrl)
            }

            Log.d("AttachmentDownloadJob", "getting input stream")

            Log.d("AttachmentDownloadJob", "inserting attachment")
            messageDataProvider.insertAttachment(
                messageId = mmsMessageId,
                attachmentId = attachment.attachmentId,
                stream = decrypted.inputStream()
            )

            if (attachment.contentType.startsWith("audio/")) {
                // process the duration
                    try {
                        InputStreamMediaDataSource(decrypted.inputStream()).use { mediaDataSource ->
                            val durationMs = (DecodedAudio.create(mediaDataSource).totalDurationMicroseconds / 1000.0).toLong()
                            messageDataProvider.updateAudioAttachmentDuration(
                                attachment.attachmentId,
                                durationMs,
                                threadID
                            )
                        }
                    } catch (e: Exception) {
                        Log.e("Loki", "Couldn't process audio attachment", e)
                    }
            }
            Log.d("AttachmentDownloadJob", "deleting tempfile")
            Log.d("AttachmentDownloadJob", "succeeding job")
            handleSuccess(dispatcherName)
        } catch (e: Exception) {
            Log.e("AttachmentDownloadJob", "Error processing attachment download", e)
            return handleFailure(e,attachment?.attachmentId)
        }
    }

    private fun handleSuccess(dispatcherName: String) {
        Log.w("AttachmentDownloadJob", "Attachment downloaded successfully.")
        delegate?.handleJobSucceeded(this, dispatcherName)
    }

    private fun handlePermanentFailure(dispatcherName: String, e: Exception) {
        delegate?.handleJobFailedPermanently(this, dispatcherName, e)
    }

    private fun handleFailure(dispatcherName: String, e: Exception) {
        delegate?.handleJobFailed(this, dispatcherName, e)
    }

    override fun serialize(): Data {
        return Data.Builder()
            .putLong(ATTACHMENT_ID_KEY, attachmentID)
            .putLong(TS_INCOMING_MESSAGE_ID_KEY, mmsMessageId)
            .build()
    }

    override fun getFactoryKey(): String {
        return KEY
    }

    @AssistedFactory
    abstract class Factory : Job.DeserializeFactory<AttachmentDownloadJob> {
        abstract fun create(
            @Assisted("attachmentID") attachmentID: Long,
            mmsMessageId: Long
        ): AttachmentDownloadJob

        override fun create(data: Data): AttachmentDownloadJob {
            return create(
                attachmentID = data.getLong(ATTACHMENT_ID_KEY),
                mmsMessageId = data.getLong(TS_INCOMING_MESSAGE_ID_KEY)
            )
        }
    }
}