package org.thoughtcrime.securesms.keyboard.emoji.search

import android.content.Context
import android.net.Uri
import dagger.hilt.android.qualifiers.ApplicationContext
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import org.session.libsession.utilities.TextSecurePreferences
import org.session.libsession.utilities.concurrent.SignalExecutors
import org.thoughtcrime.securesms.components.emoji.Emoji
import org.thoughtcrime.securesms.components.emoji.EmojiPageModel
import org.thoughtcrime.securesms.components.emoji.RecentEmojiPageModel
import org.thoughtcrime.securesms.database.EmojiSearchDatabase
import org.thoughtcrime.securesms.dependencies.DatabaseComponent
import org.thoughtcrime.securesms.emoji.EmojiSource
import java.util.function.Consumer
import javax.inject.Inject

private const val MINIMUM_QUERY_THRESHOLD = 1
private const val MINIMUM_INLINE_QUERY_THRESHOLD = 2
private const val EMOJI_SEARCH_LIMIT = 20

private val NOT_PUNCTUATION = "[A-Za-z0-9 ]".toRegex()

class EmojiSearchRepository @Inject constructor(
  private val emojiSearchDatabase: EmojiSearchDatabase
) {

  fun submitQuery(query: String, limit: Int = EMOJI_SEARCH_LIMIT): Single<List<String>> {
    val result = if (query.length >= MINIMUM_INLINE_QUERY_THRESHOLD && NOT_PUNCTUATION.matches(query.substring(query.lastIndex))) {
      Single.fromCallable { emojiSearchDatabase.query(query, limit) }
    } else {
      Single.just(emptyList())
    }

    return result.subscribeOn(Schedulers.io())
  }

  fun submitQuery(query: String, limit: Int = EMOJI_SEARCH_LIMIT, consumer: Consumer<EmojiPageModel>) {
    SignalExecutors.SERIAL.execute {
      val emoji: List<String> = emojiSearchDatabase.query(query, limit)

      val displayEmoji: List<Emoji> = emoji
        .mapNotNull { canonical -> EmojiSource.latest.canonicalToVariations[canonical] }
        .map { Emoji(it) }

      consumer.accept(EmojiSearchResultsPageModel(emoji, displayEmoji))
    }
  }

  private class EmojiSearchResultsPageModel(
    private val emoji: List<String>,
    private val displayEmoji: List<Emoji>
  ) : EmojiPageModel {
    override fun getKey(): String = ""

    override fun getIconAttr(): Int = -1

    override fun getEmoji(): List<String> = emoji

    override fun getDisplayEmoji(): List<Emoji> = displayEmoji

    override fun hasSpriteMap(): Boolean = false

    override fun getSpriteUri(): Uri? = null

    override fun isDynamic(): Boolean = false
  }
}
