package org.thoughtcrime.securesms.groups.handler

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import network.loki.messenger.libsession_util.util.GroupInfo
import network.loki.messenger.libsession_util.util.GroupMember
import org.session.libsession.utilities.ConfigFactoryProtocol
import org.session.libsession.utilities.UserConfigType
import org.session.libsession.utilities.userConfigsChanged
import org.session.libsignal.utilities.AccountId
import org.thoughtcrime.securesms.auth.LoginStateRepository
import org.thoughtcrime.securesms.dependencies.ManagerScope
import org.thoughtcrime.securesms.dependencies.OnAppStartupComponent
import java.util.EnumSet
import javax.inject.Inject
import javax.inject.Singleton

/**
 * A handler that listens for group config updates, and make sure the our group member admin state
 * is in sync with the UserGroupConfig.
 *
 * This concerns the "admin", "promotionStatus" in the GroupMemberConfig
 *
 */
@Singleton
class AdminStateSync @Inject constructor(
    private val configFactory: ConfigFactoryProtocol,
    private val loginStateRepository: LoginStateRepository,
    @param:ManagerScope private val scope: CoroutineScope
) : OnAppStartupComponent {
    private var job: Job? = null

    override fun onPostAppStarted() {
        require(job == null) { "Already started" }

        job = scope.launch {
            loginStateRepository.flowWithLoggedInState {
                configFactory.userConfigsChanged(onlyConfigTypes = setOf(UserConfigType.USER_GROUPS))
            }.collect {
                val localNumber = loginStateRepository.requireLocalNumber()

                // Go through evey user groups and if we are admin of any of the groups,
                // make sure we mark any pending group promotion status as "accepted"

                val allAdminGroups = configFactory.withUserConfigs { configs ->
                    configs.userGroups.all()
                        .asSequence()
                        .mapNotNull {
                            if ((it as? GroupInfo.ClosedGroupInfo)?.hasAdminKey() == true) {
                                AccountId(it.groupAccountId)
                            } else {
                                null
                            }
                        }
                }

                val groupToMarkAccepted = allAdminGroups
                    .filter { groupId -> isMemberPromotionPending(groupId, localNumber) }

                for (groupId in groupToMarkAccepted) {
                    configFactory.withMutableGroupConfigs(groupId) { groupConfigs ->
                        groupConfigs.groupMembers.get(localNumber)?.let { member ->
                            member.setPromotionAccepted()
                            groupConfigs.groupMembers.set(member)
                        }
                    }
                }
            }
        }
    }

    private fun isMemberPromotionPending(groupId: AccountId, localNumber: String): Boolean {
        return configFactory.withGroupConfigs(groupId) { groupConfigs ->
            val status = groupConfigs.groupMembers.get(localNumber)?.let(groupConfigs.groupMembers::status)
            status != null && status in EnumSet.of(
                GroupMember.Status.PROMOTION_SENT,
                GroupMember.Status.PROMOTION_FAILED,
                GroupMember.Status.PROMOTION_NOT_SENT
            )
        }
    }
}