@file:Suppress("NAME_SHADOWING")

package org.session.libsession.snode

import android.os.SystemClock
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.async
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.channels.SendChannel
import kotlinx.coroutines.launch
import kotlinx.coroutines.selects.onTimeout
import kotlinx.coroutines.selects.select
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.decodeFromStream
import network.loki.messenger.libsession_util.ED25519
import network.loki.messenger.libsession_util.Hash
import network.loki.messenger.libsession_util.SessionEncrypt
import nl.komponents.kovenant.Promise
import nl.komponents.kovenant.functional.bind
import nl.komponents.kovenant.functional.map
import org.session.libsession.messaging.MessagingModuleConfiguration
import org.session.libsession.snode.model.BatchResponse
import org.session.libsession.snode.model.StoreMessageResponse
import org.session.libsession.snode.utilities.asyncPromise
import org.session.libsession.snode.utilities.await
import org.session.libsession.snode.utilities.retrySuspendAsPromise
import org.session.libsession.utilities.Environment
import org.session.libsession.utilities.mapValuesNotNull
import org.session.libsession.utilities.toByteArray
import org.session.libsignal.crypto.secureRandom
import org.session.libsignal.crypto.shuffledRandom
import org.session.libsignal.database.LokiAPIDatabaseProtocol
import org.session.libsignal.utilities.Base64
import org.session.libsignal.utilities.HTTP
import org.session.libsignal.utilities.Hex
import org.session.libsignal.utilities.JsonUtil
import org.session.libsignal.utilities.Log
import org.session.libsignal.utilities.Snode
import org.session.libsignal.utilities.prettifiedDescription
import org.session.libsignal.utilities.retryWithUniformInterval
import java.util.Locale
import kotlin.collections.component1
import kotlin.collections.component2
import kotlin.collections.set
import kotlin.properties.Delegates.observable

object SnodeAPI {
    internal val database: LokiAPIDatabaseProtocol
        get() = SnodeModule.shared.storage

    private var snodeFailureCount: MutableMap<Snode, Int> = mutableMapOf()

    // the  list of "generic" nodes we use to make non swarm specific api calls
    internal var snodePool: Set<Snode>
        get() = database.getSnodePool()
        set(newValue) { database.setSnodePool(newValue) }

    @Deprecated("Use a dependency injected SnodeClock.currentTimeMills() instead")
    @JvmStatic
    val nowWithOffset
        get() = MessagingModuleConfiguration.shared.clock.currentTimeMills()

    internal var forkInfo by observable(database.getForkInfo()) { _, oldValue, newValue ->
        if (newValue > oldValue) {
            Log.d("Loki", "Setting new fork info new: $newValue, old: $oldValue")
            database.setForkInfo(newValue)
        }
    }

    // Settings
    private const val maxRetryCount = 6
    private const val minimumSnodePoolCount = 12
    private const val minimumSwarmSnodeCount = 3
    // Use port 4433 to enforce pinned certificates
    private val seedNodePort = 4443

    private val seedNodePool = when (SnodeModule.shared.environment) {
        Environment.DEV_NET -> setOf("http://sesh-net.local:1280")
        Environment.TEST_NET -> setOf("http://public.loki.foundation:38157")
        Environment.MAIN_NET -> setOf(
            "https://seed1.getsession.org:$seedNodePort",
            "https://seed2.getsession.org:$seedNodePort",
            "https://seed3.getsession.org:$seedNodePort",
        )
    }

    private const val snodeFailureThreshold = 3
    private const val useOnionRequests = true

    const val KEY_BODY = "body"
    const val KEY_CODE = "code"
    const val KEY_RESULTS = "results"
    private const val KEY_IP = "public_ip"
    private const val KEY_PORT = "storage_port"
    private const val KEY_X25519 = "pubkey_x25519"
    private const val KEY_ED25519 = "pubkey_ed25519"
    private const val KEY_VERSION = "storage_server_version"

    private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Default)

    // Error
    sealed class Error(val description: String) : Exception(description) {
        object Generic : Error("An error occurred.")
        object ClockOutOfSync : Error("Your clock is out of sync with the Service Node network.")
        object NoKeyPair : Error("Missing user key pair.")
        object SigningFailed : Error("Couldn't sign verification data.")

        // ONS
        object DecryptionFailed : Error("Couldn't decrypt ONS name.")
        object HashingFailed : Error("Couldn't compute ONS name hash.")
        object ValidationFailed : Error("ONS name validation failed.")
    }

    // Batch
    data class SnodeBatchRequestInfo(
        val method: String,
        val params: Map<String, Any>,
        @Transient
        val namespace: Int?,
    ) // assume signatures, pubkey and namespaces are attached in parameters if required

    // Internal API
    internal fun invoke(
        method: Snode.Method,
        snode: Snode,
        parameters: Map<String, Any>,
        publicKey: String? = null,
        version: Version = Version.V3
    ): RawResponsePromise = when {
        useOnionRequests -> OnionRequestAPI.sendOnionRequest(method, parameters, snode, version, publicKey).map {
            JsonUtil.fromJson(it.body ?: throw Error.Generic, Map::class.java)
        }

        else -> scope.asyncPromise {
            HTTP.execute(
                HTTP.Verb.POST,
                url = "${snode.address}:${snode.port}/storage_rpc/v1",
                parameters = buildMap {
                    this["method"] = method.rawValue
                    this["params"] = parameters
                }
            ).toString().let {
                JsonUtil.fromJson(it, Map::class.java)
            }
        }.fail { e ->
            when (e) {
                is HTTP.HTTPRequestFailedException -> handleSnodeError(e.statusCode, e.json, snode, publicKey)
                else -> Log.d("Loki", "Unhandled exception: $e.")
            }
        }
    }

    private suspend fun<Res> invokeSuspend(
        method: Snode.Method,
        snode: Snode,
        parameters: Map<String, Any>,
        responseDeserializationStrategy: DeserializationStrategy<Res>,
        publicKey: String? = null,
        version: Version = Version.V3
    ): Res = when {
        useOnionRequests -> {
            val resp = OnionRequestAPI.sendOnionRequest(method, parameters, snode, version, publicKey).await()
            (resp.body ?: throw Error.Generic).inputStream().use { inputStream ->
                MessagingModuleConfiguration.shared.json.decodeFromStream(
                    deserializer = responseDeserializationStrategy,
                    stream = inputStream
                )
            }
        }

        else -> HTTP.execute(
            HTTP.Verb.POST,
            url = "${snode.address}:${snode.port}/storage_rpc/v1",
            parameters = buildMap {
                this["method"] = method.rawValue
                this["params"] = parameters
            }
        ).toString().let {
            MessagingModuleConfiguration.shared.json.decodeFromString(
                deserializer = responseDeserializationStrategy,
                string = it
            )
        }
    }

    private val GET_RANDOM_SNODE_PARAMS = buildMap<String, Any> {
        this["method"] = "get_n_service_nodes"
        this["params"] = buildMap {
            this["active_only"] = true
            this["fields"] = sequenceOf(KEY_IP, KEY_PORT, KEY_X25519, KEY_ED25519, KEY_VERSION).associateWith { true }
        }
    }

    internal fun getRandomSnode(): Promise<Snode, Exception> =
        snodePool.takeIf { it.size >= minimumSnodePoolCount }?.secureRandom()?.let { Promise.of(it) } ?: scope.asyncPromise {
            val target = seedNodePool.random()
            Log.d("Loki", "Populating snode pool using: $target.")
            val url = "$target/json_rpc"
            val response = HTTP.execute(HTTP.Verb.POST, url, GET_RANDOM_SNODE_PARAMS, useSeedNodeConnection = true)
            val json = runCatching { JsonUtil.fromJson(response, Map::class.java) }.getOrNull()
                ?: buildMap { this["result"] = response.toString() }
            val intermediate = json["result"] as? Map<*, *> ?: throw Error.Generic
                .also { Log.d("Loki", "Failed to update snode pool, intermediate was null.") }
            val rawSnodes = intermediate["service_node_states"] as? List<*> ?: throw Error.Generic
                .also { Log.d("Loki", "Failed to update snode pool, rawSnodes was null.") }

            rawSnodes.asSequence().mapNotNull { it as? Map<*, *> }.mapNotNull { rawSnode ->
                createSnode(
                    address = rawSnode[KEY_IP] as? String,
                    port = rawSnode[KEY_PORT] as? Int,
                    ed25519Key = rawSnode[KEY_ED25519] as? String,
                    x25519Key = rawSnode[KEY_X25519] as? String,
                    version = (rawSnode[KEY_VERSION] as? List<*>)
                        ?.filterIsInstance<Int>()
                        ?.let(Snode::Version)
                ).also { if (it == null) Log.d("Loki", "Failed to parse: ${rawSnode.prettifiedDescription()}.") }
            }.toSet().also {
                Log.d("Loki", "Persisting snode pool to database.")
                snodePool = it
            }.takeUnless { it.isEmpty() }?.secureRandom() ?: throw SnodeAPI.Error.Generic
        }

    private fun createSnode(address: String?, port: Int?, ed25519Key: String?, x25519Key: String?, version: Snode.Version? = Snode.Version.ZERO): Snode? {
        return Snode(
            address?.takeUnless { it == "0.0.0.0" }?.let { "https://$it" } ?: return null,
            port ?: return null,
            Snode.KeySet(ed25519Key ?: return null, x25519Key ?: return null),
            version ?: return null
        )
    }

    internal fun dropSnodeFromSwarmIfNeeded(snode: Snode, publicKey: String) {
        database.getSwarm(publicKey)?.takeIf { snode in it }?.let {
            database.setSwarm(publicKey, it - snode)
        }
    }

    fun getSingleTargetSnode(publicKey: String): Promise<Snode, Exception> {
        // SecureRandom should be cryptographically secure
        return getSwarm(publicKey).map { it.shuffledRandom().random() }
    }

    // Public API
    suspend fun getAccountID(onsName: String): String  {
        val validationCount = 3
        val accountIDByteCount = 33
        // Hash the ONS name using BLAKE2b
        val onsName = onsName.lowercase(Locale.US)
        // Ask 3 different snodes for the Account ID associated with the given name hash
        val parameters = buildMap<String, Any> {
            this["endpoint"] = "ons_resolve"
            this["params"] = buildMap {
                this["type"] = 0
                this["name_hash"] = Base64.encodeBytes(Hash.hash32(onsName.toByteArray()))
            }
        }

        return List(validationCount) {
            scope.async {
                retryWithUniformInterval(
                    maxRetryCount = maxRetryCount,
                ) {
                    val snode = getRandomSnode().await()
                    invoke(Snode.Method.OxenDaemonRPCCall, snode, parameters).await()
                }
            }
        }.awaitAll().map { json ->
                val intermediate = json["result"] as? Map<*, *> ?: throw Error.Generic
                val hexEncodedCiphertext = intermediate["encrypted_value"] as? String ?: throw Error.Generic
                val ciphertext = Hex.fromStringCondensed(hexEncodedCiphertext)
                val nonce = (intermediate["nonce"] as? String)?.let(Hex::fromStringCondensed)
                SessionEncrypt.decryptOnsResponse(
                    lowercaseName = onsName,
                    ciphertext = ciphertext,
                    nonce = nonce
                )
            }.takeIf { it.size == validationCount && it.toSet().size == 1 }?.first()
                ?: throw Error.ValidationFailed
    }

    // the list of snodes that represent the swarm for that pubkey
    fun getSwarm(publicKey: String): Promise<Set<Snode>, Exception> =
        database.getSwarm(publicKey)?.takeIf { it.size >= minimumSwarmSnodeCount }?.let(Promise.Companion::of)
            ?: getRandomSnode().bind {
                invoke(Snode.Method.GetSwarm, it, parameters = buildMap { this["pubKey"] = publicKey }, publicKey)
            }.map {
                parseSnodes(it).toSet()
            }.success {
                database.setSwarm(publicKey, it)
            }

    /**
     * Fetch swarm nodes for the specific public key.
     *
     * Note: this differs from [getSwarm] in that it doesn't store the swarm nodes in the database.
     * This always fetches from network.
     */
    suspend fun fetchSwarmNodes(publicKey: String): List<Snode> {
        val randomNode = getRandomSnode().await()
        val response = invoke(
            method = Snode.Method.GetSwarm,
            snode = randomNode, parameters = buildMap { this["pubKey"] = publicKey },
            publicKey = publicKey
        ).await()

        return parseSnodes(response)
    }

    /**
     * Build parameters required to call authenticated storage API.
     *
     * @param auth The authentication data required to sign the request
     * @param namespace The namespace of the messages you want to retrieve. Null if not relevant.
     * @param verificationData A function that returns the data to be signed. The function takes the namespace text and timestamp as arguments.
     * @param timestamp The timestamp to be used in the request. Default is the current time.
     * @param builder A lambda that allows the user to add additional parameters to the request.
     */
    private fun buildAuthenticatedParameters(
        auth: SwarmAuth,
        namespace: Int?,
        verificationData: ((namespaceText: String, timestamp: Long) -> Any)? = null,
        timestamp: Long = nowWithOffset,
        builder: MutableMap<String, Any>.() -> Unit = {}
    ): Map<String, Any> {
        return buildMap {
            // Build user provided parameter first
            this.builder()

            if (verificationData != null) {
                // Namespace shouldn't be in the verification data if it's null or 0.
                val namespaceText = when (namespace) {
                    null, 0 -> ""
                    else -> namespace.toString()
                }

                val verifyData = when (val verify = verificationData(namespaceText, timestamp)) {
                    is String -> verify.toByteArray()
                    is ByteArray -> verify
                    else -> throw IllegalArgumentException("verificationData must return a String or ByteArray")
                }

                putAll(auth.sign(verifyData))
                put("timestamp", timestamp)
            }

            put("pubkey", auth.accountId.hexString)
            if (namespace != null && namespace != 0) {
                put("namespace", namespace)
            }

            auth.ed25519PublicKeyHex?.let { put("pubkey_ed25519", it) }
        }
    }

    fun buildAuthenticatedStoreBatchInfo(
        namespace: Int,
        message: SnodeMessage,
        auth: SwarmAuth,
    ): SnodeBatchRequestInfo {
        check(message.recipient == auth.accountId.hexString) {
            "Message sent to ${message.recipient} but authenticated with ${auth.accountId.hexString}"
        }

        val params = buildAuthenticatedParameters(
            namespace = namespace,
            auth = auth,
            verificationData = { ns, t -> "${Snode.Method.SendMessage.rawValue}$ns$t" },
        ) {
            putAll(message.toJSON())
        }

        return SnodeBatchRequestInfo(
            Snode.Method.SendMessage.rawValue,
            params,
            namespace
        )
    }

    fun buildAuthenticatedUnrevokeSubKeyBatchRequest(
        groupAdminAuth: OwnedSwarmAuth,
        subAccountTokens: List<ByteArray>,
    ): SnodeBatchRequestInfo {
        val params = buildAuthenticatedParameters(
            namespace = null,
            auth = groupAdminAuth,
            verificationData = { _, t ->
                subAccountTokens.fold(
                    "${Snode.Method.UnrevokeSubAccount.rawValue}$t".toByteArray()
                ) { acc, subAccount -> acc + subAccount }
            }
        ) {
            put("unrevoke", subAccountTokens.map(Base64::encodeBytes))
        }

        return SnodeBatchRequestInfo(
            Snode.Method.UnrevokeSubAccount.rawValue,
            params,
            null
        )
    }

    fun buildAuthenticatedRevokeSubKeyBatchRequest(
        groupAdminAuth: OwnedSwarmAuth,
        subAccountTokens: List<ByteArray>,
    ): SnodeBatchRequestInfo {
        val params = buildAuthenticatedParameters(
            namespace = null,
            auth = groupAdminAuth,
            verificationData = { _, t ->
                subAccountTokens.fold(
                    "${Snode.Method.RevokeSubAccount.rawValue}$t".toByteArray()
                ) { acc, subAccount -> acc + subAccount }
            }
        ) {
            put("revoke", subAccountTokens.map(Base64::encodeBytes))
        }

        return SnodeBatchRequestInfo(
            Snode.Method.RevokeSubAccount.rawValue,
            params,
            null
        )
    }

    /**
     * Message hashes can be shared across multiple namespaces (for a single public key destination)
     * @param publicKey the destination's identity public key to delete from (05...)
     * @param ed25519PubKey the destination's ed25519 public key to delete from. Only required for user messages.
     * @param messageHashes a list of stored message hashes to delete from all namespaces on the server
     * @param required indicates that *at least one* message in the list is deleted from the server, otherwise it will return 404
     */
    fun buildAuthenticatedDeleteBatchInfo(
        auth: SwarmAuth,
        messageHashes: List<String>,
        required: Boolean = false
    ): SnodeBatchRequestInfo {
        val params = buildAuthenticatedParameters(
            namespace = null,
            auth = auth,
            verificationData = { _, _ ->
                buildString {
                    append(Snode.Method.DeleteMessage.rawValue)
                    messageHashes.forEach(this::append)
                }
            }
        ) {
            put("messages", messageHashes)
            put("required", required)
        }

        return SnodeBatchRequestInfo(
            Snode.Method.DeleteMessage.rawValue,
            params,
            null
        )
    }

    fun buildAuthenticatedRetrieveBatchRequest(
        auth: SwarmAuth,
        lastHash: String?,
        namespace: Int = 0,
        maxSize: Int? = null
    ): SnodeBatchRequestInfo {
        val params = buildAuthenticatedParameters(
            namespace = namespace,
            auth = auth,
            verificationData = { ns, t -> "${Snode.Method.Retrieve.rawValue}$ns$t" },
        ) {
            put("last_hash", lastHash.orEmpty())
            if (maxSize != null) {
                put("max_size", maxSize)
            }
        }

        return SnodeBatchRequestInfo(
            Snode.Method.Retrieve.rawValue,
            params,
            namespace
        )
    }

    fun buildAuthenticatedAlterTtlBatchRequest(
        auth: SwarmAuth,
        messageHashes: List<String>,
        newExpiry: Long,
        shorten: Boolean = false,
        extend: Boolean = false
    ): SnodeBatchRequestInfo {
        val params =
            buildAlterTtlParams(auth, messageHashes, newExpiry, extend, shorten)
        return SnodeBatchRequestInfo(
            Snode.Method.Expire.rawValue,
            params,
            null
        )
    }

    private data class RequestInfo(
        val snode: Snode,
        val publicKey: String,
        val request: SnodeBatchRequestInfo,
        val responseType: DeserializationStrategy<*>,
        val callback: SendChannel<Result<Any>>,
        val requestTime: Long = SystemClock.elapsedRealtime(),
    )

    private val batchedRequestsSender: SendChannel<RequestInfo>

    init {
        val batchRequests = Channel<RequestInfo>()
        batchedRequestsSender = batchRequests

        val batchWindowMills = 100L

        data class BatchKey(val snodeAddress: String, val publicKey: String)

        scope.launch {
            val batches = hashMapOf<BatchKey, MutableList<RequestInfo>>()

            while (true) {
                val batch = select<List<RequestInfo>?> {
                    // If we receive a request, add it to the batch
                    batchRequests.onReceive {
                        batches.getOrPut(BatchKey(it.snode.address, it.publicKey)) { mutableListOf() }.add(it)
                        null
                    }

                    // If we have anything in the batch, look for the one that is about to expire
                    // and wait for it to expire, remove it from the batches and send it for
                    // processing.
                    if (batches.isNotEmpty()) {
                        val earliestBatch = batches.minBy { it.value.first().requestTime }
                        val deadline = earliestBatch.value.first().requestTime + batchWindowMills
                        onTimeout(
                            timeMillis = (deadline - SystemClock.elapsedRealtime()).coerceAtLeast(0)
                        ) {
                            batches.remove(earliestBatch.key)
                        }
                    }
                }

                if (batch != null) {
                    launch batch@{
                        val snode = batch.first().snode
                        val responses = try {
                            getBatchResponse(
                                snode = snode,
                                publicKey = batch.first().publicKey,
                                requests = batch.map { it.request },
                                sequence = false
                            )
                        } catch (e: Exception) {
                            for (req in batch) {
                                runCatching {
                                    req.callback.send(Result.failure(e))
                                }
                            }
                            return@batch
                        }

                        // For each response, parse the result, match it with the request then send
                        // back through the request's callback.
                        for ((req, resp) in batch.zip(responses.results)) {
                            val result = runCatching {
                                if (!resp.isSuccessful) {
                                    throw BatchResponse.Error(resp)
                                }

                                MessagingModuleConfiguration.shared.json.decodeFromJsonElement(
                                    req.responseType, resp.body)!!
                            }

                            runCatching {
                                req.callback.send(result)
                            }
                        }

                        // Close all channels in the requests just in case we don't have paired up
                        // responses.
                        for (req in batch) {
                            req.callback.close()
                        }
                    }
                }
            }
        }
    }

    suspend fun <T> sendBatchRequest(
        snode: Snode,
        publicKey: String,
        request: SnodeBatchRequestInfo,
        responseType: DeserializationStrategy<T>,
    ): T {
        val callback = Channel<Result<T>>(capacity = 1)
        @Suppress("UNCHECKED_CAST")
        batchedRequestsSender.send(RequestInfo(
            snode = snode,
            publicKey = publicKey,
            request = request,
            responseType = responseType,
            callback = callback as SendChannel<Any>
        ))
        try {
            return callback.receive().getOrThrow()
        } catch (e: CancellationException) {
            // Close the channel if the coroutine is cancelled, so the batch processing won't
            // handle this one (best effort only)
            callback.close()
            throw e
        }
    }

    suspend fun sendBatchRequest(
        snode: Snode,
        publicKey: String,
        request: SnodeBatchRequestInfo,
    ): JsonElement {
        return sendBatchRequest(snode, publicKey, request, JsonElement.serializer())
    }

    suspend fun getBatchResponse(
        snode: Snode,
        publicKey: String,
        requests: List<SnodeBatchRequestInfo>,
        sequence: Boolean = false
    ): BatchResponse {
        return invokeSuspend(
            method = if (sequence) Snode.Method.Sequence else Snode.Method.Batch,
            snode = snode,
            parameters = mapOf("requests" to requests),
            responseDeserializationStrategy = BatchResponse.serializer(),
            publicKey = publicKey
        ).also { resp ->
            // If there's a unsuccessful response, go through specific logic to handle
            // potential snode errors.
            val firstError = resp.results.firstOrNull { !it.isSuccessful }
            if (firstError != null) {
                handleSnodeError(
                    statusCode = firstError.code,
                    json = if (firstError.body is JsonObject) {
                        JsonUtil.fromJson(firstError.body.toString(), Map::class.java)
                    } else {
                        null
                    },
                    snode = snode,
                    publicKey = publicKey
                )
            }
        }
    }

    fun alterTtl(
        auth: SwarmAuth,
        messageHashes: List<String>,
        newExpiry: Long,
        extend: Boolean = false,
        shorten: Boolean = false
    ): RawResponsePromise = scope.retrySuspendAsPromise(maxRetryCount) {
        val params = buildAlterTtlParams(auth, messageHashes, newExpiry, extend, shorten)
        val snode = getSingleTargetSnode(auth.accountId.hexString).await()
        invoke(Snode.Method.Expire, snode, params, auth.accountId.hexString).await()
    }

    private fun buildAlterTtlParams(
        auth: SwarmAuth,
        messageHashes: List<String>,
        newExpiry: Long,
        extend: Boolean = false,
        shorten: Boolean = false
    ): Map<String, Any> {
        val shortenOrExtend = if (extend) "extend" else if (shorten) "shorten" else ""

        return buildAuthenticatedParameters(
            namespace = null,
            auth = auth,
            verificationData = { _, _ ->
                buildString {
                    append("expire")
                    append(shortenOrExtend)
                    append(newExpiry.toString())
                    messageHashes.forEach(this::append)
                }
            }
        ) {
            this["expiry"] = newExpiry
            this["messages"] = messageHashes
            when {
                extend -> this["extend"] = true
                shorten -> this["shorten"] = true
            }
        }
    }

    fun getNetworkTime(snode: Snode): Promise<Pair<Snode, Long>, Exception> =
        invoke(Snode.Method.Info, snode, emptyMap()).map { rawResponse ->
            val timestamp = rawResponse["timestamp"] as? Long ?: -1
            snode to timestamp
        }

    /**
     * Note: After this method returns, [auth] will not be used by any of async calls and it's afe
     * for the caller to clean up the associated resources if needed.
     */
    suspend fun sendMessage(
        message: SnodeMessage,
        auth: SwarmAuth?,
        namespace: Int = 0
    ): StoreMessageResponse {
        return retryWithUniformInterval(maxRetryCount = maxRetryCount) {
            val params = if (auth != null) {
                check(auth.accountId.hexString == message.recipient) {
                    "Message sent to ${message.recipient} but authenticated with ${auth.accountId.hexString}"
                }

                val timestamp = nowWithOffset

                buildAuthenticatedParameters(
                    auth = auth,
                    namespace = namespace,
                    verificationData = { ns, t -> "${Snode.Method.SendMessage.rawValue}$ns$t" },
                    timestamp = timestamp
                ) {
                    put("sig_timestamp", timestamp)
                    putAll(message.toJSON())
                }
            } else {
                buildMap {
                    putAll(message.toJSON())
                    if (namespace != 0) {
                        put("namespace", namespace)
                    }
                }
            }

            sendBatchRequest(
                snode = getSingleTargetSnode(message.recipient).await(),
                publicKey = message.recipient,
                request = SnodeBatchRequestInfo(
                    method = Snode.Method.SendMessage.rawValue,
                    params = params,
                    namespace = namespace
                ),
                responseType = StoreMessageResponse.serializer()
            )
        }
    }
    
    suspend fun deleteMessage(publicKey: String, swarmAuth: SwarmAuth, serverHashes: List<String>) {
        retryWithUniformInterval {
            val snode = getSingleTargetSnode(publicKey).await()
            val params = buildAuthenticatedParameters(
                auth = swarmAuth,
                namespace = null,
                verificationData = { _, _ ->
                    buildString {
                        append(Snode.Method.DeleteMessage.rawValue)
                        serverHashes.forEach(this::append)
                    }
                }
            ) {
                this["messages"] = serverHashes
            }
            val rawResponse = invoke(
                Snode.Method.DeleteMessage,
                snode,
                params,
                publicKey
            ).await()

            // thie next step is to verify the nodes on our swarm and check that the message was deleted
            // on at least one of them
            val swarms = rawResponse["swarm"] as? Map<String, Any> ?: throw (Error.Generic)

            val deletedMessages = swarms.mapValuesNotNull { (hexSnodePublicKey, rawJSON) ->
                (rawJSON as? Map<String, Any>)?.let { json ->
                    val isFailed = json["failed"] as? Boolean ?: false
                    val statusCode = json[KEY_CODE] as? String
                    val reason = json["reason"] as? String

                    if (isFailed) {
                        Log.e(
                            "Loki",
                            "Failed to delete messages from: $hexSnodePublicKey due to error: $reason ($statusCode)."
                        )
                        false
                    } else {
                        // Hashes of deleted messages
                        val hashes = json["deleted"] as List<String>
                        val signature = json["signature"] as String
                        // The signature looks like ( PUBKEY_HEX || RMSG[0] || ... || RMSG[N] || DMSG[0] || ... || DMSG[M] )
                        val message = sequenceOf(swarmAuth.accountId.hexString)
                            .plus(serverHashes)
                            .plus(hashes)
                            .toByteArray()

                        ED25519.verify(
                            ed25519PublicKey = Hex.fromStringCondensed(hexSnodePublicKey),
                            signature = Base64.decode(signature),
                            message = message,
                        )
                    }
                }
            }

            // if all the nodes returned false (the message was not deleted) then we consider this a failed scenario
            if (deletedMessages.entries.all { !it.value }) throw (Error.Generic)
        }
    }
    
    // Parsing
    private fun parseSnodes(rawResponse: Any): List<Snode> =
        (rawResponse as? Map<*, *>)
            ?.run { get("snodes") as? List<*> }
            ?.asSequence()
            ?.mapNotNull { it as? Map<*, *> }
            ?.mapNotNull {
                createSnode(
                    address = it["ip"] as? String,
                    port = (it["port"] as? String)?.toInt(),
                    ed25519Key = it[KEY_ED25519] as? String,
                    x25519Key = it[KEY_X25519] as? String
                ).apply {
                    if (this == null) Log.d(
                        "Loki",
                        "Failed to parse snode from: ${it.prettifiedDescription()}."
                    )
                }
            }?.toList() ?: listOf<Snode>().also {
            Log.d(
                "Loki",
                "Failed to parse snodes from: ${rawResponse.prettifiedDescription()}."
            )
        }

    fun deleteAllMessages(auth: SwarmAuth): Promise<Map<String, Boolean>, Exception> =
        scope.retrySuspendAsPromise(maxRetryCount) {
            val snode = getSingleTargetSnode(auth.accountId.hexString).await()
            val timestamp = MessagingModuleConfiguration.shared.clock.waitForNetworkAdjustedTime()

            val params = buildAuthenticatedParameters(
                auth = auth,
                namespace = null,
                verificationData = { _, t -> "${Snode.Method.DeleteAll.rawValue}all$t" },
                timestamp = timestamp
            ) {
                put("namespace", "all")
            }

            val rawResponse = invoke(Snode.Method.DeleteAll, snode, params, auth.accountId.hexString).await()
            parseDeletions(
                auth.accountId.hexString,
                timestamp,
                rawResponse
            )
        }


    @Suppress("UNCHECKED_CAST")
    private fun parseDeletions(userPublicKey: String, timestamp: Long, rawResponse: RawResponse): Map<String, Boolean> =
        (rawResponse["swarm"] as? Map<String, Any>)?.mapValuesNotNull { (hexSnodePublicKey, rawJSON) ->
            val json = rawJSON as? Map<String, Any> ?: return@mapValuesNotNull null
            if (json["failed"] as? Boolean == true) {
                val reason = json["reason"] as? String
                val statusCode = json[KEY_CODE] as? String
                Log.e("Loki", "Failed to delete all messages from: $hexSnodePublicKey due to error: $reason ($statusCode).")
                false
            } else {
                val hashes = (json["deleted"] as Map<String,List<String>>).flatMap { (_, hashes) -> hashes }.sorted() // Hashes of deleted messages
                val signature = json["signature"] as String
                // The signature looks like ( PUBKEY_HEX || TIMESTAMP || DELETEDHASH[0] || ... || DELETEDHASH[N] )
                val message = sequenceOf(userPublicKey, "$timestamp").plus(hashes).toByteArray()
                ED25519.verify(
                    ed25519PublicKey = Hex.fromStringCondensed(hexSnodePublicKey),
                    signature = Base64.decode(signature),
                    message = message,
                )
            }
        } ?: mapOf()

    // endregion

    // Error Handling
    internal fun handleSnodeError(statusCode: Int, json: Map<*, *>?, snode: Snode, publicKey: String? = null): Throwable? = runCatching {
        fun handleBadSnode() {
            val oldFailureCount = snodeFailureCount[snode] ?: 0
            val newFailureCount = oldFailureCount + 1
            snodeFailureCount[snode] = newFailureCount
            Log.d("Loki", "Couldn't reach snode at $snode; setting failure count to $newFailureCount.")
            if (newFailureCount >= snodeFailureThreshold) {
                Log.d("Loki", "Failure threshold reached for: $snode; dropping it.")
                publicKey?.let { dropSnodeFromSwarmIfNeeded(snode, it) }
                snodePool = (snodePool - snode).also { Log.d("Loki", "Snode pool count: ${it.count()}.") }
                snodeFailureCount -= snode
            }
        }
        when (statusCode) {
            // Usually indicates that the snode isn't up to date
            400, 500, 502, 503 -> handleBadSnode()
            406 -> {
                Log.d("Loki", "The user's clock is out of sync with the service node network.")
                throw Error.ClockOutOfSync
            }
            421 -> {
                // The snode isn't associated with the given public key anymore
                if (publicKey == null) Log.d("Loki", "Got a 421 without an associated public key.")
                else json?.let(::parseSnodes)
                    ?.takeIf { it.isNotEmpty() }
                    ?.let { database.setSwarm(publicKey, it.toSet()) }
                    ?: dropSnodeFromSwarmIfNeeded(snode, publicKey).also { Log.d("Loki", "Invalidating swarm for: $publicKey.") }
            }
            404 -> {
                Log.d("Loki", "404, probably no file found")
                throw Error.Generic
            }
            else -> {
                handleBadSnode()
                Log.d("Loki", "Unhandled response code: ${statusCode}.")
                throw Error.Generic
            }
        }
    }.exceptionOrNull()
}

// Type Aliases
typealias RawResponse = Map<*, *>
typealias RawResponsePromise = Promise<RawResponse, Exception>
