package org.session.libsession.messaging.sending_receiving

import network.loki.messenger.libsession_util.util.BlindKeyAPI
import network.loki.messenger.libsession_util.util.KeyPair
import org.session.libsession.database.StorageProtocol
import org.session.libsession.messaging.messages.Message
import org.session.libsession.messaging.messages.control.CallMessage
import org.session.libsession.messaging.messages.control.DataExtractionNotification
import org.session.libsession.messaging.messages.control.ExpirationTimerUpdate
import org.session.libsession.messaging.messages.control.GroupUpdated
import org.session.libsession.messaging.messages.control.MessageRequestResponse
import org.session.libsession.messaging.messages.control.ReadReceipt
import org.session.libsession.messaging.messages.control.TypingIndicator
import org.session.libsession.messaging.messages.control.UnsendRequest
import org.session.libsession.messaging.messages.visible.VisibleMessage
import org.session.libsession.snode.SnodeAPI
import org.session.libsignal.crypto.PushTransportDetails
import org.session.libsignal.protos.SignalServiceProtos
import org.session.libsignal.protos.SignalServiceProtos.Envelope
import org.session.libsignal.utilities.AccountId
import org.session.libsignal.utilities.Hex
import org.session.libsignal.utilities.IdPrefix
import org.session.libsignal.utilities.Log
import java.util.concurrent.TimeUnit
import javax.inject.Inject
import javax.inject.Singleton
import kotlin.math.abs

@Deprecated("This class only exists so the old BatchMessageReceiver can function. New code should use MessageHandler directly.")
@Singleton
class MessageReceiver @Inject constructor(
    private val storage: StorageProtocol,
) {

    internal sealed class Error(message: String) : Exception(message) {
        object DuplicateMessage: Error("Duplicate message.")
        object InvalidMessage: Error("Invalid message.")
        object UnknownMessage: Error("Unknown message type.")
        object UnknownEnvelopeType: Error("Unknown envelope type.")
        object DecryptionFailed : Exception("Couldn't decrypt message.")
        object InvalidSignature: Error("Invalid message signature.")
        object NoData: Error("Received an empty envelope.")
        object SenderBlocked: Error("Received a message from a blocked user.")
        object NoThread: Error("Couldn't find thread for message.")
        object SelfSend: Error("Message addressed at self.")
        object InvalidGroupPublicKey: Error("Invalid group public key.")
        object NoGroupThread: Error("No thread exists for this group.")
        object NoGroupKeyPair: Error("Missing group key pair.")
        object NoUserED25519KeyPair : Error("Couldn't find user ED25519 key pair.")
        object ExpiredMessage: Error("Message has already expired, prevent adding")

        internal val isRetryable: Boolean = when (this) {
            is DuplicateMessage, is InvalidMessage, is UnknownMessage,
            is UnknownEnvelopeType, is InvalidSignature, is NoData,
            is SenderBlocked, is SelfSend,
            is ExpiredMessage, is NoGroupThread -> false
            else -> true
        }
    }

    internal fun parse(
        data: ByteArray,
        openGroupServerID: Long?,
        isOutgoing: Boolean? = null,
        otherBlindedPublicKey: String? = null,
        openGroupPublicKey: String? = null,
        currentClosedGroups: Set<String>?,
        closedGroupSessionId: String? = null,
    ): Pair<Message, SignalServiceProtos.Content> {
        val userPublicKey = storage.getUserPublicKey()
        val isOpenGroupMessage = (openGroupServerID != null)
        var plaintext: ByteArray? = null
        var sender: String? = null
        var groupPublicKey: String? = null
        // Parse the envelope
        val envelope = Envelope.parseFrom(data) ?: throw Error.InvalidMessage
        // Decrypt the contents
        val envelopeContent = envelope.content ?: run {
            throw Error.NoData
        }

        if (isOpenGroupMessage) {
            plaintext = envelopeContent.toByteArray()
            sender = envelope.source
        } else {
            when (envelope.type) {
                SignalServiceProtos.Envelope.Type.SESSION_MESSAGE -> {
                    if (IdPrefix.fromValue(envelope.source)?.isBlinded() == true) {
                        openGroupPublicKey ?: throw Error.InvalidGroupPublicKey
                        otherBlindedPublicKey ?: throw Error.DecryptionFailed
                        val decryptionResult = MessageDecrypter.decryptBlinded(
                            envelopeContent.toByteArray(),
                            isOutgoing ?: false,
                            otherBlindedPublicKey,
                            openGroupPublicKey
                        )
                        plaintext = decryptionResult.first
                        sender = decryptionResult.second
                    } else {
                        val userX25519KeyPair = storage.getUserX25519KeyPair()
                        val decryptionResult = MessageDecrypter.decrypt(envelopeContent.toByteArray(), userX25519KeyPair)
                        plaintext = decryptionResult.first
                        sender = decryptionResult.second
                    }
                }
                SignalServiceProtos.Envelope.Type.CLOSED_GROUP_MESSAGE -> {
                    val hexEncodedGroupPublicKey = closedGroupSessionId ?: envelope.source
                    val sessionId = AccountId(hexEncodedGroupPublicKey)
                    if (sessionId.prefix == IdPrefix.GROUP) {
                        plaintext = envelopeContent.toByteArray()
                        sender = envelope.source
                        groupPublicKey = hexEncodedGroupPublicKey
                    } else {
                        if (!storage.isLegacyClosedGroup(hexEncodedGroupPublicKey)) {
                            throw Error.InvalidGroupPublicKey
                        }
                        val encryptionKeyPairs = storage.getClosedGroupEncryptionKeyPairs(hexEncodedGroupPublicKey)
                        if (encryptionKeyPairs.isEmpty()) {
                            throw Error.NoGroupKeyPair
                        }
                        // Loop through all known group key pairs in reverse order (i.e. try the latest key pair first (which'll more than
                        // likely be the one we want) but try older ones in case that didn't work)
                        var encryptionKeyPair = encryptionKeyPairs.removeAt(encryptionKeyPairs.lastIndex)
                        fun decrypt() {
                            try {
                                val decryptionResult = MessageDecrypter.decrypt(envelopeContent.toByteArray(),
                                    KeyPair(
                                        pubKey = encryptionKeyPair.publicKey.serialize(),
                                        secretKey = encryptionKeyPair.privateKey.serialize()
                                    ))
                                plaintext = decryptionResult.first
                                sender = decryptionResult.second
                            } catch (e: Exception) {
                                if (encryptionKeyPairs.isNotEmpty()) {
                                    encryptionKeyPair = encryptionKeyPairs.removeAt(encryptionKeyPairs.lastIndex)
                                    decrypt()
                                } else {
                                    Log.e("Loki", "Failed to decrypt group message", e)
                                    throw e
                                }
                            }
                        }
                        groupPublicKey = hexEncodedGroupPublicKey
                        decrypt()
                    }
                }
                else -> {
                    throw Error.UnknownEnvelopeType
                }
            }
        }
        // Parse the proto
        val proto = SignalServiceProtos.Content.parseFrom(PushTransportDetails.getStrippedPaddingMessageBody(plaintext))

        // Verify the signature timestamp inside the content is the same as in envelope.
        // If the message is from an open group, 6 hours of difference is allowed.
        if (proto.hasSigTimestampMs()) {
            val isCommunityOrCommunityInbox = openGroupServerID != null || otherBlindedPublicKey != null

            if (
                (isCommunityOrCommunityInbox && abs(proto.sigTimestampMs - envelope.timestampMs) > TimeUnit.HOURS.toMillis(6)) ||
                (!isCommunityOrCommunityInbox && proto.sigTimestampMs != envelope.timestampMs)
            ) {
                throw Error.InvalidSignature
            }
        }

        // Parse the message
        val message: Message = ReadReceipt.fromProto(proto) ?:
            TypingIndicator.fromProto(proto) ?:
            DataExtractionNotification.fromProto(proto) ?:
            ExpirationTimerUpdate.fromProto(proto, closedGroupSessionId != null) ?:
            UnsendRequest.fromProto(proto) ?:
            MessageRequestResponse.fromProto(proto) ?:
            CallMessage.fromProto(proto) ?:
            GroupUpdated.fromProto(proto) ?:
            VisibleMessage.fromProto(proto) ?: throw Error.UnknownMessage

        // Don't process the envelope any further if the sender is blocked (still visible in community chats)
        if (!isOpenGroupMessage && isBlocked(sender!!) && message.shouldDiscardIfBlocked()) {
            throw Error.SenderBlocked
        }
        val isUserBlindedSender = sender == openGroupPublicKey?.let {
            BlindKeyAPI.blind15KeyPairOrNull(
                ed25519SecretKey = storage.getUserED25519KeyPair()!!.secretKey.data,
                serverPubKey = Hex.fromStringCondensed(it),
            )
        }?.let { AccountId(IdPrefix.BLINDED, it.pubKey.data).hexString }
        val isUserSender = sender == userPublicKey

        if (isUserSender || isUserBlindedSender) {
            // Ignore self send if needed
            if (!message.isSelfSendValid) throw Error.SelfSend
            message.isSenderSelf = true
        }
        // Guard against control messages in open groups
        if (isOpenGroupMessage && message !is VisibleMessage) {
            throw Error.InvalidMessage
        }
        // Finish parsing
        message.sender = sender
        message.recipient = userPublicKey
        message.sentTimestamp = envelope.timestampMs
        message.receivedTimestamp = if (envelope.hasServerTimestampMs()) envelope.serverTimestampMs else SnodeAPI.nowWithOffset
        message.groupPublicKey = groupPublicKey
        message.openGroupServerMessageID = openGroupServerID
        // Validate
        var isValid = message.isValid()
        if (message is VisibleMessage && !isValid && proto.dataMessage.attachmentsCount != 0) { isValid = true }
        if (!isValid) {
            throw Error.InvalidMessage
        }
        // If the message failed to process the first time around we retry it later (if the error is retryable). In this case the timestamp
        // will already be in the database but we don't want to treat the message as a duplicate. The isRetry flag is a simple workaround
        // for this issue.
        if (groupPublicKey != null && groupPublicKey !in (currentClosedGroups ?: emptySet()) && IdPrefix.fromValue(groupPublicKey) != IdPrefix.GROUP) {
            throw Error.NoGroupThread
        }
        if (storage.isDuplicateMessage(envelope.timestampMs)) { throw Error.DuplicateMessage }
        storage.addReceivedMessageTimestamp(envelope.timestampMs)
        // Return
        return Pair(message, proto)
    }

}