/**
 * SPDX-License-Identifier: GPL-3.0-or-later
 * SPDX-FileCopyrightText: Copyright (C) 2025 Gijs van Tulder
 */

import { GridEdge } from "./GridEdge";
import { GridVertex } from "./GridVertex";
import { BBox, Point } from "../geom/math";
import { Polygon } from "../geom/Polygon";
import { Shape } from "./Shape";
import { SourceGrid, SourcePoint } from "./SourceGrid";
import * as zod from "zod/v4-mini";

export type TileColor = string;
export type TileColors = readonly TileColor[];

export const enum TileType {
    /**
     * A standard Tile with segments and colors.
     */
    Normal,
    /**
     * A PlaceholderTile without segments or colors,
     * can overlap other placeholder tiles.
     */
    Placeholder,
}

export const Tile_S = zod.object({
    shape: zod.int().check(zod.nonnegative()),
    polygon: Polygon.codec.def.in,
    segments: zod.optional(zod.array(Polygon.codec.def.in)),
    sourcePoint: zod.optional(zod.unknown()),
    placeholder: zod.boolean(),
    colors: zod.optional(zod.readonly(zod.array(zod.optional(zod.string())))),
});
export type Tile_S = zod.infer<typeof Tile_S>;

/**
 * A TileSegment represents a segment of a tile
 * (usually a triangle) and has a color.
 */
export class TileSegment {
    /**
     * The tile of this segment.
     */
    tile: Tile;
    /**
     * The edge index of the edge connected to this segment.
     * See tile.edge[index].
     */
    index: number;
    /**
     * The polygon of this segment.
     */
    polygon: Polygon;
    /**
     * The color of this segment.
     */
    color?: TileColor;

    /**
     * Creates a new tile segment.
     * @param tile the tile this segment is part of
     * @param index the edge index of the edge that
     *              connects this segment (tile.edge[index])
     * @param polygon the polygon shape of this segment
     * @param color the color of the new segment
     */
    constructor(
        tile: Tile,
        index: number,
        polygon: Polygon,
        color?: TileColor,
    ) {
        this.tile = tile;
        this.index = index;
        this.polygon = polygon;
        this.color = color;
    }

    /**
     * Returns the neighboring tile segments of this segment.
     *
     * If internal is true, this includes neighboring segments
     * inside and outside the tile.
     * If internal is false, only the neighbor on the external edge
     * is returned.
     *
     * If missing is true, the list will include null
     * values for any edges of the segment without a neighbor.
     * If missing is false, the list will only include
     * existing segments.
     */
    getNeighbors(internal?: boolean): TileSegment[];
    getNeighbors(internal: boolean, missing: false): TileSegment[];
    getNeighbors(internal: boolean, missing: true): (TileSegment | null)[];
    getNeighbors(
        internal?: boolean,
        missing?: boolean,
    ): (TileSegment | null)[] {
        const tile = this.tile;
        if (!tile.segments) return [];
        const n = tile.segments.length;
        const neighbors: (TileSegment | null)[] = [];
        if (internal) {
            neighbors.push(
                tile.segments[(this.index + n - 1) % n],
                tile.segments[(this.index + 1) % n],
            );
        }
        const edge = tile.edges[this.index];
        if (edge.tileA == tile && edge.tileB && edge.tileB.segments) {
            neighbors.push(edge.tileB.segments[edge.edgeIdxB!]);
        } else if (edge.tileB == tile && edge.tileA && edge.tileA.segments) {
            neighbors.push(edge.tileA.segments[edge.edgeIdxA!]);
        } else if (missing) {
            neighbors.push(null);
        }
        return neighbors;
    }
}

/**
 * A Tile represents a tile or placeholder on the grid.
 */
export class Tile {
    /**
     * The type of this tile.
     */
    tileType: TileType;
    /**
     * The source point connected to this tile.
     */
    sourcePoint?: SourcePoint;
    /**
     * The vertices forming the outline of this tile,
     * in clockwise order.
     */
    vertices!: GridVertex[];
    /**
     * The edges of this tile: clockwise order, with
     * edges[i] = vertices[i] -- vertices[i + 1].
     */
    edges!: GridEdge[];
    /**
     * The shape of this tile.
     */
    shape: Shape;
    /**
     * The polygon in grid coordinates of this tile.
     */
    polygon: Polygon;
    /**
     * The segments of this tile, or null.
     * segments[i] is connected to edge[i].
     */
    segments?: TileSegment[];
    /**
     * The area of the tile.
     */
    area: number;
    /**
     * The bounding box of the tile.
     */
    bbox: BBox;
    /**
     * The centroid coordinates of the tile.
     */
    centroid: Point;

    onUpdateColor?: (tile: Tile) => void;

    /**
     * Cache of the segment colors.
     */
    private _colors?: readonly TileColor[];

    /**
     * Constructs a new tile of the given shape and polygon.
     */
    constructor(
        shape: Shape,
        polygon: Polygon,
        segments?: Polygon[] | null,
        sourcePoint?: SourcePoint,
        tileType = TileType.Normal,
    ) {
        // basic properties
        this.tileType = tileType;
        this.sourcePoint = sourcePoint;
        this.shape = shape;
        this.polygon = polygon;
        // precompute statistics
        this.area = polygon.area;
        this.bbox = polygon.bbox;
        this.centroid = polygon.centroid;
        // generate segments
        if (segments) {
            this.segments = segments.map((s, i) => new TileSegment(this, i, s));
        }
    }

    /**
     * Serializes the tile.
     *
     * @param shapeMap the sequence of shapes to map shapes to indices
     * @returns the serialized tile
     */
    save(shapeMap: readonly Shape[]): Tile_S {
        return {
            shape: shapeMap.indexOf(this.shape),
            polygon: this.polygon.save(),
            segments: this.segments?.map((segment) => segment.polygon.save()),
            sourcePoint: this.sourcePoint?.save(),
            placeholder: this.tileType === TileType.Placeholder,
            colors: this.colors,
        };
    }

    /**
     * Restores a serialized tile.
     *
     * @param data the serialized tile
     * @param shapeMap the sequence of shapes to map indices to shapes
     * @returns the restored tile
     */
    static restore(
        data: unknown,
        shapeMap: readonly Shape[],
        sourceGrid?: SourceGrid,
    ): Tile {
        const d = Tile_S.parse(data);
        if (d.placeholder) {
            return new PlaceholderTile(
                shapeMap[d.shape],
                Polygon.restore(d.polygon),
                sourceGrid?.restorePoint(d.sourcePoint),
            );
        } else {
            const tile = new Tile(
                shapeMap[d.shape],
                Polygon.restore(d.polygon),
                d.segments?.map((segment) => Polygon.restore(segment)),
                sourceGrid?.restorePoint(d.sourcePoint),
            );
            if (d.colors && d.colors.every((c) => c)) {
                tile.colors = d.colors as TileColors;
            }
            return tile;
        }
    }

    /**
     * Returns the neighbor tiles of this tile.
     * (This does not return placeholders.)
     */
    get neighbors(): ReadonlySet<Tile> {
        const neighbors = new Set<Tile>();
        for (const edge of this.edges) {
            if (edge.tileA && edge.tileA !== this) {
                neighbors.add(edge.tileA);
            } else if (edge.tileB && edge.tileB !== this) {
                neighbors.add(edge.tileB);
            }
        }
        return neighbors;
    }

    /**
     * Returns the colors of the tile segments,
     * or undefined if no segments are defined.
     */
    get colors(): readonly TileColor[] | undefined {
        if (!this.segments || this.segments.length == 0) return undefined;
        return (this._colors ||= this.segments.map((s) => s.color!));
    }

    /**
     * Updates the colors of the tile segments.
     * Triggers a GridEventType.UpdateTileColors event.
     */
    set colors(colors: readonly TileColor[] | TileColor) {
        this._colors = undefined;
        const segments = this.segments;
        if (!segments) return;
        for (let i = 0; i < segments.length; i++) {
            segments[i].color = colors instanceof Array ? colors[i] : colors;
        }
        if (this.onUpdateColor) {
            this.onUpdateColor(this);
        }
    }
}

/**
 * A placeholder tile is a special tile that does not have colors
 * and can overlap other placeholders.
 */
export class PlaceholderTile extends Tile {
    constructor(shape: Shape, polygon: Polygon, sourcePoint?: SourcePoint) {
        super(shape, polygon, null, sourcePoint, TileType.Placeholder);
    }
}
