#!/usr/bin/env node
'use strict';

const commander = require('commander');
const str = require('./shared/pseudolocale.ffbf12a6.cjs');
const fs = require('node:fs/promises');
const path = require('node:path');

function _interopDefaultCompat (e) { return e && typeof e === 'object' && 'default' in e ? e.default : e; }

const fs__default = /*#__PURE__*/_interopDefaultCompat(fs);
const path__default = /*#__PURE__*/_interopDefaultCompat(path);

const version = "2.1.0";

function makeProgram() {
  const program = new commander.Command();
  program.version(version).option("-d, --delimiter <val>", "sets the token delimiter (default: '%')").option(
    "-S, --startDelimiter <val>",
    "sets the start token delimiter (default: delimiter)"
  ).option(
    "-E, --endDelimiter <val>",
    "sets the end token delimiter (default: delimiter)"
  ).option("-p, --prepend <val>", "sets the string start tag (default: '[!!')").option("-a, --append <val>", "sets the string end tag (default: '!!]')").option(
    "-e, --extend <amount>",
    "sets the padding percentage (default: '0')",
    parseFloat
  ).option(
    "-o, --override <char>",
    "replaces all characters with specified character"
  ).option("-s, --string <str>", "string to pseudolocalize").option("-r, --readFile <path>", "path to file to pseudolocalize").option("-w, --writeFile <path>", "path of file to write results to");
  program.on("--help", () => {
    console.log("  Examples:");
    console.log("");
    console.log("  Custom start and end tags");
    console.log("    $ pseudolocale -p '[@@' -a '@@]' -s 'test'");
    console.log(
      "    > " + str.str("test", { prepend: "[@@", append: "@@]" })
    );
    console.log("");
    const overrideOptions = {
      override: "_",
      prepend: "",
      append: ""
    };
    console.log(
      "  Replace strings with underscore to spot unlocalized strings"
    );
    console.log("    $ pseudolocale -p '' -a '' -o '_' -s 'test'");
    console.log("    > " + str.str("test", overrideOptions));
    console.log("");
    console.log("  Extend strings to ensure space for localization");
    console.log("    $ pseudolocale -e 0.3 -s 'test string'");
    console.log("    > " + str.str("test string", { extend: 0.3 }));
    console.log("");
    console.log("  Pseudolocalize all strings in a JSON file");
    console.log("    $ pseudolocale -r example.json -w example-pseudo.json");
    console.log("");
  }).action(() => command(program));
  return program;
}
async function command(program) {
  const cliOptions = program.opts();
  if (!cliOptions.string && !cliOptions.readFile) {
    console.log("  Either a string (-s) or a file (-r) must be specified.");
    program.help();
  }
  if (cliOptions.string) {
    console.log(str.str(cliOptions.string, cliOptions));
    return;
  }
  try {
    const data = await fs__default.readFile(cliOptions.readFile, { encoding: "utf8" });
    const result = JSON.stringify(
      JSON.parse(data),
      (key, value) => {
        if (typeof value === "string") {
          return str.str(value, cliOptions);
        }
        return value;
      },
      2
    );
    const dir = path__default.dirname(cliOptions.readFile);
    const ext = path__default.extname(cliOptions.readFile);
    const base = path__default.basename(cliOptions.readFile, ext);
    const out = cliOptions.writeFile || path__default.join(dir, base + "-pseudo" + ext);
    try {
      await fs__default.mkdir(path__default.dirname(out), { recursive: true });
      await fs__default.writeFile(out, result);
    } catch (e) {
      console.error("  Unable to write file [" + out + "].");
      return;
    }
  } catch (e) {
    console.error("  Unable to read file [" + cliOptions.readFile + "].");
    return;
  }
}

const program = makeProgram();
program.parseAsync(process.argv);
