"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.compileLocale = compileLocale;
const catalog_1 = require("../catalog");
const picocolors_1 = __importDefault(require("picocolors"));
const getCatalogs_1 = require("../catalog/getCatalogs");
const ProgramExit_1 = require("../ProgramExit");
const compile_1 = require("../compile");
const normalize_path_1 = __importDefault(require("normalize-path"));
const path_1 = __importDefault(require("path"));
const messages_1 = require("../messages");
const getTranslationsForCatalog_1 = require("../catalog/getTranslationsForCatalog");
async function compileLocale(catalogs, locale, options, config, doMerge, logger) {
    let mergedCatalogs = {};
    for (const catalog of catalogs) {
        const { messages, missing: missingMessages } = await (0, getTranslationsForCatalog_1.getTranslationsForCatalog)(catalog, locale, {
            fallbackLocales: config.fallbackLocales,
            sourceLocale: config.sourceLocale,
        });
        if (!options.allowEmpty &&
            locale !== config.pseudoLocale &&
            missingMessages.length > 0) {
            logger.error(picocolors_1.default.red(`Error: Failed to compile catalog for locale ${picocolors_1.default.bold(locale)}!`));
            if (options.verbose) {
                logger.error(picocolors_1.default.red("Missing translations:"));
                missingMessages.forEach((missing) => {
                    const source = missing.source || missing.source === missing.id
                        ? `: (${missing.source})`
                        : "";
                    logger.error(`${missing.id}${source}`);
                });
            }
            else {
                logger.error(picocolors_1.default.red(`Missing ${missingMessages.length} translation(s)`));
            }
            logger.error();
            throw new ProgramExit_1.ProgramExit();
        }
        if (doMerge) {
            mergedCatalogs = Object.assign(Object.assign({}, mergedCatalogs), messages);
        }
        else {
            if (!(await compileAndWrite(locale, config, options, catalog.path, messages, logger))) {
                throw new ProgramExit_1.ProgramExit();
            }
        }
    }
    if (doMerge) {
        const result = await compileAndWrite(locale, config, options, await (0, getCatalogs_1.getMergedCatalogPath)(config), mergedCatalogs, logger);
        if (!result) {
            throw new ProgramExit_1.ProgramExit();
        }
    }
}
async function compileAndWrite(locale, config, options, writePath, messages, logger) {
    const namespace = options.typescript
        ? "ts"
        : options.namespace || config.compileNamespace;
    const { source: compiledCatalog, errors } = (0, compile_1.createCompiledCatalog)(locale, messages, {
        strict: false,
        namespace,
        outputPrefix: options.outputPrefix,
        pseudoLocale: config.pseudoLocale,
        compilerBabelOptions: config.compilerBabelOptions,
    });
    if (errors.length) {
        let message = (0, messages_1.createCompilationErrorMessage)(locale, errors);
        if (options.failOnCompileError) {
            message += `These errors fail command execution because \`--strict\` option passed`;
            logger.error(picocolors_1.default.red(message));
            return false;
        }
        else {
            message += `You can fail command execution on these errors by passing \`--strict\` option`;
            logger.error(picocolors_1.default.red(message));
        }
    }
    let compiledPath = await (0, catalog_1.writeCompiled)(writePath, locale, compiledCatalog, namespace);
    compiledPath = (0, normalize_path_1.default)(path_1.default.relative(config.rootDir, compiledPath));
    options.verbose && logger.error(picocolors_1.default.green(`${locale} ⇒ ${compiledPath}`));
    return true;
}
