"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractFromFiles = extractFromFiles;
exports.mergeExtractedMessage = mergeExtractedMessage;
exports.extractFromFilesWithWorkerPool = extractFromFilesWithWorkerPool;
const picocolors_1 = __importDefault(require("picocolors"));
const path_1 = __importDefault(require("path"));
const extractors_1 = __importDefault(require("../extractors"));
const utils_1 = require("../utils");
function mergePlaceholders(prev, next) {
    const res = Object.assign({}, prev);
    // Handle case where next is null or undefined
    if (!next)
        return res;
    Object.entries(next).forEach(([key, value]) => {
        if (!res[key]) {
            res[key] = [];
        }
        if (!res[key].includes(value)) {
            res[key].push(value);
        }
        res[key].sort();
    });
    return res;
}
async function extractFromFiles(paths, config) {
    const messages = {};
    let catalogSuccess = true;
    for (const filename of paths) {
        const fileSuccess = await (0, extractors_1.default)(filename, (next) => {
            mergeExtractedMessage(next, messages, config);
        }, config);
        catalogSuccess && (catalogSuccess = fileSuccess);
    }
    if (!catalogSuccess)
        return undefined;
    return messages;
}
function mergeExtractedMessage(next, messages, config) {
    var _a;
    if (!messages[next.id]) {
        messages[next.id] = {
            message: next.message,
            context: next.context,
            placeholders: {},
            comments: [],
            origin: [],
        };
    }
    const prev = messages[next.id];
    // there might be a case when filename was not mapped from sourcemaps
    const filename = next.origin[0]
        ? path_1.default.relative(config.rootDir, next.origin[0]).replace(/\\/g, "/")
        : "";
    const origin = [filename, next.origin[1]];
    if (prev.message && next.message && prev.message !== next.message) {
        throw new Error(`Encountered different default translations for message ${picocolors_1.default.yellow(next.id)}` +
            `\n${picocolors_1.default.yellow((0, utils_1.prettyOrigin)(prev.origin))} ${prev.message}` +
            `\n${picocolors_1.default.yellow((0, utils_1.prettyOrigin)([origin]))} ${next.message}`);
    }
    messages[next.id] = Object.assign(Object.assign({}, prev), { message: (_a = prev.message) !== null && _a !== void 0 ? _a : next.message, comments: next.comment
            ? [...prev.comments, next.comment].sort()
            : prev.comments, origin: [...prev.origin, [filename, next.origin[1]]].sort((a, b) => a[0].localeCompare(b[0])), placeholders: mergePlaceholders(prev.placeholders, next.placeholders) });
}
async function extractFromFilesWithWorkerPool(workerPool, paths, config) {
    const messages = {};
    let catalogSuccess = true;
    if (!config.resolvedConfigPath) {
        throw new Error("Multithreading is only supported when lingui config loaded from file system, not passed by API");
    }
    await Promise.all(paths.map((filename) => workerPool.queue(async (worker) => {
        const result = await worker(filename, config.resolvedConfigPath);
        if (!result.success) {
            catalogSuccess = false;
        }
        else {
            result.messages.forEach((message) => {
                mergeExtractedMessage(message, messages, config);
            });
        }
    })));
    if (!catalogSuccess)
        return undefined;
    return messages;
}
