@file:Suppress("FunctionNaming") // Composable functions use PascalCase

package net.turtton.ytalarm.navigation

import android.util.Log
import androidx.compose.material3.SnackbarHostState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableLongStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavGraphBuilder
import androidx.navigation.NavHostController
import androidx.navigation.NavType
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.navArgument
import androidx.navigation.navDeepLink
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import net.turtton.ytalarm.R
import net.turtton.ytalarm.YtApplication
import net.turtton.ytalarm.database.structure.Playlist
import net.turtton.ytalarm.ui.compose.dialogs.DisplayData
import net.turtton.ytalarm.ui.compose.dialogs.DisplayDataThumbnail
import net.turtton.ytalarm.ui.compose.dialogs.MultiChoiceVideoDialog
import net.turtton.ytalarm.ui.compose.dialogs.UrlInputDialog
import net.turtton.ytalarm.ui.compose.screens.AboutPageScreen
import net.turtton.ytalarm.ui.compose.screens.AlarmListScreen
import net.turtton.ytalarm.ui.compose.screens.AllVideosScreen
import net.turtton.ytalarm.ui.compose.screens.PlaylistScreen
import net.turtton.ytalarm.ui.compose.screens.SettingsScreen
import net.turtton.ytalarm.ui.compose.screens.VideoListScreen
import net.turtton.ytalarm.ui.compose.screens.VideoPlayerScreen
import net.turtton.ytalarm.util.extensions.createImportingPlaylist
import net.turtton.ytalarm.util.extensions.updateThumbnail
import net.turtton.ytalarm.viewmodel.PlaylistViewModel
import net.turtton.ytalarm.viewmodel.PlaylistViewModelFactory
import net.turtton.ytalarm.viewmodel.VideoViewModel
import net.turtton.ytalarm.viewmodel.VideoViewModelFactory
import net.turtton.ytalarm.worker.VideoInfoDownloadWorker

/**
 * YtAlarmアプリのNavigation Graph
 *
 * @param navController ナビゲーション制御用のコントローラー
 * @param onOpenDrawer ドロワーを開くコールバック
 * @param modifier Composableに適用するModifier
 * @param startDestination 初期表示画面のルート
 */
@Composable
fun YtAlarmNavGraph(
    navController: NavHostController,
    onOpenDrawer: () -> Unit,
    modifier: Modifier = Modifier,
    startDestination: String = YtAlarmDestination.ALARM_LIST
) {
    NavHost(
        navController = navController,
        startDestination = startDestination,
        modifier = modifier
    ) {
        alarmListScreen(navController, onOpenDrawer)
        playlistScreen(navController, onOpenDrawer)
        videoListScreen(navController)
        allVideosScreen(navController, onOpenDrawer)
        videoPlayerScreen(navController)
        aboutScreen(navController)
        settingsScreen(navController)
    }
}

/**
 * アラーム一覧画面のルート定義
 *
 * Deep link（ytalarm://alarm/{alarmId}）にも対応し、
 * 指定されたアラームIDでボトムシートを開く
 */
@Suppress("UnusedParameter") // navController reserved for potential future use with deep links
private fun NavGraphBuilder.alarmListScreen(
    navController: NavHostController,
    onOpenDrawer: () -> Unit
) {
    composable(
        route = YtAlarmDestination.ALARM_LIST,
        deepLinks = listOf(
            navDeepLink { uriPattern = "ytalarm://alarm/{alarmId}" }
        ),
        arguments = listOf(
            navArgument("alarmId") {
                type = NavType.LongType
                defaultValue = 0L // 0Lはボトムシートを開かない
            }
        )
    ) { backStackEntry ->
        val alarmId = backStackEntry.arguments?.getLong("alarmId") ?: 0L
        val context = LocalContext.current

        // Pre-fetch string resource for use in lambda
        val errorInvalidAlarmId = stringResource(R.string.error_invalid_alarm_id)

        // 不正なIDのエラー処理
        // 0はボトムシートを開かない（通常のアラーム一覧表示）
        // -1Lは新規作成を意味する（許可）
        // -1L未満の負値は無効
        val initialAlarmId: Long? = when {
            alarmId == 0L -> null

            // 不正なID
            alarmId < -1L -> {
                androidx.compose.runtime.LaunchedEffect(Unit) {
                    android.widget.Toast.makeText(
                        context,
                        errorInvalidAlarmId,
                        android.widget.Toast.LENGTH_SHORT
                    ).show()
                }
                null
            }

            else -> alarmId // -1L（新規作成）または正のID（編集）
        }

        AlarmListScreen(
            onOpenDrawer = onOpenDrawer,
            initialAlarmId = initialAlarmId
        )
    }
}

/**
 * プレイリスト一覧画面のルート定義
 */
private fun NavGraphBuilder.playlistScreen(
    navController: NavHostController,
    onOpenDrawer: () -> Unit
) {
    composable(route = YtAlarmDestination.PLAYLIST) {
        PlaylistScreen(
            onNavigateToVideoList = { playlistId ->
                navController.navigate(YtAlarmDestination.videoList(playlistId))
            },
            onOpenDrawer = onOpenDrawer
        )
    }
}

/**
 * 動画一覧画面のルート定義
 */
@Suppress("LongMethod", "CyclomaticComplexMethod")
private fun NavGraphBuilder.videoListScreen(navController: NavHostController) {
    composable(
        route = YtAlarmDestination.VIDEO_LIST,
        arguments = listOf(
            navArgument("playlistId") {
                type = NavType.LongType
                defaultValue = 0L
            }
        )
    ) { backStackEntry ->
        val playlistId = backStackEntry.arguments?.getLong("playlistId") ?: 0L
        val context = LocalContext.current
        val scope = rememberCoroutineScope()
        val snackbarHostState = remember { SnackbarHostState() }

        // Pre-fetch string resources for use in lambdas
        val msgVideosAdded = stringResource(R.string.message_videos_added)
        val msgOperationFailed = stringResource(R.string.message_operation_failed)

        // ViewModels
        val videoViewModel: VideoViewModel = viewModel(
            factory = VideoViewModelFactory(
                (context.applicationContext as YtApplication).repository
            )
        )
        val playlistViewModel: PlaylistViewModel = viewModel(
            factory = PlaylistViewModelFactory(
                (context.applicationContext as YtApplication).repository
            )
        )

        // ダイアログ状態管理
        var showUrlInputDialog by remember { mutableStateOf(false) }
        var showMultiChoiceDialog by remember { mutableStateOf(false) }
        var currentPlaylistIdForDialog by remember { mutableLongStateOf(playlistId) }

        VideoListScreen(
            playlistId = playlistId,
            onNavigateBack = {
                navController.popBackStackSafely()
            },
            onNavigateToVideoPlayer = { videoId ->
                navController.navigate(YtAlarmDestination.videoPlayer(videoId, isAlarmMode = false))
            },
            onShowUrlInputDialog = { plId ->
                currentPlaylistIdForDialog = plId
                showUrlInputDialog = true
            },
            onShowMultiChoiceDialog = { plId ->
                currentPlaylistIdForDialog = plId
                showMultiChoiceDialog = true
            }
        )

        // UrlInputDialog: URLから動画/プレイリストを追加
        if (showUrlInputDialog) {
            UrlInputDialog(
                onConfirm = { url ->
                    scope.launch(Dispatchers.IO) {
                        try {
                            val targetPlaylistId = if (currentPlaylistIdForDialog != 0L) {
                                currentPlaylistIdForDialog
                            } else {
                                // 新規プレイリスト作成（Importing状態）
                                val newPlaylist = createImportingPlaylist()
                                playlistViewModel.insertAsync(newPlaylist).await()
                            }
                            // VideoInfoDownloadWorkerを使用してバックグラウンドでインポート
                            VideoInfoDownloadWorker.registerWorker(
                                context,
                                url,
                                longArrayOf(targetPlaylistId)
                            )
                            // 新規プレイリストの場合、新しいIDで画面を再ナビゲート
                            if (currentPlaylistIdForDialog == 0L) {
                                withContext(Dispatchers.Main) {
                                    showUrlInputDialog = false
                                    navController.navigate(
                                        YtAlarmDestination.videoList(targetPlaylistId)
                                    ) {
                                        popUpTo(YtAlarmDestination.videoList(0L)) {
                                            inclusive = true
                                        }
                                    }
                                }
                            } else {
                                withContext(Dispatchers.Main) {
                                    showUrlInputDialog = false
                                }
                            }
                        } catch (e: kotlinx.coroutines.CancellationException) {
                            throw e
                        } catch (@Suppress("TooGenericExceptionCaught") e: Exception) {
                            Log.e("YtAlarmNavGraph", "Failed to create playlist for URL import", e)
                            withContext(Dispatchers.Main) {
                                showUrlInputDialog = false
                                snackbarHostState.showSnackbar(msgOperationFailed)
                            }
                        }
                    }
                },
                onDismiss = { showUrlInputDialog = false }
            )
        }

        // MultiChoiceVideoDialog: 既存動画をプレイリストに追加
        if (showMultiChoiceDialog) {
            val allVideos by videoViewModel.allVideos.observeAsState(emptyList())

            MultiChoiceVideoDialog(
                displayDataList = allVideos.map { video ->
                    DisplayData(
                        id = video.id,
                        title = video.title,
                        thumbnailUrl = video.thumbnailUrl.takeIf {
                            it.isNotEmpty()
                        }?.let { DisplayDataThumbnail.Url(it) }
                    )
                },
                onConfirm = { selectedIds ->
                    scope.launch(Dispatchers.IO) {
                        try {
                            if (currentPlaylistIdForDialog == 0L) {
                                // 新規プレイリスト作成（既存動画のみなのでOriginal型）
                                var newPlaylist = Playlist(videos = selectedIds.toList())
                                newPlaylist.updateThumbnail()?.let { newPlaylist = it }
                                val newId = playlistViewModel.insertAsync(newPlaylist).await()
                                // 新しいプレイリストIDで画面を再ナビゲート
                                withContext(Dispatchers.Main) {
                                    showMultiChoiceDialog = false
                                    navController.navigate(YtAlarmDestination.videoList(newId)) {
                                        popUpTo(YtAlarmDestination.videoList(0L)) {
                                            inclusive = true
                                        }
                                    }
                                    snackbarHostState.showSnackbar(msgVideosAdded)
                                }
                            } else {
                                // 既存プレイリストに追加
                                val playlist = playlistViewModel
                                    .getFromIdAsync(currentPlaylistIdForDialog)
                                    .await()
                                if (playlist != null) {
                                    val updatedVideos = (playlist.videos + selectedIds).distinct()
                                    playlistViewModel.update(playlist.copy(videos = updatedVideos))
                                }
                                withContext(Dispatchers.Main) {
                                    snackbarHostState.showSnackbar(msgVideosAdded)
                                }
                            }
                        } catch (e: kotlinx.coroutines.CancellationException) {
                            throw e
                        } catch (e: android.database.sqlite.SQLiteException) {
                            Log.e("YtAlarmNavGraph", "Database error adding videos to playlist", e)
                            withContext(Dispatchers.Main) {
                                snackbarHostState.showSnackbar(msgOperationFailed)
                            }
                        } catch (e: IllegalStateException) {
                            Log.e("YtAlarmNavGraph", "Failed to add videos to playlist", e)
                            withContext(Dispatchers.Main) {
                                snackbarHostState.showSnackbar(msgOperationFailed)
                            }
                        }
                    }
                },
                onDismiss = { showMultiChoiceDialog = false }
            )
        }
    }
}

/**
 * 全動画一覧画面のルート定義
 */
private fun NavGraphBuilder.allVideosScreen(
    navController: NavHostController,
    onOpenDrawer: () -> Unit
) {
    composable(route = YtAlarmDestination.ALL_VIDEOS) {
        val context = LocalContext.current
        var showUrlInputDialog by remember { mutableStateOf(false) }

        AllVideosScreen(
            onOpenDrawer = onOpenDrawer,
            onNavigateToVideoPlayer = { videoId ->
                navController.navigate(YtAlarmDestination.videoPlayer(videoId, isAlarmMode = false))
            },
            onShowUrlInputDialog = {
                showUrlInputDialog = true
            }
        )

        // UrlInputDialog: URLから動画を追加
        if (showUrlInputDialog) {
            UrlInputDialog(
                onConfirm = { url ->
                    // 全動画モードでは特定のプレイリストに追加しない
                    VideoInfoDownloadWorker.registerWorker(
                        context,
                        url,
                        longArrayOf()
                    )
                    showUrlInputDialog = false
                },
                onDismiss = { showUrlInputDialog = false }
            )
        }
    }
}

/**
 * 動画プレーヤー画面のルート定義
 */
private fun NavGraphBuilder.videoPlayerScreen(navController: NavHostController) {
    composable(
        route = YtAlarmDestination.VIDEO_PLAYER,
        arguments = listOf(
            navArgument("videoId") {
                type = NavType.StringType
            },
            navArgument("isAlarmMode") {
                type = NavType.BoolType
                defaultValue = false
            }
        )
    ) { backStackEntry ->
        val videoId = backStackEntry.arguments?.getString("videoId")
        val isAlarmMode = backStackEntry.arguments?.getBoolean("isAlarmMode") ?: false

        // videoIdが不正な場合は前の画面に戻る
        if (videoId.isNullOrEmpty()) {
            androidx.compose.runtime.LaunchedEffect(Unit) {
                navController.popBackStackSafely()
            }
            return@composable
        }

        VideoPlayerScreen(
            videoId = videoId,
            isAlarmMode = isAlarmMode,
            onDismiss = {
                navController.popBackStackSafely()
            }
        )
    }
}

/**
 * About画面のルート定義
 */
private fun NavGraphBuilder.aboutScreen(navController: NavHostController) {
    composable(route = YtAlarmDestination.ABOUT) {
        AboutPageScreen(
            onNavigateBack = {
                navController.popBackStackSafely()
            }
        )
    }
}

/**
 * 設定画面のルート定義
 */
private fun NavGraphBuilder.settingsScreen(navController: NavHostController) {
    composable(route = YtAlarmDestination.SETTINGS) {
        SettingsScreen(
            onNavigateBack = {
                navController.popBackStackSafely()
            }
        )
    }
}