package net.turtton.ytalarm.worker

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import androidx.core.app.NotificationCompat
import androidx.work.Data
import androidx.work.ExistingWorkPolicy
import androidx.work.ForegroundInfo
import androidx.work.OneTimeWorkRequest
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.Operation
import androidx.work.OutOfQuotaPolicy
import androidx.work.WorkManager
import androidx.work.WorkerParameters
import arrow.core.flatMap
import arrow.core.fold
import arrow.core.raise.catch
import arrow.core.raise.either
import com.yausername.youtubedl_android.YoutubeDL
import com.yausername.youtubedl_android.YoutubeDLRequest
import kotlinx.coroutines.guava.await
import kotlinx.serialization.json.Json
import net.turtton.ytalarm.R
import net.turtton.ytalarm.database.structure.Playlist
import net.turtton.ytalarm.database.structure.Video
import net.turtton.ytalarm.util.VideoInformation
import net.turtton.ytalarm.util.extensions.copyAsFailed
import net.turtton.ytalarm.util.extensions.createImportingPlaylist
import net.turtton.ytalarm.util.extensions.deleteVideo
import net.turtton.ytalarm.util.extensions.hasUpdatingVideo
import net.turtton.ytalarm.util.extensions.insertVideos
import net.turtton.ytalarm.util.extensions.updateThumbnail

const val VIDEO_DOWNLOAD_NOTIFICATION = "net.turtton.ytalarm.VideoDLNotification"

class VideoInfoDownloadWorker(appContext: Context, workerParams: WorkerParameters) :
    CoroutineIOWorker(appContext, workerParams) {
    private val json = Json { ignoreUnknownKeys = true }

    @SuppressLint("RestrictedApi")
    override suspend fun doWork(): Result {
        val targetUrl = inputData.getString(KEY_URL) ?: return Result.failure()
        val isSyncMode = inputData.getBoolean(KEY_SYNC_MODE, false)
        val syncPlaylistId = inputData.getLong(KEY_SYNC_PLAYLIST_ID, 0L)

        if (isSyncMode && syncPlaylistId > 0) {
            return doSyncWork(targetUrl, syncPlaylistId)
        }

        var playlistArray = inputData.getLongArray(KEY_PLAYLIST)

        val stateTitle = applicationContext.getString(R.string.item_video_list_state_importing)
        val data = Video.State.Importing(Video.WorkerState.Working(id))
        var targetVideo = Video(videoId = "", title = stateTitle, stateData = data)

        val targetVideoId = repository.insert(targetVideo)
        targetVideo = repository.getVideoFromIdSync(targetVideoId)!!

        @Suppress("ControlFlowWithEmptyBody", "EmptyWhileBlock")
        while (
            WorkManager.getInstance(applicationContext)
                .getWorkInfoById(id)
                .await()
                .let { it == null || it.state.isFinished }
        ) {
        }
        playlistArray = playlistArray?.insertVideoInPlaylists(targetVideo)

        val (videos, type) = download(targetUrl)
            ?: run {
                repository.update(targetVideo.copyAsFailed(targetUrl))
                return Result.failure()
            }

        when (type) {
            is Type.Video -> {
                val video = videos.first()
                insertVideo(playlistArray, video, targetVideo)
            }

            is Type.Playlist -> {
                repository.delete(targetVideo)
                insertCloudPlaylist(playlistArray, videos, targetVideoId, type)
            }
        }

        return Result.success()
    }

    override suspend fun getForegroundInfo(): ForegroundInfo {
        val title = applicationContext.getString(R.string.notification_download_video_info_title)
        val cancel = applicationContext.getString(R.string.cancel)
        val cancelIntent = WorkManager.getInstance(applicationContext)
            .createCancelPendingIntent(id)
        val notification =
            NotificationCompat.Builder(applicationContext, VIDEO_DOWNLOAD_NOTIFICATION)
                .setSmallIcon(R.drawable.ic_download)
                .setContentTitle(title)
                .setProgress(1, 1, true)
                .addAction(R.drawable.ic_cancel, cancel, cancelIntent)
                .setSilent(true)

        return ForegroundInfo(NOTIFICATION_ID, notification.build())
    }

    private fun download(targetUrl: String): Pair<List<Video>, Type>? = either {
        catch({
            val request = YoutubeDLRequest(targetUrl)
                .addOption("--dump-single-json")
                .addOption("-f", "b")
            YoutubeDL.getInstance().execute(request) { _, _, _ -> }
        }) { error ->
            raise(error)
        }
    }.flatMap { result ->
        either {
            catch({
                json.decodeFromString<VideoInformation>(result.out)
            }) { error ->
                raise(error)
            }
        }
    }.fold(
        ifLeft = { error ->
            Log.e(WORKER_ID, "Download failed. Url: $targetUrl", error)
            null
        },
        ifRight = { videoInfo ->
            when (videoInfo.typeData) {
                is VideoInformation.Type.Video -> {
                    listOf(videoInfo.toVideo()) to Type.Video
                }

                is VideoInformation.Type.Playlist -> {
                    videoInfo.typeData
                        .entries
                        .map(VideoInformation::toVideo)
                        .let { videos ->
                            videos to Type.Playlist(videoInfo.title!!, videoInfo.url)
                        }
                }
            }
        }
    )

    private suspend fun insertVideo(
        playlistArray: LongArray?,
        newVideo: Video,
        pendingVideo: Video
    ) {
        val updatedPlaylist = checkVideoDuplication(newVideo.videoId, newVideo.domain)
            ?.let { duplicatedId ->
                repository.delete(pendingVideo)
                playlistArray?.let { repository.getPlaylistFromIdsSync(it.toList()) }
                    ?.deleteVideo(newVideo.id)
                    ?.map { playlist ->
                        val videoSet = playlist.videos.toMutableSet()
                        videoSet += duplicatedId
                        var newPlaylist = playlist.copy(videos = videoSet.toList())
                        if (newPlaylist.thumbnail is Playlist.Thumbnail.Drawable) {
                            newPlaylist.updateThumbnail()?.let {
                                newPlaylist = it
                            }
                        }
                        val currentVideos = repository.getVideoFromIdsSync(newPlaylist.videos)
                        if (!currentVideos.hasUpdatingVideo) {
                            newPlaylist = newPlaylist.copy(type = Playlist.Type.Original)
                        }
                        newPlaylist
                    }
            } ?: kotlin.run {
            val importedVideo = newVideo.copy(id = pendingVideo.id)
            repository.update(importedVideo)
            playlistArray?.let {
                repository.getPlaylistFromIdsSync(it.toList())
            }?.map { pl ->
                var playlist = pl
                var shouldUpdate = false
                val containsVideos = repository.getVideoFromIdsSync(playlist.videos)
                if (!containsVideos.hasUpdatingVideo) {
                    playlist = playlist.copy(type = Playlist.Type.Original)
                    shouldUpdate = true
                }
                if (playlist.thumbnail is Playlist.Thumbnail.Drawable) {
                    playlist.updateThumbnail()?.also {
                        playlist = it
                        shouldUpdate = true
                    }
                }
                playlist.takeIf { shouldUpdate }
            }
        }

        updatedPlaylist?.filterNotNull()
            .takeIf { !it.isNullOrEmpty() }
            ?.let {
                repository.update(it)
            }
    }

    private suspend fun insertCloudPlaylist(
        playlistArray: LongArray?,
        videos: List<Video>,
        deletedVideoId: Long,
        type: Type.Playlist
    ) {
        val targetIds = mutableListOf<Long>()
        val newVideos = videos.filter {
            checkVideoDuplication(it.videoId, it.domain)
                ?.also { duplicatedId -> targetIds += duplicatedId }
                .let { duplicatedId -> duplicatedId == null }
        }
        targetIds += repository.insert(newVideos)

        playlistArray?.let { _ ->
            var playlists = repository.getPlaylistFromIdsSync(playlistArray.toList())
            playlists = playlists.deleteVideo(deletedVideoId)
            playlists = checkPlaylistSyncRule(playlists, targetIds)
            playlists = playlists.insertVideos(targetIds)

            var playlistType = Playlist.Type.CloudPlaylist(type.url, id)
            playlists = playlists.map { playlist ->
                if (playlist.type is Playlist.Type.CloudPlaylist) {
                    playlistType = playlistType.copy(syncRule = playlist.type.syncRule)
                }
                var newPlaylist = playlist.copy(type = playlistType, title = type.title)
                if (newPlaylist.thumbnail is Playlist.Thumbnail.Drawable) {
                    newPlaylist.updateThumbnail()?.let {
                        newPlaylist = it
                    }
                }
                newPlaylist
            }
            repository.update(playlists)
        }
    }

    private fun checkPlaylistSyncRule(
        playlists: List<Playlist>,
        targetIds: List<Long>
    ): List<Playlist> = playlists.map { immutablePlaylist ->
        var playlist = immutablePlaylist
        val playlistType = playlist.type
        val expectedRule = Playlist.SyncRule.DELETE_IF_NOT_EXIST
        if (playlistType is Playlist.Type.CloudPlaylist && playlistType.syncRule == expectedRule) {
            val currentVideos = playlist.videos
            val removeTarget = currentVideos.filterNot { targetIds.contains(it) }
            val thumbnail = playlist.thumbnail
            if (thumbnail is Playlist.Thumbnail.Video && removeTarget.contains(thumbnail.id)) {
                val newThumbnail = targetIds.firstOrNull()?.let {
                    Playlist.Thumbnail.Video(it)
                } ?: run {
                    val noImage = R.drawable.ic_no_image
                    Playlist.Thumbnail.Drawable(noImage)
                }
                playlist = playlist.copy(thumbnail = newThumbnail)
            }
            playlist = playlist.copy(videos = emptyList())
        }
        playlist
    }

    private suspend fun LongArray.insertVideoInPlaylists(video: Video) = map {
        val playlist =
            if (it == 0L) {
                createImportingPlaylist()
            } else {
                repository.getPlaylistFromIdSync(it) ?: return@map null
            }
        val newList = playlist.videos.toMutableSet().apply { add(video.id) }.toList()
        val newPlaylist = playlist.copy(videos = newList)
        if (it == 0L) {
            repository.insert(newPlaylist)
        } else {
            repository.update(newPlaylist)
            it
        }
    }.filterNotNull().toLongArray()

    private suspend fun checkVideoDuplication(videoId: String, domain: String): Long? =
        repository.getVideoFromVideoIdSync(videoId)?.let {
            if (it.domain == domain) {
                it.id
            } else {
                null
            }
        }

    private suspend fun doSyncWork(targetUrl: String, playlistId: Long): Result {
        val playlist = repository.getPlaylistFromIdSync(playlistId)
            ?: return Result.failure()

        val playlistType = playlist.type as? Playlist.Type.CloudPlaylist
            ?: return Result.failure()

        val (videos, type) = download(targetUrl)
            ?: return Result.failure()

        if (type !is Type.Playlist) {
            return Result.failure()
        }

        syncCloudPlaylist(playlist, playlistType, videos)
        return Result.success()
    }

    private suspend fun syncCloudPlaylist(
        playlist: Playlist,
        playlistType: Playlist.Type.CloudPlaylist,
        newVideos: List<Video>
    ) {
        val targetIds = mutableListOf<Long>()
        val videosToInsert = newVideos.filter { video ->
            checkVideoDuplication(video.videoId, video.domain)
                ?.also { duplicatedId -> targetIds += duplicatedId }
                .let { it == null }
        }
        targetIds += repository.insert(videosToInsert)

        var updatedPlaylist = playlist

        when (playlistType.syncRule) {
            Playlist.SyncRule.ALWAYS_ADD -> {
                val currentVideos = playlist.videos.toMutableSet()
                currentVideos.addAll(targetIds)
                updatedPlaylist = updatedPlaylist.copy(videos = currentVideos.toList())
            }

            Playlist.SyncRule.DELETE_IF_NOT_EXIST -> {
                val thumbnail = playlist.thumbnail
                val newThumbnail = if (thumbnail is Playlist.Thumbnail.Video &&
                    !targetIds.contains(thumbnail.id)
                ) {
                    targetIds.firstOrNull()?.let { Playlist.Thumbnail.Video(it) }
                        ?: Playlist.Thumbnail.Drawable(R.drawable.ic_no_image)
                } else {
                    thumbnail
                }
                updatedPlaylist = updatedPlaylist.copy(
                    videos = targetIds,
                    thumbnail = newThumbnail
                )
            }
        }

        val updatedType = playlistType.copy(workerId = id)
        updatedPlaylist = updatedPlaylist.copy(
            type = updatedType,
            lastUpdated = java.util.Calendar.getInstance()
        )

        repository.update(updatedPlaylist)
    }

    private sealed interface Type {
        object Video : Type

        data class Playlist(val title: String, val url: String) : Type
    }

    companion object {
        private const val NOTIFICATION_ID = 1
        const val WORKER_ID = "VideoDownloadWorker"
        private const val KEY_URL = "DownloadUrl"
        private const val KEY_PLAYLIST = "PlaylistId"
        private const val KEY_SYNC_MODE = "SyncMode"
        private const val KEY_SYNC_PLAYLIST_ID = "SyncPlaylistId"

        fun registerSyncWorker(
            context: Context,
            playlistId: Long,
            playlistUrl: String
        ): OneTimeWorkRequest {
            val data = Data.Builder()
                .putString(KEY_URL, playlistUrl)
                .putBoolean(KEY_SYNC_MODE, true)
                .putLong(KEY_SYNC_PLAYLIST_ID, playlistId)
                .build()
            val request = OneTimeWorkRequestBuilder<VideoInfoDownloadWorker>()
                .setInputData(data)
                .setExpedited(OutOfQuotaPolicy.RUN_AS_NON_EXPEDITED_WORK_REQUEST)
                .build()
            WorkManager.getInstance(context).enqueueUniqueWork(
                "SyncWorker_$playlistId",
                ExistingWorkPolicy.REPLACE,
                request
            )
            return request
        }

        fun registerWorker(
            context: Context,
            targetUrl: String,
            targetPlaylists: LongArray = longArrayOf()
        ): OneTimeWorkRequest {
            val (request, task) = prepareWorker(targetUrl, targetPlaylists)
            WorkManager.getInstance(context).task()
            return request
        }

        fun prepareWorker(
            targetUrl: String,
            targetPlaylists: LongArray = longArrayOf()
        ): Pair<OneTimeWorkRequest, EnqueueTask> {
            val data = Data.Builder().putString(KEY_URL, targetUrl)
            if (targetPlaylists.isNotEmpty()) {
                data.putLongArray(KEY_PLAYLIST, targetPlaylists)
            }
            val request = OneTimeWorkRequestBuilder<VideoInfoDownloadWorker>()
                .setInputData(data.build())
                .setExpedited(OutOfQuotaPolicy.RUN_AS_NON_EXPEDITED_WORK_REQUEST)
                .build()
            val enqueueTask: EnqueueTask = {
                enqueueUniqueWork(
                    WORKER_ID,
                    ExistingWorkPolicy.APPEND_OR_REPLACE,
                    request
                )
            }
            return request to enqueueTask
        }
    }
}

typealias EnqueueTask = WorkManager.() -> Operation