/*
 * This file is part of GNU Taler
 * (C) 2025 Taler Systems S.A.
 *
 * GNU Taler is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 3, or (at your option) any later version.
 *
 * GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

package net.taler.wallet.main

import net.taler.common.CurrencySpecification
import net.taler.wallet.PrefsViewMode
import net.taler.wallet.ViewModeType
import net.taler.wallet.balances.ScopeInfo
import net.taler.wallet.transactions.TransactionStateFilter

sealed class ViewMode {
    abstract fun toPrefs(): PrefsViewMode

    data object Assets: ViewMode() {
        override fun toPrefs(): PrefsViewMode = PrefsViewMode
            .newBuilder()
            .setType(ViewModeType.ASSETS)
            .build()
    }

    data class Transactions(
        val selectedScope: ScopeInfo,
        val selectedSpec: CurrencySpecification? = null,
        val searchQuery: String? = null,
        val stateFilter: TransactionStateFilter? = null,
    ): ViewMode() {
        override fun toPrefs(): PrefsViewMode = PrefsViewMode
            .newBuilder()
            .setType(ViewModeType.TRANSACTIONS)
            .setSelectedScope(selectedScope.toPrefs())
            .apply {
                if (this@Transactions.searchQuery != null)
                    setSearchQuery(this@Transactions.searchQuery)
                if (this@Transactions.stateFilter != null)
                    setStateFilter(this@Transactions.stateFilter.toPrefs())
            }.build()
    }

    companion object {
        fun fromPrefs(prefs: PrefsViewMode): ViewMode = when (prefs.type) {
            ViewModeType.ASSETS, ViewModeType.UNRECOGNIZED -> Assets

            ViewModeType.TRANSACTIONS -> Transactions(
                // TODO: better null handling
                selectedScope = ScopeInfo.fromPrefs(prefs.selectedScope)!!,
                searchQuery = prefs.searchQuery,
                stateFilter = TransactionStateFilter.fromPrefs(prefs.stateFilter),
            )
        }
    }
}