// Copyright (C) 1992, 2019  Roland Horsch and others:
// Original data by K. Simonsen (1992, plus errata, plus data from http://www.unicode.org):
// -- Simonsen, K., "Character Mnemonics and Character Sets",
//    RFC 1345, DOI 10.17487/RFC1345, June 1992, <https://www.rfc-editor.org/info/rfc1345>.
// -- http://www.unicode.org (cmp. https://www.unicode.org/Public/5.2.0/ucd/Blocks.txt).
// For the presented form: Copyright (C) 2019  Roland Horsch <gx work s{at}mai l.de>.
// License: GPLv3-or-later - see LICENSE file (or www.gnu.org/licenses/gpl.html),
// plus the compatible full texts for further details.
// ABSOLUTELY NO WARRANTY. Formatted by Netbeans Ext plugin.

package com.gitlab.dibdib.picked.common;

/** Used in Dibdib with '^' instead of '.': '.' and '`' are needed as markers with narrow glyphs. */
public final class Rfc1345 {

  /*
  In the two-character mnemonics, all invariant graphic character in
  the ISO 646 character codes except "&" are used, i.e. the following
  characters:

       ! "     %   ' ( ) * + , - . / 0 1 2 3 4 5 6 7 8 9 : ; < = > ?
       A B C D E F G H I J K L M N O P Q R S T U V W X Y Z       _
       a b c d e f g h i j k l m n o p q r s t u v w x y z

  The character "_" is not used as the first character.
  [Note: '_' is not used in combination with: '{}[]()<>/'.]

  After a letter, the second character signifies the following:

       Exclamation mark           ! Grave
       Apostrophe                 ' Acute accent
       Greater-Than sign          > Circumflex accent
       Question Mark              ? tilde
       Hyphen-Minus               - Macron
       Left parenthesis           ( Breve
       Full Stop                  . Dot Above
       Colon                      : Diaeresis
       Comma                      , Cedilla
       Underline                  _ Underline
       Solidus                    / Stroke
       Quotation mark             " Double acute accent
       Semicolon                  ; Ogonek
       Less-Than sign             < Caron
       Zero                       0 Ring above
       Two                        2 Hook
       Nine                       9 Horn

       Equals                     = Cyrillic
       Asterisk                   * Greek
       Percent sign               % Greek/Cyrillic special
       Plus                       + smalls: Arabic, capitals: Hebrew
       Three                      3 some Latin/Greek/Cyrillic letters
       Four                       4 Bopomofo
       Five                       5 Hiragana
       Six                        6 Katakana
  */

  public static final String[] kRfc1345 = {
    "NU", "\u0000", // NULL (NUL)
    "SH", "\u0001", // START OF HEADING (SOH)
    "SX", "\u0002", // START OF TEXT (STX)
    "EX", "\u0003", // END OF TEXT (ETX)
    "ET", "\u0004", // END OF TRANSMISSION (EOT)
    "EQ", "\u0005", // ENQUIRY (ENQ)
    "AK", "\u0006", // ACKNOWLEDGE (ACK)
    "BL", "\u0007", // BELL (BEL)
    "BS", "\u0008", // BACKSPACE (BS)
    "HT", "\u0009", // CHARACTER TABULATION (HT)
    "LF", "\n", // LINE FEED (LF)
    "VT", "\u000b", // LINE TABULATION (VT)
    "FF", "\u000c", // FORM FEED (FF)
    "CR", "\r", // CARRIAGE RETURN (CR)
    "SO", "\u000e", // SHIFT OUT (SO)
    "SI", "\u000f", // SHIFT IN (SI)
    "DL", "\u0010", // DATALINK ESCAPE (DLE)
    "D1", "\u0011", // DEVICE CONTROL ONE (DC1)
    "D2", "\u0012", // DEVICE CONTROL TWO (DC2)
    "D3", "\u0013", // DEVICE CONTROL THREE (DC3)
    "D4", "\u0014", // DEVICE CONTROL FOUR (DC4)
    "NK", "\u0015", // NEGATIVE ACKNOWLEDGE (NAK)
    "SY", "\u0016", // SYNCRONOUS IDLE (SYN)
    "EB", "\u0017", // END OF TRANSMISSION BLOCK (ETB)
    "CN", "\u0018", // CANCEL (CAN)
    "EM", "\u0019", // END OF MEDIUM (EM)
    "SB", "\u001a", // SUBSTITUTE (SUB)
    "EC", "\u001b", // ESCAPE (ESC)
    "FS", "\u001c", // FILE SEPARATOR (IS4)
    "GS", "\u001d", // GROUP SEPARATOR (IS3)
    "RS", "\u001e", // RECORD SEPARATOR (IS2)
    "US", "\u001f", // UNIT SEPARATOR (IS1)

    "PA", "\u0080", // PADDING CHARACTER (PAD)
    "HO", "\u0081", // HIGH OCTET PRESET (HOP)
    "BH", "\u0082", // BREAK PERMITTED HERE (BPH)
    "NH", "\u0083", // NO BREAK HERE (NBH)
    "IN", "\u0084", // INDEX (IND)
    "NL", "\u0085", // NEXT LINE (NEL)
    "SA", "\u0086", // START OF SELECTED AREA (SSA)
    "ES", "\u0087", // END OF SELECTED AREA (ESA)
    "HS", "\u0088", // CHARACTER TABULATION SET (HTS)
    "HJ", "\u0089", // CHARACTER TABULATION WITH JUSTIFICATION (HTJ)
    "VS", "\u008a", // LINE TABULATION SET (VTS)
    "PD", "\u008b", // PARTIAL LINE FORWARD (PLD)
    "PU", "\u008c", // PARTIAL LINE BACKWARD (PLU)
    "RI", "\u008d", // REVERSE LINE FEED (RI)
    "S2", "\u008e", // SINGLE-SHIFT TWO (SS2)
    "S3", "\u008f", // SINGLE-SHIFT THREE (SS3)
    "DC", "\u0090", // DEVICE CONTROL STRING (DCS)
    "P1", "\u0091", // PRIVATE USE ONE (PU1)
    "P2", "\u0092", // PRIVATE USE TWO (PU2)
    "TS", "\u0093", // SET TRANSMIT STATE (STS)
    "CC", "\u0094", // CANCEL CHARACTER (CCH)
    "MW", "\u0095", // MESSAGE WAITING (MW)
    "SG", "\u0096", // START OF GUARDED AREA (SPA)
    "EG", "\u0097", // END OF GUARDED AREA (EPA)
    "SS", "\u0098", // START OF STRING (SOS)
    "GC", "\u0099", // SINGLE GRAPHIC CHARACTER INTRODUCER (SGCI)
    "SC", "\u009a", // SINGLE CHARACTER INTRODUCER (SCI)
    "CI", "\u009b", // CONTROL SEQUENCE INTRODUCER (CSI)
    "ST", "\u009c", // STRING TERMINATOR (ST)
    "OC", "\u009d", // OPERATING SYSTEM COMMAND (OSC)
    "PM", "\u009e", // PRIVACY MESSAGE (PM)
    "AC", "\u009f", // APPLICATION PROGRAM COMMAND (APC) e000 indicates unfinished (Mnemonic)

    "SP", "\u0020", // SPACE
    "!", "\u0021", // EXCLAMATION MARK
    "\"", "\"", // QUOTATION MARK
    "Nb", "\u0023", // NUMBER SIGN
    "DO", "\u0024", // DOLLAR SIGN
    "%", "\u0025", // PERCENT SIGN
    "&", "\u0026", // AMPERSAND
    "'", "\u0027", // APOSTROPHE
    "(", "\u0028", // LEFT PARENTHESIS
    ")", "\u0029", // RIGHT PARENTHESIS
    "*", "\u002a", // ASTERISK
    "+", "\u002b", // PLUS SIGN
    ",", "\u002c", // COMMA
    "-", "\u002d", // HYPHEN-MINUS
    ".", "\u002e", // FULL STOP
    "/", "\u002f", // SOLIDUS
    "0", "\u0030", // DIGIT ZERO
    "1", "\u0031", // DIGIT ONE
    "2", "\u0032", // DIGIT TWO
    "3", "\u0033", // DIGIT THREE
    "4", "\u0034", // DIGIT FOUR
    "5", "\u0035", // DIGIT FIVE
    "6", "\u0036", // DIGIT SIX
    "7", "\u0037", // DIGIT SEVEN
    "8", "\u0038", // DIGIT EIGHT
    "9", "\u0039", // DIGIT NINE
    ":", "\u003a", // COLON
    ";", "\u003b", // SEMICOLON
    "<", "\u003c", // LESS-THAN SIGN
    "=", "\u003d", // EQUALS SIGN
    ">", "\u003e", // GREATER-THAN SIGN
    "?", "\u003f", // QUESTION MARK
    "At", "\u0040", // COMMERCIAL AT
    "A", "\u0041", // LATIN CAPITAL LETTER A
    "B", "\u0042", // LATIN CAPITAL LETTER B
    "C", "\u0043", // LATIN CAPITAL LETTER C
    "D", "\u0044", // LATIN CAPITAL LETTER D
    "E", "\u0045", // LATIN CAPITAL LETTER E
    "F", "\u0046", // LATIN CAPITAL LETTER F
    "G", "\u0047", // LATIN CAPITAL LETTER G
    "H", "\u0048", // LATIN CAPITAL LETTER H
    "I", "\u0049", // LATIN CAPITAL LETTER I
    "J", "\u004a", // LATIN CAPITAL LETTER J
    "K", "\u004b", // LATIN CAPITAL LETTER K
    "L", "\u004c", // LATIN CAPITAL LETTER L
    "M", "\u004d", // LATIN CAPITAL LETTER M
    "N", "\u004e", // LATIN CAPITAL LETTER N
    "O", "\u004f", // LATIN CAPITAL LETTER O
    "P", "\u0050", // LATIN CAPITAL LETTER P
    "Q", "\u0051", // LATIN CAPITAL LETTER Q
    "R", "\u0052", // LATIN CAPITAL LETTER R
    "S", "\u0053", // LATIN CAPITAL LETTER S
    "T", "\u0054", // LATIN CAPITAL LETTER T
    "U", "\u0055", // LATIN CAPITAL LETTER U
    "V", "\u0056", // LATIN CAPITAL LETTER V
    "W", "\u0057", // LATIN CAPITAL LETTER W
    "X", "\u0058", // LATIN CAPITAL LETTER X
    "Y", "\u0059", // LATIN CAPITAL LETTER Y
    "Z", "\u005a", // LATIN CAPITAL LETTER Z
    "<(", "\u005b", // LEFT SQUARE BRACKET
    "//", "\\", // REVERSE SOLIDUS
    ")>", "\u005d", // RIGHT SQUARE BRACKET
    "'>", "\u005e", // CIRCUMFLEX ACCENT
    "_", "\u005f", // LOW LINE
    "'!", "\u0060", // GRAVE ACCENT
    "a", "\u0061", // LATIN SMALL LETTER A
    "b", "\u0062", // LATIN SMALL LETTER B
    "c", "\u0063", // LATIN SMALL LETTER C
    "d", "\u0064", // LATIN SMALL LETTER D
    "e", "\u0065", // LATIN SMALL LETTER E
    "f", "\u0066", // LATIN SMALL LETTER F
    "g", "\u0067", // LATIN SMALL LETTER G
    "h", "\u0068", // LATIN SMALL LETTER H
    "i", "\u0069", // LATIN SMALL LETTER I
    "j", "\u006a", // LATIN SMALL LETTER J
    "k", "\u006b", // LATIN SMALL LETTER K
    "l", "\u006c", // LATIN SMALL LETTER L
    "m", "\u006d", // LATIN SMALL LETTER M
    "n", "\u006e", // LATIN SMALL LETTER N
    "o", "\u006f", // LATIN SMALL LETTER O
    "p", "\u0070", // LATIN SMALL LETTER P
    "q", "\u0071", // LATIN SMALL LETTER Q
    "r", "\u0072", // LATIN SMALL LETTER R
    "s", "\u0073", // LATIN SMALL LETTER S
    "t", "\u0074", // LATIN SMALL LETTER T
    "u", "\u0075", // LATIN SMALL LETTER U
    "v", "\u0076", // LATIN SMALL LETTER V
    "w", "\u0077", // LATIN SMALL LETTER W
    "x", "\u0078", // LATIN SMALL LETTER X
    "y", "\u0079", // LATIN SMALL LETTER Y
    "z", "\u007a", // LATIN SMALL LETTER Z
    "(!", "\u007b", // LEFT CURLY BRACKET
    "!!", "\u007c", // VERTICAL LINE
    "!)", "\u007d", // RIGHT CURLY BRACKET
    "'?", "\u007e", // TILDE

    "DT", "\u007f", // DELETE (DEL)

    "NS", "\u00a0", // NO-BREAK SPACE
    "!I", "\u00a1", // INVERTED EXCLAMATION MARK
    "Ct", "\u00a2", // CENT SIGN
    "Pd", "\u00a3", // POUND SIGN
    "Cu", "\u00a4", // CURRENCY SIGN
    "Ye", "\u00a5", // YEN SIGN
    "BB", "\u00a6", // BROKEN BAR
    "SE", "\u00a7", // SECTION SIGN
    "':", "\u00a8", // DIAERESIS
    "Co", "\u00a9", // COPYRIGHT SIGN
    "-a", "\u00aa", // FEMININE ORDINAL INDICATOR
    "<<", "\u00ab", // LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
    "NO", "\u00ac", // NOT SIGN
    "--", "\u00ad", // SOFT HYPHEN
    "Rg", "\u00ae", // REGISTERED SIGN
    "'m", "\u00af", // MACRON
    "DG", "\u00b0", // DEGREE SIGN
    "+-", "\u00b1", // PLUS-MINUS SIGN
    "2S", "\u00b2", // SUPERSCRIPT TWO
    "3S", "\u00b3", // SUPERSCRIPT THREE
    "''", "\u00b4", // ACUTE ACCENT
    "My", "\u00b5", // MICRO SIGN
    "PI", "\u00b6", // PILCROW SIGN
    ".M", "\u00b7", // MIDDLE DOT
    "',", "\u00b8", // CEDILLA
    "1S", "\u00b9", // SUPERSCRIPT ONE
    "-o", "\u00ba", // MASCULINE ORDINAL INDICATOR
    ">>", "\u00bb", // RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
    "14", "\u00bc", // VULGAR FRACTION ONE QUARTER
    "12", "\u00bd", // VULGAR FRACTION ONE HALF
    "34", "\u00be", // VULGAR FRACTION THREE QUARTERS
    "?I", "\u00bf", // INVERTED QUESTION MARK
    "A!", "\u00c0", // LATIN CAPITAL LETTER A WITH GRAVE
    "A'", "\u00c1", // LATIN CAPITAL LETTER A WITH ACUTE
    "A>", "\u00c2", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX
    "A?", "\u00c3", // LATIN CAPITAL LETTER A WITH TILDE
    "A:", "\u00c4", // LATIN CAPITAL LETTER A WITH DIAERESIS
    "AA", "\u00c5", // LATIN CAPITAL LETTER A WITH RING ABOVE
    "AE", "\u00c6", // LATIN CAPITAL LETTER AE
    "C,", "\u00c7", // LATIN CAPITAL LETTER C WITH CEDILLA
    "E!", "\u00c8", // LATIN CAPITAL LETTER E WITH GRAVE
    "E'", "\u00c9", // LATIN CAPITAL LETTER E WITH ACUTE
    "E>", "\u00ca", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX
    "E:", "\u00cb", // LATIN CAPITAL LETTER E WITH DIAERESIS
    "I!", "\u00cc", // LATIN CAPITAL LETTER I WITH GRAVE
    "I'", "\u00cd", // LATIN CAPITAL LETTER I WITH ACUTE
    "I>", "\u00ce", // LATIN CAPITAL LETTER I WITH CIRCUMFLEX
    "I:", "\u00cf", // LATIN CAPITAL LETTER I WITH DIAERESIS
    "D-", "\u00d0", // LATIN CAPITAL LETTER ETH (Icelandic)
    "N?", "\u00d1", // LATIN CAPITAL LETTER N WITH TILDE
    "O!", "\u00d2", // LATIN CAPITAL LETTER O WITH GRAVE
    "O'", "\u00d3", // LATIN CAPITAL LETTER O WITH ACUTE
    "O>", "\u00d4", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX
    "O?", "\u00d5", // LATIN CAPITAL LETTER O WITH TILDE
    "O:", "\u00d6", // LATIN CAPITAL LETTER O WITH DIAERESIS
    "*X", "\u00d7", // MULTIPLICATION SIGN
    "O/", "\u00d8", // LATIN CAPITAL LETTER O WITH STROKE
    "U!", "\u00d9", // LATIN CAPITAL LETTER U WITH GRAVE
    "U'", "\u00da", // LATIN CAPITAL LETTER U WITH ACUTE
    "U>", "\u00db", // LATIN CAPITAL LETTER U WITH CIRCUMFLEX
    "U:", "\u00dc", // LATIN CAPITAL LETTER U WITH DIAERESIS
    "Y'", "\u00dd", // LATIN CAPITAL LETTER Y WITH ACUTE
    "TH", "\u00de", // LATIN CAPITAL LETTER THORN (Icelandic)
    "ss", "\u00df", // LATIN SMALL LETTER SHARP S (German)
    "a!", "\u00e0", // LATIN SMALL LETTER A WITH GRAVE
    "a'", "\u00e1", // LATIN SMALL LETTER A WITH ACUTE
    "a>", "\u00e2", // LATIN SMALL LETTER A WITH CIRCUMFLEX
    "a?", "\u00e3", // LATIN SMALL LETTER A WITH TILDE
    "a:", "\u00e4", // LATIN SMALL LETTER A WITH DIAERESIS
    "aa", "\u00e5", // LATIN SMALL LETTER A WITH RING ABOVE
    "ae", "\u00e6", // LATIN SMALL LETTER AE
    "c,", "\u00e7", // LATIN SMALL LETTER C WITH CEDILLA
    "e!", "\u00e8", // LATIN SMALL LETTER E WITH GRAVE
    "e'", "\u00e9", // LATIN SMALL LETTER E WITH ACUTE
    "e>", "\u00ea", // LATIN SMALL LETTER E WITH CIRCUMFLEX
    "e:", "\u00eb", // LATIN SMALL LETTER E WITH DIAERESIS
    "i!", "\u00ec", // LATIN SMALL LETTER I WITH GRAVE
    "i'", "\u00ed", // LATIN SMALL LETTER I WITH ACUTE
    "i>", "\u00ee", // LATIN SMALL LETTER I WITH CIRCUMFLEX
    "i:", "\u00ef", // LATIN SMALL LETTER I WITH DIAERESIS
    "d-", "\u00f0", // LATIN SMALL LETTER ETH (Icelandic)
    "n?", "\u00f1", // LATIN SMALL LETTER N WITH TILDE
    "o!", "\u00f2", // LATIN SMALL LETTER O WITH GRAVE
    "o'", "\u00f3", // LATIN SMALL LETTER O WITH ACUTE
    "o>", "\u00f4", // LATIN SMALL LETTER O WITH CIRCUMFLEX
    "o?", "\u00f5", // LATIN SMALL LETTER O WITH TILDE
    "o:", "\u00f6", // LATIN SMALL LETTER O WITH DIAERESIS
    "-:", "\u00f7", // DIVISION SIGN
    "o/", "\u00f8", // LATIN SMALL LETTER O WITH STROKE
    "u!", "\u00f9", // LATIN SMALL LETTER U WITH GRAVE
    "u'", "\u00fa", // LATIN SMALL LETTER U WITH ACUTE
    "u>", "\u00fb", // LATIN SMALL LETTER U WITH CIRCUMFLEX
    "u:", "\u00fc", // LATIN SMALL LETTER U WITH DIAERESIS
    "y'", "\u00fd", // LATIN SMALL LETTER Y WITH ACUTE
    "th", "\u00fe", // LATIN SMALL LETTER THORN (Icelandic)
    "y:", "\u00ff", // LATIN SMALL LETTER Y WITH DIAERESIS
    "A-", "\u0100", // LATIN CAPITAL LETTER A WITH MACRON
    "a-", "\u0101", // LATIN SMALL LETTER A WITH MACRON
    "A(", "\u0102", // LATIN CAPITAL LETTER A WITH BREVE
    "a(", "\u0103", // LATIN SMALL LETTER A WITH BREVE
    "A;", "\u0104", // LATIN CAPITAL LETTER A WITH OGONEK
    "a;", "\u0105", // LATIN SMALL LETTER A WITH OGONEK
    "C'", "\u0106", // LATIN CAPITAL LETTER C WITH ACUTE
    "c'", "\u0107", // LATIN SMALL LETTER C WITH ACUTE
    "C>", "\u0108", // LATIN CAPITAL LETTER C WITH CIRCUMFLEX
    "c>", "\u0109", // LATIN SMALL LETTER C WITH CIRCUMFLEX
    "C.", "\u010a", // LATIN CAPITAL LETTER C WITH DOT ABOVE
    "c.", "\u010b", // LATIN SMALL LETTER C WITH DOT ABOVE
    "C<", "\u010c", // LATIN CAPITAL LETTER C WITH CARON
    "c<", "\u010d", // LATIN SMALL LETTER C WITH CARON
    "D<", "\u010e", // LATIN CAPITAL LETTER D WITH CARON
    "d<", "\u010f", // LATIN SMALL LETTER D WITH CARON
    "D/", "\u0110", // LATIN CAPITAL LETTER D WITH STROKE
    "d/", "\u0111", // LATIN SMALL LETTER D WITH STROKE
    "E-", "\u0112", // LATIN CAPITAL LETTER E WITH MACRON
    "e-", "\u0113", // LATIN SMALL LETTER E WITH MACRON
    "E(", "\u0114", // LATIN CAPITAL LETTER E WITH BREVE
    "e(", "\u0115", // LATIN SMALL LETTER E WITH BREVE
    "E.", "\u0116", // LATIN CAPITAL LETTER E WITH DOT ABOVE
    "e.", "\u0117", // LATIN SMALL LETTER E WITH DOT ABOVE
    "E;", "\u0118", // LATIN CAPITAL LETTER E WITH OGONEK
    "e;", "\u0119", // LATIN SMALL LETTER E WITH OGONEK
    "E<", "\u011a", // LATIN CAPITAL LETTER E WITH CARON
    "e<", "\u011b", // LATIN SMALL LETTER E WITH CARON
    "G>", "\u011c", // LATIN CAPITAL LETTER G WITH CIRCUMFLEX
    "g>", "\u011d", // LATIN SMALL LETTER G WITH CIRCUMFLEX
    "G(", "\u011e", // LATIN CAPITAL LETTER G WITH BREVE
    "g(", "\u011f", // LATIN SMALL LETTER G WITH BREVE
    "G.", "\u0120", // LATIN CAPITAL LETTER G WITH DOT ABOVE
    "g.", "\u0121", // LATIN SMALL LETTER G WITH DOT ABOVE
    "G,", "\u0122", // LATIN CAPITAL LETTER G WITH CEDILLA
    "g,", "\u0123", // LATIN SMALL LETTER G WITH CEDILLA
    "H>", "\u0124", // LATIN CAPITAL LETTER H WITH CIRCUMFLEX
    "h>", "\u0125", // LATIN SMALL LETTER H WITH CIRCUMFLEX
    "H/", "\u0126", // LATIN CAPITAL LETTER H WITH STROKE
    "h/", "\u0127", // LATIN SMALL LETTER H WITH STROKE
    "I?", "\u0128", // LATIN CAPITAL LETTER I WITH TILDE
    "i?", "\u0129", // LATIN SMALL LETTER I WITH TILDE
    "I-", "\u012a", // LATIN CAPITAL LETTER I WITH MACRON
    "i-", "\u012b", // LATIN SMALL LETTER I WITH MACRON
    "I(", "\u012c", // LATIN CAPITAL LETTER I WITH BREVE
    "i(", "\u012d", // LATIN SMALL LETTER I WITH BREVE
    "I;", "\u012e", // LATIN CAPITAL LETTER I WITH OGONEK
    "i;", "\u012f", // LATIN SMALL LETTER I WITH OGONEK
    "I.", "\u0130", // LATIN CAPITAL LETTER I WITH DOT ABOVE
    "i.", "\u0131", // LATIN SMALL LETTER I DOTLESS
    "IJ", "\u0132", // LATIN CAPITAL LIGATURE IJ
    "ij", "\u0133", // LATIN SMALL LIGATURE IJ
    "J>", "\u0134", // LATIN CAPITAL LETTER J WITH CIRCUMFLEX
    "j>", "\u0135", // LATIN SMALL LETTER J WITH CIRCUMFLEX
    "K,", "\u0136", // LATIN CAPITAL LETTER K WITH CEDILLA
    "k,", "\u0137", // LATIN SMALL LETTER K WITH CEDILLA
    "kk", "\u0138", // LATIN SMALL LETTER KRA (Greenlandic)
    "L'", "\u0139", // LATIN CAPITAL LETTER L WITH ACUTE
    "l'", "\u013a", // LATIN SMALL LETTER L WITH ACUTE
    "L,", "\u013b", // LATIN CAPITAL LETTER L WITH CEDILLA
    "l,", "\u013c", // LATIN SMALL LETTER L WITH CEDILLA
    "L<", "\u013d", // LATIN CAPITAL LETTER L WITH CARON
    "l<", "\u013e", // LATIN SMALL LETTER L WITH CARON
    "L.", "\u013f", // LATIN CAPITAL LETTER L WITH MIDDLE DOT
    "l.", "\u0140", // LATIN SMALL LETTER L WITH MIDDLE DOT
    "L/", "\u0141", // LATIN CAPITAL LETTER L WITH STROKE
    "l/", "\u0142", // LATIN SMALL LETTER L WITH STROKE
    "N'", "\u0143", // LATIN CAPITAL LETTER N WITH ACUTE
    "n'", "\u0144", // LATIN SMALL LETTER N WITH ACUTE
    "N,", "\u0145", // LATIN CAPITAL LETTER N WITH CEDILLA
    "n,", "\u0146", // LATIN SMALL LETTER N WITH CEDILLA
    "N<", "\u0147", // LATIN CAPITAL LETTER N WITH CARON
    "n<", "\u0148", // LATIN SMALL LETTER N WITH CARON
    "'n", "\u0149", // LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
    "NG", "\u014a", // LATIN CAPITAL LETTER ENG (Lappish)
    "ng", "\u014b", // LATIN SMALL LETTER ENG (Lappish)
    "O-", "\u014c", // LATIN CAPITAL LETTER O WITH MACRON
    "o-", "\u014d", // LATIN SMALL LETTER O WITH MACRON
    "O(", "\u014e", // LATIN CAPITAL LETTER O WITH BREVE
    "o(", "\u014f", // LATIN SMALL LETTER O WITH BREVE
    "O\"", "\u0150", // LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
    "o\"", "\u0151", // LATIN SMALL LETTER O WITH DOUBLE ACUTE
    "OE", "\u0152", // LATIN CAPITAL LIGATURE OE
    "oe", "\u0153", // LATIN SMALL LIGATURE OE
    "R'", "\u0154", // LATIN CAPITAL LETTER R WITH ACUTE
    "r'", "\u0155", // LATIN SMALL LETTER R WITH ACUTE
    "R,", "\u0156", // LATIN CAPITAL LETTER R WITH CEDILLA
    "r,", "\u0157", // LATIN SMALL LETTER R WITH CEDILLA
    "R<", "\u0158", // LATIN CAPITAL LETTER R WITH CARON
    "r<", "\u0159", // LATIN SMALL LETTER R WITH CARON
    "S'", "\u015a", // LATIN CAPITAL LETTER S WITH ACUTE
    "s'", "\u015b", // LATIN SMALL LETTER S WITH ACUTE
    "S>", "\u015c", // LATIN CAPITAL LETTER S WITH CIRCUMFLEX
    "s>", "\u015d", // LATIN SMALL LETTER S WITH CIRCUMFLEX
    "S,", "\u015e", // LATIN CAPITAL LETTER S WITH CEDILLA
    "s,", "\u015f", // LATIN SMALL LETTER S WITH CEDILLA
    "S<", "\u0160", // LATIN CAPITAL LETTER S WITH CARON
    "s<", "\u0161", // LATIN SMALL LETTER S WITH CARON
    "T,", "\u0162", // LATIN CAPITAL LETTER T WITH CEDILLA
    "t,", "\u0163", // LATIN SMALL LETTER T WITH CEDILLA
    "T<", "\u0164", // LATIN CAPITAL LETTER T WITH CARON
    "t<", "\u0165", // LATIN SMALL LETTER T WITH CARON
    "T/", "\u0166", // LATIN CAPITAL LETTER T WITH STROKE
    "t/", "\u0167", // LATIN SMALL LETTER T WITH STROKE
    "U?", "\u0168", // LATIN CAPITAL LETTER U WITH TILDE
    "u?", "\u0169", // LATIN SMALL LETTER U WITH TILDE
    "U-", "\u016a", // LATIN CAPITAL LETTER U WITH MACRON
    "u-", "\u016b", // LATIN SMALL LETTER U WITH MACRON
    "U(", "\u016c", // LATIN CAPITAL LETTER U WITH BREVE
    "u(", "\u016d", // LATIN SMALL LETTER U WITH BREVE
    "U0", "\u016e", // LATIN CAPITAL LETTER U WITH RING ABOVE
    "u0", "\u016f", // LATIN SMALL LETTER U WITH RING ABOVE
    "U\"", "\u0170", // LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
    "u\"", "\u0171", // LATIN SMALL LETTER U WITH DOUBLE ACUTE
    "U;", "\u0172", // LATIN CAPITAL LETTER U WITH OGONEK
    "u;", "\u0173", // LATIN SMALL LETTER U WITH OGONEK
    "W>", "\u0174", // LATIN CAPITAL LETTER W WITH CIRCUMFLEX
    "w>", "\u0175", // LATIN SMALL LETTER W WITH CIRCUMFLEX
    "Y>", "\u0176", // LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
    "y>", "\u0177", // LATIN SMALL LETTER Y WITH CIRCUMFLEX
    "Y:", "\u0178", // LATIN CAPITAL LETTER Y WITH DIAERESIS
    "Z'", "\u0179", // LATIN CAPITAL LETTER Z WITH ACUTE
    "z'", "\u017a", // LATIN SMALL LETTER Z WITH ACUTE
    "Z.", "\u017b", // LATIN CAPITAL LETTER Z WITH DOT ABOVE
    "z.", "\u017c", // LATIN SMALL LETTER Z WITH DOT ABOVE
    "Z<", "\u017d", // LATIN CAPITAL LETTER Z WITH CARON
    "z<", "\u017e", // LATIN SMALL LETTER Z WITH CARON
    "O9", "\u01a0", // LATIN CAPITAL LETTER O WITH HORN
    "o9", "\u01a1", // LATIN SMALL LETTER O WITH HORN
    "OI", "\u01a2", // LATIN CAPITAL LETTER OI
    "oi", "\u01a3", // LATIN SMALL LETTER OI
    "yr", "\u01a6", // LATIN LETTER YR
    "U9", "\u01af", // LATIN CAPITAL LETTER U WITH HORN
    "u9", "\u01b0", // LATIN SMALL LETTER U WITH HORN
    "Z/", "\u01b5", // LATIN CAPITAL LETTER Z WITH STROKE
    "z/", "\u01b6", // LATIN SMALL LETTER Z WITH STROKE
    "ED", "\u01b7", // LATIN CAPITAL LETTER EZH
    "A<", "\u01cd", // LATIN CAPITAL LETTER A WITH CARON
    "a<", "\u01ce", // LATIN SMALL LETTER A WITH CARON
    "I<", "\u01cf", // LATIN CAPITAL LETTER I WITH CARON
    "i<", "\u01d0", // LATIN SMALL LETTER I WITH CARON
    "O<", "\u01d1", // LATIN CAPITAL LETTER O WITH CARON
    "o<", "\u01d2", // LATIN SMALL LETTER O WITH CARON
    "U<", "\u01d3", // LATIN CAPITAL LETTER U WITH CARON
    "u<", "\u01d4", // LATIN SMALL LETTER U WITH CARON
    "U:-", "\u01d5", // LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
    "u:-", "\u01d6", // LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
    "U:'", "\u01d7", // LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
    "u:'", "\u01d8", // LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
    "U:<", "\u01d9", // LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
    "u:<", "\u01da", // LATIN SMALL LETTER U WITH DIAERESIS AND CARON
    "U:!", "\u01db", // LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
    "u:!", "\u01dc", // LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
    "A1", "\u01de", // LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
    "a1", "\u01df", // LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
    "A7", "\u01e0", // LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
    "a7", "\u01e1", // LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
    "A3", "\u01e2", // LATIN CAPITAL LETTER AE WITH MACRON
    "a3", "\u01e3", // LATIN SMALL LETTER AE WITH MACRON
    "G/", "\u01e4", // LATIN CAPITAL LETTER G WITH STROKE
    "g/", "\u01e5", // LATIN SMALL LETTER G WITH STROKE
    "G<", "\u01e6", // LATIN CAPITAL LETTER G WITH CARON
    "g<", "\u01e7", // LATIN SMALL LETTER G WITH CARON
    "K<", "\u01e8", // LATIN CAPITAL LETTER K WITH CARON
    "k<", "\u01e9", // LATIN SMALL LETTER K WITH CARON
    "O;", "\u01ea", // LATIN CAPITAL LETTER O WITH OGONEK
    "o;", "\u01eb", // LATIN SMALL LETTER O WITH OGONEK
    "O1", "\u01ec", // LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
    "o1", "\u01ed", // LATIN SMALL LETTER O WITH OGONEK AND MACRON
    "EZ", "\u01ee", // LATIN CAPITAL LETTER EZH WITH CARON
    "ez", "\u01ef", // LATIN SMALL LETTER EZH WITH CARON
    "j<", "\u01f0", // LATIN SMALL LETTER J WITH CARON
    "G'", "\u01f4", // LATIN CAPITAL LETTER G WITH ACUTE
    "g'", "\u01f5", // LATIN SMALL LETTER G WITH ACUTE
    "AA'", "\u01fa", // LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
    "aa'", "\u01fb", // LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
    "AE'", "\u01fc", // LATIN CAPITAL LETTER AE WITH ACUTE
    "ae'", "\u01fd", // LATIN SMALL LETTER AE WITH ACUTE
    "O/'", "\u01fe", // LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
    "o/'", "\u01ff", // LATIN SMALL LETTER O WITH STROKE AND ACUTE
    ";S", "\u02bf", // MODIFIER LETTER LEFT HALF RING
    "'<", "\u02c7", // CARON
    "'(", "\u02d8", // BREVE
    "'.", "\u02d9", // DOT ABOVE
    "'0", "\u02da", // RING ABOVE
    "';", "\u02db", // OGONEK
    "'\"", "\u02dd", // DOUBLE ACUTE ACCENT
    "A%", "\u0386", // GREEK CAPITAL LETTER ALPHA WITH ACUTE
    "E%", "\u0388", // GREEK CAPITAL LETTER EPSILON WITH ACUTE
    "Y%", "\u0389", // GREEK CAPITAL LETTER ETA WITH ACUTE
    "I%", "\u038a", // GREEK CAPITAL LETTER IOTA WITH ACUTE
    "O%", "\u038c", // GREEK CAPITAL LETTER OMICRON WITH ACUTE
    "U%", "\u038e", // GREEK CAPITAL LETTER UPSILON WITH ACUTE
    "W%", "\u038f", // GREEK CAPITAL LETTER OMEGA WITH ACUTE
    "i3", "\u0390", // GREEK SMALL LETTER IOTA WITH ACUTE AND DIAERESIS
    "A*", "\u0391", // GREEK CAPITAL LETTER ALPHA
    "B*", "\u0392", // GREEK CAPITAL LETTER BETA
    "G*", "\u0393", // GREEK CAPITAL LETTER GAMMA
    "D*", "\u0394", // GREEK CAPITAL LETTER DELTA
    "E*", "\u0395", // GREEK CAPITAL LETTER EPSILON
    "Z*", "\u0396", // GREEK CAPITAL LETTER ZETA
    "Y*", "\u0397", // GREEK CAPITAL LETTER ETA
    "H*", "\u0398", // GREEK CAPITAL LETTER THETA
    "I*", "\u0399", // GREEK CAPITAL LETTER IOTA
    "K*", "\u039a", // GREEK CAPITAL LETTER KAPPA
    "L*", "\u039b", // GREEK CAPITAL LETTER LAMDA
    "M*", "\u039c", // GREEK CAPITAL LETTER MU
    "N*", "\u039d", // GREEK CAPITAL LETTER NU
    "C*", "\u039e", // GREEK CAPITAL LETTER XI
    "O*", "\u039f", // GREEK CAPITAL LETTER OMICRON
    "P*", "\u03a0", // GREEK CAPITAL LETTER PI
    "R*", "\u03a1", // GREEK CAPITAL LETTER RHO
    "S*", "\u03a3", // GREEK CAPITAL LETTER SIGMA
    "T*", "\u03a4", // GREEK CAPITAL LETTER TAU
    "U*", "\u03a5", // GREEK CAPITAL LETTER UPSILON
    "F*", "\u03a6", // GREEK CAPITAL LETTER PHI
    "X*", "\u03a7", // GREEK CAPITAL LETTER CHI
    "Q*", "\u03a8", // GREEK CAPITAL LETTER PSI
    "W*", "\u03a9", // GREEK CAPITAL LETTER OMEGA
    "J*", "\u03aa", // GREEK CAPITAL LETTER IOTA WITH DIAERESIS
    "V*", "\u03ab", // GREEK CAPITAL LETTER UPSILON WITH DIAERESIS
    "a%", "\u03ac", // GREEK SMALL LETTER ALPHA WITH ACUTE
    "e%", "\u03ad", // GREEK SMALL LETTER EPSILON WITH ACUTE
    "y%", "\u03ae", // GREEK SMALL LETTER ETA WITH ACUTE
    "i%", "\u03af", // GREEK SMALL LETTER IOTA WITH ACUTE
    "u3", "\u03b0", // GREEK SMALL LETTER UPSILON WITH ACUTE AND DIAERESIS
    "a*", "\u03b1", // GREEK SMALL LETTER ALPHA
    "b*", "\u03b2", // GREEK SMALL LETTER BETA
    "g*", "\u03b3", // GREEK SMALL LETTER GAMMA
    "d*", "\u03b4", // GREEK SMALL LETTER DELTA
    "e*", "\u03b5", // GREEK SMALL LETTER EPSILON
    "z*", "\u03b6", // GREEK SMALL LETTER ZETA
    "y*", "\u03b7", // GREEK SMALL LETTER ETA
    "h*", "\u03b8", // GREEK SMALL LETTER THETA
    "i*", "\u03b9", // GREEK SMALL LETTER IOTA
    "k*", "\u03ba", // GREEK SMALL LETTER KAPPA
    "l*", "\u03bb", // GREEK SMALL LETTER LAMDA
    "m*", "\u03bc", // GREEK SMALL LETTER MU
    "n*", "\u03bd", // GREEK SMALL LETTER NU
    "c*", "\u03be", // GREEK SMALL LETTER XI
    "o*", "\u03bf", // GREEK SMALL LETTER OMICRON
    "p*", "\u03c0", // GREEK SMALL LETTER PI
    "r*", "\u03c1", // GREEK SMALL LETTER RHO
    "*s", "\u03c2", // GREEK SMALL LETTER FINAL SIGMA
    "s*", "\u03c3", // GREEK SMALL LETTER SIGMA
    "t*", "\u03c4", // GREEK SMALL LETTER TAU
    "u*", "\u03c5", // GREEK SMALL LETTER UPSILON
    "f*", "\u03c6", // GREEK SMALL LETTER PHI
    "x*", "\u03c7", // GREEK SMALL LETTER CHI
    "q*", "\u03c8", // GREEK SMALL LETTER PSI
    "w*", "\u03c9", // GREEK SMALL LETTER OMEGA
    "j*", "\u03ca", // GREEK SMALL LETTER IOTA WITH DIAERESIS
    "v*", "\u03cb", // GREEK SMALL LETTER UPSILON WITH DIAERESIS
    "o%", "\u03cc", // GREEK SMALL LETTER OMICRON WITH ACUTE
    "u%", "\u03cd", // GREEK SMALL LETTER UPSILON WITH ACUTE
    "w%", "\u03ce", // GREEK SMALL LETTER OMEGA WITH ACUTE
    "'G", "\u03d8", // GREEK NUMERAL SIGN
    ",G", "\u03d9", // GREEK LOWER NUMERAL SIGN
    "T3", "\u03da", // GREEK CAPITAL LETTER STIGMA
    "t3", "\u03db", // GREEK SMALL LETTER STIGMA
    "M3", "\u03dc", // GREEK CAPITAL LETTER DIGAMMA
    "m3", "\u03dd", // GREEK SMALL LETTER DIGAMMA
    "K3", "\u03de", // GREEK CAPITAL LETTER KOPPA
    "k3", "\u03df", // GREEK SMALL LETTER KOPPA
    "P3", "\u03e0", // GREEK CAPITAL LETTER SAMPI
    "p3", "\u03e1", // GREEK SMALL LETTER SAMPI
    "'%", "\u03f4", // ACUTE ACCENT AND DIAERESIS (Tonos and Dialytika)
    "j3", "\u03f5", // GREEK IOTA BELOW
    "IO", "\u0401", // CYRILLIC CAPITAL LETTER IO
    "D%", "\u0402", // CYRILLIC CAPITAL LETTER DJE (Serbocroatian)
    "G%", "\u0403", // CYRILLIC CAPITAL LETTER GJE (Macedonian)
    "IE", "\u0404", // CYRILLIC CAPITAL LETTER UKRAINIAN IE
    "DS", "\u0405", // CYRILLIC CAPITAL LETTER DZE (Macedonian)
    "II", "\u0406", // CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
    "YI", "\u0407", // CYRILLIC CAPITAL LETTER YI (Ukrainian)
    "J%", "\u0408", // CYRILLIC CAPITAL LETTER JE
    "LJ", "\u0409", // CYRILLIC CAPITAL LETTER LJE
    "NJ", "\u040a", // CYRILLIC CAPITAL LETTER NJE
    "Ts", "\u040b", // CYRILLIC CAPITAL LETTER TSHE (Serbocroatian)
    "KJ", "\u040c", // CYRILLIC CAPITAL LETTER KJE (Macedonian)
    "V%", "\u040e", // CYRILLIC CAPITAL LETTER SHORT U (Byelorussian)
    "DZ", "\u040f", // CYRILLIC CAPITAL LETTER DZHE
    "A=", "\u0410", // CYRILLIC CAPITAL LETTER A
    "B=", "\u0411", // CYRILLIC CAPITAL LETTER BE
    "V=", "\u0412", // CYRILLIC CAPITAL LETTER VE
    "G=", "\u0413", // CYRILLIC CAPITAL LETTER GHE
    "D=", "\u0414", // CYRILLIC CAPITAL LETTER DE
    "E=", "\u0415", // CYRILLIC CAPITAL LETTER IE
    "Z%", "\u0416", // CYRILLIC CAPITAL LETTER ZHE
    "Z=", "\u0417", // CYRILLIC CAPITAL LETTER ZE
    "I=", "\u0418", // CYRILLIC CAPITAL LETTER I
    "J=", "\u0419", // CYRILLIC CAPITAL LETTER SHORT I
    "K=", "\u041a", // CYRILLIC CAPITAL LETTER KA
    "L=", "\u041b", // CYRILLIC CAPITAL LETTER EL
    "M=", "\u041c", // CYRILLIC CAPITAL LETTER EM
    "N=", "\u041d", // CYRILLIC CAPITAL LETTER EN
    "O=", "\u041e", // CYRILLIC CAPITAL LETTER O
    "P=", "\u041f", // CYRILLIC CAPITAL LETTER PE
    "R=", "\u0420", // CYRILLIC CAPITAL LETTER ER
    "S=", "\u0421", // CYRILLIC CAPITAL LETTER ES
    "T=", "\u0422", // CYRILLIC CAPITAL LETTER TE
    "U=", "\u0423", // CYRILLIC CAPITAL LETTER U
    "F=", "\u0424", // CYRILLIC CAPITAL LETTER EF
    "H=", "\u0425", // CYRILLIC CAPITAL LETTER HA
    "C=", "\u0426", // CYRILLIC CAPITAL LETTER TSE
    "C%", "\u0427", // CYRILLIC CAPITAL LETTER CHE
    "S%", "\u0428", // CYRILLIC CAPITAL LETTER SHA
    "Sc", "\u0429", // CYRILLIC CAPITAL LETTER SHCHA
    "=\"", "\u042a", // CYRILLIC CAPITAL LETTER HARD SIGN
    "Y=", "\u042b", // CYRILLIC CAPITAL LETTER YERU
    "%\"", "\u042c", // CYRILLIC CAPITAL LETTER SOFT SIGN
    "JE", "\u042d", // CYRILLIC CAPITAL LETTER E
    "JU", "\u042e", // CYRILLIC CAPITAL LETTER YU
    "JA", "\u042f", // CYRILLIC CAPITAL LETTER YA
    "a=", "\u0430", // CYRILLIC SMALL LETTER A
    "b=", "\u0431", // CYRILLIC SMALL LETTER BE
    "v=", "\u0432", // CYRILLIC SMALL LETTER VE
    "g=", "\u0433", // CYRILLIC SMALL LETTER GHE
    "d=", "\u0434", // CYRILLIC SMALL LETTER DE
    "e=", "\u0435", // CYRILLIC SMALL LETTER IE
    "z%", "\u0436", // CYRILLIC SMALL LETTER ZHE
    "z=", "\u0437", // CYRILLIC SMALL LETTER ZE
    "i=", "\u0438", // CYRILLIC SMALL LETTER I
    "j=", "\u0439", // CYRILLIC SMALL LETTER SHORT I
    "k=", "\u043a", // CYRILLIC SMALL LETTER KA
    "l=", "\u043b", // CYRILLIC SMALL LETTER EL
    "m=", "\u043c", // CYRILLIC SMALL LETTER EM
    "n=", "\u043d", // CYRILLIC SMALL LETTER EN
    "o=", "\u043e", // CYRILLIC SMALL LETTER O
    "p=", "\u043f", // CYRILLIC SMALL LETTER PE
    "r=", "\u0440", // CYRILLIC SMALL LETTER ER
    "s=", "\u0441", // CYRILLIC SMALL LETTER ES
    "t=", "\u0442", // CYRILLIC SMALL LETTER TE
    "u=", "\u0443", // CYRILLIC SMALL LETTER U
    "f=", "\u0444", // CYRILLIC SMALL LETTER EF
    "h=", "\u0445", // CYRILLIC SMALL LETTER HA
    "c=", "\u0446", // CYRILLIC SMALL LETTER TSE
    "c%", "\u0447", // CYRILLIC SMALL LETTER CHE
    "s%", "\u0448", // CYRILLIC SMALL LETTER SHA
    "sc", "\u0449", // CYRILLIC SMALL LETTER SHCHA
    "='", "\u044a", // CYRILLIC SMALL LETTER HARD SIGN
    "y=", "\u044b", // CYRILLIC SMALL LETTER YERU
    "%'", "\u044c", // CYRILLIC SMALL LETTER SOFT SIGN
    "je", "\u044d", // CYRILLIC SMALL LETTER E
    "ju", "\u044e", // CYRILLIC SMALL LETTER YU
    "ja", "\u044f", // CYRILLIC SMALL LETTER YA
    "io", "\u0451", // CYRILLIC SMALL LETTER IO
    "d%", "\u0452", // CYRILLIC SMALL LETTER DJE (Serbocroatian)
    "g%", "\u0453", // CYRILLIC SMALL LETTER GJE (Macedonian)
    "ie", "\u0454", // CYRILLIC SMALL LETTER UKRAINIAN IE
    "ds", "\u0455", // CYRILLIC SMALL LETTER DZE (Macedonian)
    "ii", "\u0456", // CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
    "yi", "\u0457", // CYRILLIC SMALL LETTER YI (Ukrainian)
    "j%", "\u0458", // CYRILLIC SMALL LETTER JE
    "lj", "\u0459", // CYRILLIC SMALL LETTER LJE
    "nj", "\u045a", // CYRILLIC SMALL LETTER NJE
    "ts", "\u045b", // CYRILLIC SMALL LETTER TSHE (Serbocroatian)
    "kj", "\u045c", // CYRILLIC SMALL LETTER KJE (Macedonian)
    "v%", "\u045e", // CYRILLIC SMALL LETTER SHORT U (Byelorussian)
    "dz", "\u045f", // CYRILLIC SMALL LETTER DZHE
    "Y3", "\u0462", // CYRILLIC CAPITAL LETTER YAT
    "y3", "\u0463", // CYRILLIC SMALL LETTER YAT
    "O3", "\u046a", // CYRILLIC CAPITAL LETTER BIG YUS
    "o3", "\u046b", // CYRILLIC SMALL LETTER BIG YUS
    "F3", "\u0472", // CYRILLIC CAPITAL LETTER FITA
    "f3", "\u0473", // CYRILLIC SMALL LETTER FITA
    "V3", "\u0474", // CYRILLIC CAPITAL LETTER IZHITSA
    "v3", "\u0475", // CYRILLIC SMALL LETTER IZHITSA
    "C3", "\u0480", // CYRILLIC CAPITAL LETTER KOPPA
    "c3", "\u0481", // CYRILLIC SMALL LETTER KOPPA
    "G3", "\u0490", // CYRILLIC CAPITAL LETTER GHE WITH UPTURN
    "g3", "\u0491", // CYRILLIC SMALL LETTER GHE WITH UPTURN
    "A+", "\u05d0", // HEBREW LETTER ALEF
    "B+", "\u05d1", // HEBREW LETTER BET
    "G+", "\u05d2", // HEBREW LETTER GIMEL
    "D+", "\u05d3", // HEBREW LETTER DALET
    "H+", "\u05d4", // HEBREW LETTER HE
    "W+", "\u05d5", // HEBREW LETTER VAV
    "Z+", "\u05d6", // HEBREW LETTER ZAYIN
    "X+", "\u05d7", // HEBREW LETTER HET
    "Tj", "\u05d8", // HEBREW LETTER TET
    "J+", "\u05d9", // HEBREW LETTER YOD
    "K%", "\u05da", // HEBREW LETTER FINAL KAF
    "K+", "\u05db", // HEBREW LETTER KAF
    "L+", "\u05dc", // HEBREW LETTER LAMED
    "M%", "\u05dd", // HEBREW LETTER FINAL MEM
    "M+", "\u05de", // HEBREW LETTER MEM
    "N%", "\u05df", // HEBREW LETTER FINAL NUN
    "N+", "\u05e0", // HEBREW LETTER NUN
    "S+", "\u05e1", // HEBREW LETTER SAMEKH
    "E+", "\u05e2", // HEBREW LETTER AYIN
    "P%", "\u05e3", // HEBREW LETTER FINAL PE
    "P+", "\u05e4", // HEBREW LETTER PE
    "Zj", "\u05e5", // HEBREW LETTER FINAL TSADI
    "ZJ", "\u05e6", // HEBREW LETTER TSADI
    "Q+", "\u05e7", // HEBREW LETTER QOF
    "R+", "\u05e8", // HEBREW LETTER RESH
    "Sh", "\u05e9", // HEBREW LETTER SHIN
    "T+", "\u05ea", // HEBREW LETTER TAV
    ",+", "\u060c", // ARABIC COMMA
    ";+", "\u061b", // ARABIC SEMICOLON
    "?+", "\u061f", // ARABIC QUESTION MARK
    "H'", "\u0621", // ARABIC LETTER HAMZA
    "aM", "\u0622", // ARABIC LETTER ALEF WITH MADDA ABOVE
    "aH", "\u0623", // ARABIC LETTER ALEF WITH HAMZA ABOVE
    "wH", "\u0624", // ARABIC LETTER WAW WITH HAMZA ABOVE
    "ah", "\u0625", // ARABIC LETTER ALEF WITH HAMZA BELOW
    "yH", "\u0626", // ARABIC LETTER YEH WITH HAMZA ABOVE
    "a+", "\u0627", // ARABIC LETTER ALEF
    "b+", "\u0628", // ARABIC LETTER BEH
    "tm", "\u0629", // ARABIC LETTER TEH MARBUTA
    "t+", "\u062a", // ARABIC LETTER TEH
    "tk", "\u062b", // ARABIC LETTER THEH
    "g+", "\u062c", // ARABIC LETTER JEEM
    "hk", "\u062d", // ARABIC LETTER HAH
    "x+", "\u062e", // ARABIC LETTER KHAH
    "d+", "\u062f", // ARABIC LETTER DAL
    "dk", "\u0630", // ARABIC LETTER THAL
    "r+", "\u0631", // ARABIC LETTER REH
    "z+", "\u0632", // ARABIC LETTER ZAIN
    "s+", "\u0633", // ARABIC LETTER SEEN
    "sn", "\u0634", // ARABIC LETTER SHEEN
    "c+", "\u0635", // ARABIC LETTER SAD
    "dd", "\u0636", // ARABIC LETTER DAD
    "tj", "\u0637", // ARABIC LETTER TAH
    "zH", "\u0638", // ARABIC LETTER ZAH
    "e+", "\u0639", // ARABIC LETTER AIN
    "i+", "\u063a", // ARABIC LETTER GHAIN
    "++", "\u0640", // ARABIC TATWEEL
    "f+", "\u0641", // ARABIC LETTER FEH
    "q+", "\u0642", // ARABIC LETTER QAF
    "k+", "\u0643", // ARABIC LETTER KAF
    "l+", "\u0644", // ARABIC LETTER LAM
    "m+", "\u0645", // ARABIC LETTER MEEM
    "n+", "\u0646", // ARABIC LETTER NOON
    "h+", "\u0647", // ARABIC LETTER HEH
    "w+", "\u0648", // ARABIC LETTER WAW
    "j+", "\u0649", // ARABIC LETTER ALEF MAKSURA
    "y+", "\u064a", // ARABIC LETTER YEH
    ":+", "\u064b", // ARABIC FATHATAN
    "\"+", "\u064c", // ARABIC DAMMATAN
    "=+", "\u064d", // ARABIC KASRATAN
    "/+", "\u064e", // ARABIC FATHA
    "'+", "\u064f", // ARABIC DAMMA
    "1+", "\u0650", // ARABIC KASRA
    "3+", "\u0651", // ARABIC SHADDA
    "0+", "\u0652", // ARABIC SUKUN
    "aS", "\u0670", // SUPERSCRIPT ARABIC LETTER ALEF
    "p+", "\u067e", // ARABIC LETTER PEH
    "v+", "\u06a4", // ARABIC LETTER VEH
    "gf", "\u06af", // ARABIC LETTER GAF
    "0a", "\u06f0", // EASTERN ARABIC-INDIC DIGIT ZERO
    "1a", "\u06f1", // EASTERN ARABIC-INDIC DIGIT ONE
    "2a", "\u06f2", // EASTERN ARABIC-INDIC DIGIT TWO
    "3a", "\u06f3", // EASTERN ARABIC-INDIC DIGIT THREE
    "4a", "\u06f4", // EASTERN ARABIC-INDIC DIGIT FOUR
    "5a", "\u06f5", // EASTERN ARABIC-INDIC DIGIT FIVE
    "6a", "\u06f6", // EASTERN ARABIC-INDIC DIGIT SIX
    "7a", "\u06f7", // EASTERN ARABIC-INDIC DIGIT SEVEN
    "8a", "\u06f8", // EASTERN ARABIC-INDIC DIGIT EIGHT
    "9a", "\u06f9", // EASTERN ARABIC-INDIC DIGIT NINE
    "A-0", "\u1e00", // LATIN CAPITAL LETTER A WITH RING BELOW
    "a-0", "\u1e01", // LATIN SMALL LETTER A WITH RING BELOW
    "B.", "\u1e02", // LATIN CAPITAL LETTER B WITH DOT ABOVE
    "b.", "\u1e03", // LATIN SMALL LETTER B WITH DOT ABOVE
    "B-.", "\u1e04", // LATIN CAPITAL LETTER B WITH DOT BELOW
    "b-.", "\u1e05", // LATIN SMALL LETTER B WITH DOT BELOW
    "B_", "\u1e06", // LATIN CAPITAL LETTER B WITH LINE BELOW
    "b_", "\u1e07", // LATIN SMALL LETTER B WITH LINE BELOW
    "C,'", "\u1e08", // LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
    "c,'", "\u1e09", // LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
    "D.", "\u1e0a", // LATIN CAPITAL LETTER D WITH DOT ABOVE
    "d.", "\u1e0b", // LATIN SMALL LETTER D WITH DOT ABOVE
    "D-.", "\u1e0c", // LATIN CAPITAL LETTER D WITH DOT BELOW
    "d-.", "\u1e0d", // LATIN SMALL LETTER D WITH DOT BELOW
    "D_", "\u1e0e", // LATIN CAPITAL LETTER D WITH LINE BELOW
    "d_", "\u1e0f", // LATIN SMALL LETTER D WITH LINE BELOW
    "D,", "\u1e10", // LATIN CAPITAL LETTER D WITH CEDILLA
    "d,", "\u1e11", // LATIN SMALL LETTER D WITH CEDILLA
    "D->", "\u1e12", // LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
    "d->", "\u1e13", // LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
    "E-!", "\u1e14", // LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
    "e-!", "\u1e15", // LATIN SMALL LETTER E WITH MACRON AND GRAVE
    "E-'", "\u1e16", // LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
    "e-'", "\u1e17", // LATIN SMALL LETTER E WITH MACRON AND ACUTE
    "E->", "\u1e18", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
    "e->", "\u1e19", // LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
    "E-?", "\u1e1a", // LATIN CAPITAL LETTER E WITH TILDE BELOW
    "e-?", "\u1e1b", // LATIN SMALL LETTER E WITH TILDE BELOW
    "E,(", "\u1e1c", // LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
    "e,(", "\u1e1d", // LATIN SMALL LETTER E WITH CEDILLA AND BREVE
    "F.", "\u1e1e", // LATIN CAPITAL LETTER F WITH DOT ABOVE
    "f.", "\u1e1f", // LATIN SMALL LETTER F WITH DOT ABOVE
    "G-", "\u1e20", // LATIN CAPITAL LETTER G WITH MACRON
    "g-", "\u1e21", // LATIN SMALL LETTER G WITH MACRON
    "H.", "\u1e22", // LATIN CAPITAL LETTER H WITH DOT ABOVE
    "h.", "\u1e23", // LATIN SMALL LETTER H WITH DOT ABOVE
    "H-.", "\u1e24", // LATIN CAPITAL LETTER H WITH DOT BELOW
    "h-.", "\u1e25", // LATIN SMALL LETTER H WITH DOT BELOW
    "H:", "\u1e26", // LATIN CAPITAL LETTER H WITH DIAERESIS
    "h:", "\u1e27", // LATIN SMALL LETTER H WITH DIAERESIS
    "H,", "\u1e28", // LATIN CAPITAL LETTER H WITH CEDILLA
    "h,", "\u1e29", // LATIN SMALL LETTER H WITH CEDILLA
    "H-(", "\u1e2a", // LATIN CAPITAL LETTER H WITH BREVE BELOW
    "h-(", "\u1e2b", // LATIN SMALL LETTER H WITH BREVE BELOW
    "I-?", "\u1e2c", // LATIN CAPITAL LETTER I WITH TILDE BELOW
    "i-?", "\u1e2d", // LATIN SMALL LETTER I WITH TILDE BELOW
    "I:'", "\u1e2e", // LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
    "i:'", "\u1e2f", // LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
    "K'", "\u1e30", // LATIN CAPITAL LETTER K WITH ACUTE
    "k'", "\u1e31", // LATIN SMALL LETTER K WITH ACUTE
    "K-.", "\u1e32", // LATIN CAPITAL LETTER K WITH DOT BELOW
    "k-.", "\u1e33", // LATIN SMALL LETTER K WITH DOT BELOW
    "K_", "\u1e34", // LATIN CAPITAL LETTER K WITH LINE BELOW
    "k_", "\u1e35", // LATIN SMALL LETTER K WITH LINE BELOW
    "L-.", "\u1e36", // LATIN CAPITAL LETTER L WITH DOT BELOW
    "l-.", "\u1e37", // LATIN SMALL LETTER L WITH DOT BELOW
    "L--.", "\u1e38", // LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
    "l--.", "\u1e39", // LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
    "L_", "\u1e3a", // LATIN CAPITAL LETTER L WITH LINE BELOW
    "l_", "\u1e3b", // LATIN SMALL LETTER L WITH LINE BELOW
    "L->", "\u1e3c", // LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
    "l->", "\u1e3d", // LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
    "M'", "\u1e3e", // LATIN CAPITAL LETTER M WITH ACUTE
    "m'", "\u1e3f", // LATIN SMALL LETTER M WITH ACUTE
    "M.", "\u1e40", // LATIN CAPITAL LETTER M WITH DOT ABOVE
    "m.", "\u1e41", // LATIN SMALL LETTER M WITH DOT ABOVE
    "M-.", "\u1e42", // LATIN CAPITAL LETTER M WITH DOT BELOW
    "m-.", "\u1e43", // LATIN SMALL LETTER M WITH DOT BELOW
    "N.", "\u1e44", // LATIN CAPITAL LETTER N WITH DOT ABOVE
    "n.", "\u1e45", // LATIN SMALL LETTER N WITH DOT ABOVE
    "N-.", "\u1e46", // LATIN CAPITAL LETTER N WITH DOT BELOW
    "n-.", "\u1e47", // LATIN SMALL LETTER N WITH DOT BELOW
    "N_", "\u1e48", // LATIN CAPITAL LETTER N WITH LINE BELOW
    "n_", "\u1e49", // LATIN SMALL LETTER N WITH LINE BELOW
    "N->", "\u1e4a", // LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
    "n->", "\u1e4b", // LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
    "O?'", "\u1e4c", // LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
    "o?'", "\u1e4d", // LATIN SMALL LETTER O WITH TILDE AND ACUTE
    "O?:", "\u1e4e", // LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
    "o?:", "\u1e4f", // LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
    "O-!", "\u1e50", // LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
    "o-!", "\u1e51", // LATIN SMALL LETTER O WITH MACRON AND GRAVE
    "O-'", "\u1e52", // LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
    "o-'", "\u1e53", // LATIN SMALL LETTER O WITH MACRON AND ACUTE
    "P'", "\u1e54", // LATIN CAPITAL LETTER P WITH ACUTE
    "p'", "\u1e55", // LATIN SMALL LETTER P WITH ACUTE
    "P.", "\u1e56", // LATIN CAPITAL LETTER P WITH DOT ABOVE
    "p.", "\u1e57", // LATIN SMALL LETTER P WITH DOT ABOVE
    "R.", "\u1e58", // LATIN CAPITAL LETTER R WITH DOT ABOVE
    "r.", "\u1e59", // LATIN SMALL LETTER R WITH DOT ABOVE
    "R-.", "\u1e5a", // LATIN CAPITAL LETTER R WITH DOT BELOW
    "r-.", "\u1e5b", // LATIN SMALL LETTER R WITH DOT BELOW
    "R--.", "\u1e5c", // LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
    "r--.", "\u1e5d", // LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
    "R_", "\u1e5e", // LATIN CAPITAL LETTER R WITH LINE BELOW
    "r_", "\u1e5f", // LATIN SMALL LETTER R WITH LINE BELOW
    "S.", "\u1e60", // LATIN CAPITAL LETTER S WITH DOT ABOVE
    "s.", "\u1e61", // LATIN SMALL LETTER S WITH DOT ABOVE
    "S-.", "\u1e62", // LATIN CAPITAL LETTER S WITH DOT BELOW
    "s-.", "\u1e63", // LATIN SMALL LETTER S WITH DOT BELOW
    "S'.", "\u1e64", // LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
    "s'.", "\u1e65", // LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
    "S<.", "\u1e66", // LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
    "s<.", "\u1e67", // LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
    "S.-.", "\u1e68", // LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
    "s.-.", "\u1e69", // LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
    "T.", "\u1e6a", // LATIN CAPITAL LETTER T WITH DOT ABOVE
    "t.", "\u1e6b", // LATIN SMALL LETTER T WITH DOT ABOVE
    "T-.", "\u1e6c", // LATIN CAPITAL LETTER T WITH DOT BELOW
    "t-.", "\u1e6d", // LATIN SMALL LETTER T WITH DOT BELOW
    "T_", "\u1e6e", // LATIN CAPITAL LETTER T WITH LINE BELOW
    "t_", "\u1e6f", // LATIN SMALL LETTER T WITH LINE BELOW
    "T->", "\u1e70", // LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
    "t->", "\u1e71", // LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
    "U--:", "\u1e72", // LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
    "u--:", "\u1e73", // LATIN SMALL LETTER U WITH DIAERESIS BELOW
    "U-?", "\u1e74", // LATIN CAPITAL LETTER U WITH TILDE BELOW
    "u-?", "\u1e75", // LATIN SMALL LETTER U WITH TILDE BELOW
    "U->", "\u1e76", // LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
    "u->", "\u1e77", // LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
    "U?'", "\u1e78", // LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
    "u?'", "\u1e79", // LATIN SMALL LETTER U WITH TILDE AND ACUTE
    "U-:", "\u1e7a", // LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
    "u-:", "\u1e7b", // LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
    "V?", "\u1e7c", // LATIN CAPITAL LETTER V WITH TILDE
    "v?", "\u1e7d", // LATIN SMALL LETTER V WITH TILDE
    "V-.", "\u1e7e", // LATIN CAPITAL LETTER V WITH DOT BELOW
    "v-.", "\u1e7f", // LATIN SMALL LETTER V WITH DOT BELOW
    "W!", "\u1e80", // LATIN CAPITAL LETTER W WITH GRAVE
    "w!", "\u1e81", // LATIN SMALL LETTER W WITH GRAVE
    "W'", "\u1e82", // LATIN CAPITAL LETTER W WITH ACUTE
    "w'", "\u1e83", // LATIN SMALL LETTER W WITH ACUTE
    "W:", "\u1e84", // LATIN CAPITAL LETTER W WITH DIAERESIS
    "w:", "\u1e85", // LATIN SMALL LETTER W WITH DIAERESIS
    "W.", "\u1e86", // LATIN CAPITAL LETTER W WITH DOT ABOVE
    "w.", "\u1e87", // LATIN SMALL LETTER W WITH DOT ABOVE
    "W-.", "\u1e88", // LATIN CAPITAL LETTER W WITH DOT BELOW
    "w-.", "\u1e89", // LATIN SMALL LETTER W WITH DOT BELOW
    "X.", "\u1e8a", // LATIN CAPITAL LETTER X WITH DOT ABOVE
    "x.", "\u1e8b", // LATIN SMALL LETTER X WITH DOT ABOVE
    "X:", "\u1e8c", // LATIN CAPITAL LETTER X WITH DIAERESIS
    "x:", "\u1e8d", // LATIN SMALL LETTER X WITH DIAERESIS
    "Y.", "\u1e8e", // LATIN CAPITAL LETTER Y WITH DOT ABOVE
    "y.", "\u1e8f", // LATIN SMALL LETTER Y WITH DOT ABOVE
    "Z>", "\u1e90", // LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
    "z>", "\u1e91", // LATIN SMALL LETTER Z WITH CIRCUMFLEX
    "Z-.", "\u1e92", // LATIN CAPITAL LETTER Z WITH DOT BELOW
    "z-.", "\u1e93", // LATIN SMALL LETTER Z WITH DOT BELOW
    "Z_", "\u1e94", // LATIN CAPITAL LETTER Z WITH LINE BELOW
    "z_", "\u1e95", // LATIN SMALL LETTER Z WITH LINE BELOW
    "h_", "\u1e96", // LATIN SMALL LETTER H WITH LINE BELOW
    "t:", "\u1e97", // LATIN SMALL LETTER T WITH DIAERESIS
    "w0", "\u1e98", // LATIN SMALL LETTER W WITH RING ABOVE
    "y0", "\u1e99", // LATIN SMALL LETTER Y WITH RING ABOVE
    "A-.", "\u1ea0", // LATIN CAPITAL LETTER A WITH DOT BELOW
    "a-.", "\u1ea1", // LATIN SMALL LETTER A WITH DOT BELOW
    "A2", "\u1ea2", // LATIN CAPITAL LETTER A WITH HOOK ABOVE
    "a2", "\u1ea3", // LATIN SMALL LETTER A WITH HOOK ABOVE
    "A>'", "\u1ea4", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
    "a>'", "\u1ea5", // LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
    "A>!", "\u1ea6", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
    "a>!", "\u1ea7", // LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
    "A>2", "\u1ea8", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    "a>2", "\u1ea9", // LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
    "A>?", "\u1eaa", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
    "a>?", "\u1eab", // LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
    "A>-.", "\u1eac", // LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    "a>-.", "\u1ead", // LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
    "A('", "\u1eae", // LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
    "a('", "\u1eaf", // LATIN SMALL LETTER A WITH BREVE AND ACUTE
    "A(!", "\u1eb0", // LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
    "a(!", "\u1eb1", // LATIN SMALL LETTER A WITH BREVE AND GRAVE
    "A(2", "\u1eb2", // LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
    "a(2", "\u1eb3", // LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
    "A(?", "\u1eb4", // LATIN CAPITAL LETTER A WITH BREVE AND TILDE
    "a(?", "\u1eb5", // LATIN SMALL LETTER A WITH BREVE AND TILDE
    "A(-.", "\u1eb6", // LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
    "a(-.", "\u1eb7", // LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
    "E-.", "\u1eb8", // LATIN CAPITAL LETTER E WITH DOT BELOW
    "e-.", "\u1eb9", // LATIN SMALL LETTER E WITH DOT BELOW
    "E2", "\u1eba", // LATIN CAPITAL LETTER E WITH HOOK ABOVE
    "e2", "\u1ebb", // LATIN SMALL LETTER E WITH HOOK ABOVE
    "E?", "\u1ebc", // LATIN CAPITAL LETTER E WITH TILDE
    "e?", "\u1ebd", // LATIN SMALL LETTER E WITH TILDE
    "E>'", "\u1ebe", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
    "e>'", "\u1ebf", // LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
    "E>!", "\u1ec0", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
    "e>!", "\u1ec1", // LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
    "E>2", "\u1ec2", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    "e>2", "\u1ec3", // LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
    "E>?", "\u1ec4", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
    "e>?", "\u1ec5", // LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
    "E>-.", "\u1ec6", // LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    "e>-.", "\u1ec7", // LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
    "I2", "\u1ec8", // LATIN CAPITAL LETTER I WITH HOOK ABOVE
    "i2", "\u1ec9", // LATIN SMALL LETTER I WITH HOOK ABOVE
    "I-.", "\u1eca", // LATIN CAPITAL LETTER I WITH DOT BELOW
    "i-.", "\u1ecb", // LATIN SMALL LETTER I WITH DOT BELOW
    "O-.", "\u1ecc", // LATIN CAPITAL LETTER O WITH DOT BELOW
    "o-.", "\u1ecd", // LATIN SMALL LETTER O WITH DOT BELOW
    "O2", "\u1ece", // LATIN CAPITAL LETTER O WITH HOOK ABOVE
    "o2", "\u1ecf", // LATIN SMALL LETTER O WITH HOOK ABOVE
    "O>'", "\u1ed0", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
    "o>'", "\u1ed1", // LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
    "O>!", "\u1ed2", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
    "o>!", "\u1ed3", // LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
    "O>2", "\u1ed4", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    "o>2", "\u1ed5", // LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
    "O>?", "\u1ed6", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
    "o>?", "\u1ed7", // LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
    "O>-.", "\u1ed8", // LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    "o>-.", "\u1ed9", // LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
    "O9'", "\u1eda", // LATIN CAPITAL LETTER O WITH HORN AND ACUTE
    "o9'", "\u1edb", // LATIN SMALL LETTER O WITH HORN AND ACUTE
    "O9!", "\u1edc", // LATIN CAPITAL LETTER O WITH HORN AND GRAVE
    "o9!", "\u1edd", // LATIN SMALL LETTER O WITH HORN AND GRAVE
    "O92", "\u1ede", // LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
    "o92", "\u1edf", // LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
    "O9?", "\u1ee0", // LATIN CAPITAL LETTER O WITH HORN AND TILDE
    "o9?", "\u1ee1", // LATIN SMALL LETTER O WITH HORN AND TILDE
    "O9-.", "\u1ee2", // LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
    "o9-.", "\u1ee3", // LATIN SMALL LETTER O WITH HORN AND DOT BELOW
    "U-.", "\u1ee4", // LATIN CAPITAL LETTER U WITH DOT BELOW
    "u-.", "\u1ee5", // LATIN SMALL LETTER U WITH DOT BELOW
    "U2", "\u1ee6", // LATIN CAPITAL LETTER U WITH HOOK ABOVE
    "u2", "\u1ee7", // LATIN SMALL LETTER U WITH HOOK ABOVE
    "U9'", "\u1ee8", // LATIN CAPITAL LETTER U WITH HORN AND ACUTE
    "u9'", "\u1ee9", // LATIN SMALL LETTER U WITH HORN AND ACUTE
    "U9!", "\u1eea", // LATIN CAPITAL LETTER U WITH HORN AND GRAVE
    "u9!", "\u1eeb", // LATIN SMALL LETTER U WITH HORN AND GRAVE
    "U92", "\u1eec", // LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
    "u92", "\u1eed", // LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
    "U9?", "\u1eee", // LATIN CAPITAL LETTER U WITH HORN AND TILDE
    "u9?", "\u1eef", // LATIN SMALL LETTER U WITH HORN AND TILDE
    "U9-.", "\u1ef0", // LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
    "u9-.", "\u1ef1", // LATIN SMALL LETTER U WITH HORN AND DOT BELOW
    "Y!", "\u1ef2", // LATIN CAPITAL LETTER Y WITH GRAVE
    "y!", "\u1ef3", // LATIN SMALL LETTER Y WITH GRAVE
    "Y-.", "\u1ef4", // LATIN CAPITAL LETTER Y WITH DOT BELOW
    "y-.", "\u1ef5", // LATIN SMALL LETTER Y WITH DOT BELOW
    "Y2", "\u1ef6", // LATIN CAPITAL LETTER Y WITH HOOK ABOVE
    "y2", "\u1ef7", // LATIN SMALL LETTER Y WITH HOOK ABOVE
    "Y?", "\u1ef8", // LATIN CAPITAL LETTER Y WITH TILDE
    "y?", "\u1ef9", // LATIN SMALL LETTER Y WITH TILDE
    ";'", "\u1f00", // GREEK DASIA AND ACUTE ACCENT
    ",'", "\u1f01", // GREEK PSILI AND ACUTE ACCENT
    ";!", "\u1f02", // GREEK DASIA AND VARIA
    ",!", "\u1f03", // GREEK PSILI AND VARIA
    "?;", "\u1f04", // GREEK DASIA AND PERISPOMENI
    "?,", "\u1f05", // GREEK PSILI AND PERISPOMENI
    "!:", "\u1f06", // GREEK DIAERESIS AND VARIA
    "?:", "\u1f07", // GREEK DIAERESIS AND PERISPOMENI
    "1N", "\u2002", // EN SPACE
    "1M", "\u2003", // EM SPACE
    "3M", "\u2004", // THREE-PER-EM SPACE
    "4M", "\u2005", // FOUR-PER-EM SPACE
    "6M", "\u2006", // SIX-PER-EM SPACE
    "1T", "\u2009", // THIN SPACE
    "1H", "\u200a", // HAIR SPACE
    "-1", "\u2010", // HYPHEN
    "-N", "\u2013", // EN DASH
    "-M", "\u2014", // EM DASH
    "-3", "\u2015", // HORIZONTAL BAR
    "!2", "\u2016", // DOUBLE VERTICAL LINE
    "=2", "\u2017", // DOUBLE LOW LINE
    "'6", "\u2018", // LEFT SINGLE QUOTATION MARK
    "'9", "\u2019", // RIGHT SINGLE QUOTATION MARK
    ".9", "\u201a", // SINGLE LOW-9 QUOTATION MARK
    "9'", "\u201b", // SINGLE HIGH-REVERSED-9 QUOTATION MARK
    "\"6", "\u201c", // LEFT DOUBLE QUOTATION MARK
    "\"9", "\u201d", // RIGHT DOUBLE QUOTATION MARK
    ":9", "\u201e", // DOUBLE LOW-9 QUOTATION MARK
    "9\"", "\u201f", // DOUBLE HIGH-REVERSED-9 QUOTATION MARK
    "/-", "\u2020", // DAGGER
    "/=", "\u2021", // DOUBLE DAGGER
    "..", "\u2025", // TWO DOT LEADER
    "%0", "\u2030", // PER MILLE SIGN
    "1'", "\u2032", // PRIME
    "2'", "\u2033", // DOUBLE PRIME
    "3'", "\u2034", // TRIPLE PRIME
    "1\"", "\u2035", // REVERSED PRIME
    "2\"", "\u2036", // REVERSED DOUBLE PRIME
    "3\"", "\u2037", // REVERSED TRIPLE PRIME
    "Ca", "\u2038", // CARET
    "<1", "\u2039", // SINGLE LEFT-POINTING ANGLE QUOTATION MARK
    ">1", "\u203a", // SINGLE RIGHT-POINTING ANGLE QUOTATION MARK
    ":X", "\u203b", // REFERENCE MARK
    "!*2", "\u203c", // DOUBLE EXCLAMATION MARK
    "'-", "\u203e", // OVERLINE
    "/f", "\u2044", // FRACTION SLASH
    "0S", "\u2070", // SUPERSCRIPT DIGIT ZERO
    "4S", "\u2074", // SUPERSCRIPT DIGIT FOUR
    "5S", "\u2075", // SUPERSCRIPT DIGIT FIVE
    "6S", "\u2076", // SUPERSCRIPT DIGIT SIX
    "7S", "\u2077", // SUPERSCRIPT DIGIT SEVEN
    "8S", "\u2078", // SUPERSCRIPT DIGIT EIGHT
    "9S", "\u2079", // SUPERSCRIPT DIGIT NINE
    "+S", "\u207a", // SUPERSCRIPT PLUS SIGN
    "-S", "\u207b", // SUPERSCRIPT MINUS
    "=S", "\u207c", // SUPERSCRIPT EQUALS SIGN
    "(S", "\u207d", // SUPERSCRIPT LEFT PARENTHESIS
    ")S", "\u207e", // SUPERSCRIPT RIGHT PARENTHESIS
    "nS", "\u207f", // SUPERSCRIPT LATIN SMALL LETTER N
    "0s", "\u2080", // SUBSCRIPT DIGIT ZERO
    "1s", "\u2081", // SUBSCRIPT DIGIT ONE
    "2s", "\u2082", // SUBSCRIPT DIGIT TWO
    "3s", "\u2083", // SUBSCRIPT DIGIT THREE
    "4s", "\u2084", // SUBSCRIPT DIGIT FOUR
    "5s", "\u2085", // SUBSCRIPT DIGIT FIVE
    "6s", "\u2086", // SUBSCRIPT DIGIT SIX
    "7s", "\u2087", // SUBSCRIPT DIGIT SEVEN
    "8s", "\u2088", // SUBSCRIPT DIGIT EIGHT
    "9s", "\u2089", // SUBSCRIPT DIGIT NINE
    "+s", "\u208a", // SUBSCRIPT PLUS SIGN
    "-s", "\u208b", // SUBSCRIPT MINUS
    "=s", "\u208c", // SUBSCRIPT EQUALS SIGN
    "(s", "\u208d", // SUBSCRIPT LEFT PARENTHESIS
    ")s", "\u208e", // SUBSCRIPT RIGHT PARENTHESIS
    "Li", "\u20a4", // LIRA SIGN
    "Pt", "\u20a7", // PESETA SIGN
    "W=", "\u20a9", // WON SIGN
    "oC", "\u2103", // DEGREE CENTIGRADE
    "co", "\u2105", // CARE OF
    "oF", "\u2109", // DEGREE FAHRENHEIT
    "N0", "\u2116", // NUMERO SIGN
    "PO", "\u2117", // SOUND RECORDING COPYRIGHT
    "Rx", "\u211e", // PRESCRIPTION TAKE
    "SM", "\u2120", // SERVICE MARK
    "TM", "\u2122", // TRADE MARK SIGN
    "Om", "\u2126", // OHM SIGN
    "AO", "\u212b", // ANGSTROEM SIGN
    "13", "\u2153", // VULGAR FRACTION ONE THIRD
    "23", "\u2154", // VULGAR FRACTION TWO THIRDS
    "15", "\u2155", // VULGAR FRACTION ONE FIFTH
    "25", "\u2156", // VULGAR FRACTION TWO FIFTHS
    "35", "\u2157", // VULGAR FRACTION THREE FIFTHS
    "45", "\u2158", // VULGAR FRACTION FOUR FIFTHS
    "16", "\u2159", // VULGAR FRACTION ONE SIXTH
    "56", "\u215a", // VULGAR FRACTION FIVE SIXTHS
    "18", "\u215b", // VULGAR FRACTION ONE EIGHTH
    "38", "\u215c", // VULGAR FRACTION THREE EIGHTHS
    "58", "\u215d", // VULGAR FRACTION FIVE EIGHTHS
    "78", "\u215e", // VULGAR FRACTION SEVEN EIGHTHS
    "1R", "\u2160", // ROMAN NUMERAL ONE
    "2R", "\u2161", // ROMAN NUMERAL TWO
    "3R", "\u2162", // ROMAN NUMERAL THREE
    "4R", "\u2163", // ROMAN NUMERAL FOUR
    "5R", "\u2164", // ROMAN NUMERAL FIVE
    "6R", "\u2165", // ROMAN NUMERAL SIX
    "7R", "\u2166", // ROMAN NUMERAL SEVEN
    "8R", "\u2167", // ROMAN NUMERAL EIGHT
    "9R", "\u2168", // ROMAN NUMERAL NINE
    "aR", "\u2169", // ROMAN NUMERAL TEN
    "bR", "\u216a", // ROMAN NUMERAL ELEVEN
    "cR", "\u216b", // ROMAN NUMERAL TWELVE
    "50R", "\u216c", // ROMAN NUMERAL FIFTY
    "100R", "\u216d", // ROMAN NUMERAL ONE HUNDRED
    "500R", "\u216e", // ROMAN NUMERAL FIVE HUNDRED
    "1000R", "\u216f", // ROMAN NUMERAL ONE THOUSAND
    "1r", "\u2170", // SMALL ROMAN NUMERAL ONE
    "2r", "\u2171", // SMALL ROMAN NUMERAL TWO
    "3r", "\u2172", // SMALL ROMAN NUMERAL THREE
    "4r", "\u2173", // SMALL ROMAN NUMERAL FOUR
    "5r", "\u2174", // SMALL ROMAN NUMERAL FIVE
    "6r", "\u2175", // SMALL ROMAN NUMERAL SIX
    "7r", "\u2176", // SMALL ROMAN NUMERAL SEVEN
    "8r", "\u2177", // SMALL ROMAN NUMERAL EIGHT
    "9r", "\u2178", // SMALL ROMAN NUMERAL NINE
    "ar", "\u2179", // SMALL ROMAN NUMERAL TEN
    "br", "\u217a", // SMALL ROMAN NUMERAL ELEVEN
    "cr", "\u217b", // SMALL ROMAN NUMERAL TWELVE
    "50r", "\u217c", // SMALL ROMAN NUMERAL FIFTY
    "100r", "\u217d", // SMALL ROMAN NUMERAL ONE HUNDRED
    "500r", "\u217e", // SMALL ROMAN NUMERAL FIVE HUNDRED
    "1000r", "\u217f", // SMALL ROMAN NUMERAL ONE THOUSAND
    "1000RCD", "\u2180", // ROMAN NUMERAL ONE THOUSAND C D
    "5000R", "\u2181", // ROMAN NUMERAL FIVE THOUSAND
    "10000R", "\u2182", // ROMAN NUMERAL TEN THOUSAND
    "<-", "\u2190", // LEFTWARDS ARROW
    "-!", "\u2191", // UPWARDS ARROW
    "->", "\u2192", // RIGHTWARDS ARROW
    "-v", "\u2193", // DOWNWARDS ARROW
    "<>", "\u2194", // LEFT RIGHT ARROW
    "UD", "\u2195", // UP DOWN ARROW
    "<!!", "\u2196", // NORTH WEST ARROW
    "//>", "\u2197", // NORTH EAST ARROW
    "!!>", "\u2198", // SOUTH EAST ARROW
    "<//", "\u2199", // SOUTH WEST ARROW
    "<=", "\u21d0", // LEFTWARDS DOUBLE ARROW
    "=>", "\u21d2", // RIGHTWARDS DOUBLE ARROW
    "==", "\u21d4", // LEFT RIGHT DOUBLE ARROW
    "FA", "\u2200", // FOR ALL
    "dP", "\u2202", // PARTIAL DIFFERENTIAL
    "TE", "\u2203", // THERE EXISTS
    "/0", "\u2205", // EMPTY SET
    "DE", "\u2206", // INCREMENT
    "NB", "\u2207", // NABLA
    "(-", "\u2208", // ELEMENT OF
    "-)", "\u220b", // CONTAINS AS MEMBER
    "*P", "\u220f", // N-ARY PRODUCT
    "+Z", "\u2211", // N-ARY SUMMATION
    "-2", "\u2212", // MINUS SIGN
    "-+", "\u2213", // MINUS-OR-PLUS SIGN
    "*-", "\u2217", // ASTERISK OPERATOR
    "Ob", "\u2218", // RING OPERATOR
    "Sb", "\u2219", // BULLET OPERATOR
    "RT", "\u221a", // SQUARE ROOT
    "0(", "\u221d", // PROPORTIONAL TO
    "00", "\u221e", // INFINITY
    "-L", "\u221f", // RIGHT ANGLE
    "-V", "\u2220", // ANGLE
    "PP", "\u2225", // PARALLEL TO
    "AN", "\u2227", // LOGICAL AND
    "OR", "\u2228", // LOGICAL OR
    "(U", "\u2229", // INTERSECTION
    ")U", "\u222a", // UNION
    "In", "\u222b", // INTEGRAL
    "DI", "\u222c", // DOUBLE INTEGRAL
    "Io", "\u222e", // CONTOUR INTEGRAL
    ".:", "\u2234", // THEREFORE
    ":.", "\u2235", // BECAUSE
    ":R", "\u2236", // RATIO
    "::", "\u2237", // PROPORTION
    "?1", "\u223c", // TILDE OPERATOR
    "CG", "\u223e", // INVERTED LAZY S
    "?-", "\u2243", // ASYMPTOTICALLY EQUAL TO
    "?=", "\u2245", // APPROXIMATELY EQUAL TO
    "?2", "\u2248", // ALMOST EQUAL TO
    "=?", "\u224c", // ALL EQUAL TO
    "HI", "\u2253", // IMAGE OF OR APPROXIMATELY EQUAL TO
    "!=", "\u2260", // NOT EQUAL TO
    "=3", "\u2261", // IDENTICAL TO
    "=<", "\u2264", // LESS-THAN OR EQUAL TO
    ">=", "\u2265", // GREATER-THAN OR EQUAL TO
    "<*", "\u226a", // MUCH LESS-THAN
    "*>", "\u226b", // MUCH GREATER-THAN
    "!<", "\u226e", // NOT LESS-THAN
    "!>", "\u226f", // NOT GREATER-THAN
    "(C", "\u2282", // SUBSET OF
    ")C", "\u2283", // SUPERSET OF
    "(_", "\u2286", // SUBSET OF OR EQUAL TO
    ")_", "\u2287", // SUPERSET OF OR EQUAL TO
    "0.", "\u2299", // CIRCLED DOT OPERATOR
    "02", "\u229a", // CIRCLED RING OPERATOR
    "-T", "\u22a5", // UP TACK
    ".P", "\u22c5", // DOT OPERATOR
    ":3", "\u22ee", // VERTICAL ELLIPSIS
    ".3", "\u22ef", // MIDLINE HORIZONTAL ELLIPSIS
    "Eh", "\u2302", // HOUSE
    "<7", "\u2308", // LEFT CEILING
    ">7", "\u2309", // RIGHT CEILING
    "7<", "\u230a", // LEFT FLOOR
    "7>", "\u230b", // RIGHT FLOOR
    "NI", "\u2310", // REVERSED NOT SIGN
    "(A", "\u2312", // ARC
    "TR", "\u2315", // TELEPHONE RECORDER
    "Iu", "\u2320", // TOP HALF INTEGRAL
    "Il", "\u2321", // BOTTOM HALF INTEGRAL
    "</", "\u2329", // LEFT-POINTING ANGLE BRACKET
    "/>", "\u232a", // RIGHT-POINTING ANGLE BRACKET
    "Vs", "\u2423", // OPEN BOX
    "1h", "\u2440", // OCR HOOK
    "3h", "\u2441", // OCR CHAIR
    "2h", "\u2442", // OCR FORK
    "4h", "\u2443", // OCR INVERTED FORK
    "1j", "\u2446", // OCR BRANCH BANK IDENTIFICATION
    "2j", "\u2447", // OCR AMOUNT OF CHECK
    "3j", "\u2448", // OCR DASH
    "4j", "\u2449", // OCR CUSTOMER ACCOUNT NUMBER
    /*
    "1-o", "\u2460", // CIRCLED DIGIT ONE
    "2-o", "\u2461", // CIRCLED DIGIT TWO
    "3-o", "\u2462", // CIRCLED DIGIT THREE
    "4-o", "\u2463", // CIRCLED DIGIT FOUR
    "5-o", "\u2464", // CIRCLED DIGIT FIVE
    "6-o", "\u2465", // CIRCLED DIGIT SIX
    "7-o", "\u2466", // CIRCLED DIGIT SEVEN
    "8-o", "\u2467", // CIRCLED DIGIT EIGHT
    "9-o", "\u2468", // CIRCLED DIGIT NINE
    "10-o", "\u2469", // CIRCLED NUMBER TEN
    "11-o", "\u246a", // CIRCLED NUMBER ELEVEN
    "12-o", "\u246b", // CIRCLED NUMBER TWELVE
    "13-o", "\u246c", // CIRCLED NUMBER THIRTEEN
    "14-o", "\u246d", // CIRCLED NUMBER FOURTEEN
    "15-o", "\u246e", // CIRCLED NUMBER FIFTEEN
    "16-o", "\u246f", // CIRCLED NUMBER SIXTEEN
    "17-o", "\u2470", // CIRCLED NUMBER SEVENTEEN
    "18-o", "\u2471", // CIRCLED NUMBER EIGHTEEN
    "19-o", "\u2472", // CIRCLED NUMBER NINETEEN
    "20-o", "\u2473", // CIRCLED NUMBER TWENTY
    "(1)", "\u2474", // PARENTHESIZED DIGIT ONE
    "(2)", "\u2475", // PARENTHESIZED DIGIT TWO
    "(3)", "\u2476", // PARENTHESIZED DIGIT THREE
    "(4)", "\u2477", // PARENTHESIZED DIGIT FOUR
    "(5)", "\u2478", // PARENTHESIZED DIGIT FIVE
    "(6)", "\u2479", // PARENTHESIZED DIGIT SIX
    "(7)", "\u247a", // PARENTHESIZED DIGIT SEVEN
    "(8)", "\u247b", // PARENTHESIZED DIGIT EIGHT
    "(9)", "\u247c", // PARENTHESIZED DIGIT NINE
    "(10)", "\u247d", // PARENTHESIZED NUMBER TEN
    "(11)", "\u247e", // PARENTHESIZED NUMBER ELEVEN
    "(12)", "\u247f", // PARENTHESIZED NUMBER TWELVE
    "(13)", "\u2480", // PARENTHESIZED NUMBER THIRTEEN
    "(14)", "\u2481", // PARENTHESIZED NUMBER FOURTEEN
    "(15)", "\u2482", // PARENTHESIZED NUMBER FIFTEEN
    "(16)", "\u2483", // PARENTHESIZED NUMBER SIXTEEN
    "(17)", "\u2484", // PARENTHESIZED NUMBER SEVENTEEN
    "(18)", "\u2485", // PARENTHESIZED NUMBER EIGHTEEN
    "(19)", "\u2486", // PARENTHESIZED NUMBER NINETEEN
    "(20)", "\u2487", // PARENTHESIZED NUMBER TWENTY
    "1.", "\u2488", // DIGIT ONE FULL STOP
    "2.", "\u2489", // DIGIT TWO FULL STOP
    "3.", "\u248a", // DIGIT THREE FULL STOP
    "4.", "\u248b", // DIGIT FOUR FULL STOP
    "5.", "\u248c", // DIGIT FIVE FULL STOP
    "6.", "\u248d", // DIGIT SIX FULL STOP
    "7.", "\u248e", // DIGIT SEVEN FULL STOP
    "8.", "\u248f", // DIGIT EIGHT FULL STOP
    "9.", "\u2490", // DIGIT NINE FULL STOP
    "10.", "\u2491", // NUMBER TEN FULL STOP
    "11.", "\u2492", // NUMBER ELEVEN FULL STOP
    "12.", "\u2493", // NUMBER TWELVE FULL STOP
    "13.", "\u2494", // NUMBER THIRTEEN FULL STOP
    "14.", "\u2495", // NUMBER FOURTEEN FULL STOP
    "15.", "\u2496", // NUMBER FIFTEEN FULL STOP
    "16.", "\u2497", // NUMBER SIXTEEN FULL STOP
    "17.", "\u2498", // NUMBER SEVENTEEN FULL STOP
    "18.", "\u2499", // NUMBER EIGHTEEN FULL STOP
    "19.", "\u249a", // NUMBER NINETEEN FULL STOP
    "20.", "\u249b", // NUMBER TWENTY FULL STOP
    "(a)", "\u249c", // PARENTHESIZED LATIN SMALL LETTER A
    "(b)", "\u249d", // PARENTHESIZED LATIN SMALL LETTER B
    "(c)", "\u249e", // PARENTHESIZED LATIN SMALL LETTER C
    "(d)", "\u249f", // PARENTHESIZED LATIN SMALL LETTER D
    "(e)", "\u24a0", // PARENTHESIZED LATIN SMALL LETTER E
    "(f)", "\u24a1", // PARENTHESIZED LATIN SMALL LETTER F
    "(g)", "\u24a2", // PARENTHESIZED LATIN SMALL LETTER G
    "(h)", "\u24a3", // PARENTHESIZED LATIN SMALL LETTER H
    "(i)", "\u24a4", // PARENTHESIZED LATIN SMALL LETTER I
    "(j)", "\u24a5", // PARENTHESIZED LATIN SMALL LETTER J
    "(k)", "\u24a6", // PARENTHESIZED LATIN SMALL LETTER K
    "(l)", "\u24a7", // PARENTHESIZED LATIN SMALL LETTER L
    "(m)", "\u24a8", // PARENTHESIZED LATIN SMALL LETTER M
    "(n)", "\u24a9", // PARENTHESIZED LATIN SMALL LETTER N
    "(o)", "\u24aa", // PARENTHESIZED LATIN SMALL LETTER O
    "(p)", "\u24ab", // PARENTHESIZED LATIN SMALL LETTER P
    "(q)", "\u24ac", // PARENTHESIZED LATIN SMALL LETTER Q
    "(r)", "\u24ad", // PARENTHESIZED LATIN SMALL LETTER R
    "(s)", "\u24ae", // PARENTHESIZED LATIN SMALL LETTER S
    "(t)", "\u24af", // PARENTHESIZED LATIN SMALL LETTER T
    "(u)", "\u24b0", // PARENTHESIZED LATIN SMALL LETTER U
    "(v)", "\u24b1", // PARENTHESIZED LATIN SMALL LETTER V
    "(w)", "\u24b2", // PARENTHESIZED LATIN SMALL LETTER W
    "(x)", "\u24b3", // PARENTHESIZED LATIN SMALL LETTER X
    "(y)", "\u24b4", // PARENTHESIZED LATIN SMALL LETTER Y
    "(z)", "\u24b5", // PARENTHESIZED LATIN SMALL LETTER Z
    "A-o", "\u24b6", // CIRCLED LATIN CAPITAL LETTER A
    "B-o", "\u24b7", // CIRCLED LATIN CAPITAL LETTER B
    "C-o", "\u24b8", // CIRCLED LATIN CAPITAL LETTER C
    "D-o", "\u24b9", // CIRCLED LATIN CAPITAL LETTER D
    "E-o", "\u24ba", // CIRCLED LATIN CAPITAL LETTER E
    "F-o", "\u24bb", // CIRCLED LATIN CAPITAL LETTER F
    "G-o", "\u24bc", // CIRCLED LATIN CAPITAL LETTER G
    "H-o", "\u24bd", // CIRCLED LATIN CAPITAL LETTER H
    "I-o", "\u24be", // CIRCLED LATIN CAPITAL LETTER I
    "J-o", "\u24bf", // CIRCLED LATIN CAPITAL LETTER J
    "K-o", "\u24c0", // CIRCLED LATIN CAPITAL LETTER K
    "L-o", "\u24c1", // CIRCLED LATIN CAPITAL LETTER L
    "M-o", "\u24c2", // CIRCLED LATIN CAPITAL LETTER M
    "N-o", "\u24c3", // CIRCLED LATIN CAPITAL LETTER N
    "O-o", "\u24c4", // CIRCLED LATIN CAPITAL LETTER O
    "P-o", "\u24c5", // CIRCLED LATIN CAPITAL LETTER P
    "Q-o", "\u24c6", // CIRCLED LATIN CAPITAL LETTER Q
    "R-o", "\u24c7", // CIRCLED LATIN CAPITAL LETTER R
    "S-o", "\u24c8", // CIRCLED LATIN CAPITAL LETTER S
    "T-o", "\u24c9", // CIRCLED LATIN CAPITAL LETTER T
    "U-o", "\u24ca", // CIRCLED LATIN CAPITAL LETTER U
    "V-o", "\u24cb", // CIRCLED LATIN CAPITAL LETTER V
    "W-o", "\u24cc", // CIRCLED LATIN CAPITAL LETTER W
    "X-o", "\u24cd", // CIRCLED LATIN CAPITAL LETTER X
    "Y-o", "\u24ce", // CIRCLED LATIN CAPITAL LETTER Y
    "Z-o", "\u24cf", // CIRCLED LATIN CAPITAL LETTER Z
    "a-o", "\u24d0", // CIRCLED LATIN SMALL LETTER A
    "b-o", "\u24d1", // CIRCLED LATIN SMALL LETTER B
    "c-o", "\u24d2", // CIRCLED LATIN SMALL LETTER C
    "d-o", "\u24d3", // CIRCLED LATIN SMALL LETTER D
    "e-o", "\u24d4", // CIRCLED LATIN SMALL LETTER E
    "f-o", "\u24d5", // CIRCLED LATIN SMALL LETTER F
    "g-o", "\u24d6", // CIRCLED LATIN SMALL LETTER G
    "h-o", "\u24d7", // CIRCLED LATIN SMALL LETTER H
    "i-o", "\u24d8", // CIRCLED LATIN SMALL LETTER I
    "j-o", "\u24d9", // CIRCLED LATIN SMALL LETTER J
    "k-o", "\u24da", // CIRCLED LATIN SMALL LETTER K
    "l-o", "\u24db", // CIRCLED LATIN SMALL LETTER L
    "m-o", "\u24dc", // CIRCLED LATIN SMALL LETTER M
    "n-o", "\u24dd", // CIRCLED LATIN SMALL LETTER N
    "o-o", "\u24de", // CIRCLED LATIN SMALL LETTER O
    "p-o", "\u24df", // CIRCLED LATIN SMALL LETTER P
    "q-o", "\u24e0", // CIRCLED LATIN SMALL LETTER Q
    "r-o", "\u24e1", // CIRCLED LATIN SMALL LETTER R
    "s-o", "\u24e2", // CIRCLED LATIN SMALL LETTER S
    "t-o", "\u24e3", // CIRCLED LATIN SMALL LETTER T
    "u-o", "\u24e4", // CIRCLED LATIN SMALL LETTER U
    "v-o", "\u24e5", // CIRCLED LATIN SMALL LETTER V
    "w-o", "\u24e6", // CIRCLED LATIN SMALL LETTER W
    "x-o", "\u24e7", // CIRCLED LATIN SMALL LETTER X
    "y-o", "\u24e8", // CIRCLED LATIN SMALL LETTER Y
    "z-o", "\u24e9", // CIRCLED LATIN SMALL LETTER Z
    "0-o", "\u24ea", // CIRCLED DIGIT ZERO
    */
    "hh", "\u2500", // BOX DRAWINGS LIGHT HORIZONTAL
    "HH", "\u2501", // BOX DRAWINGS HEAVY HORIZONTAL
    "vv", "\u2502", // BOX DRAWINGS LIGHT VERTICAL
    "VV", "\u2503", // BOX DRAWINGS HEAVY VERTICAL
    "3-", "\u2504", // BOX DRAWINGS LIGHT TRIPLE DASH HORIZONTAL
    "3_", "\u2505", // BOX DRAWINGS HEAVY TRIPLE DASH HORIZONTAL
    "3!", "\u2506", // BOX DRAWINGS LIGHT TRIPLE DASH VERTICAL
    "3/", "\u2507", // BOX DRAWINGS HEAVY TRIPLE DASH VERTICAL
    "4-", "\u2508", // BOX DRAWINGS LIGHT QUADRUPLE DASH HORIZONTAL
    "4_", "\u2509", // BOX DRAWINGS HEAVY QUADRUPLE DASH HORIZONTAL
    "4!", "\u250a", // BOX DRAWINGS LIGHT QUADRUPLE DASH VERTICAL
    "4/", "\u250b", // BOX DRAWINGS HEAVY QUADRUPLE DASH VERTICAL
    "dr", "\u250c", // BOX DRAWINGS LIGHT DOWN AND RIGHT
    "dR", "\u250d", // BOX DRAWINGS DOWN LIGHT AND RIGHT HEAVY
    "Dr", "\u250e", // BOX DRAWINGS DOWN HEAVY AND RIGHT LIGHT
    "DR", "\u250f", // BOX DRAWINGS HEAVY DOWN AND RIGHT
    "dl", "\u2510", // BOX DRAWINGS LIGHT DOWN AND LEFT
    "dL", "\u2511", // BOX DRAWINGS DOWN LIGHT AND LEFT HEAVY
    "Dl", "\u2512", // BOX DRAWINGS DOWN HEAVY AND LEFT LIGHT
    "LD", "\u2513", // BOX DRAWINGS HEAVY DOWN AND LEFT
    "ur", "\u2514", // BOX DRAWINGS LIGHT UP AND RIGHT
    "uR", "\u2515", // BOX DRAWINGS UP LIGHT AND RIGHT HEAVY
    "Ur", "\u2516", // BOX DRAWINGS UP HEAVY AND RIGHT LIGHT
    "UR", "\u2517", // BOX DRAWINGS HEAVY UP AND RIGHT
    "ul", "\u2518", // BOX DRAWINGS LIGHT UP AND LEFT
    "uL", "\u2519", // BOX DRAWINGS UP LIGHT AND LEFT HEAVY
    "Ul", "\u251a", // BOX DRAWINGS UP HEAVY AND LEFT LIGHT
    "UL", "\u251b", // BOX DRAWINGS HEAVY UP AND LEFT
    "vr", "\u251c", // BOX DRAWINGS LIGHT VERTICAL AND RIGHT
    "vR", "\u251d", // BOX DRAWINGS VERTICAL LIGHT AND RIGHT HEAVY
    "Udr", "\u251e", // BOX DRAWINGS UP HEAVY AND RIGHT DOWN LIGHT
    "uDr", "\u251f", // BOX DRAWINGS DOWN HEAVY AND RIGHT UP LIGHT
    "Vr", "\u2520", // BOX DRAWINGS VERTICAL HEAVY AND RIGHT LIGHT
    "UdR", "\u2521", // BOX DRAWINGS DOWN LIGHT AND RIGHT UP HEAVY
    "uDR", "\u2522", // BOX DRAWINGS UP LIGHT AND RIGHT DOWN HEAVY
    "VR", "\u2523", // BOX DRAWINGS HEAVY VERTICAL AND RIGHT
    "vl", "\u2524", // BOX DRAWINGS LIGHT VERTICAL AND LEFT
    "vL", "\u2525", // BOX DRAWINGS VERTICAL LIGHT AND LEFT HEAVY
    "Udl", "\u2526", // BOX DRAWINGS UP HEAVY AND LEFT DOWN LIGHT
    "uDl", "\u2527", // BOX DRAWINGS DOWN HEAVY AND LEFT UP LIGHT
    "Vl", "\u2528", // BOX DRAWINGS VERTICAL HEAVY AND LEFT LIGHT
    "UdL", "\u2529", // BOX DRAWINGS DOWN LIGHT AND LEFT UP HEAVY
    "uDL", "\u252a", // BOX DRAWINGS UP LIGHT AND LEFT DOWN HEAVY
    "VL", "\u252b", // BOX DRAWINGS HEAVY VERTICAL AND LEFT
    "dh", "\u252c", // BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
    "dLr", "\u252d", // BOX DRAWINGS LEFT HEAVY AND RIGHT DOWN LIGHT
    "dlR", "\u252e", // BOX DRAWINGS RIGHT HEAVY AND LEFT DOWN LIGHT
    "dH", "\u252f", // BOX DRAWINGS DOWN LIGHT AND HORIZONTAL HEAVY
    "Dh", "\u2530", // BOX DRAWINGS DOWN HEAVY AND HORIZONTAL LIGHT
    "DLr", "\u2531", // BOX DRAWINGS RIGHT LIGHT AND LEFT DOWN HEAVY
    "DlR", "\u2532", // BOX DRAWINGS LEFT LIGHT AND RIGHT DOWN HEAVY
    "DH", "\u2533", // BOX DRAWINGS HEAVY DOWN AND HORIZONTAL
    "uh", "\u2534", // BOX DRAWINGS LIGHT UP AND HORIZONTAL
    "uLr", "\u2535", // BOX DRAWINGS LEFT HEAVY AND RIGHT UP LIGHT
    "ulR", "\u2536", // BOX DRAWINGS RIGHT HEAVY AND LEFT UP LIGHT
    "uH", "\u2537", // BOX DRAWINGS UP LIGHT AND HORIZONTAL HEAVY
    "Uh", "\u2538", // BOX DRAWINGS UP HEAVY AND HORIZONTAL LIGHT
    "ULr", "\u2539", // BOX DRAWINGS RIGHT LIGHT AND LEFT UP HEAVY
    "UlR", "\u253a", // BOX DRAWINGS LEFT LIGHT AND RIGHT UP HEAVY
    "UH", "\u253b", // BOX DRAWINGS HEAVY UP AND HORIZONTAL
    "vh", "\u253c", // BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
    "vLr", "\u253d", // BOX DRAWINGS LEFT HEAVY AND RIGHT VERTICAL LIGHT
    "vlR", "\u253e", // BOX DRAWINGS RIGHT HEAVY AND LEFT VERTICAL LIGHT
    "vH", "\u253f", // BOX DRAWINGS VERTICAL LIGHT AND HORIZONTAL HEAVY
    "Udh", "\u2540", // BOX DRAWINGS UP HEAVY AND DOWN HORIZONTAL LIGHT
    "uDh", "\u2541", // BOX DRAWINGS DOWN HEAVY AND UP HORIZONTAL LIGHT
    "Vh", "\u2542", // BOX DRAWINGS VERTICAL HEAVY AND HORIZONTAL LIGHT
    "UdLr", "\u2543", // BOX DRAWINGS LEFT UP HEAVY AND RIGHT DOWN LIGHT
    "UdlR", "\u2544", // BOX DRAWINGS RIGHT UP HEAVY AND LEFT DOWN LIGHT
    "uDLr", "\u2545", // BOX DRAWINGS LEFT DOWN HEAVY AND RIGHT UP LIGHT
    "uDlR", "\u2546", // BOX DRAWINGS RIGHT DOWN HEAVY AND LEFT UP LIGHT
    "UdH", "\u2547", // BOX DRAWINGS DOWN LIGHT AND UP HORIZONTAL HEAVY
    "uDH", "\u2548", // BOX DRAWINGS UP LIGHT AND DOWN HORIZONTAL HEAVY
    "VLr", "\u2549", // BOX DRAWINGS RIGHT LIGHT AND LEFT VERTICAL HEAVY
    "VlR", "\u254a", // BOX DRAWINGS LEFT LIGHT AND RIGHT VERTICAL HEAVY
    "VH", "\u254b", // BOX DRAWINGS HEAVY VERTICAL AND HORIZONTAL
    "FD", "\u2571", // BOX DRAWINGS LIGHT DIAGONAL UPPER RIGHT TO LOWER LEFT
    "BD", "\u2572", // BOX DRAWINGS LIGHT DIAGONAL UPPER LEFT TO LOWER RIGHT
    "TB", "\u2580", // UPPER HALF BLOCK
    "LB", "\u2584", // LOWER HALF BLOCK
    "FB", "\u2588", // FULL BLOCK
    "lB", "\u258c", // LEFT HALF BLOCK
    "RB", "\u2590", // RIGHT HALF BLOCK
    ".S", "\u2591", // LIGHT SHADE
    ":S", "\u2592", // MEDIUM SHADE
    "?S", "\u2593", // DARK SHADE
    "fS", "\u25a0", // BLACK SQUARE
    "OS", "\u25a1", // WHITE SQUARE
    "RO", "\u25a2", // WHITE SQUARE WITH ROUNDED CORNERS
    "Rr", "\u25a3", // WHITE SQUARE CONTAINING BLACK SMALL SQUARE
    "RF", "\u25a4", // SQUARE WITH HORIZONTAL FILL
    "RY", "\u25a5", // SQUARE WITH VERTICAL FILL
    "RH", "\u25a6", // SQUARE WITH ORTHOGONAL CROSSHATCH FILL
    "RZ", "\u25a7", // SQUARE WITH UPPER LEFT TO LOWER RIGHT FILL
    "RK", "\u25a8", // SQUARE WITH UPPER RIGHT TO LOWER LEFT FILL
    "RX", "\u25a9", // SQUARE WITH DIAGONAL CROSSHATCH FILL
    "sB", "\u25aa", // BLACK SMALL SQUARE
    "SR", "\u25ac", // BLACK RECTANGLE
    "Or", "\u25ad", // WHITE RECTANGLE
    "UT", "\u25b2", // BLACK UP-POINTING TRIANGLE
    "uT", "\u25b3", // WHITE UP-POINTING TRIANGLE
    "PR", "\u25b6", // BLACK RIGHT-POINTING TRIANGLE
    "Tr", "\u25b7", // WHITE RIGHT-POINTING TRIANGLE
    "Dt", "\u25bc", // BLACK DOWN-POINTING TRIANGLE
    "dT", "\u25bd", // WHITE DOWN-POINTING TRIANGLE
    "PL", "\u25c0", // BLACK LEFT-POINTING TRIANGLE
    "Tl", "\u25c1", // WHITE LEFT-POINTING TRIANGLE
    "Db", "\u25c6", // BLACK DIAMOND
    "Dw", "\u25c7", // WHITE DIAMOND
    "LZ", "\u25ca", // LOZENGE
    "0m", "\u25cb", // WHITE CIRCLE
    "0o", "\u25ce", // BULLSEYE
    "0M", "\u25cf", // BLACK CIRCLE
    "0L", "\u25d0", // CIRCLE WITH LEFT HALF BLACK
    "0R", "\u25d1", // CIRCLE WITH RIGHT HALF BLACK
    "Sn", "\u25d8", // INVERSE BULLET
    "Ic", "\u25d9", // INVERSE WHITE CIRCLE
    "Fd", "\u25e2", // BLACK LOWER RIGHT TRIANGLE
    "Bd", "\u25e3", // BLACK LOWER LEFT TRIANGLE
    "*2", "\u2605", // BLACK STAR
    "*1", "\u2606", // WHITE STAR
    "TEL", "\u260e", // BLACK TELEPHONE
    "tel", "\u260f", // WHITE TELEPHONE
    "<H", "\u261c", // WHITE LEFT POINTING INDEX
    ">H", "\u261e", // WHITE RIGHT POINTING INDEX
    "0u", "\u263a", // WHITE SMILING FACE
    "0U", "\u263b", // BLACK SMILING FACE
    "SU", "\u263c", // WHITE SUN WITH RAYS
    "Fm", "\u2640", // FEMALE SIGN
    "Ml", "\u2642", // MALE SIGN
    "cS", "\u2660", // BLACK SPADE SUIT
    "cH", "\u2661", // WHITE HEART SUIT
    "cD", "\u2662", // WHITE DIAMOND SUIT
    "cC", "\u2663", // BLACK CLUB SUIT
    "cS-", "\u2664", // WHITE SPADE SUIT
    "cH-", "\u2665", // BLACK HEART SUIT
    "cD-", "\u2666", // BLACK DIAMOND SUIT
    "cC-", "\u2667", // WHITE CLUB SUIT
    "Md", "\u2669", // QUARTER NOTE
    "M8", "\u266a", // EIGHTH NOTE
    "M2", "\u266b", // BARRED EIGHTH NOTES
    "M16", "\u266c", // BARRED SIXTEENTH NOTES
    "Mb", "\u266d", // MUSIC FLAT SIGN
    "Mx", "\u266e", // MUSIC NATURAL SIGN
    "MX", "\u266f", // MUSIC SHARP SIGN
    "OK", "\u2713", // CHECK MARK
    "XX", "\u2717", // BALLOT X
    "-X", "\u2720", // MALTESE CROSS
    "IS", "\u3000", // IDEOGRAPHIC SPACE
    ",_", "\u3001", // IDEOGRAPHIC COMMA
    "._", "\u3002", // IDEOGRAPHIC PERIOD
    "+\"", "\u3003", // DITTO MARK
    "+_", "\u3004", // IDEOGRAPHIC DITTO MARK
    "*_", "\u3005", // IDEOGRAPHIC ITERATION MARK
    ";_", "\u3006", // IDEOGRAPHIC CLOSING MARK
    "0_", "\u3007", // IDEOGRAPHIC NUMBER ZERO
    "<+", "\u300a", // LEFT DOUBLE ANGLE BRACKET
    ">+", "\u300b", // RIGHT DOUBLE ANGLE BRACKET
    "<'", "\u300c", // LEFT CORNER BRACKET
    ">'", "\u300d", // RIGHT CORNER BRACKET
    "<\"", "\u300e", // LEFT WHITE CORNER BRACKET
    ">\"", "\u300f", // RIGHT WHITE CORNER BRACKET
    "(\"", "\u3010", // LEFT BLACK LENTICULAR BRACKET
    ")\"", "\u3011", // RIGHT BLACK LENTICULAR BRACKET
    "=T", "\u3012", // POSTAL MARK
    "=_", "\u3013", // GETA MARK
    "('", "\u3014", // LEFT TORTOISE SHELL BRACKET
    ")'", "\u3015", // RIGHT TORTOISE SHELL BRACKET
    "(I", "\u3016", // LEFT WHITE LENTICULAR BRACKET
    ")I", "\u3017", // RIGHT WHITE LENTICULAR BRACKET
    "-?", "\u301c", // WAVE DASH
    "=T:)", "\u3020", // POSTAL MARK FACE
    "A5", "\u3041", // HIRAGANA LETTER SMALL A
    "a5", "\u3042", // HIRAGANA LETTER A
    "I5", "\u3043", // HIRAGANA LETTER SMALL I
    "i5", "\u3044", // HIRAGANA LETTER I
    "U5", "\u3045", // HIRAGANA LETTER SMALL U
    "u5", "\u3046", // HIRAGANA LETTER U
    "E5", "\u3047", // HIRAGANA LETTER SMALL E
    "e5", "\u3048", // HIRAGANA LETTER E
    "O5", "\u3049", // HIRAGANA LETTER SMALL O
    "o5", "\u304a", // HIRAGANA LETTER O
    "ka", "\u304b", // HIRAGANA LETTER KA
    "ga", "\u304c", // HIRAGANA LETTER GA
    "ki", "\u304d", // HIRAGANA LETTER KI
    "gi", "\u304e", // HIRAGANA LETTER GI
    "ku", "\u304f", // HIRAGANA LETTER KU
    "gu", "\u3050", // HIRAGANA LETTER GU
    "ke", "\u3051", // HIRAGANA LETTER KE
    "ge", "\u3052", // HIRAGANA LETTER GE
    "ko", "\u3053", // HIRAGANA LETTER KO
    "go", "\u3054", // HIRAGANA LETTER GO
    "sa", "\u3055", // HIRAGANA LETTER SA
    "za", "\u3056", // HIRAGANA LETTER ZA
    "si", "\u3057", // HIRAGANA LETTER SI
    "zi", "\u3058", // HIRAGANA LETTER ZI
    "su", "\u3059", // HIRAGANA LETTER SU
    "zu", "\u305a", // HIRAGANA LETTER ZU
    "se", "\u305b", // HIRAGANA LETTER SE
    "ze", "\u305c", // HIRAGANA LETTER ZE
    "so", "\u305d", // HIRAGANA LETTER SO
    "zo", "\u305e", // HIRAGANA LETTER ZO
    "ta", "\u305f", // HIRAGANA LETTER TA
    "da", "\u3060", // HIRAGANA LETTER DA
    "ti", "\u3061", // HIRAGANA LETTER TI
    "di", "\u3062", // HIRAGANA LETTER DI
    "tU", "\u3063", // HIRAGANA LETTER SMALL TU
    "tu", "\u3064", // HIRAGANA LETTER TU
    "du", "\u3065", // HIRAGANA LETTER DU
    "te", "\u3066", // HIRAGANA LETTER TE
    "de", "\u3067", // HIRAGANA LETTER DE
    "to", "\u3068", // HIRAGANA LETTER TO
    "do", "\u3069", // HIRAGANA LETTER DO
    "na", "\u306a", // HIRAGANA LETTER NA
    "ni", "\u306b", // HIRAGANA LETTER NI
    "nu", "\u306c", // HIRAGANA LETTER NU
    "ne", "\u306d", // HIRAGANA LETTER NE
    "no", "\u306e", // HIRAGANA LETTER NO
    "ha", "\u306f", // HIRAGANA LETTER HA
    "ba", "\u3070", // HIRAGANA LETTER BA
    "pa", "\u3071", // HIRAGANA LETTER PA
    "hi", "\u3072", // HIRAGANA LETTER HI
    "bi", "\u3073", // HIRAGANA LETTER BI
    "pi", "\u3074", // HIRAGANA LETTER PI
    "hu", "\u3075", // HIRAGANA LETTER HU
    "bu", "\u3076", // HIRAGANA LETTER BU
    "pu", "\u3077", // HIRAGANA LETTER PU
    "he", "\u3078", // HIRAGANA LETTER HE
    "be", "\u3079", // HIRAGANA LETTER BE
    "pe", "\u307a", // HIRAGANA LETTER PE
    "ho", "\u307b", // HIRAGANA LETTER HO
    "bo", "\u307c", // HIRAGANA LETTER BO
    "po", "\u307d", // HIRAGANA LETTER PO
    "ma", "\u307e", // HIRAGANA LETTER MA
    "mi", "\u307f", // HIRAGANA LETTER MI
    "mu", "\u3080", // HIRAGANA LETTER MU
    "me", "\u3081", // HIRAGANA LETTER ME
    "mo", "\u3082", // HIRAGANA LETTER MO
    "yA", "\u3083", // HIRAGANA LETTER SMALL YA
    "ya", "\u3084", // HIRAGANA LETTER YA
    "yU", "\u3085", // HIRAGANA LETTER SMALL YU
    "yu", "\u3086", // HIRAGANA LETTER YU
    "yO", "\u3087", // HIRAGANA LETTER SMALL YO
    "yo", "\u3088", // HIRAGANA LETTER YO
    "ra", "\u3089", // HIRAGANA LETTER RA
    "ri", "\u308a", // HIRAGANA LETTER RI
    "ru", "\u308b", // HIRAGANA LETTER RU
    "re", "\u308c", // HIRAGANA LETTER RE
    "ro", "\u308d", // HIRAGANA LETTER RO
    "wA", "\u308e", // HIRAGANA LETTER SMALL WA
    "wa", "\u308f", // HIRAGANA LETTER WA
    "wi", "\u3090", // HIRAGANA LETTER WI
    "we", "\u3091", // HIRAGANA LETTER WE
    "wo", "\u3092", // HIRAGANA LETTER WO
    "n5", "\u3093", // HIRAGANA LETTER N
    "vu", "\u3094", // HIRAGANA LETTER VU
    "\"5", "\u309b", // KATAKANA-HIRAGANA VOICED SOUND MARK
    "05", "\u309c", // KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
    "*5", "\u309d", // HIRAGANA ITERATION MARK
    "+5", "\u309e", // HIRAGANA VOICED ITERATION MARK
    "a6", "\u30a1", // KATAKANA LETTER SMALL A
    "A6", "\u30a2", // KATAKANA LETTER A
    "i6", "\u30a3", // KATAKANA LETTER SMALL I
    "I6", "\u30a4", // KATAKANA LETTER I
    "u6", "\u30a5", // KATAKANA LETTER SMALL U
    "U6", "\u30a6", // KATAKANA LETTER U
    "e6", "\u30a7", // KATAKANA LETTER SMALL E
    "E6", "\u30a8", // KATAKANA LETTER E
    "o6", "\u30a9", // KATAKANA LETTER SMALL O
    "O6", "\u30aa", // KATAKANA LETTER O
    "Ka", "\u30ab", // KATAKANA LETTER KA
    "Ga", "\u30ac", // KATAKANA LETTER GA
    "Ki", "\u30ad", // KATAKANA LETTER KI
    "Gi", "\u30ae", // KATAKANA LETTER GI
    "Ku", "\u30af", // KATAKANA LETTER KU
    "Gu", "\u30b0", // KATAKANA LETTER GU
    "Ke", "\u30b1", // KATAKANA LETTER KE
    "Ge", "\u30b2", // KATAKANA LETTER GE
    "Ko", "\u30b3", // KATAKANA LETTER KO
    "Go", "\u30b4", // KATAKANA LETTER GO
    "Sa", "\u30b5", // KATAKANA LETTER SA
    "Za", "\u30b6", // KATAKANA LETTER ZA
    "Si", "\u30b7", // KATAKANA LETTER SI
    "Zi", "\u30b8", // KATAKANA LETTER ZI
    "Su", "\u30b9", // KATAKANA LETTER SU
    "Zu", "\u30ba", // KATAKANA LETTER ZU
    "Se", "\u30bb", // KATAKANA LETTER SE
    "Ze", "\u30bc", // KATAKANA LETTER ZE
    "So", "\u30bd", // KATAKANA LETTER SO
    "Zo", "\u30be", // KATAKANA LETTER ZO
    "Ta", "\u30bf", // KATAKANA LETTER TA
    "Da", "\u30c0", // KATAKANA LETTER DA
    "Ti", "\u30c1", // KATAKANA LETTER TI
    "Di", "\u30c2", // KATAKANA LETTER DI
    "TU", "\u30c3", // KATAKANA LETTER SMALL TU
    "Tu", "\u30c4", // KATAKANA LETTER TU
    "Du", "\u30c5", // KATAKANA LETTER DU
    "Te", "\u30c6", // KATAKANA LETTER TE
    "De", "\u30c7", // KATAKANA LETTER DE
    "To", "\u30c8", // KATAKANA LETTER TO
    "Do", "\u30c9", // KATAKANA LETTER DO
    "Na", "\u30ca", // KATAKANA LETTER NA
    "Ni", "\u30cb", // KATAKANA LETTER NI
    "Nu", "\u30cc", // KATAKANA LETTER NU
    "Ne", "\u30cd", // KATAKANA LETTER NE
    "No", "\u30ce", // KATAKANA LETTER NO
    "Ha", "\u30cf", // KATAKANA LETTER HA
    "Ba", "\u30d0", // KATAKANA LETTER BA
    "Pa", "\u30d1", // KATAKANA LETTER PA
    "Hi", "\u30d2", // KATAKANA LETTER HI
    "Bi", "\u30d3", // KATAKANA LETTER BI
    "Pi", "\u30d4", // KATAKANA LETTER PI
    "Hu", "\u30d5", // KATAKANA LETTER HU
    "Bu", "\u30d6", // KATAKANA LETTER BU
    "Pu", "\u30d7", // KATAKANA LETTER PU
    "He", "\u30d8", // KATAKANA LETTER HE
    "Be", "\u30d9", // KATAKANA LETTER BE
    "Pe", "\u30da", // KATAKANA LETTER PE
    "Ho", "\u30db", // KATAKANA LETTER HO
    "Bo", "\u30dc", // KATAKANA LETTER BO
    "Po", "\u30dd", // KATAKANA LETTER PO
    "Ma", "\u30de", // KATAKANA LETTER MA
    "Mi", "\u30df", // KATAKANA LETTER MI
    "Mu", "\u30e0", // KATAKANA LETTER MU
    "Me", "\u30e1", // KATAKANA LETTER ME
    "Mo", "\u30e2", // KATAKANA LETTER MO
    "YA", "\u30e3", // KATAKANA LETTER SMALL YA
    "Ya", "\u30e4", // KATAKANA LETTER YA
    "YU", "\u30e5", // KATAKANA LETTER SMALL YU
    "Yu", "\u30e6", // KATAKANA LETTER YU
    "YO", "\u30e7", // KATAKANA LETTER SMALL YO
    "Yo", "\u30e8", // KATAKANA LETTER YO
    "Ra", "\u30e9", // KATAKANA LETTER RA
    "Ri", "\u30ea", // KATAKANA LETTER RI
    "Ru", "\u30eb", // KATAKANA LETTER RU
    "Re", "\u30ec", // KATAKANA LETTER RE
    "Ro", "\u30ed", // KATAKANA LETTER RO
    "WA", "\u30ee", // KATAKANA LETTER SMALL WA
    "Wa", "\u30ef", // KATAKANA LETTER WA
    "Wi", "\u30f0", // KATAKANA LETTER WI
    "We", "\u30f1", // KATAKANA LETTER WE
    "Wo", "\u30f2", // KATAKANA LETTER WO
    "N6", "\u30f3", // KATAKANA LETTER N
    "Vu", "\u30f4", // KATAKANA LETTER VU
    "KA", "\u30f5", // KATAKANA LETTER SMALL KA
    "KE", "\u30f6", // KATAKANA LETTER SMALL KE
    "Va", "\u30f7", // KATAKANA LETTER VA
    "Vi", "\u30f8", // KATAKANA LETTER VI
    "Ve", "\u30f9", // KATAKANA LETTER VE
    "Vo", "\u30fa", // KATAKANA LETTER VO
    ".6", "\u30fb", // KATAKANA MIDDLE DOT
    "-6", "\u30fc", // KATAKANA-HIRAGANA PROLONGED SOUND MARK
    "*6", "\u30fd", // KATAKANA ITERATION MARK
    "+6", "\u30fe", // KATAKANA VOICED ITERATION MARK
    "b4", "\u3105", // BOPOMOFO LETTER B
    "p4", "\u3106", // BOPOMOFO LETTER P
    "m4", "\u3107", // BOPOMOFO LETTER M
    "f4", "\u3108", // BOPOMOFO LETTER F
    "d4", "\u3109", // BOPOMOFO LETTER D
    "t4", "\u310a", // BOPOMOFO LETTER T
    "n4", "\u310b", // BOPOMOFO LETTER N
    "l4", "\u310c", // BOPOMOFO LETTER L
    "g4", "\u310d", // BOPOMOFO LETTER G
    "k4", "\u310e", // BOPOMOFO LETTER K
    "h4", "\u310f", // BOPOMOFO LETTER H
    "j4", "\u3110", // BOPOMOFO LETTER J
    "q4", "\u3111", // BOPOMOFO LETTER Q
    "x4", "\u3112", // BOPOMOFO LETTER X
    "zh", "\u3113", // BOPOMOFO LETTER ZH
    "ch", "\u3114", // BOPOMOFO LETTER CH
    "sh", "\u3115", // BOPOMOFO LETTER SH
    "r4", "\u3116", // BOPOMOFO LETTER R
    "z4", "\u3117", // BOPOMOFO LETTER Z
    "c4", "\u3118", // BOPOMOFO LETTER C
    "s4", "\u3119", // BOPOMOFO LETTER S
    "a4", "\u311a", // BOPOMOFO LETTER A
    "o4", "\u311b", // BOPOMOFO LETTER O
    "e4", "\u311c", // BOPOMOFO LETTER E
    "eh4", "\u311d", // BOPOMOFO LETTER EH
    "ai", "\u311e", // BOPOMOFO LETTER AI
    "ei", "\u311f", // BOPOMOFO LETTER EI
    "au", "\u3120", // BOPOMOFO LETTER AU
    "ou", "\u3121", // BOPOMOFO LETTER OU
    "an", "\u3122", // BOPOMOFO LETTER AN
    "en", "\u3123", // BOPOMOFO LETTER EN
    "aN", "\u3124", // BOPOMOFO LETTER ANG
    "eN", "\u3125", // BOPOMOFO LETTER ENG
    "er", "\u3126", // BOPOMOFO LETTER ER
    "i4", "\u3127", // BOPOMOFO LETTER I
    "u4", "\u3128", // BOPOMOFO LETTER U
    "iu", "\u3129", // BOPOMOFO LETTER IU
    "v4", "\u312a", // BOPOMOFO LETTER V
    "nG", "\u312b", // BOPOMOFO LETTER NG
    "gn", "\u312c", // BOPOMOFO LETTER GN
    "(JU)", "\u321c", // PARENTHESIZED HANGUL JU
    "1c", "\u3220", // PARENTHESIZED IDEOGRAPH ONE
    "2c", "\u3221", // PARENTHESIZED IDEOGRAPH TWO
    "3c", "\u3222", // PARENTHESIZED IDEOGRAPH THREE
    "4c", "\u3223", // PARENTHESIZED IDEOGRAPH FOUR
    "5c", "\u3224", // PARENTHESIZED IDEOGRAPH FIVE
    "6c", "\u3225", // PARENTHESIZED IDEOGRAPH SIX
    "7c", "\u3226", // PARENTHESIZED IDEOGRAPH SEVEN
    "8c", "\u3227", // PARENTHESIZED IDEOGRAPH EIGHT
    "9c", "\u3228", // PARENTHESIZED IDEOGRAPH NINE
    "10c", "\u3229", // PARENTHESIZED IDEOGRAPH TEN
    "KSC", "\u327f", // KOREAN STANDARD SYMBOL
    "ff", "\ufb00", // LATIN SMALL LIGATURE FF
    "fi", "\ufb01", // LATIN SMALL LIGATURE FI
    "fl", "\ufb02", // LATIN SMALL LIGATURE FL
    "ffi", "\ufb03", // LATIN SMALL LIGATURE FFI
    "ffl", "\ufb04", // LATIN SMALL LIGATURE FFL
    "ft", "\ufb05", // LATIN SMALL LIGATURE FT
    "st", "\ufb06", // LATIN SMALL LIGATURE ST
    "3+;", "\ufe7d", // ARABIC SHADDA MEDIAL FORM
    "aM.", "\ufe82", // ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM
    "aH.", "\ufe84", // ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM
    "a+-", "\ufe8d", // ARABIC LETTER ALEF ISOLATED FORM
    "a+.", "\ufe8e", // ARABIC LETTER ALEF FINAL FORM
    "b+-", "\ufe8f", // ARABIC LETTER BEH ISOLATED FORM
    "b+,", "\ufe90", // ARABIC LETTER BEH INITIAL FORM
    "b+;", "\ufe91", // ARABIC LETTER BEH MEDIAL FORM
    "b+.", "\ufe92", // ARABIC LETTER BEH FINAL FORM
    "tm-", "\ufe93", // ARABIC LETTER TEH MARBUTA ISOLATED FORM
    "tm.", "\ufe94", // ARABIC LETTER TEH MARBUTA FINAL FORM
    "t+-", "\ufe95", // ARABIC LETTER TEH ISOLATED FORM
    "t+,", "\ufe96", // ARABIC LETTER TEH INITIAL FORM
    "t+;", "\ufe97", // ARABIC LETTER TEH MEDIAL FORM
    "t+.", "\ufe98", // ARABIC LETTER TEH FINAL FORM
    "tk-", "\ufe99", // ARABIC LETTER THEH ISOLATED FORM
    "tk,", "\ufe9a", // ARABIC LETTER THEH INITIAL FORM
    "tk;", "\ufe9b", // ARABIC LETTER THEH MEDIAL FORM
    "tk.", "\ufe9c", // ARABIC LETTER THEH FINAL FORM
    "g+-", "\ufe9d", // ARABIC LETTER JEEM ISOLATED FORM
    "g+,", "\ufe9e", // ARABIC LETTER JEEM INITIAL FORM
    "g+;", "\ufe9f", // ARABIC LETTER JEEM MEDIAL FORM
    "g+.", "\ufea0", // ARABIC LETTER JEEM FINAL FORM
    "hk-", "\ufea1", // ARABIC LETTER HAH ISOLATED FORM
    "hk,", "\ufea2", // ARABIC LETTER HAH INITIAL FORM
    "hk;", "\ufea3", // ARABIC LETTER HAH MEDIAL FORM
    "hk.", "\ufea4", // ARABIC LETTER HAH FINAL FORM
    "x+-", "\ufea5", // ARABIC LETTER KHAH ISOLATED FORM
    "x+,", "\ufea6", // ARABIC LETTER KHAH INITIAL FORM
    "x+;", "\ufea7", // ARABIC LETTER KHAH MEDIAL FORM
    "x+.", "\ufea8", // ARABIC LETTER KHAH FINAL FORM
    "d+-", "\ufea9", // ARABIC LETTER DAL ISOLATED FORM
    "d+.", "\ufeaa", // ARABIC LETTER DAL FINAL FORM
    "dk-", "\ufeab", // ARABIC LETTER THAL ISOLATED FORM
    "dk.", "\ufeac", // ARABIC LETTER THAL FINAL FORM
    "r+-", "\ufead", // ARABIC LETTER REH ISOLATED FORM
    "r+.", "\ufeae", // ARABIC LETTER REH FINAL FORM
    "z+-", "\ufeaf", // ARABIC LETTER ZAIN ISOLATED FORM
    "z+.", "\ufeb0", // ARABIC LETTER ZAIN FINAL FORM
    "s+-", "\ufeb1", // ARABIC LETTER SEEN ISOLATED FORM
    "s+,", "\ufeb2", // ARABIC LETTER SEEN INITIAL FORM
    "s+;", "\ufeb3", // ARABIC LETTER SEEN MEDIAL FORM
    "s+.", "\ufeb4", // ARABIC LETTER SEEN FINAL FORM
    "sn-", "\ufeb5", // ARABIC LETTER SHEEN ISOLATED FORM
    "sn,", "\ufeb6", // ARABIC LETTER SHEEN INITIAL FORM
    "sn;", "\ufeb7", // ARABIC LETTER SHEEN MEDIAL FORM
    "sn.", "\ufeb8", // ARABIC LETTER SHEEN FINAL FORM
    "c+-", "\ufeb9", // ARABIC LETTER SAD ISOLATED FORM
    "c+,", "\ufeba", // ARABIC LETTER SAD INITIAL FORM
    "c+;", "\ufebb", // ARABIC LETTER SAD MEDIAL FORM
    "c+.", "\ufebc", // ARABIC LETTER SAD FINAL FORM
    "dd-", "\ufebd", // ARABIC LETTER DAD ISOLATED FORM
    "dd,", "\ufebe", // ARABIC LETTER DAD INITIAL FORM
    "dd;", "\ufebf", // ARABIC LETTER DAD MEDIAL FORM
    "dd.", "\ufec0", // ARABIC LETTER DAD FINAL FORM
    "tj-", "\ufec1", // ARABIC LETTER TAH ISOLATED FORM
    "tj,", "\ufec2", // ARABIC LETTER TAH INITIAL FORM
    "tj;", "\ufec3", // ARABIC LETTER TAH MEDIAL FORM
    "tj.", "\ufec4", // ARABIC LETTER TAH FINAL FORM
    "zH-", "\ufec5", // ARABIC LETTER ZAH ISOLATED FORM
    "zH,", "\ufec6", // ARABIC LETTER ZAH INITIAL FORM
    "zH;", "\ufec7", // ARABIC LETTER ZAH MEDIAL FORM
    "zH.", "\ufec8", // ARABIC LETTER ZAH FINAL FORM
    "e+-", "\ufec9", // ARABIC LETTER AIN ISOLATED FORM
    "e+,", "\ufeca", // ARABIC LETTER AIN INITIAL FORM
    "e+;", "\ufecb", // ARABIC LETTER AIN MEDIAL FORM
    "e+.", "\ufecc", // ARABIC LETTER AIN FINAL FORM
    "i+-", "\ufecd", // ARABIC LETTER GHAIN ISOLATED FORM
    "i+,", "\ufece", // ARABIC LETTER GHAIN INITIAL FORM
    "i+;", "\ufecf", // ARABIC LETTER GHAIN MEDIAL FORM
    "i+.", "\ufed0", // ARABIC LETTER GHAIN FINAL FORM
    "f+-", "\ufed1", // ARABIC LETTER FEH ISOLATED FORM
    "f+,", "\ufed2", // ARABIC LETTER FEH INITIAL FORM
    "f+;", "\ufed3", // ARABIC LETTER FEH MEDIAL FORM
    "f+.", "\ufed4", // ARABIC LETTER FEH FINAL FORM
    "q+-", "\ufed5", // ARABIC LETTER QAF ISOLATED FORM
    "q+,", "\ufed6", // ARABIC LETTER QAF INITIAL FORM
    "q+;", "\ufed7", // ARABIC LETTER QAF MEDIAL FORM
    "q+.", "\ufed8", // ARABIC LETTER QAF FINAL FORM
    "k+-", "\ufed9", // ARABIC LETTER KAF ISOLATED FORM
    "k+,", "\ufeda", // ARABIC LETTER KAF INITIAL FORM
    "k+;", "\ufedb", // ARABIC LETTER KAF MEDIAL FORM
    "k+.", "\ufedc", // ARABIC LETTER KAF FINAL FORM
    "l+-", "\ufedd", // ARABIC LETTER LAM ISOLATED FORM
    "l+,", "\ufede", // ARABIC LETTER LAM INITIAL FORM
    "l+;", "\ufedf", // ARABIC LETTER LAM MEDIAL FORM
    "l+.", "\ufee0", // ARABIC LETTER LAM FINAL FORM
    "m+-", "\ufee1", // ARABIC LETTER MEEM ISOLATED FORM
    "m+,", "\ufee2", // ARABIC LETTER MEEM INITIAL FORM
    "m+;", "\ufee3", // ARABIC LETTER MEEM MEDIAL FORM
    "m+.", "\ufee4", // ARABIC LETTER MEEM FINAL FORM
    "n+-", "\ufee5", // ARABIC LETTER NOON ISOLATED FORM
    "n+,", "\ufee6", // ARABIC LETTER NOON INITIAL FORM
    "n+;", "\ufee7", // ARABIC LETTER NOON MEDIAL FORM
    "n+.", "\ufee8", // ARABIC LETTER NOON FINAL FORM
    "h+-", "\ufee9", // ARABIC LETTER HEH ISOLATED FORM
    "h+,", "\ufeea", // ARABIC LETTER HEH INITIAL FORM
    "h+;", "\ufeeb", // ARABIC LETTER HEH MEDIAL FORM
    "h+.", "\ufeec", // ARABIC LETTER HEH FINAL FORM
    "w+-", "\ufeed", // ARABIC LETTER WAW ISOLATED FORM
    "w+.", "\ufeee", // ARABIC LETTER WAW FINAL FORM
    "j+-", "\ufeef", // ARABIC LETTER ALEF MAKSURA ISOLATED FORM
    "j+.", "\ufef0", // ARABIC LETTER ALEF MAKSURA FINAL FORM
    "y+-", "\ufef1", // ARABIC LETTER YEH ISOLATED FORM
    "y+,", "\ufef2", // ARABIC LETTER YEH INITIAL FORM
    "y+;", "\ufef3", // ARABIC LETTER YEH MEDIAL FORM
    "y+.", "\ufef4", // ARABIC LETTER YEH FINAL FORM
    "lM-", "\ufef5", // ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM
    "lM.", "\ufef6", // ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM
    "lH-", "\ufef7", // ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM
    "lH.", "\ufef8", // ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM
    "lh-", "\ufef9", // ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM
    "lh.", "\ufefa", // ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM
    "la-", "\ufefb", // ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM
    "la.", "\ufefc", // ARABIC LIGATURE LAM WITH ALEF FINAL FORM
    /*
      "/c", "\ue001", // JOIN THIS LINE WITH NEXT LINE (Mnemonic)
      "UA", "\ue002", // Unit space A (ISO-IR-8-1 064)
      "UB", "\ue003", // Unit space B (ISO-IR-8-1 096)
      "\"3", "\ue004", // NON-SPACING UMLAUT (ISO-IR-38 201) (character part)
      "\"1", "\ue005", // NON-SPACING DIAERESIS WITH ACCENT (ISO-IR-70 192) (character part)
      "\"!", "\ue006", // NON-SPACING GRAVE ACCENT (ISO-IR-103 193) (character part)
      "\"'", "\ue007", // NON-SPACING ACUTE ACCENT (ISO-IR-103 194) (character part)
      "\">", "\ue008", // NON-SPACING CIRCUMFLEX ACCENT (ISO-IR-103 195) (character part)
      "\"?", "\ue009", // NON-SPACING TILDE (ISO-IR-103 196) (character part)
      "\"-", "\ue00a", // NON-SPACING MACRON (ISO-IR-103 197) (character part)
      "\"(", "\ue00b", // NON-SPACING BREVE (ISO-IR-103 198) (character part)
      "\".", "\ue00c", // NON-SPACING DOT ABOVE (ISO-IR-103 199) (character part)
      "\":", "\ue00d", // NON-SPACING DIAERESIS (ISO-IR-103 200) (character part)
      "\"0", "\ue00e", // NON-SPACING RING ABOVE (ISO-IR-103 202) (character part)
      "\"\"", "\ue00f", // NON-SPACING DOUBLE ACCUTE (ISO-IR-103 204) (character part)
      "\"<", "\ue010", // NON-SPACING CARON (ISO-IR-103 206) (character part)
      "\",", "\ue011", // NON-SPACING CEDILLA (ISO-IR-103 203) (character part)
      "\";", "\ue012", // NON-SPACING OGONEK (ISO-IR-103 206) (character part)
      "\"_", "\ue013", // NON-SPACING LOW LINE (ISO-IR-103 204) (character part)
      "\"=", "\ue014", // NON-SPACING DOUBLE LOW LINE (ISO-IR-38 217) (character part)
      "\"/", "\ue015", // NON-SPACING LONG SOLIDUS (ISO-IR-128 201) (character part)
      "\"i", "\ue016", // GREEK NON-SPACING IOTA BELOW (ISO-IR-55 39) (character part)
      "\"d", "\ue017", // GREEK NON-SPACING DASIA PNEUMATA (ISO-IR-55 38) (character part)
      "\"p", "\ue018", // GREEK NON-SPACING PSILI PNEUMATA (ISO-IR-55 37) (character part)
      ";;", "\ue019", // GREEK DASIA PNEUMATA (ISO-IR-18 92)
      ",,", "\ue01a", // GREEK PSILI PNEUMATA (ISO-IR-18 124)
      "b3", "\ue01b", // GREEK SMALL LETTER MIDDLE BETA (ISO-IR-18 99)
      "Ci", "\ue01c", // CIRCLE (ISO-IR-83 0294)
      "f(", "\ue01d", // FUNCTION SIGN (ISO-IR-143 221)
      "ed", "\ue01e", // LATIN SMALL LETTER EZH (ISO-IR-158 142)
      "am", "\ue01f", // ANTE MERIDIAM SIGN (ISO-IR-149 0267)
      "pm", "\ue020", // POST MERIDIAM SIGN (ISO-IR-149 0268)
      "Tel", "\ue021", // TEL COMPATIBILITY SIGN (ISO-IR-149 0269)
      "a+:", "\ue022", // ARABIC LETTER ALEF FINAL FORM COMPATIBILITY (IBM868 144)
      "Fl", "\ue023", // DUTCH GUILDER SIGN (IBM437 159)
      "GF", "\ue024", // GAMMA FUNCTION SIGN (ISO-10646-1DIS 032/032/037/122)
      ">V", "\ue025", // RIGHTWARDS VECTOR ABOVE (ISO-10646-1DIS 032/032/038/046)
      "!*", "\ue026", // GREEK VARIA (ISO-10646-1DIS 032/032/042/164)
      "?*", "\ue027", // GREEK PERISPOMENI (ISO-10646-1DIS 032/032/042/165)
      "J<", "\ue028", // LATIN CAPITAL LETTER J WITH CARON (lowercase: 000/000/001/240)
    */
  };
  public static final String[] kUnicodeBlocks = {
    "\u0000", "C0 Controls and Basic Latin (Basic Latin)", // 007F
    "\u0080", "C1 Controls and Latin-1 Supplement (Latin-1 Supplement)", // 00FF
    "\u0100", "Latin Extended-A", // 017F
    "\u0180", "Latin Extended-B", // 024F
    "\u0250", "IPA Extensions", // 02AF
    "\u02B0", "Spacing Modifier Letters", // 02FF
    "\u0300", "Combining Diacritical Marks", // 036F
    "\u0370", "Greek and Coptic", // 03FF
    "\u0400", "Cyrillic", // 04FF
    "\u0530", "Armenian", // 058F
    "\u0590", "Hebrew", // 05FF
    "\u0600", "Arabic", // 06FF
    "\u0700", "Syriac", // 074F
    "\u0780", "Thaana", // 07BF
    "\u0900", "Devanagari", // 097F
    "\u0980", "Bengali", // 09FF
    "\u0A00", "Gurmukhi", // 0A7F
    "\u0A80", "Gujarati", // 0AFF
    "\u0B00", "Oriya", // 0B7F
    "\u0B80", "Tamil", // 0BFF
    "\u0C00", "Telugu", // 0C7F
    "\u0C80", "Kannada", // 0CFF
    "\u0D00", "Malayalam", // 0D7F
    "\u0D80", "Sinhala", // 0DFF
    "\u0E00", "Thai", // 0E7F
    "\u0E80", "Lao", // 0EFF
    "\u0F00", "Tibetan", // 0FFF
    "\u1000", "Myanmar", // 109F
    "\u10A0", "Georgian", // 10FF
    "\u1100", "Hangul Jamo", // 11FF
    "\u1200", "Ethiopic", // 137F
    "\u13A0", "Cherokee", // 13FF
    "\u1400", "Unified Canadian Aboriginal Syllabics", // 167F
    "\u1680", "Ogham", // 169F
    "\u16A0", "Runic", // 16FF
    "\u1780", "Khmer", // 17FF
    "\u1800", "Mongolian", // 18AF
    "\u1E00", "Latin Extended Additional", // 1EFF
    "\u1F00", "Greek Extended", // 1FFF
    "\u2000", "General Punctuation", // 206F
    "\u2070", "Superscripts and Subscripts", // 209F
    "\u20A0", "Currency Symbols", // 20CF
    "\u20D0", "Combining Diacritical Marks for Symbols", // 20FF
    "\u2100", "Letterlike Symbols", // 214F
    "\u2150", "Number Forms", // 218F
    "\u2190", "Arrows", // 21FF
    "\u2200", "Mathematical Operators", // 22FF
    "\u2300", "Miscellaneous Technical", // 23FF
    "\u2400", "Control Pictures", // 243F
    "\u2440", "Optical Character Recognition", // 245F
    "\u2460", "Enclosed Alphanumerics", // 24FF
    "\u2500", "Box Drawing", // 257F
    "\u2580", "Block Elements", // 259F
    "\u25A0", "Geometric Shapes", // 25FF
    "\u2600", "Miscellaneous Symbols", // 26FF
    "\u2700", "Dingbats", // 27BF
    "\u2800", "Braille Patterns", // 28FF
    "\u2E80", "CJK Radicals Supplement", // 2EFF
    "\u2F00", "Kangxi Radicals", // 2FDF
    "\u2FF0", "Ideographic Description Characters", // 2FFF
    "\u3000", "CJK Symbols and Punctuation", // 303F
    "\u3040", "Hiragana", // 309F
    "\u30A0", "Katakana", // 30FF
    "\u3100", "Bopomofo", // 312F
    "\u3130", "Hangul Compatibility Jamo", // 318F
    "\u3190", "Kanbun", // 319F
    "\u31A0", "Bopomofo Extended", // 31BF
    "\u3200", "Enclosed CJK Letters and Months", // 32FF
    "\u3300", "CJK Compatibility", // 33FF
    "\u3400", "CJK Unified Ideographs Extension A", // 4DB5
    "\u4E00", "CJK Unified Ideographs", // 9FA5
    "\uA000", "Yi Syllables", // A48F
    "\uA490", "Yi Radicals", // A4CF
    "\uAC00", "Hangul Syllables", // D7A3
    "\uD800", "High Surrogates", // DB7F
    "\uDB80", "High Private Use Surrogates", // DBFF
    "\uDC00", "Low Surrogates", // DFFF
    "\uE000", "Private Use Area", // F8FF
    "\uF900", "CJK Compatibility Ideographs", // FAFF
    "\uFB00", "Alphabetic Presentation Forms", // FB4F
    "\uFB50", "Arabic Presentation Forms-A", // FDFF
    "\uFE20", "Combining Half Marks", // FE2F
    "\uFE30", "CJK Compatibility Forms", // FE4F
    "\uFE50", "Small Form Variants", // FE6F
    "\uFE70", "Arabic Presentation Forms-B", // FEFF
    "\uFF00", "Halfwidth and Fullwidth Forms", // FFEF
    "\uFFF0", "Specials", // FFFF
  };

  /* Supported by many fonts ... */
  public static char[] kBasic80 = {
    // ' ' 0x21
    '!', 0xa1, 0, 0xa5,
    ',', // extra
    0xa7, 0xa9, 0xab, 0xac, 0xae, 0xaf, 0xb0, 0xb1, 0xb4, 0, 0xb8, 0xba, 0, 0xbf,
    '"', 0xd7, 0xd8, 0xf7, 0xf8, 0x127, 0x148, 0x14a, 0x14b, 0x152, 0x153, 0x192, 0x1a0, 0x1af, 0x1e6, 0x1e7,
    '#', 0x250, 0x251, 0x252, 0x254, 0x259, 0x25b, 0x25c, 0x263, 0x264, 0x265, 0x26f, 0x279, 0x27e, 0x281,
    0x283, 0x28c, 0x28d, 0x292, 0x294, 0x295, 0x296, 0x298,
    '$', 0x2b0, 0, 0x2c1, 0x2c6, 0x2c7, 0x2c9,
    '%', 0x2d8, 0, 0x2dd, 0x305, 0x1e9e, 0x2013, 0x2014, 0x2015, 0x2018, 0, 0x2020,
    '&', 0x2026, 0x2030, 0x20ac, 0x2105, 0x2116, 0x2122, 0x2126, 0x212e, 0x2190, 0, 0x2195,
    0x2200, 0x2203, 0x220a, 0x220f, 0x2211, 0x221a, 0x221e, 0x2227, 0, 0x222a,
    '\'', 0x222b,
    '(', 0x2248, 0x2260, 0x2261, 0x2264, 0x2265, 0x2300, 0x2302,
    ')', 0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524, 0x252c, 0x2534, 0x253c,
    '*', 0x2550, 0, 0x255d,
    '+', 0x255e, 0, 0x256c,
    // ',' 0xa7,
    '-', 0x256d, 0, 0x2570,
    // '.' ('Z'+1)
    '/', 0x2580, 0x2584, 0x2588, 0x258c, 0x2590, 0x2591, 0x2592, 0x2593, 0x2c6f, 0x2c70,
  };

  public static char[] kIpaShapesSorted = {
    'A', 'a', 'B', 'b', 'C', 'c', 'D', 'd', 'E',
    'e', 'F', 'f', 'G', 'g', 'H', 'h', 'I', 'i', 'J', 'j', 'K', 'k', 'L', 'l',
    'M', 'm', 'N', 'n', 'O', 'o', 'P', 'p', 'Q', 'q', 'R', 'r',
    'S', 's', 'T', 't', 'U', 'u', 'V', 'v', 'W', 'w', 'X', 'x', 'Y', 'y', 'Z', 'z',
    '\u00a2', '\u00c6', '\u00d0', '\u00d8', '\u00e6', '\u00f0', '\u00f8',
    '\u0124', '\u0126', '\u0127', '\u013f', '\u0141', '\u014a', '\u014b', '\u0152', '\u0152', '\u0153', '\u0174', '\u017f', '\u0186',
    '\u018e', '\u018f', '\u0190', '\u0194', '\u0197', '\u019c', '\u019f', '\u01a6', '\u01a9', '\u01b1', '\u01b2', '\u01b7',
    '\u0241', '\u0244', '\u0245', '\u0250', '\u0251', '\u0252', '\u0254', '\u0258', '\u0259',
    '\u025b', '\u025c', '\u025e', '\u0262', '\u0263', '\u0264', '\u0266', '\u0268', '\u026a', '\u026c', '\u026e', '\u026f',
    '\u0270', '\u0271', '\u0274', '\u0275', '\u0276', '\u0278', '\u0279', '\u027e', '\u0280', '\u0281', '\u0283', '\u0289', '\u028a',
    '\u028b', '\u028c', '\u028d', '\u028e', '\u028f', '\u0292', '\u0294', '\u0295', '\u0299', '\u029f', '\u0392', '\u0398', '\u039b',
    '\u03a6', '\u03a7', '\u03b2', '\u03b8', '\u03c7', '\u03da', '\u0411', '\u0417', '\u0418', '\u0423', '\u0429', '\u042e', '\u042f',
    '\u0474', '\u0475', '\u04ae', '\u04de', '\u2c6d', '\u2C6E', '\u2c6f', '\u2c70', '\uA7Ac', '\ua7ad', '\ua7ae',
  };

  public static char[] kIpaShapes = {

    1, 'P', 'B', 'T', 'D', 'K', 'G', 'Q', '\uA7Ac', /* + */'C', '\u0241',
    2, 'p', 'b', 't', 'd', 'k', 'g', 'q', '\u0262', /* + */'c', '\u0294',

    3, 'M', '\u2C6E', 'N', '\u014a', '\u0418',
    4, 'm', '\u0271', 'n', '\u014b', '\u0274',

    5, '\u0411', 'R', '\u01a6',
    6, '\u0299', 'r', '\u0280',

    7, '\u0474', '\u017f',
    8, '\u0475', '\u027e',

    9, '\u03a6', '\u0392', 'F', 'V', 'S', 'Z', 'X', '\u0194', '\u03a7', '\u042f', '\u0126',
    '\u03da', 'H', '\u0124', /* + */'\u0398', '\u00d0', '\u01a9', '\u01b7', '\u0174',
    10, '\u0278', '\u03b2', 'f', 'v', 's', 'z', 'x', '\u0263', '\u03c7', '\u0281', '\u0127',
    '\u0295', 'h', '\u0266', /* + */'\u03b8', '\u00f0', '\u0283', '\u0292', '\u028d',

    11, '\ua7ad', '\u0141',
    12, '\u026c', '\u026e',

    13, '\u01b2', '\u00a2', '\u0429', /* + */'J', 'W',
    14, '\u028b', '\u0279', '\u0270', /* + */'j', 'w',

    15, 'L', '\u013f', /* + */'\u039b',
    16, 'l', '\u029f', /* + */'\u028e',

    17, 'I', 'Y', '\u0197', '\u0244', '\u019c', 'U',
    18, 'i', 'y', '\u0268', '\u0289', '\u026f', 'u',

    19, '\ua7ae', '\u0423', '\u01b1',
    20, '\u026a', '\u028f', '\u028a',

    21, 'E', '\u00d8', '\u018e', '\u019f', '\u04ae', 'O',
    22, 'e', '\u00f8', '\u0258', '\u0275', '\u0264', 'o',

    23, '\u018f',
    24, '\u0259',

    25, '\u0190', '\u0152', '\u0417', '\u04de', '\u0245', '\u0186',
    26, '\u025b', '\u0153', '\u025c', '\u025e', '\u028c', '\u0254',

    27, '\u00c6', '\u2c6f',
    28, '\u00e6', '\u0250',

    29, 'A', '\u042e', '\u2c6d', '\u2c70',
    30, 'a', '\u0276', '\u0251', '\u0252',

  };
}
