import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_errors.dart';
import 'package:natinfo_flutter/shared/data_sources/source_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

void main() {
  group('SourceLoader', () {
    test('falls back to secondary source on failure', () async {
      final loader = _buildLoader(
        specs: [
          _spec('api-primary', type: SourceType.api, priority: 1),
          _spec(
            'asset-fallback',
            type: SourceType.asset,
            priority: 2,
            requiresNetwork: false,
          ),
        ],
        adapters: [
          _FailingAdapter(SourceType.api),
          _SuccessfulAdapter(SourceType.asset),
        ],
      );

      final result = await loader.load('dataset');
      expect(result.spec.id, 'asset-fallback');
    });

    test('throws descriptive error when required source unavailable', () async {
      final loader = _buildLoader(
        specs: [
          _spec(
            'api-required',
            type: SourceType.api,
            requiredAtBuild: true,
            requiresNetwork: true,
          ),
        ],
        adapters: [_SuccessfulAdapter(SourceType.api)],
      );

      expect(
        () => loader.load('dataset', allowNetwork: false),
        throwsA(isA<SourceUnavailableException>()),
      );
    });

    test('logs checksum mismatch but still returns result', () async {
      final logs = <String>[];
      final loader = _buildLoader(
        specs: [
          _spec(
            'asset',
            type: SourceType.asset,
            requiresNetwork: false,
            checksum: 'deadbeef',
            checksumAlgo: 'md5',
          ),
        ],
        adapters: [
          _SuccessfulAdapter(SourceType.asset, bytes: [1, 2, 3]),
        ],
        logs: logs,
      );

      final result = await loader.load('dataset', allowNetwork: false);
      expect(result.spec.id, 'asset');
      expect(
        logs.where((entry) => entry.contains('Checksum mismatch')),
        isNotEmpty,
      );
    });

    test('logs schema mismatch when expected schema differs', () async {
      final logs = <String>[];
      final loader = _buildLoader(
        specs: [_spec('api', type: SourceType.api, schemaVersion: 1)],
        adapters: [_SuccessfulAdapter(SourceType.api)],
        logs: logs,
      );

      await loader.load('dataset', expectedSchemaVersion: 2);
      expect(
        logs.where((entry) => entry.contains('Schema version mismatch')),
        isNotEmpty,
      );
    });
  });
}

SourceLoader _buildLoader({
  required List<SourceSpec> specs,
  required List<SourceAdapter> adapters,
  List<String>? logs,
}) {
  final registry = SourceRegistry(specs);
  final resolver = SourceResolver(registry);
  return SourceLoader(
    resolver: resolver,
    adapters: adapters,
    logger:
        logs == null
            ? null
            : SourceLogger(tag: 'TestSources', handler: logs.add),
  );
}

SourceSpec _spec(
  String id, {
  required SourceType type,
  int priority = 1,
  bool requiredAtBuild = false,
  bool requiresNetwork = true,
  int schemaVersion = 1,
  String? checksum,
  String? checksumAlgo,
}) {
  return SourceSpec(
    id: id,
    dataset: 'dataset',
    name: 'Test dataset',
    type: type,
    uri: Uri.parse('https://example.com/$id'),
    scope: 'test',
    priority: priority,
    requiredAtBuild: requiredAtBuild,
    requiresNetwork: requiresNetwork,
    schemaVersion: schemaVersion,
    checksum: checksum,
    checksumAlgo: checksumAlgo,
  );
}

class _SuccessfulAdapter extends SourceAdapter {
  _SuccessfulAdapter(this._type, {List<int>? bytes})
    : _bytes = bytes ?? const [0];

  final SourceType _type;
  final List<int> _bytes;

  @override
  SourceType get type => _type;

  @override
  Future<SourceLoadResult> load(SourceSpec spec) async {
    final bytes = List<int>.from(_bytes);
    return SourceLoadResult(
      spec: spec,
      bytes: bytes,
      integrity: verifyIntegrity(spec, bytes),
    );
  }
}

class _FailingAdapter extends SourceAdapter {
  _FailingAdapter(this._type);

  final SourceType _type;

  @override
  SourceType get type => _type;

  @override
  Future<SourceLoadResult> load(SourceSpec spec) async {
    throw Exception('failed');
  }
}
