import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';
import 'package:provider/provider.dart';

void main() {
  late AuthProvider signedOutAuth;

  setUp(() async {
    signedOutAuth = await _buildAuthProvider();
  });

  testWidgets('records history via repository when tapping an item', (
    tester,
  ) async {
    final repository = _RecordingNatinfRepository();
    var tapHandled = false;
    const natinf = Natinf(
      numeroNatinf: '12345',
      qualificationInfraction: 'Test',
      dacgSource: 0,
    );

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: signedOutAuth),
          Provider<NatinfRepository>.value(value: repository),
        ],
        child: MaterialApp(
          home: Scaffold(
            body: NatinfList(
              natinfList: const [natinf],
              onNatinfTap: (_, __) async {
                tapHandled = true;
              },
            ),
          ),
        ),
      ),
    );

    await tester.tap(find.byType(InkWell));
    await tester.pumpAndSettle();

    expect(repository.recordedNumero, '12345');
    expect(tapHandled, isTrue);
  });

  testWidgets(
    'keeps animation progress when rebuilding with the same list content',
    (tester) async {
      final repository = _RecordingNatinfRepository();

      await tester.pumpWidget(
        _buildNatinfListHost(
          natinfs: [_testNatinf('100'), _testNatinf('200')],
          repository: repository,
          authProvider: signedOutAuth,
        ),
      );

      await tester.pump();
      await tester.pump(const Duration(milliseconds: 80));
      final initialProgress = _controllerProgress(tester);
      expect(initialProgress, greaterThan(0));
      expect(initialProgress, lessThan(0.5));

      await tester.pumpWidget(
        _buildNatinfListHost(
          natinfs: [_testNatinf('100'), _testNatinf('200')],
          repository: repository,
          authProvider: signedOutAuth,
        ),
      );
      await tester.pump(const Duration(milliseconds: 50));

      final progressAfterSameContent = _controllerProgress(tester);
      expect(progressAfterSameContent, greaterThan(initialProgress));

      await tester.pumpWidget(
        _buildNatinfListHost(
          natinfs: [_testNatinf('100'), _testNatinf('200'), _testNatinf('300')],
          repository: repository,
          authProvider: signedOutAuth,
        ),
      );
      await tester.pump();

      final progressAfterListChange = _controllerProgress(tester);
      expect(progressAfterListChange, lessThan(0.1));
    },
  );
}

class _RecordingNatinfRepository extends NatinfRepository {
  String? recordedNumero;

  @override
  Future<void> recordHistoryEntry(String numero) async {
    recordedNumero = numero;
  }
}

Future<AuthProvider> _buildAuthProvider() async {
  final storage = InMemoryAuthStorage();
  final repository = AuthRepository(
    baseUri: Uri.parse('https://natinfo.app/api'),
    storage: storage,
    httpClient: MockClient((request) async => http.Response('{}', 200)),
  );
  final provider = AuthProvider(repository);
  await provider.bootstrap();
  return provider;
}

Widget _buildNatinfListHost({
  required List<Natinf> natinfs,
  required AuthProvider authProvider,
  NatinfRepository? repository,
}) {
  return MultiProvider(
    providers: [
      ChangeNotifierProvider(create: (_) => ThemeProvider()),
      ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
      Provider<NatinfRepository>.value(
        value: repository ?? _RecordingNatinfRepository(),
      ),
    ],
    child: MaterialApp(home: Scaffold(body: NatinfList(natinfList: natinfs))),
  );
}

Natinf _testNatinf(String numero) => Natinf(
  numeroNatinf: numero,
  qualificationInfraction: 'Test $numero',
  dacgSource: 0,
);

double _controllerProgress(WidgetTester tester) {
  final state = tester.state(find.byType(NatinfList));
  // ignore: avoid_dynamic_calls
  return (state as dynamic).animationProgress as double;
}
