import 'dart:convert';

import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/data/favorites_sync_service.dart';
import 'package:sembast/sembast_memory.dart';

class _StubAuthRepository extends AuthRepository {
  _StubAuthRepository({required http.Client client})
    : _client = client,
      super(
        baseUri: Uri.parse('https://example.com'),
        storage: InMemoryAuthStorage(),
        httpClient: client,
      );

  final http.Client _client;
  bool ensureCalled = false;

  @override
  Future<void> ensureValidAccessToken() async {
    ensureCalled = true;
  }

  @override
  http.Client get authorizedClient => _client;
}

void main() {
  late DatabaseHelper db;

  setUp(() async {
    db = DatabaseHelper();
    final path = 'memory_${DateTime.now().microsecondsSinceEpoch}.db';
    await db.initDbForTesting(databaseFactoryMemory, path);
  });

  test('pushLocal builds snapshot and persists server state', () async {
    final folderId = await db.createFavouriteFolder(name: 'Root');
    await db.addFavourites('123');
    await db.addNatinfToFolder(folderId, '456');

    late Map<String, dynamic> postedBody;
    final client = MockClient((request) async {
      postedBody = jsonDecode(request.body) as Map<String, dynamic>;
      final incomingSnapshot = postedBody['snapshot'] as Map<String, dynamic>;
      final responseSnapshot = {
        'snapshot': {
          ...incomingSnapshot,
          'version': 'server-version',
          'checksum': 'abc',
        },
      };
      return http.Response(
        jsonEncode(responseSnapshot),
        200,
        headers: {'Content-Type': 'application/json'},
      );
    });
    final auth = _StubAuthRepository(client: client);
    final service = FavoritesSyncService(
      baseUri: Uri.parse('https://api.example.com'),
      authRepository: auth,
      dbHelper: db,
      client: client,
    );

    await service.pushLocal(forceReplace: true);

    expect(auth.ensureCalled, isTrue);
    expect(postedBody['forceReplace'], isTrue);
    final snapshot = postedBody['snapshot'] as Map<String, dynamic>;
    final items = snapshot['items'] as List<dynamic>;
    final folders = snapshot['folders'] as List<dynamic>;
    expect(items, isNotEmpty);
    expect(folders, isNotEmpty);

    final folderUuid = folders.first['id'];
    final folderItem = items.firstWhere(
      (item) => item['folderId'] == folderUuid,
    );
    expect(folderItem['numero'], '456');

    final state = await db.getFavoritesState();
    expect(state['version'], 'server-version');
    expect(state['checksum'], 'abc');
    expect(state['lastSyncAt'], isNotNull);
    expect(state['deviceId'], isNotEmpty);

    final rootFavs = await db.getFavourites();
    expect(rootFavs.map((m) => m['numero_natinf']), contains('123'));
    final folderItems = await db.getFavouriteFolderItemsRaw(
      includeDeleted: false,
    );
    expect(folderItems.first['folder_uuid'], folderUuid);
  });

  test('pullFromServer applies snapshot and saves metadata', () async {
    final client = MockClient((request) async {
      final responseSnapshot = {
        'snapshot': {
          'version': 'server-version',
          'checksum': 'xyz',
          'folders': [],
          'items': [],
        },
      };
      return http.Response(
        jsonEncode(responseSnapshot),
        200,
        headers: {'Content-Type': 'application/json'},
      );
    });
    final auth = _StubAuthRepository(client: client);
    final service = FavoritesSyncService(
      baseUri: Uri.parse('https://api.example.com'),
      authRepository: auth,
      dbHelper: db,
      client: client,
    );

    await service.pullFromServer();

    expect(auth.ensureCalled, isTrue);
    final state = await db.getFavoritesState();
    expect(state['version'], 'server-version');
    expect(state['checksum'], 'xyz');
    expect(state['deviceId'], isNotEmpty);
  });

  test('applySnapshotLocally resets and hydrates favorites', () async {
    final initialFolder = await db.createFavouriteFolder(name: 'Local');
    await db.addNatinfToFolder(initialFolder, 'local-num');
    await db.addFavourites('root-local');

    final service = FavoritesSyncService(
      baseUri: Uri.parse('https://api.example.com'),
      authRepository: _StubAuthRepository(
        client: MockClient((_) async {
          return http.Response('{}', 200);
        }),
      ),
      dbHelper: db,
    );

    final snapshot = {
      'version': 'server-v1',
      'checksum': 'ccc',
      'folders': [
        {
          'id': 'folder-1',
          'parentId': null,
          'name': 'Server',
          'position': 0,
          'updatedAt': '2024-01-01T00:00:00Z',
          'deleted': false,
        },
      ],
      'items': [
        {
          'id': 'fav-1',
          'folderId': null,
          'numero': '321',
          'position': 0,
          'updatedAt': '2024-01-01T00:00:00Z',
          'deleted': false,
        },
        {
          'id': 'fav-2',
          'folderId': 'folder-1',
          'numero': '654',
          'position': 0,
          'updatedAt': '2024-01-02T00:00:00Z',
          'deleted': false,
        },
      ],
    };

    await service.applySnapshotLocally(snapshot);

    final folders = await db.getAllFavouriteFolders();
    expect(folders, hasLength(1));
    expect(folders.first.uuid, 'folder-1');

    final rootFavs = await db.getFavourites();
    expect(rootFavs.map((m) => m['numero_natinf']), contains('321'));
    expect(
      rootFavs.map((m) => m['numero_natinf']),
      isNot(contains('root-local')),
    );

    final folderItems = await db.getFavouriteFolderItemsRaw(
      includeDeleted: false,
    );
    expect(folderItems, hasLength(1));
    expect(folderItems.first['numero_natinf'], '654');
    expect(folderItems.first['folder_uuid'], 'folder-1');
  });
}
